/*
 * NXP i.MX8M Plus SNVS-RTC driver
 *
 * Copyright (c) 2021 Atmark Techno, Inc
 * Written by Makoto Sato <makoto.sato@atmark-techno.com>
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <command.h>
#include <linux/compat.h>
#include <rtc.h>

#include <asm/io.h>
#include <asm/arch/imx-regs.h>

#define SNVS_LPCR			(SNVS_HP_BASE_ADDR + 0x38)
#define SNVS_LPSRTCMR		(SNVS_HP_BASE_ADDR + 0x50)
#define SNVS_LPSRTCLR		(SNVS_HP_BASE_ADDR + 0x54)

#define SNVS_LPCR_SRTC_ENV	(1 << 0)

#define CNTR_TO_SECS_SH		15

static int snvs_rtc_enable(void)
{
	int timeout = 1000;
	u32 lpcr;

	lpcr = readl(SNVS_LPCR);
	if (lpcr & SNVS_LPCR_SRTC_ENV)
		return 0;

	writel(lpcr | SNVS_LPCR_SRTC_ENV, SNVS_LPCR);
	while (--timeout) {
		lpcr = readl(SNVS_LPCR);
		if (lpcr & SNVS_LPCR_SRTC_ENV)
			break;
	}

	if (!timeout)
		return -1;

	return 0;
}

static u64 rtc_read_lpsrt(void)
{
	u32 msb, lsb;

	msb = readl(SNVS_LPSRTCMR);
	lsb = readl(SNVS_LPSRTCLR);
	return (u64) msb << 32 | lsb;
}

static u32 rtc_read_lp_counter(void)
{
	u64 read1, read2;
	unsigned int timeout = 100;

	read1 = rtc_read_lpsrt();
	do {
		read2 = read1;
		read1 = rtc_read_lpsrt();
	} while (read1 != read2 && --timeout);
	if (!timeout)
		return -1;

	return (u32) (read1 >> CNTR_TO_SECS_SH);
}

int rtc_get(struct rtc_time *tm)
{
	int ret;
	u32 time;
	char *toff;

	ret = snvs_rtc_enable();
	if (ret < 0)
		return ret;

	time = rtc_read_lp_counter();
	if (time < 0)
		return time;

	toff = env_get("timeoffset");
	if (toff)
		time += simple_strtol(toff, NULL, 10);

	rtc_to_tm(time, tm);

	return 0;
}
