/*
 * Logging support
 *
 * Copyright (c) 2021 Atmark Techno, Inc
 * Written by Makoto Sato <makoto.sato@atmark-techno.com>
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <log.h>
#include <fs.h>
#include <mmc.h>
#include <stdlib.h>
#include <asm/io.h>
#include <rtc.h>
#include <dm/uclass.h>

DECLARE_GLOBAL_DATA_PTR;

#define NEW_LINE_SIZE		(CONFIG_SYS_CBSIZE + 64)

#ifdef CONFIG_TARGET_ARMADILLO_X2
#define DEVICE_NUM		2
#define DEVICE_PARTITION	"2.5"
#elif CONFIG_TARGET_ARMADILLO_640
#define DEVICE_NUM		0
#define DEVICE_PARTITION	"0.5"
#endif
#define INTERFACE_NAME		"mmc"
#define FILE_NAME		"/atlog"
#define OLD_FILE_NAME		FILE_NAME ".1"
#define BOOT_PART_NUM		0
#define GP_PART_NUM		5
#define FILE_SIZE_LIMIT		(3 * 1024 * 1024) /* 3.0 MB */

static const char mon_name[12][4] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
static unsigned long int recursive_lock;

static inline int log_file_fs_set_blk_dev(void)
{
	int ret;

	ret = fs_set_blk_dev(INTERFACE_NAME, DEVICE_PARTITION, FS_TYPE_FAT);
	if (ret) {
		printf("Could not set blkdev %s %s (ext)\n",
			INTERFACE_NAME, DEVICE_PARTITION);
	}
	return ret;
}

static int log_file_fs_exists(const char *filename)
{
	int ret;

	ret = log_file_fs_set_blk_dev();
	if (ret)
		return ret;

	return fs_exists(filename);
}

static int log_file_fs_size(const char *filename, loff_t *size)
{
	int ret;

	ret = log_file_fs_set_blk_dev();
	if (ret)
		return ret;

	return fs_size(filename, size);
}

static int log_file_fs_read(const char *filename, ulong addr, loff_t offset,
			    loff_t len, loff_t *actread)
{
	int ret;

	ret = log_file_fs_set_blk_dev();
	if (ret)
		return ret;

	return fs_read(filename, addr, offset, len, actread);
}

static int log_file_fs_write(const char *filename, ulong addr, loff_t offset,
			     loff_t len, loff_t *actwrite)
{
	int ret;

	ret = log_file_fs_set_blk_dev();
	if (ret)
		return ret;

	return fs_write(filename, addr, offset, len, actwrite);
}

static int log_file_rotate(void)
{
	char *buf = NULL;
	loff_t len;
	loff_t size;
	int ret;

	ret = log_file_fs_exists(FILE_NAME);
	if (!ret)
		return 0;

	ret = log_file_fs_size(FILE_NAME, &size);
	if (ret < 0)
		goto error;

	if (size < FILE_SIZE_LIMIT)
		return 0;

	buf = (char *) malloc(size + 1);
	if (buf == NULL)
		goto error;

	ret = log_file_fs_read(FILE_NAME, (ulong) buf, 0, size, &len);
	if (ret < 0)
		goto error;

	ret = log_file_fs_write(OLD_FILE_NAME, (ulong) buf, 0, size, &len);
	if (ret < 0)
		goto error;

	free(buf);

	return 1;

error:
	if (buf)
		free(buf);
	return -1;
}

static void log_file_rtc_get(struct rtc_time *tm) {
	int ret;
	struct udevice *dev;

	ret = uclass_get_device(UCLASS_RTC, 0, &dev);
	if (ret) {
                printf("could not get rtc device: invalid date will be used in logs\n");
		goto error;
        }

	ret = dm_rtc_get(dev, tm);
	if (ret < 0) {
		printf("Could not get time: invalid date will be used in logs\n");
		goto error;
	}
	return;

error:
	/* initialize date to 01-01-1970 00:00:00 */
	rtc_to_tm(0, tm);
}

static int log_file_emit(struct log_device *ldev, struct log_rec *rec)
{
	loff_t len;
	loff_t size;
	loff_t offset;
	int ret;
	char new_line[NEW_LINE_SIZE];
	size_t new_line_len;
	struct mmc *mmc;
	struct rtc_time tm;

	if (test_and_set_bit(0, &recursive_lock))
		return 0;

	mmc = find_mmc_device(DEVICE_NUM);
	if (!mmc) {
		// log before mmc init
		goto error_silent;
	}

	log_file_rtc_get(&tm);
	new_line_len = snprintf(new_line, NEW_LINE_SIZE,
				"%s %2d %02d:%02d:%02d armadillo %s uboot: %s",
				mon_name[tm.tm_mon - 1], tm.tm_mday,
				tm.tm_hour, tm.tm_min, tm.tm_sec,
				log_get_level_name(rec->level), rec->msg);

	ret = mmc_init(mmc);
	if (ret < 0) {
		printf("Could not init mmc: %d\n", ret);
		goto error;
	}

	ret = mmc_switch_part(mmc, GP_PART_NUM);
	if (ret < 0) {
		printf("Could not switch mmc %d to part %d\n",
		       DEVICE_NUM, GP_PART_NUM);
		goto error;
	}

	ret = log_file_rotate();
	if (ret < 0)
		goto error;

	size = 0;
	/* if file was rotated keep size at 0 */
	ret = !ret && log_file_fs_exists(FILE_NAME);
	if (ret) {
		ret = log_file_fs_size(FILE_NAME, &size);
		if (ret < 0)
			goto error;
	}

	offset = size;
	ret = log_file_fs_write(FILE_NAME, (ulong) new_line, offset, new_line_len, &len);
	if (ret < 0)
		goto error;

	mmc_switch_part(mmc, BOOT_PART_NUM);

	test_and_clear_bit(0, &recursive_lock);

	return 0;

error:
	if (mmc)
		mmc_switch_part(mmc, BOOT_PART_NUM);
	printf("** Unable to write file %s **\n", FILE_NAME);
error_silent:
	test_and_clear_bit(0, &recursive_lock);

	return 1;
}

LOG_DRIVER(file) = {
	.name	= "file",
	.emit	= log_file_emit,
};
