// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright 2019 NXP
 */

#include <common.h>
#include <errno.h>
#include <miiphy.h>
#include <netdev.h>
#include <asm/io.h>
#include <asm/mach-imx/iomux-v3.h>
#include <asm-generic/gpio.h>
#include <asm/arch/imx8mp_pins.h>
#include <asm/arch/sys_proto.h>
#include <asm/mach-imx/gpio.h>
#include <asm/mach-imx/mxc_i2c.h>
#include <asm/arch/clock.h>
#include <spl.h>
#include <asm/mach-imx/dma.h>
#include <power/pmic.h>
#include <power/pca9450.h>
#include "../common/tcpc.h"
#include <usb.h>
#include <dwc3-uboot.h>
#include <mmc.h>
#include <linux/bitfield.h>
#include <linux/libfdt.h>
#include <fdt_support.h>
#include <fuse.h>
#include "board_kotai.h"
#ifdef CONFIG_FSL_CAAM_KB
#include <fsl_caam.h>
#endif

DECLARE_GLOBAL_DATA_PTR;

// flags for gd->board_type
#define BOARD_TYPE_ETH_DETECTED 0x1
#define BOARD_TYPE_ONE_ETH      0x2

enum ax2_ethernet_phy {
	AX2_ETHERNET_PHY_KSZ9131 = 0,
	AX2_ETHERNET_PHY_88E1512,
};

static enum ax2_ethernet_phy ethphy_type;

#define UART_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_FSEL1)
#define WDOG_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_ODE | PAD_CTL_PUE | PAD_CTL_PE)

static iomux_v3_cfg_t const uart_pads[] = {
	MX8MP_PAD_UART2_RXD__UART2_DCE_RX | MUX_PAD_CTRL(UART_PAD_CTRL),
	MX8MP_PAD_UART2_TXD__UART2_DCE_TX | MUX_PAD_CTRL(UART_PAD_CTRL),
};

static iomux_v3_cfg_t const wdog_pads[] = {
	MX8MP_PAD_GPIO1_IO02__WDOG1_WDOG_B  | MUX_PAD_CTRL(WDOG_PAD_CTRL),
};

#ifdef CONFIG_NAND_MXS

static void setup_gpmi_nand(void)
{
	init_nand_clk();
}
#endif

uint mmc_get_env_part(struct mmc *mmc)
{
	uint part = EXT_CSD_EXTRACT_BOOT_PART(mmc->part_config);

	if (part > 2)
		/* Fallback to user partition if not boot0/boot1 */
		part = 0;

	return part;
}

#ifdef CONFIG_DWC_ETH_QOS

#define EQOS_LDO_PAD IMX_GPIO_NR(4, 18)
#define EQOS_LED3_PAD IMX_GPIO_NR(4, 0)
#define EQOS_INT_N IMX_GPIO_NR(4, 21)
static iomux_v3_cfg_t const eqos_pads[] = {
	MX8MP_PAD_SAI1_TXD6__GPIO4_IO18 | MUX_PAD_CTRL(NO_PAD_CTRL),
	MX8MP_PAD_SAI1_RXFS__GPIO4_IO00 | MUX_PAD_CTRL(NO_PAD_CTRL),
};
static iomux_v3_cfg_t const eqos_check_pad =
	MX8MP_PAD_SAI2_RXFS__GPIO4_IO21 | MUX_PAD_CTRL(PAD_CTL_PE /* PULL DOWN */);
static iomux_v3_cfg_t const eqos_check_pad_release =
	MX8MP_PAD_SAI2_RXFS__GPIO4_IO21 | MUX_PAD_CTRL(0 /* PULL DISABLE */);

static void setup_iomux_eqos(void)
{
	imx_iomux_v3_setup_multiple_pads(eqos_pads,
					 ARRAY_SIZE(eqos_pads));
	imx_iomux_v3_setup_pad(eqos_check_pad);

	gpio_request(EQOS_LDO_PAD, "eqos_ldo");
	gpio_direction_output(EQOS_LDO_PAD, 1);

	/* Turn off the led */
	gpio_request(EQOS_LED3_PAD, "eqos_led3");
	gpio_direction_output(EQOS_LED3_PAD, 1);

	gpio_request(EQOS_INT_N, "eqos_int_n");
	gpio_direction_input(EQOS_INT_N);
}

static bool is_high_g1(void)
{
	u32 val;
	int ret;

	ret = fuse_read(EFUSE_KOTAI_BANK, EFUSE_KOTAI_WORD, &val);
	if (ret) {
		printf("%s: failed to read kotai-code(ret = %d)\n",
		       __func__, ret);
		return false;
	}

	switch (val) {
	case EFUSE_KOTAI_HIGH_G1_ES1:
		return true;
	}

	return false;
}

static bool eqos_exists(void)
{
	int val;
	int ret;

	if (gd->board_type & BOARD_TYPE_ETH_DETECTED) {
		return (gd->board_type & BOARD_TYPE_ONE_ETH) == 0;
	}

	/* for EQOS_INT_N pull-up */
	gpio_direction_output(EQOS_LDO_PAD, 1);
	mdelay(1); /* at least 500 us for TLV758P startup time (typ) */

	/*
	 * If the phy is on the board, EQOS_INT_N will be pulled up on
	 * the board, if not, it will be pulled down by the iomux pad
	 * setting. So the value of EQOS_INT_N is:
	 *
	 *  - 1 if the phy is on the board.
	 *  - 0 if the phy is not on the board.
	 *  - -1 or lesser if an internal error occurrs.
	 */
	val = gpio_get_value(EQOS_INT_N);
	if (val < 0) {
		printf("%s: failed to get gpio(EQOS_INT_N) value.(ret=%d)\n",
		       __func__, val);
	}
	ret = val == 1;
	gd->board_type |= BOARD_TYPE_ETH_DETECTED | (ret == 0 ? BOARD_TYPE_ONE_ETH : 0);

	if (!ret) {
		gpio_direction_output(EQOS_LED3_PAD, 0);
		gpio_direction_output(EQOS_LDO_PAD, 0);
	}

	return ret;
}

static int setup_eqos(void)
{
	struct iomuxc_gpr_base_regs *gpr =
		(struct iomuxc_gpr_base_regs *)IOMUXC_GPR_BASE_ADDR;
	int ret = 0;

	setup_iomux_eqos();

	if (eqos_exists()) {
		/* set INTF as RGMII, enable RGMII TXC clock */
		clrsetbits_le32(&gpr->gpr[1],
				IOMUXC_GPR_GPR1_GPR_ENET_QOS_INTF_SEL_MASK,
				BIT(16));
		setbits_le32(&gpr->gpr[1], BIT(19) | BIT(21));

		ret = set_clk_eqos(ENET_125MHZ);
	}

	return ret;
}
#endif

#ifdef CONFIG_POST
/* POST uses snvs registers that require some initialization to use */
#define SNVS_LP_BASE 0x30370000
#define SNVS_LPPGDR (SNVS_LP_BASE + 0x64)
#define SNVS_LPPGDR_VALUE 0x41736166
#define SNVS_LPSR (SNVS_LP_BASE + 0x4c)
static void setup_snvs(void)
{
	/* set power glitch detector register to its nominal value */
	writel(SNVS_LPPGDR_VALUE, SNVS_LPPGDR);
	/* reset status register glitch detector state */
	writel(~0, SNVS_LPSR);
}
#endif

int board_early_init_f(void)
{
	struct wdog_regs *wdog = (struct wdog_regs *)WDOG1_BASE_ADDR;

	imx_iomux_v3_setup_multiple_pads(wdog_pads, ARRAY_SIZE(wdog_pads));

	set_wdog_reset(wdog);

	imx_iomux_v3_setup_multiple_pads(uart_pads, ARRAY_SIZE(uart_pads));

	init_uart_clk(1);

#ifdef CONFIG_DWC_ETH_QOS
	setup_iomux_eqos();
#endif

#ifdef CONFIG_POST
	setup_snvs();
#endif

	return 0;
}

#ifdef CONFIG_OF_BOARD_SETUP
static int fdt_setprop_string_by_alias(void *blob, char *alias, char *prop, char *val)
{
	int offset;
	int ret;

	offset = fdt_path_offset(blob, alias);
	if (offset < 0) {
		printf("%s is BADPATH\n", alias);
		return -1;
	}

	ret = fdt_setprop_string(blob, offset, prop, val);
	if (ret < 0) {
		printf("failed to set %s to %s(ret = %d)\n",
		       prop, val, ret);
		return -1;
	}

	return 0;
}

static int fix_fdt_ethphy(void *blob, char *eth_alias, char *ethphy)
{
	int eth_offset, phy_offset, len, phy_phandle;
	char path[256];
	int ret;

	eth_offset = fdt_path_offset(blob, eth_alias);
	if (eth_offset < 0) {
		printf("%s is BADPATH(alias)\n", eth_alias);
		return -1;
	}
	ret = fdt_get_path(blob, eth_offset, path, sizeof(path));
	if (ret) {
		printf("failed to fdt_get_path(ret=%d)\n", ret);
		return -1;
	}

	len = strlen(path);
	strncpy(path + len, ethphy, sizeof(path) - len);
	len = strlen(path);

	phy_offset = fdt_path_offset(blob, path);
	if (phy_offset < 0) {
		printf("%s is BADPATH\n", path);
		return -1;
	}

	phy_phandle = cpu_to_fdt32(fdt_get_phandle(blob, phy_offset));
	if (!phy_phandle) {
		printf("not found phandle(%s)\n", path);
		return -1;
	}

	ret = fdt_setprop(blob, eth_offset, "phy-handle", &phy_phandle,
			  sizeof(phy_phandle));
	if (ret < 0) {
		printf("failed to fdt_setprop (ret=%d)\n", ret);
		return -1;
	}

	return 0;
}

#ifdef CONFIG_FSL_CAAM_KB
static int fdt_add_kaslr_seed(void *blob)
{
	uint8_t buf[8];
	int chosen_offset, ret;

	caam_open();
	if (caam_hwrng(buf, sizeof(buf))) {
		printf("Failed to get random number, skipping KASLR seed\n");
		return 0;
	}

	chosen_offset = fdt_find_or_add_subnode(blob, 0, "chosen");
	if (chosen_offset < 0) {
		printf("Could not find or create chosen node\n");
		return chosen_offset;
	}
	ret = fdt_setprop(blob, chosen_offset, "kaslr-seed", buf, sizeof(buf));
	if (ret < 0) {
		printf("Could not set kaslr-seed\n");
		return ret;
	}

	return 0;
}
#endif

/* fix linux fdt */
static void board_fixup_ethernet(void *blob)
{
	switch (ethphy_type) {
	case AX2_ETHERNET_PHY_88E1512:
		fix_fdt_ethphy(blob, "ethernet0", "/mdio/fec-phy@0");
		fdt_setprop_string_by_alias(blob, "ethernet0/mdio/fec-phy@3", "status", "broken");
		fix_fdt_ethphy(blob, "ethernet1", "/mdio/eqos-phy@0");
		fdt_setprop_string_by_alias(blob, "ethernet1/mdio/eqos-phy@3", "status", "broken");
		fdt_setprop_string_by_alias(blob, "/leds/enet_led3",
					    "linux,default-trigger",
					    "stmmac-1:00:1Gbps");
		fdt_setprop_string_by_alias(blob, "/leds/enet1_led3",
					    "linux,default-trigger",
					    "30be0000.ethernet-1:00:1Gbps");
		break;
	case AX2_ETHERNET_PHY_KSZ9131:
		fdt_setprop_string_by_alias(blob, "ethernet0/mdio/fec-phy@0", "status", "broken");
		fdt_setprop_string_by_alias(blob, "ethernet1/mdio/eqos-phy@0", "status", "broken");
		break;
	default:
		break;
	}
}

int ft_board_setup(void *blob, bd_t *bd)
{
	char *model_name = NULL;
	u32 val;
	int ret;

#ifdef CONFIG_IMX8M_DRAM_INLINE_ECC
	int rc;
	phys_addr_t ecc0_start = 0xb0000000;
	phys_addr_t ecc1_start = 0x130000000;
	phys_addr_t ecc2_start = 0x1b0000000;
	size_t ecc_size = 0x10000000;

	rc = add_res_mem_dt_node(blob, "ecc", ecc0_start, ecc_size);
	if (rc < 0) {
		printf("Could not create ecc0 reserved-memory node.\n");
		return rc;
	}

	rc = add_res_mem_dt_node(blob, "ecc", ecc1_start, ecc_size);
	if (rc < 0) {
		printf("Could not create ecc1 reserved-memory node.\n");
		return rc;
	}

	rc = add_res_mem_dt_node(blob, "ecc", ecc2_start, ecc_size);
	if (rc < 0) {
		printf("Could not create ecc2 reserved-memory node.\n");
		return rc;
	}
#endif
#ifdef CONFIG_DWC_ETH_QOS
	if (eqos_exists()) {
		/* EQOS_INT_N should not be pulled up/down because it is pulled
		 * up on the board.
		 */
		imx_iomux_v3_setup_pad(eqos_check_pad_release);
	} else {
		fdt_set_status_by_alias(blob, "ethernet1", FDT_STATUS_DISABLED, 0);
	}
#endif

	/* high_g1 does not have fec and the only phy is 88E1512. */
	if (!is_high_g1())
		board_fixup_ethernet(blob);

	ret = fuse_read(EFUSE_KOTAI_BANK, EFUSE_KOTAI_WORD, &val);
	if (ret) {
		printf("failed to read kotai-code\n");
		return -1;
	}

	switch (val) {
	case EFUSE_KOTAI_X2:
	case EFUSE_KOTAI_X2ES1:
	case EFUSE_KOTAI_X2ES2:
		model_name = "Atmark-Techno Armadillo-X2 Board";
		/* not populated */
		fdt_setprop_string_by_alias(blob, "i2c2/gpio@43", "status", "broken");
		break;
	case EFUSE_KOTAI_G4:
	case EFUSE_KOTAI_G4ES0:
	case EFUSE_KOTAI_G4ES1:
	case EFUSE_KOTAI_G4ES2:
	case EFUSE_KOTAI_HIGH_G1_ES1:
		break;
	default:
		printf("unknown board.\n");
		break;
	}

	if (model_name) {
		ret = fdt_setprop_string(blob, 0, "model", model_name);
		if (ret < 0) {
			printf("failed to set model(%s)(ret = %d)\n",
			       model_name, ret);
		}
	}

	ret = 0;
#ifdef CONFIG_FSL_CAAM_KB
	ret = fdt_add_kaslr_seed(blob);
#endif

	return ret;
}
#endif

#ifdef CONFIG_FEC_MXC
#define FEC_LDO_PAD IMX_GPIO_NR(4, 19)
#define FEC_LED3_PAD IMX_GPIO_NR(4, 20)
static iomux_v3_cfg_t const fec1_pads[] = {
	MX8MP_PAD_SAI1_TXD7__GPIO4_IO19 | MUX_PAD_CTRL(NO_PAD_CTRL),
	MX8MP_PAD_SAI1_MCLK__GPIO4_IO20 | MUX_PAD_CTRL(NO_PAD_CTRL),
};

static void setup_iomux_fec(void)
{
	imx_iomux_v3_setup_multiple_pads(fec1_pads,
					 ARRAY_SIZE(fec1_pads));

	gpio_request(FEC_LDO_PAD, "fec1_ldo");
	gpio_direction_output(FEC_LDO_PAD, 1);

	/* at least 500 us for TLV758P startup time (typ) */
	/* at least 15 ms (5 ms is a margin) for FPF1107 startup time */
	mdelay(15);

	/* Turn off the led */
	gpio_request(FEC_LED3_PAD, "fec_led3");
	gpio_direction_output(FEC_LED3_PAD, 1);
}

static int setup_fec(void)
{
	setup_iomux_fec();
	return 0;
}
#endif

#if defined(CONFIG_FEC_MXC) || defined(CONFIG_DWC_ETH_QOS)
#define AX2_ETH_LED_STATE_ACTIVITY	0
#define AX2_ETH_LED_STATE_LINK100	1
#define AX2_ETH_LED_STATE_FORCEOFF	2

/* LED MODE field is 4-bit length */
#define KSZ9131_LINK_ACTIVITY		0
#define KSZ9131_LINK1000_ACTIVITY	1
#define KSZ9131_LINK100_ACTIVITY	2
#define KSZ9131_LINK10_ACTIVITY		3
#define KSZ9131_LINK100_1000_ACTIVITY	4
#define KSZ9131_LINK10_1000_ACTIVITY	5
#define KSZ9131_LINK10_100_ACTIVITY	6
#define KSZ9131_DUPLEX_COLLISION	8
#define KSZ9131_COLLISION		9
#define KSZ9131_ACTIVITY		10
#define KSZ9131_AUTO_NEGOTIATION_FAULT	12
#define KSZ9131_FORCE_LED_OFF		14
#define KSZ9131_FORCE_LED_ON		15

/* Used only in software. Not written directly to register. Setting
 * 0x10 here disables activity and collision in led behaviour register.
 */
#define KSZ9131_COMBINATION_DISABLES(x)	(x | 0x10)
#define KSZ9131_LINK		KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK_ACTIVITY)
#define KSZ9131_LINK1000	KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK1000_ACTIVITY)
#define KSZ9131_LINK100		KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK100_ACTIVITY)
#define KSZ9131_LINK10		KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK10_ACTIVITY)
#define KSZ9131_LINK100_1000	KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK100_1000_ACTIVITY)
#define KSZ9131_LINK10_1000	KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK10_1000_ACTIVITY)
#define KSZ9131_LINK10_100	KSZ9131_COMBINATION_DISABLES(KSZ9131_LINK10_100_ACTIVITY)
#define KSZ9131_DUPLEX		KSZ9131_COMBINATION_DISABLES(KSZ9131_DUPLEX_COLLISION)

#define MII_KSZ9031_LED_MODE		0x1A
#define KSZ9031_LED_MODE_KSZ9031	BIT(14)
#define MII_KSZPHY_LED_MODE_SELECT	0x16
#define KSZPHY_LED_MODE_SELECT_LED1	GENMASK(3, 0)
#define KSZPHY_LED_MODE_SELECT_LED2	GENMASK(7, 4)
#define MII_KSZPHY_LED_BEHAVIOR		0x17
#define KSZPHY_LED_BEHAVIOR_LED_COMBINATION_DISABLES	0x03
#define KSZPHY_LED_BEHAVIOR_LED1_COMBINATION_DISABLES	BIT(0)
#define KSZPHY_LED_BEHAVIOR_LED2_COMBINATION_DISABLES	BIT(1)
#define KSZPHY_LED_BEHAVIOR_PULSING	BIT(12)
#define KSZPHY_LED_BEHAVIOR_ENHANCED_LED_MODE	BIT(15)

void ksz9131_setup_led(struct phy_device *phydev, unsigned led1, unsigned led2)
{
	unsigned value;

	/* Extended LED mode */
	value = phy_read(phydev, MDIO_DEVAD_NONE, MII_KSZ9031_LED_MODE);
	value &= ~KSZ9031_LED_MODE_KSZ9031;
	phy_write(phydev, MDIO_DEVAD_NONE, MII_KSZ9031_LED_MODE, value);

	/* LED mode select */
	value = FIELD_PREP(KSZPHY_LED_MODE_SELECT_LED1, led1);
	value |= FIELD_PREP(KSZPHY_LED_MODE_SELECT_LED2, led2);
	phy_write(phydev, MDIO_DEVAD_NONE, MII_KSZPHY_LED_MODE_SELECT, value);

	/* LED behavior settings */
	value = phy_read(phydev, MDIO_DEVAD_NONE, MII_KSZPHY_LED_BEHAVIOR);
	value |= KSZPHY_LED_BEHAVIOR_ENHANCED_LED_MODE;
	value &= ~KSZPHY_LED_BEHAVIOR_PULSING;
	value &= ~KSZPHY_LED_BEHAVIOR_LED_COMBINATION_DISABLES;
#undef FIELD_MAX
#define FIELD_MAX(_mask)	FIELD_GET(_mask, _mask)
	if (led1 > FIELD_MAX(KSZPHY_LED_MODE_SELECT_LED1))
		value |= KSZPHY_LED_BEHAVIOR_LED1_COMBINATION_DISABLES;
	if (led2 > FIELD_MAX(KSZPHY_LED_MODE_SELECT_LED2))
		value |= KSZPHY_LED_BEHAVIOR_LED2_COMBINATION_DISABLES;
	phy_write(phydev, MDIO_DEVAD_NONE, MII_KSZPHY_LED_BEHAVIOR, value);
}

/* LED Control Registers at Page 3 */
#define MIIM_88E151x_LED_FN_CTRL	16
#define MIIM_88E151x_LEDx_FN_FLD(x, f)	(((f) & 0xf) << ((x) * 4))
#define MIIM_88E151x_LED_FN_ACT		0x3
#define MIIM_88E151x_LED_FN_100_LINK	0x7
#define MIIM_88E151x_LED_FN_FORCE_OFF	0x8
#define MIIM_88E151x_LED_POL_CTRL	17
#define MIIM_88E151x_LEDx_POL_FLD(x, f)	(((f) & 0x3) << ((x) * 2))
#define MIIM_88E151x_LED_POL_LOW_TRI	0x2
static void m88e1512_setup_led(struct phy_device *phydev, unsigned int led0,
			       unsigned int led1)
{
	int val;

	val = phydev->drv->readext(phydev, 0, 3, MIIM_88E151x_LED_FN_CTRL);
	val &= ~(MIIM_88E151x_LEDx_FN_FLD(1, 0xf) |
		 MIIM_88E151x_LEDx_FN_FLD(0, 0xf));
	val |= MIIM_88E151x_LEDx_FN_FLD(1, led1) |
		MIIM_88E151x_LEDx_FN_FLD(0, led0);
	phydev->drv->writeext(phydev, 0, 3, MIIM_88E151x_LED_FN_CTRL, val);

	val = phydev->drv->readext(phydev, 0, 3, MIIM_88E151x_LED_POL_CTRL);
	val &= ~(MIIM_88E151x_LEDx_POL_FLD(1, 0x3) |
		 MIIM_88E151x_LEDx_POL_FLD(0, 0x3));
	val |= MIIM_88E151x_LEDx_POL_FLD(1, MIIM_88E151x_LED_POL_LOW_TRI) |
		MIIM_88E151x_LEDx_POL_FLD(0, MIIM_88E151x_LED_POL_LOW_TRI);
	phydev->drv->writeext(phydev, 0, 3, MIIM_88E151x_LED_POL_CTRL, val);
}

static unsigned int led_state_ax2_to_ksz9131(unsigned int state)
{
	switch (state) {
	case AX2_ETH_LED_STATE_ACTIVITY:
		return KSZ9131_LINK_ACTIVITY;
	case AX2_ETH_LED_STATE_LINK100:
		return KSZ9131_LINK100;
	case AX2_ETH_LED_STATE_FORCEOFF:
		return KSZ9131_FORCE_LED_OFF;
	default:
		printf("Unknown eth led state %d\n", state);
		return KSZ9131_FORCE_LED_OFF;
	}
}

static unsigned int led_state_ax2_to_88e151x(unsigned int state)
{
	switch (state) {
	case AX2_ETH_LED_STATE_ACTIVITY:
		return MIIM_88E151x_LED_FN_ACT;
	case AX2_ETH_LED_STATE_LINK100:
		return MIIM_88E151x_LED_FN_100_LINK;
	case AX2_ETH_LED_STATE_FORCEOFF:
		return MIIM_88E151x_LED_FN_FORCE_OFF;
	default:
		printf("Unknown eth led state %d\n", state);
		return MIIM_88E151x_LED_FN_FORCE_OFF;
	}
}

static void setup_led(struct phy_device *phydev, unsigned int led1,
		      unsigned int led2)
{
	if (strcmp(phydev->drv->name, "Micrel ksz9131") == 0) {
		ksz9131_setup_led(phydev, led_state_ax2_to_ksz9131(led1),
				  led_state_ax2_to_ksz9131(led2));
	}
	if (strcmp(phydev->drv->name, "Marvell 88E151x") == 0) {
		m88e1512_setup_led(phydev, led_state_ax2_to_88e151x(led1),
				   led_state_ax2_to_88e151x(led2));
	}
}

/* General Control Register 1 at Page 18 */
#define MIIM_88E151x_GENERAL_CTRL	20
#define MIIM_88E151x_MODE_MASK		0x7
#define MIIM_88E151x_MODE_RGMII_COPPER	0x0
static void m88e1512_rgmii_copper_mode(struct phy_device *phydev)
{
	int val;

	val = phydev->drv->readext(phydev, 0, 18, MIIM_88E151x_GENERAL_CTRL);
	val &= ~MIIM_88E151x_MODE_MASK;
	val |= MIIM_88E151x_MODE_RGMII_COPPER;
	phydev->drv->writeext(phydev, 0, 18, MIIM_88E151x_GENERAL_CTRL, val);
}

int board_phy_startup(struct phy_device *phydev)
{
	unsigned led;

	if (phydev->drv->startup)
		phydev->drv->startup(phydev);
	setup_led(phydev, AX2_ETH_LED_STATE_ACTIVITY,
		  AX2_ETH_LED_STATE_LINK100);

	if (!strncmp(phydev->bus->name, "FEC", 3))
		led = FEC_LED3_PAD;
	else
		led = EQOS_LED3_PAD;
	if (phydev->speed == SPEED_1000)
		gpio_direction_output(led, 0);
	else
		gpio_direction_output(led, 1);

	return 0;
}

int board_phy_config(struct phy_device *phydev)
{
	if (phydev->drv->config)
		phydev->drv->config(phydev);
	setup_led(phydev, AX2_ETH_LED_STATE_FORCEOFF,
		  AX2_ETH_LED_STATE_FORCEOFF);

	if (strcmp(phydev->drv->name, "Marvell 88E151x") == 0) {
		m88e1512_rgmii_copper_mode(phydev);
		ethphy_type = AX2_ETHERNET_PHY_88E1512;
	}

	return 0;
}

int board_phy_shutdown(struct phy_device *phydev)
{
	if (phydev->drv->shutdown)
		phydev->drv->shutdown(phydev);
	setup_led(phydev, AX2_ETH_LED_STATE_FORCEOFF,
		  AX2_ETH_LED_STATE_FORCEOFF);

	if (!strncmp(phydev->bus->name, "FEC", 3))
		gpio_direction_output(FEC_LED3_PAD, 1);
	else
		gpio_direction_output(EQOS_LED3_PAD, 1);

	return 0;
}
#endif

#ifdef CONFIG_USB_TCPC
struct tcpc_port port1;
struct tcpc_port port2;

static int setup_pd_switch(uint8_t i2c_bus, uint8_t addr)
{
	struct udevice *bus;
	struct udevice *i2c_dev = NULL;
	int ret;
	uint8_t valb;

	ret = uclass_get_device_by_seq(UCLASS_I2C, i2c_bus, &bus);
	if (ret) {
		printf("%s: Can't find bus\n", __func__);
		return -EINVAL;
	}

	ret = dm_i2c_probe(bus, addr, 0, &i2c_dev);
	if (ret) {
		printf("%s: Can't find device id=0x%x\n",
			__func__, addr);
		return -ENODEV;
	}

	ret = dm_i2c_read(i2c_dev, 0xB, &valb, 1);
	if (ret) {
		printf("%s dm_i2c_read failed, err %d\n", __func__, ret);
		return -EIO;
	}
	valb |= 0x4; /* Set DB_EXIT to exit dead battery mode */
	ret = dm_i2c_write(i2c_dev, 0xB, (const uint8_t *)&valb, 1);
	if (ret) {
		printf("%s dm_i2c_write failed, err %d\n", __func__, ret);
		return -EIO;
	}

	/* Set OVP threshold to 23V */
	valb = 0x6;
	ret = dm_i2c_write(i2c_dev, 0x8, (const uint8_t *)&valb, 1);
	if (ret) {
		printf("%s dm_i2c_write failed, err %d\n", __func__, ret);
		return -EIO;
	}

	return 0;
}

int pd_switch_snk_enable(struct tcpc_port *port)
{
	if (port == &port1) {
		debug("Setup pd switch on port 1\n");
		return setup_pd_switch(1, 0x72);
	} else
		return -EINVAL;
}

/* Port2 is the power supply, port 1 does not support power */
struct tcpc_port_config port1_config = {
	.i2c_bus = 1, /*i2c2*/
	.addr = 0x50,
	.port_type = TYPEC_PORT_UFP,
	.max_snk_mv = 20000,
	.max_snk_ma = 3000,
	.max_snk_mw = 45000,
	.op_snk_mv = 15000,
	.switch_setup_func = &pd_switch_snk_enable,
	.disable_pd = true,
};

struct tcpc_port_config port2_config = {
	.i2c_bus = 2, /*i2c3*/
	.addr = 0x50,
	.port_type = TYPEC_PORT_UFP,
	.max_snk_mv = 20000,
	.max_snk_ma = 3000,
	.max_snk_mw = 45000,
	.op_snk_mv = 15000,
};

#define USB_TYPEC_SEL IMX_GPIO_NR(4, 20)
#define USB_TYPEC_EN IMX_GPIO_NR(2, 20)

static iomux_v3_cfg_t ss_mux_gpio[] = {
	MX8MP_PAD_SAI1_MCLK__GPIO4_IO20 | MUX_PAD_CTRL(NO_PAD_CTRL),
	MX8MP_PAD_SD2_WP__GPIO2_IO20 | MUX_PAD_CTRL(NO_PAD_CTRL),
};

void ss_mux_select(enum typec_cc_polarity pol)
{
	if (pol == TYPEC_POLARITY_CC1)
		gpio_direction_output(USB_TYPEC_SEL, 0);
	else
		gpio_direction_output(USB_TYPEC_SEL, 1);
}

static int setup_typec(void)
{
	int ret;
	struct gpio_desc per_12v_desc;

	debug("tcpc_init port 2\n");
	ret = tcpc_init(&port2, port2_config, NULL);
	if (ret) {
		printf("%s: tcpc port2 init failed, err=%d\n",
		       __func__, ret);
	} else if (tcpc_pd_sink_check_charging(&port2)) {
		printf("Power supply on USB2\n");

		/* Enable PER 12V, any check before it? */
		ret = dm_gpio_lookup_name("gpio@20_1", &per_12v_desc);
		if (ret) {
			printf("%s lookup gpio@20_1 failed ret = %d\n", __func__, ret);
			return -ENODEV;
		}

		ret = dm_gpio_request(&per_12v_desc, "per_12v_en");
		if (ret) {
			printf("%s request per_12v failed ret = %d\n", __func__, ret);
			return -EIO;
		}

		/* Enable PER 12V regulator */
		dm_gpio_set_dir_flags(&per_12v_desc, GPIOD_IS_OUT | GPIOD_IS_OUT_ACTIVE);
	}

	debug("tcpc_init port 1\n");
	imx_iomux_v3_setup_multiple_pads(ss_mux_gpio, ARRAY_SIZE(ss_mux_gpio));
	gpio_request(USB_TYPEC_SEL, "typec_sel");
	gpio_request(USB_TYPEC_EN, "typec_en");
	gpio_direction_output(USB_TYPEC_EN, 0);

	ret = tcpc_init(&port1, port1_config, &ss_mux_select);
	if (ret) {
		printf("%s: tcpc port1 init failed, err=%d\n",
		       __func__, ret);
	} else {
		return ret;
	}

	return ret;
}
#endif

#ifdef CONFIG_USB_DWC3

#define USB_PHY_CTRL0			0xF0040
#define USB_PHY_CTRL0_REF_SSP_EN	BIT(2)

#define USB_PHY_CTRL1			0xF0044
#define USB_PHY_CTRL1_RESET		BIT(0)
#define USB_PHY_CTRL1_COMMONONN		BIT(1)
#define USB_PHY_CTRL1_ATERESET		BIT(3)
#define USB_PHY_CTRL1_VDATSRCENB0	BIT(19)
#define USB_PHY_CTRL1_VDATDETENB0	BIT(20)

#define USB_PHY_CTRL2			0xF0048
#define USB_PHY_CTRL2_TXENABLEN0	BIT(8)

#define USB_PHY_CTRL6			0xF0058

#define HSIO_GPR_BASE                               (0x32F10000U)
#define HSIO_GPR_REG_0                              (HSIO_GPR_BASE)
#define HSIO_GPR_REG_0_USB_CLOCK_MODULE_EN_SHIFT    (1)
#define HSIO_GPR_REG_0_USB_CLOCK_MODULE_EN          (0x1U << HSIO_GPR_REG_0_USB_CLOCK_MODULE_EN_SHIFT)


static struct dwc3_device dwc3_device_data = {
#ifdef CONFIG_SPL_BUILD
	.maximum_speed = USB_SPEED_HIGH,
#else
	.maximum_speed = USB_SPEED_SUPER,
#endif
	.base = USB1_BASE_ADDR,
	.dr_mode = USB_DR_MODE_PERIPHERAL,
	.index = 0,
	.power_down_scale = 2,
};

int usb_gadget_handle_interrupts(void)
{
	dwc3_uboot_handle_interrupt(0);
	return 0;
}

static void dwc3_nxp_usb_phy_init(struct dwc3_device *dwc3)
{
	u32 RegData;

	/* enable usb clock via hsio gpr */
	RegData = readl(HSIO_GPR_REG_0);
	RegData |= HSIO_GPR_REG_0_USB_CLOCK_MODULE_EN;
	writel(RegData, HSIO_GPR_REG_0);

	/* USB3.0 PHY signal fsel for 100M ref */
	RegData = readl(dwc3->base + USB_PHY_CTRL0);
	RegData = (RegData & 0xfffff81f) | (0x2a<<5);
	writel(RegData, dwc3->base + USB_PHY_CTRL0);

	RegData = readl(dwc3->base + USB_PHY_CTRL6);
	RegData &=~0x1;
	writel(RegData, dwc3->base + USB_PHY_CTRL6);

	RegData = readl(dwc3->base + USB_PHY_CTRL1);
	RegData &= ~(USB_PHY_CTRL1_VDATSRCENB0 | USB_PHY_CTRL1_VDATDETENB0 |
			USB_PHY_CTRL1_COMMONONN);
	RegData |= USB_PHY_CTRL1_RESET | USB_PHY_CTRL1_ATERESET;
	writel(RegData, dwc3->base + USB_PHY_CTRL1);

	RegData = readl(dwc3->base + USB_PHY_CTRL0);
	RegData |= USB_PHY_CTRL0_REF_SSP_EN;
	writel(RegData, dwc3->base + USB_PHY_CTRL0);

	RegData = readl(dwc3->base + USB_PHY_CTRL2);
	RegData |= USB_PHY_CTRL2_TXENABLEN0;
	writel(RegData, dwc3->base + USB_PHY_CTRL2);

	RegData = readl(dwc3->base + USB_PHY_CTRL1);
	RegData &= ~(USB_PHY_CTRL1_RESET | USB_PHY_CTRL1_ATERESET);
	writel(RegData, dwc3->base + USB_PHY_CTRL1);
}
#endif

#if defined(CONFIG_USB_DWC3) || defined(CONFIG_USB_XHCI_IMX8M)

#ifndef CONFIG_YAKUSHIMA_EVA_BOARD
#define USB2422_I2C_BUS_ADDR	1
#define USB2422_I2C_SLAVE_ADDR	0x2C

#define USB2422_REG_STATUS_CMD		0xFF
#define USB2422_STATUS_CMD_INTF_PW_DN	0x4
#define USB2422_STATUS_CMD_RESET	0x2
#define USB2422_STATUS_CMD_USB_ATTACH	0x1

/* Let usb2422 hub signal a usb attach event to an upstream device.
 * And then the registers ranged 0x00-0xFE are write-protected until
 * next RESET_N signal. */
static int setup_usb2422(void)
{
	struct udevice *bus;
	struct udevice *i2c_dev;
	const uint8_t wbuf[] = {
		1,
		USB2422_STATUS_CMD_USB_ATTACH,
	};
	int retries;
	int ret;

	ret = uclass_get_device_by_seq(UCLASS_I2C, USB2422_I2C_BUS_ADDR, &bus);
	if (ret) {
		printf("%s: Can't find bus\n", __func__);
		return -EINVAL;
	}

	/* dm_i2c_probe() is failed with ret == -EREMOTEIO if usb2422
	 * is during waking up. So we retry it. (typ. 2 times) */
	retries = 20;
	do {
		ret = dm_i2c_probe(bus, USB2422_I2C_SLAVE_ADDR, 0, &i2c_dev);
		if (ret == 0)
			break;
		if (ret != -EREMOTEIO)
			break;
		mdelay(1);
	} while (retries--);
	if (ret) {
		printf("%s: Can't find device id=0x%x(ret=%d)\n",
		       __func__, USB2422_I2C_SLAVE_ADDR, ret);
		return -ENODEV;
	}

	ret = dm_i2c_write(i2c_dev, USB2422_REG_STATUS_CMD, wbuf,
			   ARRAY_SIZE(wbuf));
	if (ret) {
		printf("%s dm_i2c_write failed, err %d\n", __func__, ret);
		return -EIO;
	}

	return 0;
}
#endif

#define USB_GPIO_PAD_CTRL (IOMUX_CONFIG_SION | PAD_CTL_FSEL0 | PAD_CTL_DSE0)

static iomux_v3_cfg_t const usb2_hub_reset_pads[] = {
	MX8MP_PAD_GPIO1_IO10__GPIO1_IO10  | MUX_PAD_CTRL(USB_GPIO_PAD_CTRL),
};

#define USB1_PWR_EN IMX_GPIO_NR(1, 9)
#define USB2_HUB_RESET_N IMX_GPIO_NR(1, 10)
#define HIGH_G1_USB1_HUB_RESET_N USB1_PWR_EN
#define HIGH_G1_USB2_HUB_RESET_N USB2_HUB_RESET_N
int board_usb_init(int index, enum usb_init_type init)
{
	int ret = 0;
	imx8m_usb_power(index, true);

	if (index == 0 && init == USB_INIT_DEVICE) {
#ifdef CONFIG_USB_TCPC
		ret = tcpc_setup_ufp_mode(&port1);
		if (ret)
			return ret;
#endif
		dwc3_nxp_usb_phy_init(&dwc3_device_data);
		return dwc3_uboot_init(&dwc3_device_data);
	} else if (index == 0 && init == USB_INIT_HOST) {
#ifdef CONFIG_USB_TCPC
		ret = tcpc_setup_dfp_mode(&port1);
#endif
		if (!is_high_g1()) {
			/* Enable USB1 5V VBUS */
			gpio_request(USB1_PWR_EN, "usb1_pwr_en");
			gpio_direction_output(USB1_PWR_EN, 1);
		} else {
			/* Enable USB1 by unresetting the USB HUB */
			gpio_request(HIGH_G1_USB1_HUB_RESET_N, "usb1_hub_reset_n");
			gpio_direction_output(HIGH_G1_USB1_HUB_RESET_N, 0);
		}
		return ret;
	} else if (index == 1 && init == USB_INIT_HOST) {
		/* Enable USB2 by unresetting the USB HUB */
		imx_iomux_v3_setup_multiple_pads(usb2_hub_reset_pads,
			ARRAY_SIZE(usb2_hub_reset_pads));
		if (!is_high_g1()) {
			gpio_request(USB2_HUB_RESET_N, "usb2_hub_reset_n");
			gpio_direction_output(USB2_HUB_RESET_N, 1);
#ifndef CONFIG_YAKUSHIMA_EVA_BOARD
			ret = setup_usb2422();
			return ret;
#endif
		} else {
			gpio_request(HIGH_G1_USB2_HUB_RESET_N, "usb2_hub_reset_n");
			gpio_direction_output(HIGH_G1_USB2_HUB_RESET_N, 0);
		}
	}

	return 0;
}

int board_usb_cleanup(int index, enum usb_init_type init)
{
	int ret = 0;
	if (index == 0 && init == USB_INIT_DEVICE) {
		dwc3_uboot_exit(index);
	} else if (index == 0 && init == USB_INIT_HOST) {
#ifdef CONFIG_USB_TCPC
		ret = tcpc_disable_src_vbus(&port1);
#endif
		if (!is_high_g1()) {
			/* On yakushima-eva board, do not set USB1_PWR_EN to
			 * low because of signal collision. */
#ifndef CONFIG_YAKUSHIMA_EVA_BOARD
			/* Disable USB1 5V VBUS */
			gpio_direction_output(USB1_PWR_EN, 0);
#endif
		} else {
			/* Disable USB1 */
			gpio_request(HIGH_G1_USB1_HUB_RESET_N, "usb1_hub_reset_n");
			gpio_direction_output(HIGH_G1_USB1_HUB_RESET_N, 1);
		}
		return ret;
	} else if (index == 1 && init == USB_INIT_HOST) {
		/* Disable USB2 */
		if (!is_high_g1()) {
			gpio_request(USB2_HUB_RESET_N, "usb2_hub_reset_n");
			gpio_direction_output(USB2_HUB_RESET_N, 0);
		} else {
			gpio_request(HIGH_G1_USB2_HUB_RESET_N, "usb2_hub_reset_n");
			gpio_direction_output(HIGH_G1_USB2_HUB_RESET_N, 1);
		}
	}

	imx8m_usb_power(index, false);

	return ret;
}

#ifdef CONFIG_USB_TCPC
/* Not used so far */
int board_typec_get_mode(int index)
{
	int ret = 0;
	enum typec_cc_polarity pol;
	enum typec_cc_state state;

	if (index == 0) {
		tcpc_setup_ufp_mode(&port1);

		ret = tcpc_get_cc_status(&port1, &pol, &state);
		if (!ret) {
			if (state == TYPEC_STATE_SRC_RD_RA || state == TYPEC_STATE_SRC_RD)
				return USB_INIT_HOST;
		}

		return USB_INIT_DEVICE;
	} else {
		return USB_INIT_HOST;
	}
}
#endif
#endif

void log_wdt_reset(void)
{
	struct pmic *p;

	p = pmic_get("PCA9450");
	if (!p) {
		printf("reset cause: could not get pmic\n");
		return;
	}
	if (pmic_probe(p)) {
		printf("reset cause: could not probe pmic\n");
		return;
	}

	u32 pwron_stat;
	if (pmic_reg_read(p, PCA9450_PWRON_STAT, &pwron_stat) < 0) {
		printf("reset cause: failed to query pmic poweron status\n");
		return;
	}

	if (pwron_stat & 0x40) {
		printf("reset cause: watchdog\n");
		log_warning("reset by wdt\n");
	} else if (pwron_stat & 0x10) {
		printf("reset cause: button\n");
	} else if (pwron_stat & 0x20) {
		printf("reset cause: normal reboot\n");
	} else if (pwron_stat & 0x80) {
		printf("reset cause: first boot since power on\n");
	} else {
		printf("reset cause: unknown value %x\n", pwron_stat);
	}
}

#define LED1_GPIO IMX_GPIO_NR(1, 14)
#define PWRON_GPIO IMX_GPIO_NR(1, 6)
#define HIGH_G1_NET_LED1 IMX_GPIO_NR(4, 27)
#define HIGH_G1_NET_LED2 IMX_GPIO_NR(4, 29)
void setup_indicators(void)
{
	/* turn on LED1 gpio until linux takes over */
	gpio_request(LED1_GPIO, "led1");
	gpio_direction_output(LED1_GPIO, 1);
	if (!is_high_g1()) {
		/* turn on PWRON_IND gpio until linux takes over */
		gpio_request(PWRON_GPIO, "pwron");
		gpio_direction_output(PWRON_GPIO, 1);
	} else {
		/* turn off NET_LED[1|2] gpio until linux takes over */
		gpio_request(HIGH_G1_NET_LED1, "net_led1");
		gpio_direction_output(HIGH_G1_NET_LED1, 0);
		gpio_request(HIGH_G1_NET_LED2, "net_led2");
		gpio_direction_output(HIGH_G1_NET_LED2, 0);
	}
}

#define FSL_SIP_GPC			0xC2000000
#define FSL_SIP_CONFIG_GPC_PM_DOMAIN	0x3
#define DISPMIX				13
#define MIPI				15

int board_init(void)
{
	if (power_pca9450b_init(I2C_PMIC))
		printf("power init failed\n");

#ifdef CONFIG_USB_TCPC
	setup_typec();
#endif

#ifdef CONFIG_FEC_MXC
	if (!is_high_g1())
		setup_fec();
#endif

#ifdef CONFIG_DWC_ETH_QOS
	/* clock, pin, gpr */
	setup_eqos();
#endif

#ifdef CONFIG_NAND_MXS
	setup_gpmi_nand();
#endif

#if defined(CONFIG_USB_DWC3) || defined(CONFIG_USB_XHCI_IMX8M)
	init_usb_clk();
#endif

	setup_indicators();

	/* enable the dispmix & mipi phy power domain */
	call_imx_sip(FSL_SIP_GPC, FSL_SIP_CONFIG_GPC_PM_DOMAIN, DISPMIX, true, 0);
	call_imx_sip(FSL_SIP_GPC, FSL_SIP_CONFIG_GPC_PM_DOMAIN, MIPI, true, 0);

	return 0;
}

static int check_mmc_autodetect(void)
{
	char *autodetect_str = env_get("mmcautodetect");

	if ((autodetect_str != NULL) &&
			(strcmp(autodetect_str, "yes") == 0)) {
		return 1;
	}

	return 0;
}

void at_board_late_mmc_env_init(void) {
	char buf[100];
	int part = 1;
	u32 dev_no = mmc_get_env_dev();

	if (!check_mmc_autodetect())
		return;

	/* evk board:
	 * emmc    -> dev_no = 2, ext csd bootpart = 1/2
	 * sd card -> dev_no = 1, ext csd bootpart = 7
	 *
	 * atmark image has kernel/dtb in p1/2 and rootfs in p1/2 as well,
	 * so use part as is directly for both
	 */
	if (dev_no == 2) {
		/* emmc, get boot partition from extcsd */
		struct mmc *mmc;

		mmc = find_mmc_device(dev_no);

		if (!mmc) {
			printf("booting on emmc but no mmc device found, skipping mmcautodetect\n");
			return;
		}
		if (!mmc->has_init) {
			printf("mmc dev found but hasn't been init, skipping mmcautodetect\n");
			return;
		}

		part = EXT_CSD_EXTRACT_BOOT_PART(mmc->part_config);
	} else {
		/* sd card, get boot partition from env */
		part = env_get_ulong("mmcpart", 10, 1);
	}

	/* device where to load linux from */
	snprintf(buf, sizeof(buf), "%d", dev_no);
	env_set("mmcdev", buf);
	/* partition index where linux/dtb are stored */
	snprintf(buf, sizeof(buf), "%d", part);
	env_set("mmcpart", buf);
	/* kernel boot command */
	snprintf(buf, sizeof(buf), "/dev/mmcblk%dp%d rootwait ro",
		 mmc_map_to_kernel_blk(dev_no), part);
	env_set("mmcroot", buf);

	snprintf(buf, sizeof(buf), "mmc dev %d", dev_no);
	run_command(buf, 0);
}

static void set_serial(void)
{
	uint32_t product_id_val, lot_val;
	char serial[13 /* product (4), lot (8), \0 (1) */];
	char *old_serial;
	int ret;

	ret = fuse_read(EFUSE_KOTAI_BANK, EFUSE_KOTAI_WORD, &product_id_val);
	if (ret) {
		printf("Error: Can not read seihin kanri id\n");
		return;
	}

	ret = fuse_read(EFUSE_KOTAI_BANK, EFUSE_KOTAI_WORD + 1, &lot_val);
	if (ret) {
		printf("Error: Can not read lot number\n");
		return;
	}

	if (product_id_val == 0 && lot_val == 0) {
		printf("Error: serial number not written in fuses\n");
		return;
	}

	snprintf(serial, sizeof(serial), "%04X%08X", product_id_val, lot_val);

	old_serial = env_get("serial#");
	if (old_serial && old_serial[0] != '\0' && strcmp(serial, old_serial)) {
		/* board previously booted somewhere else, clear mac addresses */
		printf("serial# changed: Clearing MAC addresses in env\n");
		env_set("ethaddr", NULL);
		env_set("eth1addr", NULL);
	}

	env_set("serial#", serial);
}

int board_late_init(void)
{
	/* late init because requires mmc */
	log_wdt_reset();

#ifdef CONFIG_ENV_IS_IN_MMC
	at_board_late_mmc_env_init();
#endif
#ifdef CONFIG_ENV_VARS_UBOOT_RUNTIME_CONFIG
	env_set("board_name", "EVK");
	env_set("board_rev", "iMX8MP");
#endif
	set_serial();

	return 0;
}

#ifdef CONFIG_IMX_BOOTAUX
ulong board_get_usable_ram_top(ulong total_size)
{
	/* Reserve 16M memory used by M core vring/buffer, which begins at 16MB before optee */
	if (rom_pointer[1])
		return gd->ram_top - SZ_16M;

	return gd->ram_top;
}
#endif

#ifdef CONFIG_FSL_FASTBOOT
#ifdef CONFIG_ANDROID_RECOVERY
int is_recovery_key_pressing(void)
{
	return 0; /*TODO*/
}
#endif /*CONFIG_ANDROID_RECOVERY*/
#endif /*CONFIG_FSL_FASTBOOT*/

#ifdef CONFIG_ANDROID_SUPPORT
bool is_power_key_pressed(void) {
	return (bool)(!!(readl(SNVS_HPSR) & (0x1 << 6)));
}
#endif

#ifdef CONFIG_SPL_MMC_SUPPORT

#define UBOOT_RAW_SECTOR_OFFSET 0x40
unsigned long spl_mmc_get_uboot_raw_sector(struct mmc *mmc)
{
	u32 boot_dev = spl_boot_device();
	switch (boot_dev) {
		case BOOT_DEVICE_MMC2:
			return CONFIG_SYS_MMCSD_RAW_MODE_U_BOOT_SECTOR - UBOOT_RAW_SECTOR_OFFSET;
		default:
			return CONFIG_SYS_MMCSD_RAW_MODE_U_BOOT_SECTOR;
	}
}
#endif

#ifdef CONFIG_OF_BOARD_FIXUP
/* fix uboot fdt */
int board_fix_fdt(void* blob)
{
#ifdef CONFIG_FEC_MXC
	/* high_g1 does not have fec so it will be disabled. */
	if (is_high_g1())
		fdt_set_status_by_alias(blob, "ethernet0", FDT_STATUS_DISABLED, 0);
#endif

#ifdef CONFIG_DWC_ETH_QOS
	if (!eqos_exists()) {
		fdt_set_status_by_alias(blob, "ethernet1", FDT_STATUS_DISABLED, 0);
	}
#endif

	return 0;
}
#endif
