// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (C) 2023 Atmark Techno Inc.
 */

#include <stdbool.h>
#include <stdio.h>
#include <env.h>
#include <i2c.h>
#include <malloc.h>
#include <dm/uclass.h>
#include <linux/libfdt.h>
#include "board_addition.h"

#define I2C_BUS	(3)

/* EEPROM I2C slave address */
#define EEPROM_ADDR1	(0x57) /* 1st */
#define EEPROM_ADDR2	(0x56) /* 2nd */
#define EEPROM_ADDR3	(0x55) /* 3rd */
#define EEPROM_ADDR4	(0x53) /* 4th */

#define A6E_ADDITIONAL_BOARD_MAX	(4)

/* EEPROM Format */
#define POS_VENDOR	(0x00)
#define POS_PRODUCT	(0x02)
#define POS_REVISION	(0x04)

/* Vendor ID's */
#define VENDOR_ATMARK_TECHNO	(0x0001)

/* Product ID's */
#define PRODUCT_ATMARK_TECHNO_DI8AI4	(0x000E)

/* revision */
#define DI8AI4_REV_THREE		(0x0003)

#define GET_EEPROM_ERROR		(0xFFFF)

static const unsigned char add_board_i2c_addrs[A6E_ADDITIONAL_BOARD_MAX] = {
	EEPROM_ADDR1,
	EEPROM_ADDR2,
	EEPROM_ADDR3,
	EEPROM_ADDR4,
};

static const char *a6e_di8ai4[A6E_ADDITIONAL_BOARD_MAX] = {
	"armadillo-iotg-a6e-di8ai4-1st.dtbo",
	"armadillo-iotg-a6e-di8ai4-2nd.dtbo",
	"armadillo-iotg-a6e-di8ai4-3rd.dtbo",
	"armadillo-iotg-a6e-di8ai4-4th.dtbo",
};

#ifdef CONFIG_OF_BOARD_SETUP
static const char *a6e_di8ai4_adc_alias[A6E_ADDITIONAL_BOARD_MAX] = {
	"i2c3/adc@49",
	"i2c3/adc@48",
	"i2c3/adc@4B",
	"i2c3/adc@4A",
};
#endif

static bool disable_di8ai4_board_irq;
static int di8ai4_boards_count;
static unsigned short di8ai4_product_id;

static int eeprom_read(unsigned char i2c_addr, unsigned char addr, void *data,
		       unsigned short size, struct udevice *i2c_dev)
{
	int ret;
	unsigned char *pdata = (unsigned char *)data;

#ifdef CONFIG_DM_I2C
	ret = dm_i2c_read(i2c_dev, addr, pdata, size);
#else
	ret = i2c_read(i2c_addr, addr, pdata, size);
#endif
	if (ret) {
		printf("%s: i2c read failed ret:%d.\n", __func__, ret);
		return ret;
	}
	return 0;
}

static bool is_vendor_atmark_techno(unsigned char i2c_addr, struct udevice *i2c_dev)
{
	unsigned short	vendor_id_eeprom;
	unsigned short	vendor_id;

	if (eeprom_read(i2c_addr, POS_VENDOR, &vendor_id_eeprom, 2, i2c_dev))
		return false;

	vendor_id = be16_to_cpu(vendor_id_eeprom);
	if (vendor_id == VENDOR_ATMARK_TECHNO)
		return true;

	return false;
}

static unsigned short get_eeprom_data(unsigned short i2c_addr, struct udevice *i2c_dev,
				      unsigned short format)
{
	unsigned short	eeprom_data;

	if (eeprom_read(i2c_addr, format, &eeprom_data, 2, i2c_dev))
		return GET_EEPROM_ERROR;

	return be16_to_cpu(eeprom_data);
}

#ifdef CONFIG_OF_BOARD_SETUP
static int del_ftd_interrupt(void *blob)
{
	int i;
	int offset;
	int ret = 0;

	if (disable_di8ai4_board_irq) {
		for (i = 0; i < di8ai4_boards_count; i++) {
			offset = fdt_path_offset(blob, a6e_di8ai4_adc_alias[i]);
			if (offset < 0) {
				printf("%s is BADPATH(alias)\n", a6e_di8ai4_adc_alias[i]);
				return -1;
			}
			ret = fdt_delprop(blob, offset, "interrupt-parent");
			if (ret) {
				printf("failed to fdt_delprop interrupt-parent (ret=%d)\n", ret);
				break;
			}
			ret = fdt_delprop(blob, offset, "interrupts");
			if (ret) {
				printf("failed to fdt_delprop interrupts (ret=%d)\n", ret);
				break;
			}
			if (i == 0) {
				ret = fdt_delprop(blob, offset, "wakeup-source");
				if (ret) {
					printf("failed to fdt_delprop wakeup-source (ret=%d)\n",
					       ret);
					break;
				}
			}
		}

		if (!ret)
			printf("Disabled additional board interrupts.\n");
	}

	return ret;
}

static int append_add_board_data(void *blob)
{
	int ret = 0;
	int offset;

	if (di8ai4_boards_count) {
		offset = fdt_path_offset(blob, "/chosen");

		ret = fdt_appendprop_u32(blob, offset, "atmark,extboard-count",
					 di8ai4_boards_count);
		if (ret) {
			printf("failed to fdt_appendprop extboard-count (ret=%d)\n", ret);
			return ret;
		}

		/* If it becomes possible to connect multiple types of expansion boards
		 * in the future, it will be necessary to increase "atmark,extboard-info".
		 * For backward compatibility, make sure to define "atmark,extboard-info" as 1
		 * and add "atmark,extboard-info[2-4]".
		 */
		ret = fdt_appendprop_u32(blob, offset, "atmark,extboard-info",
					 di8ai4_product_id);
		if (ret)
			printf("failed to fdt_appendprop extboard-info (ret=%d)\n", ret);
	}

	return ret;
}

/* fix linux fdt */
int ft_board_setup(void *blob, bd_t *bd)
{
	int ret = 0;

	ret = del_ftd_interrupt(blob);
	if (ret)
		return ret;

	return append_add_board_data(blob);
}
#endif

static int remove_additional_dtbo(char *buf)
{
	int i;
	char *p;
	char *p_space;
	char *p_src;
	char *tail;
	int total_len;
	int current_len;

	if (!strlen(buf))
		return 0;

	total_len = strlen(buf) + 1;

	tail = buf + total_len - 1;

	for (i = 0; i < A6E_ADDITIONAL_BOARD_MAX; i++) {
		p = strstr(buf, a6e_di8ai4[i]);
		if (p) {
			p_space = strstr(p, " ");
			if (p_space)
				current_len = p_space - p;
			else
				current_len = strlen(p);

			if (p == buf) {
				/* top */
				if (total_len == (current_len + 1)) {
					/* only this dtbo */
					return 0;
				}
				p_src = buf + current_len + 1;
				memmove(p, p_src, total_len - 1 - (current_len + 1));
				total_len -= current_len + 1;
				tail -= current_len + 1;
				*tail = '\0';
			} else if (total_len == ((p - buf) + current_len + 1)) {
				/* last */
				p--;	/* remove space before */
				total_len -= current_len + 1;
				tail -= current_len + 1;
				*tail = '\0';
			} else {
				/* intermediate */
				p_src = p + current_len + 1;
				memmove(p, p_src, total_len - 1 - (p_src - buf));
				total_len -= current_len + 1;
				tail -= current_len + 1;
				*tail = '\0';
			}
		}
	}
	return total_len;
}

static void check_apply_additional_board(struct udevice *i2c_bus)
{
	char *buf = NULL;
	unsigned short product_id;
	unsigned short revision;
	unsigned char i2c_addr;
	struct udevice *i2c_dev = NULL;
	int i;
	int len = 0;
	int ret;
	bool is_continue = true;
	bool found = false;
	bool is_revision3 = false;

	buf = env_get("fdt_overlays");
	if (buf) {
		buf = strdup(buf);
		len = remove_additional_dtbo(buf);
		if (!len) {
			free(buf);
			buf = NULL;
		}
	}

	for (i = 0; i < A6E_ADDITIONAL_BOARD_MAX; i++) {
		i2c_addr = add_board_i2c_addrs[i];

#if CONFIG_DM_I2C
		ret = dm_i2c_probe(i2c_bus, i2c_addr, 0, &i2c_dev);
#else
		ret = i2c_probe(i2c_addr);
#endif
		if (ret)
			break;

		if (!is_vendor_atmark_techno(i2c_addr, i2c_dev))
			break;

		product_id = get_eeprom_data(i2c_addr, i2c_dev, POS_PRODUCT);
		switch (product_id) {
		case PRODUCT_ATMARK_TECHNO_DI8AI4:
			if (len) {
				/* put space for after 2nd dtbo */
				buf = realloc(buf, len + 1);
				snprintf(&buf[len - 1], 2, " ");
				len++;
			}

			if (buf) {
				buf = realloc(buf,
					      len + strlen(a6e_di8ai4[i]) + 1);
				len += snprintf(&buf[len - 1],
						strlen(a6e_di8ai4[i]) + 1,
						"%s", a6e_di8ai4[i]);
			} else {
				buf = strdup(a6e_di8ai4[i]);
				len += strlen(a6e_di8ai4[i]) + 1;
			}
			found = true;
			di8ai4_product_id = product_id;
			di8ai4_boards_count = i + 1;

			revision = get_eeprom_data(i2c_addr, i2c_dev, POS_REVISION);
			switch (revision) {
			case DI8AI4_REV_THREE:
				is_revision3 = true;
			default:
				break;
			}
			break;
		default:
			/* The +Di8+Ai4 board is not recognized unless it is
			 * added in order from the 1st.
			 */
			is_continue = false;
			break;
		}

		if (!is_continue)
			break;
	}

	/* If revision 3 exists when two or more expansion boards
	 * are connected, revision 3 will be corrupted
	 * when an interrupt from the expansion board occurs.
	 */
	if (i > 1 && is_revision3)
		disable_di8ai4_board_irq = true;

	if (len && found)
		env_set("fdt_overlays", buf);

	free(buf);
}

static bool setup_i2c(struct udevice **i2c_bus)
{
	int ret;

#ifdef CONFIG_DM_I2C
	ret = uclass_get_device_by_seq(UCLASS_I2C, I2C_BUS, i2c_bus);
#else
	ret = i2c_set_bus_num(I2C_BUS);
#endif
	if (ret)
		return false;

	return true;
}

static int check_fdt_autodetect(void)
{
	char *autodetect_str = env_get("fdtautodetect");

	if (autodetect_str &&
	    (strcmp(autodetect_str, "yes") == 0)) {
		return 1;
	}

	return 0;
}

int detect_additional_boards(void)
{
	struct udevice *i2c_bus = NULL;

	if (!check_fdt_autodetect())
		return 0;

	if (!setup_i2c(&i2c_bus))
		return 0;

	check_apply_additional_board(i2c_bus);

	return 0;
}
