/*
 * Board functions for Atmark Techno Armadillo-640
 *
 * Copyright (C) 2017 Yasushi SHOJI <yashi@atmark-techno.com>
 *
 * SPDX-License-Identifier:	GPL-2.0+
 *
 */

#include <stdbool.h>
#include <linux/types.h>
#include <asm/types.h>
#include <asm/u-boot.h>
#include <asm/global_data.h>
#include <asm/arch/sys_proto.h>
#include <asm/arch/iomux.h>
#include <asm/arch/mx6-pins.h>
#include <asm/gpio.h>
#include <fuse.h>
#include <mmc.h>
#include <memalign.h>
#include "fec_mxc.h"
#include "sdhc_mxc.h"
#include "board_kotai.h"
#include <rtc.h>
#include <i2c.h>
#include <dm/uclass.h>
#include <power/pfuze3000_pmic.h>

#ifdef CONFIG_A6_SUBBOARD
#include "a6_subboard.h"
#endif
#ifndef CONFIG_A600_BASE
#include "board_addition.h"
#endif

DECLARE_GLOBAL_DATA_PTR;

#define GPIO_LED1 IMX_GPIO_NR(1, 22)
#define GPIO_LED2 IMX_GPIO_NR(1, 23)
#define GPIO_LED3 IMX_GPIO_NR(1, 17)
#define YELLOW_LED IMX_GPIO_NR(1, 18)

#define BOOT_CFG1 IMX_GPIO_NR(3, 10)
#define BOOT_CFG2 IMX_GPIO_NR(3, 16)

#define RTC_SEC_REG_ADDR	0x00
#define RTC_MIN_REG_ADDR	0x01
#define RTC_HR_REG_ADDR		0x02
#define RTC_DAY_REG_ADDR	0x03
#define RTC_DATE_REG_ADDR	0x04
#define RTC_MON_REG_ADDR	0x05
#define RTC_YR_REG_ADDR		0x06

#define RTC_FLAG_REG_ADDR	0x0E
#define RTC_FLAG_BIT_V1F	BIT(0)
#define RTC_FLAG_BIT_V2F	BIT(1)

#define RTC_CTL_REG_ADDR	0x0F
#define RTC_CTL_BIT_RST		BIT(0)

#define RTC_I2C_BUS 1
#define RTC_I2C_ADDR 0x32
#define RV8803_FLAG                     0x0E
#define RV8803_CTRL                     0x0F

#define RV8803_FLAG_AF                  BIT(3)
#define RV8803_FLAG_TF                  BIT(4)
#define RV8803_FLAG_UF                  BIT(5)

#define RV8803_CTRL_EIE                 BIT(2)
#define RV8803_CTRL_AIE                 BIT(3)
#define RV8803_CTRL_TIE                 BIT(4)
#define RV8803_CTRL_UIE                 BIT(5)

#define MAINTENANCE_MODE_GPIO_CON9_1 IMX_GPIO_NR(1, 22)
#define MAINTENANCE_MODE_GPIO_SW1 IMX_GPIO_NR(1, 10)
static iomux_v3_cfg_t maintenance_mode_gpio_pads[] = {
	MX6_PAD_UART2_CTS_B__GPIO1_IO22 | MUX_PAD_CTRL(PAD_CTL_PUS_22K_UP | PAD_CTL_PUE),
	MX6_PAD_JTAG_MOD__GPIO1_IO10 | MUX_PAD_CTRL(PAD_CTL_HYS | PAD_CTL_PUS_100K_DOWN | PAD_CTL_DSE_DISABLE),
};

static void setup_maintenance_sw(void)
{
	imx_iomux_v3_setup_multiple_pads(maintenance_mode_gpio_pads,
					 ARRAY_SIZE(maintenance_mode_gpio_pads));

	gpio_request(MAINTENANCE_MODE_GPIO_CON9_1, "U-Boot key");
	gpio_direction_input(MAINTENANCE_MODE_GPIO_CON9_1);

	gpio_request(MAINTENANCE_MODE_GPIO_SW1, "U-Boot key");
	gpio_direction_input(MAINTENANCE_MODE_GPIO_SW1);
}

static void setup_rtc_disarm_alarm(void)
{
	/* rv8803 interrupt line is connected to ONOFF signal (as well as gpio),
	 * so if we power up from rv8803 alarm the board will power back off
	 * after 5s unless we clear the alarm
	 */
	int ret;

#ifdef CONFIG_DM_I2C
	struct udevice *bus;
	struct udevice *i2c_dev = NULL;
#endif
	uint8_t ctrl, flag;


#ifdef CONFIG_DM_I2C
	ret = uclass_get_device_by_seq(UCLASS_I2C, RTC_I2C_BUS, &bus);
	if (ret) {
		printf("%s: Can't find bus\n", __func__);
		return;
	}
#else
	ret = i2c_set_bus_num(RTC_I2C_BUS);
	if (ret) {
		printf("can't set bus num: %d\n", ret);
		return;
	}
#endif

#ifdef CONFIG_DM_I2C
	ret = dm_i2c_probe(bus, RTC_I2C_ADDR, 0, &i2c_dev);
#else
	ret = i2c_probe(RTC_I2C_ADDR);
#endif
	if (ret) {
		printf("%s: Can't find device id=0x%x\n",
			__func__, RTC_I2C_ADDR);
		return;
	}

#ifdef CONFIG_DM_I2C
	ret = dm_i2c_read(i2c_dev, RV8803_FLAG, &flag, 1);
#else
	ret = i2c_read(RTC_I2C_ADDR, RV8803_FLAG, 1, &flag, 1);
#endif
	if (ret) {
		printf("%s i2c read flag failed, err %d\n", __func__, ret);
		return;
	}
	if (flag & (RV8803_FLAG_AF|RV8803_FLAG_TF|RV8803_FLAG_UF))
		printf("rv8803 rtc woken by interrupt\n");

#ifdef CONFIG_DM_I2C
	ret = dm_i2c_read(i2c_dev, RV8803_CTRL, &ctrl, 1);
#else
	ret = i2c_read(RTC_I2C_ADDR, RV8803_CTRL, 1, &ctrl, 1);
#endif
	if (ret) {
		printf("%s i2c read ctrl failed, err %d\n", __func__, ret);
		return;
	}
	ctrl &= ~(RV8803_CTRL_EIE|RV8803_CTRL_AIE|RV8803_CTRL_TIE|RV8803_CTRL_UIE);
#ifdef CONFIG_DM_I2C
	ret = dm_i2c_write(i2c_dev, RV8803_CTRL, (const uint8_t *)&ctrl, 1);
#else
	ret = i2c_write(RTC_I2C_ADDR, RV8803_CTRL, 1, &ctrl, 1);
#endif
	if (ret) {
		printf("%s i2c write ctrl failed, err %d\n", __func__, ret);
		return;
	}

	/* assert reset */
	ret = dm_i2c_reg_write(i2c_dev, RTC_CTL_REG_ADDR, RTC_CTL_BIT_RST);
	if (ret < 0)
		return;

	/* clear all flags */
	ret = dm_i2c_reg_write(i2c_dev, RTC_FLAG_REG_ADDR, 0);
	if (ret < 0)
		return;

	/* clear reset */
	ret = dm_i2c_reg_write(i2c_dev, RTC_CTL_REG_ADDR, 0);
	if (ret < 0)
		return;
}

int board_mmc_get_env_part(int devno)
{
	struct mmc *mmc = find_mmc_device(devno);
	int part = EXT_CSD_EXTRACT_BOOT_PART(mmc->part_config);

	if (part > 2)
		/* Fallback to user partition if not boot0/boot1 */
		part = 0;

	return part;
}

int board_init(void)
{
	setup_maintenance_sw();
	setup_fec();
	setup_rtc_disarm_alarm();

	return 0;
}

static int check_mmc_autodetect(void)
{
	char *autodetect_str = env_get("mmcautodetect");

	if ((autodetect_str != NULL) &&
			(strcmp(autodetect_str, "yes") == 0)) {
		return 1;
	}

	return 0;
}

void at_board_late_mmc_env_init(void) {
	char buf[100];
	int part = 1;
	u32 dev_no = mmc_get_env_dev();

	if (!check_mmc_autodetect())
		return;

	/* evk board:
	 * emmc    -> dev_no = 0, ext csd bootpart = 1/2
	 * sd card -> dev_no = 1, ext csd bootpart = 7
	 *
	 * atmark image has kernel/dtb in p1/2 and rootfs in p1/2 as well,
	 * so use part as is directly for both
	 */
	if (dev_no == 0) {
		/* emmc, get boot partition from extcsd */
		struct mmc *mmc;

		mmc = find_mmc_device(dev_no);

		if (!mmc) {
			printf("booting on emmc but no mmc device found, skipping mmcautodetect\n");
			return;
		}
		if (!mmc->has_init) {
			printf("mmc dev found but hasn't been init, skipping mmcautodetect\n");
			return;
		}

		part = EXT_CSD_EXTRACT_BOOT_PART(mmc->part_config);
	} else {
		/* sd card, get boot partition from env */
		part = env_get_ulong("mmcpart", 10, 1);
	}

	/* device where to load linux from */
	snprintf(buf, sizeof(buf), "%d", dev_no);
	env_set("mmcdev", buf);
	/* partition index where linux/dtb are stored */
	snprintf(buf, sizeof(buf), "%d", part);
	env_set("mmcpart", buf);
	/* kernel boot command */
	snprintf(buf, sizeof(buf), "/dev/mmcblk%dp%d rootwait ro",
		 mmc_map_to_kernel_blk(dev_no), part);
	env_set("mmcroot", buf);

	snprintf(buf, sizeof(buf), "mmc dev %d", dev_no);
	run_command(buf, 0);
}

#ifdef CONFIG_A600_BASE
static int check_fdt_autodetect(void)
{
	char *autodetect_str = env_get("fdtautodetect");

	if ((autodetect_str != NULL) &&
	    (strcmp(autodetect_str, "yes") == 0)) {
		return 1;
	}

	return 0;
}

static void set_fdt_file(uint32_t product_id_val)
{
	if (!check_fdt_autodetect())
		return;

	/* force overwrite */
	switch (product_id_val) {
	case 0x0097: /* fallthrough */
	case 0x009C: /* Armadillo-640 */
		env_set("fdt_file", "boot/armadillo-640.dtb");
		break;
	case 0x00B4: /* fallthrough */
	case 0x00B7: /* Armadillo-610 */
		env_set("fdt_file", "boot/armadillo-610.dtb");
		break;
	default: /* unknown board */
		printf("%s: unknown board.\n", __func__);
		break;
	}
}
#endif

static void set_board_info(uint32_t *product_id_val)
{
	uint32_t lot_val;
	char serial[13 /* product (4), lot (8), \0 (1) */];
	char *old_serial;
	int ret;

	ret = fuse_read(EFUSE_KOTAI_BANK, EFUSE_KOTAI_WORD, product_id_val);
	if (ret) {
		printf("Error: Can not read seihin kanri id\n");
		return;
	}

	ret = fuse_read(EFUSE_KOTAI_BANK, EFUSE_KOTAI_WORD + 1, &lot_val);
	if (ret) {
		printf("Error: Can not read lot number\n");
		return;
	}

	if (*product_id_val == 0 && lot_val == 0) {
		printf("Error: serial number not written in fuses\n");
		return;
	}

	snprintf(serial, sizeof(serial), "%04X%08X", *product_id_val, lot_val);

	old_serial = env_get("serial#");
	if (old_serial && old_serial[0] != '\0' && strcmp(serial, old_serial)) {
		/* board previously booted somewhere else, clear mac addresses */
		printf("serial# changed: Clearing MAC addresses in env\n");
		env_set("ethaddr", NULL);
	}

	env_set("serial#", serial);
}

int board_late_init(void)
{
	uint32_t product_id_val;

#ifdef CONFIG_ENV_IS_IN_MMC
	at_board_late_mmc_env_init();
#endif
#ifdef CONFIG_A6_SUBBOARD
	a6_subboard_late_init();
#endif
	set_board_info(&product_id_val);
#ifdef CONFIG_A600_BASE
	set_fdt_file(product_id_val);
#endif

#ifndef CONFIG_A600_BASE
	/* A6E only */
	detect_additional_boards();
#endif

	return 0;
}

int dram_init(void)
{
	gd->ram_size = imx_ddr_size();
	return 0;
}

#ifdef CONFIG_BOOTSTAGE
extern unsigned long __weak timer_read_counter(void);
ulong timer_get_boot_us(void)
{
	return timer_read_counter();
}
#endif

#if defined(CONFIG_SUPPORT_MAINTENANCE_MODE)
int is_maintenance_mode(void)
{
	return !gpio_get_value(MAINTENANCE_MODE_GPIO_CON9_1) || gpio_get_value(MAINTENANCE_MODE_GPIO_SW1);
}
#endif
