/* SPDX-License-Identifier: GPL-2.0+ */
/*
 * (C) Copyright 2015-2016 Freescale Semiconductor, Inc.
 * (C) Copyright 2017-2019,2020 NXP
 * (C) Copyright 2017 MicroSys Electronics GmbH
 *
 */

#ifndef __ARCH_ARM_MACH_S32V234_SIUL_H__
#define __ARCH_ARM_MACH_S32V234_SIUL_H__

#include <asm/io.h>
#include "ddr.h"

#define SIUL2_MIDR1				(SIUL2_BASE_ADDR + 0x00000004)
#define SIUL2_MIDR2				(SIUL2_BASE_ADDR + 0x00000008)
#define SIUL2_DISR0				(SIUL2_BASE_ADDR + 0x00000010)
#define SIUL2_DIRER0				(SIUL2_BASE_ADDR + 0x00000018)
#define SIUL2_DIRSR0				(SIUL2_BASE_ADDR + 0x00000020)
#define SIUL2_IREER0				(SIUL2_BASE_ADDR + 0x00000028)
#define SIUL2_IFEER0				(SIUL2_BASE_ADDR + 0x00000030)
#define SIUL2_IFER0				(SIUL2_BASE_ADDR + 0x00000038)

#define SIUL2_IFMCR_BASE			(SIUL2_BASE_ADDR + 0x00000040)
#define SIUL2_IFMCRn(i)				(SIUL2_IFMCR_BASE + 4 * (i))

#define SIUL2_IFCPR				(SIUL2_BASE_ADDR + 0x000000C0)

/* SIUL2_MSCR specifications as stated in Reference Manual:
 * 0 - 359 Output Multiplexed Signal Configuration Registers
 * 512- 1023 Input Multiplexed Signal Configuration Registers */
#define SIUL2_MSCR_BASE			(SIUL2_BASE_ADDR + 0x00000240)
#define SIUL2_MSCRn(i)			(SIUL2_MSCR_BASE + 4 * (i))

#define SIUL2_IMCR_BASE			(SIUL2_BASE_ADDR + 0x00000A40)
#define SIUL2_IMCRn(i)			(SIUL2_IMCR_BASE +  4 * (i))

#define SIUL2_GPDO_BASE			(SIUL2_BASE_ADDR + 0x00001300)
#define SIUL2_GPDOn(i)			(SIUL2_GPDO_BASE + (i))

#define SIUL2_GPDI_BASE			(SIUL2_BASE_ADDR + 0x00001500)
#define SIUL2_GPDIn(i)			(SIUL2_GPDI_BASE + (i))

#define SIUL2_PGPDO_BASE		(SIUL2_BASE_ADDR + 0x00001700)
#define SIUL2_PGPDOn(i)			(SIUL2_PGPDO_BASE + \
						(((i) / 2) * 4) + \
						(((i) % 2) ? 0 : 2))

#define SIUL2_PGPDI_BASE		(SIUL2_BASE_ADDR + 0x00001740)
#define SIUL2_PGPDIn(i)			(SIUL2_PGPDI_BASE + \
						(((i) / 2) * 4) + \
						(((i) % 2) ? 0 : 2))

#define SIUL2_MPGPDO_BASE		(SIUL2_BASE_ADDR + 0x00001780)
#define SIUL2_MPGPDOn(i)		(SIUL2_MPGPDO_BASE + 4 * (i))

/* GPIO */
/* 163 GPIOs in output mode, we assume the GPIO number is in range */
#define SIUL2_GPDO_for_GPIO(i)		(((i) & (~0x3)) >> 2)
#define SIUL2_GPDO_PDO_off_for_GPIO(i)	(((i) & (0x3))
#define SIUL2_PDOn(i)			(SIUL2_GPDOn(SIUL2_GPDO_for_GPIO(i) + \
						SIUL2_GPDO_PDO_off_for_GPIO(i))
#define SIUL2_GPIO_VALUE0		(0x00)
#define SIUL2_GPIO_VALUE1		(0x01)

/* SIUL2_MSCR masks */
#define SIUL2_MSCR_DDR_DO_TRIM(v)	((v) & 0xC0000000)
#define SIUL2_MSCR_DDR_DO_TRIM_MIN	(0 << 30)
#define SIUL2_MSCR_DDR_DO_TRIM_50PS	(1 << 30)
#define SIUL2_MSCR_DDR_DO_TRIM_100PS	(2 << 30)
#define SIUL2_MSCR_DDR_DO_TRIM_150PS	(3 << 30)

#define SIUL2_MSCR_DDR_INPUT(v)		((v) & 0x20000000)
#define SIUL2_MSCR_DDR_INPUT_CMOS	(0 << 29)
#define SIUL2_MSCR_DDR_INPUT_DIFF_DDR	(1 << 29)

#define SIUL2_MSCR_DDR_SEL(v)		((v) & 0x18000000)
#define SIUL2_MSCR_DDR_SEL_DDR3		(0 << 27)
#define SIUL2_MSCR_DDR_SEL_LPDDR2	(2 << 27)

#define SIUL2_MSCR_DDR_ODT(v)		((v) & 0x07000000)
#define SIUL2_MSCR_DDR_ODT_120ohm	(1 << 24)
#define SIUL2_MSCR_DDR_ODT_60ohm	(2 << 24)
#define SIUL2_MSCR_DDR_ODT_40ohm	(3 << 24)
#define SIUL2_MSCR_DDR_ODT_30ohm	(4 << 24)
#define SIUL2_MSCR_DDR_ODT_24ohm	(5 << 24)
#define SIUL2_MSCR_DDR_ODT_20ohm	(6 << 24)
#define SIUL2_MSCR_DDR_ODT_17ohm	(7 << 24)

#define SIUL2_MSCR_DCYCLE_TRIM(v)	((v) & 0x00C00000)
#define SIUL2_MSCR_DCYCLE_TRIM_NONE	(0 << 22)
#define SIUL2_MSCR_DCYCLE_TRIM_LEFT	(1 << 22)
#define SIUL2_MSCR_DCYCLE_TRIM_RIGHT	(2 << 22)

#define SIUL2_MSCR_OBE(v)		((v) & 0x00200000)
#define SIUL2_MSCR_OBE_EN		(1 << 21)

#define SIUL2_MSCR_ODE(v)		((v) & 0x00100000)
#define SIUL2_MSCR_ODE_EN		(1 << 20)

#define SIUL2_MSCR_IBE(v)		((v) & 0x00010000)
#define SIUL2_MSCR_IBE_EN		(1 << 19)

#define SIUL2_MSCR_HYS(v)		((v) & 0x00400000)
#define SIUL2_MSCR_HYS_EN		(1 << 18)

#define SIUL2_MSCR_INV(v)		((v) & 0x00020000)
#define SIUL2_MSCR_INV_EN		(1 << 17)

#define SIUL2_MSCR_PKE(v)		((v) & 0x00010000)
#define SIUL2_MSCR_PKE_EN		(1 << 16)

#define SIUL2_MSCR_SRE(v)		((v) & 0x0000C000)
#define SIUL2_MSCR_SRE_SPEED_LOW_50	(0 << 14)
#define SIUL2_MSCR_SRE_SPEED_LOW_100	(1 << 14)
#define SIUL2_MSCR_SRE_SPEED_HIGH_100	(2 << 14)
#define SIUL2_MSCR_SRE_SPEED_HIGH_200	(3 << 14)

#define SIUL2_MSCR_PUE(v)		((v) & 0x00002000)
#define SIUL2_MSCR_PUE_EN		(1 << 13)

#define SIUL2_MSCR_PUS(v)		((v) & 0x00001800)
#define SIUL2_MSCR_PUS_100K_DOWN	(0 << 11)
#define SIUL2_MSCR_PUS_50K_UP		BIT(11)
#define SIUL2_MSCR_PUS_100K_UP		(2 << 11)
#define SIUL2_MSCR_PUS_33K_UP		(3 << 11)

#define SIUL2_MSCR_DSE(v)		((v) & 0x00000700)
#define SIUL2_MSCR_DSE_240ohm		(1 << 8)
#define SIUL2_MSCR_DSE_120ohm		(2 << 8)
#define SIUL2_MSCR_DSE_80ohm		(3 << 8)
#define SIUL2_MSCR_DSE_60ohm		(4 << 8)
#define SIUL2_MSCR_DSE_48ohm		(5 << 8)
#define SIUL2_MSCR_DSE_40ohm		(6 << 8)
#define SIUL2_MSCR_DSE_34ohm		(7 << 8)

#define SIUL2_MSCR_CRPOINT_TRIM(v)	((v) & 0x000000C0)
#define SIUL2_MSCR_CRPOINT_TRIM_1	(1 << 6)

#define SIUL2_MSCR_SMC(v)		((v) & 0x00000020)
#define SIUL2_MSCR_MUX_MODE(v)		((v) & 0x0000000f)
#define SIUL2_MSCR_MUX_MODE_ALT0	(0x0)
#define SIUL2_MSCR_MUX_MODE_ALT1	(0x1)
#define SIUL2_MSCR_MUX_MODE_ALT2	(0x2)
#define SIUL2_MSCR_MUX_MODE_ALT3	(0x3)

/* UART settings */

/* UART0 */
/* TXD */
#define SIUL2_MSCR_PA12		12
/* RXD */
#define SIUL2_MSCR_PA11		11
#define SIUL2_IMCR_UART0_RXD	200

/* UART1 */
/* TXD */
#define SIUL2_MSCR_PA14		14
/* RXD */
#define SIUL2_MSCR_PA13		13
#define SIUL2_IMCR_UART1_RXD	202

/* UART MSCR settings */
#define SIUL2_MSCR_PORT_CTRL_UART_TXD	\
	(SIUL2_MSCR_OBE_EN |		\
	 SIUL2_MSCR_PUS_100K_UP |	\
	 SIUL2_MSCR_DSE_60ohm |		\
	 SIUL2_MSCR_SRE_SPEED_LOW_100 |	\
	 SIUL2_MSCR_MUX_MODE_ALT1)
#define SIUL2_MSCR_PORT_CTRL_UART_RXD	\
	(SIUL2_MSCR_PUE_EN |		\
	 SIUL2_MSCR_IBE_EN |		\
	 SIUL2_MSCR_DCYCLE_TRIM_RIGHT)

/* UART IMCR mux modes */
#define SIUL2_IMCR_UART_RXD_to_pad	(SIUL2_MSCR_MUX_MODE_ALT2)

/* uSDHC settings */
#define SIUL2_USDHC_PAD_CTRL_BASE	(SIUL2_MSCR_SRE_SPEED_HIGH_200 | SIUL2_MSCR_OBE_EN |	\
						SIUL2_MSCR_DSE_34ohm | SIUL2_MSCR_PKE_EN | SIUL2_MSCR_IBE_EN |		\
						SIUL2_MSCR_PUS_100K_UP | SIUL2_MSCR_PUE_EN )
#define SIUL2_USDHC_PAD_CTRL_CMD	(SIUL2_USDHC_PAD_CTRL_BASE | SIUL2_MSCR_MUX_MODE_ALT1)
#define SIUL2_USDHC_PAD_CTRL_CLK	(SIUL2_USDHC_PAD_CTRL_BASE | SIUL2_MSCR_MUX_MODE_ALT2)
#define SIUL2_USDHC_PAD_CTRL_DAT0_3	(SIUL2_USDHC_PAD_CTRL_BASE | SIUL2_MSCR_MUX_MODE_ALT2)
#define SIUL2_USDHC_PAD_CTRL_DAT4_7	(SIUL2_USDHC_PAD_CTRL_BASE | SIUL2_MSCR_MUX_MODE_ALT3)

/* I2C settings */
#define SIUL2_MSCR_PA15			15
#define SIUL2_MSCR_PB0			16
#define SIUL2_MSCR_PB1			17
#define SIUL2_MSCR_PB2			18
#define SIUL2_MSCR_PB3			19
#define SIUL2_MSCR_PB4			20
#define SIUL2_MSCR_PG3			99
#define SIUL2_MSCR_PG4			100
#define SIUL2_MSCR_PG5			101
#define SIUL2_MSCR_PG6			102

#define SIUL2_IMCR_I2C0_DATA		269
#define SIUL2_IMCR_I2C0_CLK		268
#define SIUL2_IMCR_I2C1_DATA		271
#define SIUL2_IMCR_I2C1_CLK		270
#define SIUL2_IMCR_I2C2_DATA		273
#define SIUL2_IMCR_I2C2_CLK		272

/* I2C0 - Serial Data Input AC15 */
#define SIUL2_PAD_CTRL_I2C_COMMON \
	(SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_IBE_EN | \
	 SIUL2_MSCR_ODE_EN | \
	 SIUL2_MSCR_DSE_34ohm)

#define SIUL2_PAD_CTRL_I2C0_MSCR_SDA_AC15 \
	(SIUL2_MSCR_MUX_MODE_ALT2 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C0_IMCR_SDA_AC15	(SIUL2_MSCR_MUX_MODE_ALT3)

/* I2C0 - Serial Clock Input AE15 */
#define SIUL2_PAD_CTRL_I2C0_MSCR_SCLK_AE15 \
	(SIUL2_MSCR_MUX_MODE_ALT2 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C0_IMCR_SCLK_AE15	(SIUL2_MSCR_MUX_MODE_ALT3)

/* I2C0 - Serial Data Input F11 */
#define SIUL2_PAD_CTRL_I2C0_MSCR_SDA_F11 \
	(SIUL2_MSCR_MUX_MODE_ALT1 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C0_IMCR_SDA_F11	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C0 - Serial Clock Input F12 */
#define SIUL2_PAD_CTRL_I2C0_MSCR_SCLK_F12 \
	(SIUL2_MSCR_MUX_MODE_ALT1 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C0_IMCR_SCLK_F12	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C1 - Serial Data Input */
#define SIUL2_PAD_CTRL_I2C1_MSCR_SDA \
	(SIUL2_MSCR_MUX_MODE_ALT2 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C1_IMCR_SDA	(SIUL2_MSCR_MUX_MODE_ALT3)

/* I2C1 - Serial Clock Input */
#define SIUL2_PAD_CTRL_I2C1_MSCR_SCLK \
	(SIUL2_MSCR_MUX_MODE_ALT2 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C1_IMCR_SCLK	(SIUL2_MSCR_MUX_MODE_ALT3)

/* I2C2 - Serial Data Input */
#define SIUL2_PAD_CTRL_I2C2_MSCR_SDA \
	(SIUL2_MSCR_MUX_MODE_ALT1 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C2_IMCR_SDA	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C2 - Serial Clock Input */
#define SIUL2_PAD_CTRL_I2C2_MSCR_SCLK \
	(SIUL2_MSCR_MUX_MODE_ALT1 | \
	 SIUL2_PAD_CTRL_I2C_COMMON)
#define SIUL2_PAD_CTRL_I2C2_IMCR_SCLK	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C settings MPXS32V234-R1 */
/* I2C0 - Serial Data Input */
#define SIUL2_PAD_CTRL_I2C0_MSCR1_SDA		\
		(SIUL2_MSCR_MUX_MODE_ALT1 |	\
		 SIUL2_MSCR_OBE_EN |		\
		 SIUL2_MSCR_IBE_EN |		\
		 SIUL2_MSCR_ODE_EN |		\
		 SIUL2_MSCR_DSE_34ohm)
#define SIUL2_PAD_CTRL_I2C0_IMCR1_SDA	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C0 - Serial Clock Input */
#define SIUL2_PAD_CTRL_I2C0_MSCR1_SCLK		\
		(SIUL2_MSCR_MUX_MODE_ALT1 |	\
		 SIUL2_MSCR_OBE_EN |		\
		 SIUL2_MSCR_IBE_EN |		\
		 SIUL2_MSCR_ODE_EN |		\
		 SIUL2_MSCR_DSE_34ohm)
#define SIUL2_PAD_CTRL_I2C0_IMCR1_SCLK	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C1 - Serial Data Input */
#define SIUL2_PAD_CTRL_I2C1_MSCR1_SDA		\
		(SIUL2_MSCR_MUX_MODE_ALT1 |	\
		 SIUL2_MSCR_OBE_EN |		\
		 SIUL2_MSCR_IBE_EN |		\
		 SIUL2_MSCR_ODE_EN |		\
		 SIUL2_MSCR_DSE_34ohm)
#define SIUL2_PAD_CTRL_I2C1_IMCR1_SDA	(SIUL2_MSCR_MUX_MODE_ALT2)

/* I2C1 - Serial Clock Input */
#define SIUL2_PAD_CTRL_I2C1_MSCR1_SCLK		\
		(SIUL2_MSCR_MUX_MODE_ALT1 |	\
		 SIUL2_MSCR_OBE_EN |		\
		 SIUL2_MSCR_IBE_EN |		\
		 SIUL2_MSCR_ODE_EN |		\
		 SIUL2_MSCR_DSE_34ohm)
#define SIUL2_PAD_CTRL_I2C1_IMCR1_SCLK	(SIUL2_MSCR_MUX_MODE_ALT2)

/* ENET settings */

#define SIUL2_MSCR_PC13		45
#define SIUL2_MSCR_PC14		46
#define SIUL2_MSCR_PC14_IN	981

#define SIUL2_MSCR_PC15_SWITCH	47
#define SIUL2_MSCR_PC15		47
#define SIUL2_MSCR_PC15_IN	978

#define SIUL2_MSCR_PD0		48
#define SIUL2_MSCR_PD0_IN	979

#define SIUL2_MSCR_PD1		49
#define SIUL2_MSCR_PD1_IN	974

#define SIUL2_MSCR_PD2		50
#define SIUL2_MSCR_PD2_IN	975

#define SIUL2_MSCR_PD3		51
#define SIUL2_MSCR_PD3_IN	976

#define SIUL2_MSCR_PD4		52
#define SIUL2_MSCR_PD4_IN	977

#define SIUL2_MSCR_PD5		53
#define SIUL2_MSCR_PD5_IN	973

#define SIUL2_MSCR_PD6		54
#define SIUL2_MSCR_PD6_IN	970

#define SIUL2_MSCR_PD7		55
#define SIUL2_MSCR_PD8		56
#define SIUL2_MSCR_PD9		57
#define SIUL2_MSCR_PD10		58
#define SIUL2_MSCR_PD11		59

/* ENET - SIUL2_MSCR_ENET_MDC = 0x20c701 */
#define SIUL2_MSCR_ENET_MDC \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 |\
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_MDIO = 0x28c701 */
#define SIUL2_MSCR_ENET_MDIO \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_IBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_MDIO_IN = 0x00000002 */
#define SIUL2_MSCR_ENET_MDIO_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_TX_CLK_SWITCH = 0x20c701 */
#define SIUL2_MSCR_ENET_TX_CLK_SWITCH \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_TX_CLK = 0x00203701 */
#define SIUL2_MSCR_ENET_TX_CLK \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_PUS_100K_UP | \
	 SIUL2_MSCR_PUE_EN | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_TX_CLK_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_TX_CLK_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_CLK = 0x8c700 */
#define SIUL2_MSCR_ENET_RX_CLK \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_CLK_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_RX_CLK_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_D0 = 0x8c700 */
#define SIUL2_MSCR_ENET_RX_D0 \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_D0_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_RX_D0_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_D1 = 0x8c700 */
#define SIUL2_MSCR_ENET_RX_D1 \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_D1_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_RX_D1_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_D2 = 0x8c700 */
#define SIUL2_MSCR_ENET_RX_D2 \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_D2_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_RX_D2_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_D3 = 0x8c700 */
#define SIUL2_MSCR_ENET_RX_D3 \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_D3_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_RX_D3_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_DV = 0x8c700 */
#define SIUL2_MSCR_ENET_RX_DV \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_DV_IN = 0x00000002*/
#define SIUL2_MSCR_ENET_RX_DV_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_RX_ER = 0x0008c700 */
#define SIUL2_MSCR_ENET_RX_ER \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* ENET - SIUL2_MSCR_ENET_RX_ER_IN = 0x00000002 */
#define SIUL2_MSCR_ENET_RX_ER_IN	SIUL2_MSCR_MUX_MODE_ALT2

/* ENET - SIUL2_MSCR_ENET_TX_D0 = 0x20c701 */
#define SIUL2_MSCR_ENET_TX_D0 \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_TX_D1 = 0x20c701 */
#define SIUL2_MSCR_ENET_TX_D1 \
	(SIUL2_MSCR_DSE_34ohm |	\
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_TX_D2 = 0x20c701 */
#define SIUL2_MSCR_ENET_TX_D2 \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_TX_D3 = 0x20c701 */
#define SIUL2_MSCR_ENET_TX_D3 \
	(SIUL2_MSCR_DSE_34ohm |	\
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_TX_EN = 0x20c701 */
#define SIUL2_MSCR_ENET_TX_EN \
	(SIUL2_MSCR_DSE_34ohm |	\
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_OBE_EN | \
	 SIUL2_MSCR_MUX_MODE_ALT1)

/* ENET - SIUL2_MSCR_ENET_RMII_CLK_REF_IP = 0x0008c700 */
#define SIUL2_MSCR_ENET_RMII_CLK_REF_IP \
	(SIUL2_MSCR_DSE_34ohm | \
	 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
	 SIUL2_MSCR_IBE_EN)

/* QuadSPI settings */

#define SIUL2_PK6_MSCR	150
#define SIUL2_PK6_MSCR_MUX_MODE_QSPI_A_SCK	SIUL2_MSCR_MUX_MODE_ALT1

#define SIUL2_PK5_MSCR	149
#define SIUL2_PK5_MSCR_MUX_MODE_QSPI_A_CS0	SIUL2_MSCR_MUX_MODE_ALT1

#define SIUL2_PK13_MSCR	157
#define SIUL2_PK13_MSCR_MUX_MODE_QSPI_B_SCK	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PK13_MSCR_MUX_MODE_QSPI_CK2	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PK12_MSCR	156
#define SIUL2_PK12_MSCR_MUX_MODE_QSPI_B_CS0	SIUL2_MSCR_MUX_MODE_ALT1

#define SIUL2_PORT_MSCR_CTRL_QSPI_AB_DATA_BASE	\
		(SIUL2_MSCR_DSE_34ohm |		\
		SIUL2_MSCR_SRE_SPEED_HIGH_200 |	\
		SIUL2_MSCR_IBE_EN |		\
		SIUL2_MSCR_OBE_EN)
#define SIUL2_PORT_MSCR_CTRL_QSPI_A_DATA0_3	\
		(SIUL2_MSCR_MUX_MODE_ALT1 |	\
		SIUL2_PORT_MSCR_CTRL_QSPI_AB_DATA_BASE)
#define SIUL2_PORT_MSCR_CTRL_QSPI_A_DATA4_7	\
		(SIUL2_MSCR_MUX_MODE_ALT2 |	\
		SIUL2_PORT_MSCR_CTRL_QSPI_AB_DATA_BASE)

#define SIUL2_PORT_MSCR_CTRL_QSPI_B_DATA0_3	\
		(SIUL2_MSCR_MUX_MODE_ALT1 |	\
		SIUL2_PORT_MSCR_CTRL_QSPI_AB_DATA_BASE)

#define SIUL2_PORT_IMCR_MUX_MODE_QSPI_A_DATA0_7	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PORT_IMCR_MUX_MODE_QSPI_B_DATA0_3	SIUL2_MSCR_MUX_MODE_ALT2

/* 0x0009E000 + SIUL2_MSCR_PUS_100K_DOWN */
#define SIUL2_PORT_MSCR_CTRL_QSPI_A_DQS	\
		(SIUL2_MSCR_IBE_EN |	\
		SIUL2_MSCR_PKE_EN |	\
		SIUL2_MSCR_SRE_SPEED_HIGH_200 |	\
		SIUL2_MSCR_PUE_EN |	\
		SIUL2_MSCR_PUS_100K_DOWN)

#define SIUL2_PORT_MSCR_CTRL_QSPI_CLK_BASE	\
		(	\
		SIUL2_MSCR_DSE_34ohm |	\
		SIUL2_MSCR_PUS_100K_UP |	\
		SIUL2_MSCR_SRE_SPEED_HIGH_200 |	\
		SIUL2_MSCR_OBE_EN)

#define SIUL2_PK11_MSCR	155
#define SIUL2_PK11_MSCR_MUX_MODE_QSPI_A_DATA3	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PK11_IMCR_QSPI_A_DATA3	(823 - 512)
#define SIUL2_PK11_IMCR_MUX_MODE_QSPI_A_DATA3	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PK10_MSCR	154
#define SIUL2_PK10_MSCR_MUX_MODE_QSPI_A_DATA2	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PK10_IMCR_QSPI_A_DATA2	(822 - 512)
#define SIUL2_PK10_IMCR_MUX_MODE_QSPI_A_DATA2	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PK9_MSCR	153
#define SIUL2_PK9_MSCR_MUX_MODE_QSPI_A_DATA1	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PK9_IMCR_QSPI_A_DATA1	(821 - 512)
#define SIUL2_PK9_IMCR_MUX_MODE_QSPI_A_DATA1	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PK8_MSCR	152
#define SIUL2_PK8_MSCR_MUX_MODE_QSPI_A_DATA0	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PK8_IMCR_QSPI_A_DATA0	(820 - 512)
#define SIUL2_PK8_IMCR_MUX_MODE_QSPI_A_DATA0	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PK7_MSCR	151
#define SIUL2_PK7_IMCR_QSPI_A_DQS	(819 - 512)
#define SIUL2_PK7_IMCR_MUX_MODE_QSPI_A_DQS	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PL2_MSCR	162
#define SIUL2_PL2_MSCR_MUX_MODE_QSPI_B_DATA3	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PL2_MSCR_MUX_MODE_QSPI_A_DATA7	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PL2_IMCR_QSPI_A_DATA7	(827 - 512)
#define SIUL2_PL2_IMCR_QSPI_B_DATA3	(832 - 512)
#define SIUL2_PL2_IMCR_MUX_MODE_QSPI_A_DATA7	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PL2_IMCR_MUX_MODE_QSPI_B_DATA3	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PL1_MSCR	161
#define SIUL2_PL1_MSCR_MUX_MODE_QSPI_B_DATA2	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PL1_MSCR_MUX_MODE_QSPI_A_DATA6	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PL1_IMCR_QSPI_A_DATA6	(826 - 512)
#define SIUL2_PL1_IMCR_QSPI_B_DATA2	(831 - 512)
#define SIUL2_PL1_IMCR_MUX_MODE_QSPI_A_DATA6	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PL1_IMCR_MUX_MODE_QSPI_B_DATA2	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PL0_MSCR	160
#define SIUL2_PL0_MSCR_MUX_MODE_QSPI_B_DATA1	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PL0_MSCR_MUX_MODE_QSPI_A_DATA5	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PL0_IMCR_QSPI_A_DATA5	(825 - 512)
#define SIUL2_PL0_IMCR_QSPI_B_DATA1	(830 - 512)
#define SIUL2_PL0_IMCR_MUX_MODE_QSPI_A_DATA5	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PL0_IMCR_MUX_MODE_QSPI_B_DATA1	SIUL2_MSCR_MUX_MODE_ALT2

#define SIUL2_PK15_MSCR	159
#define SIUL2_PK15_MSCR_MUX_MODE_QSPI_B_DATA0	SIUL2_MSCR_MUX_MODE_ALT1
#define SIUL2_PK15_MSCR_MUX_MODE_QSPI_A_DATA4	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PK15_IMCR_QSPI_A_DATA4	(824 - 512)
#define SIUL2_PK15_IMCR_QSPI_B_DATA0	(829 - 512)
#define SIUL2_PK15_IMCR_MUX_MODE_QSPI_A_DATA4	SIUL2_MSCR_MUX_MODE_ALT2
#define SIUL2_PK15_IMCR_MUX_MODE_QSPI_B_DATA0	SIUL2_MSCR_MUX_MODE_ALT2

/* DSPI Settings */

#define SIUL2_MSCR_PB5  21
#define SIUL2_MSCR_PB6  22
#define SIUL2_MSCR_PB7  23
#define SIUL2_MSCR_PB8  24
#define SIUL2_MSCR_PC0  32
#define SIUL2_MSCR_PC1  33
#define SIUL2_MSCR_PC2  34
#define SIUL2_MSCR_PC3  35

#define SIUL2_PAD_CTRL_MSCR_CSx	 \
				(SIUL2_MSCR_OBE_EN | SIUL2_MSCR_PUS_100K_UP | \
				 SIUL2_MSCR_DSE_34ohm | SIUL2_MSCR_PUE_EN)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_CS0_OUT	(SIUL2_PAD_CTRL_MSCR_CSx | \
				SIUL2_MSCR_MUX_MODE_ALT1)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_CS4_OUT	(SIUL2_PAD_CTRL_MSCR_CSx | \
				SIUL2_MSCR_MUX_MODE_ALT3)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_CS5_OUT	(SIUL2_PAD_CTRL_MSCR_CSx | \
				SIUL2_MSCR_MUX_MODE_ALT3)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_CS6_OUT	(SIUL2_PAD_CTRL_MSCR_CSx | \
				SIUL2_MSCR_MUX_MODE_ALT3)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_CS7_OUT	(SIUL2_PAD_CTRL_MSCR_CSx | \
				SIUL2_MSCR_MUX_MODE_ALT2)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_SOUT_OUT	(SIUL2_MSCR_OBE_EN | \
				SIUL2_MSCR_DSE_34ohm | SIUL2_MSCR_MUX_MODE_ALT1)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_SCK_OUT	(SIUL2_MSCR_OBE_EN | \
				SIUL2_MSCR_DSE_34ohm | SIUL2_MSCR_MUX_MODE_ALT1)

#define SIUL2_PAD_CTRL_DSPI0_MSCR_SIN_OUT	(SIUL2_MSCR_PUE_EN | \
				SIUL2_MSCR_IBE_EN | SIUL2_MSCR_PUS_50K_UP)

#define SIUL2_PB7_IMCR_SPI0_SIN    (800 - 512)

#define SIUL2_PAD_CTRL_DSPI0_IMCR_SIN_IN	(SIUL2_MSCR_MUX_MODE_ALT2)

#ifdef CONFIG_FSL_DCU_FB

/* DCU Settings */

#define SIUL2_MSCR_PH8  120
#define SIUL2_MSCR_PH9  121
#define SIUL2_MSCR_PH10 122
#define SIUL2_MSCR_PH12 124
#define SIUL2_MSCR_PH13 125
#define SIUL2_MSCR_PH14 126
#define SIUL2_MSCR_PH15 127
#define SIUL2_MSCR_PJ0  128
#define SIUL2_MSCR_PJ1  129
#define SIUL2_MSCR_PJ2  130
#define SIUL2_MSCR_PJ3  131
#define SIUL2_MSCR_PJ4  132
#define SIUL2_MSCR_PJ5  133
#define SIUL2_MSCR_PJ6  134
#define SIUL2_MSCR_PJ7  135
#define SIUL2_MSCR_PJ8  136
#define SIUL2_MSCR_PJ9  137
#define SIUL2_MSCR_PJ10 138
#define SIUL2_MSCR_PJ11 139
#define SIUL2_MSCR_PJ12 140
#define SIUL2_MSCR_PJ13 141
#define SIUL2_MSCR_PJ14 142
#define SIUL2_MSCR_PJ15 143
#define SIUL2_MSCR_PK0  144
#define SIUL2_MSCR_PK1  145
#define SIUL2_MSCR_PK2  146
#define SIUL2_MSCR_PK3  147
#define SIUL2_MSCR_PK4  148

/* DCU CFG = 0x20C101 */
#define SIUL2_MSCR_DCU_CFG      (SIUL2_MSCR_DSE_80ohm | \
				 SIUL2_MSCR_SRE_SPEED_HIGH_200 | \
				 SIUL2_MSCR_OBE_EN | SIUL2_MSCR_IBE_EN | \
				 SIUL2_MSCR_MUX_MODE_ALT1)

#endif /* CONFIG_FSL_DCU_FB */

/* SIUL2_MIDR1 masks */
#define SIUL2_MIDR1_MINOR_MASK		(0xF << 0)
#define SIUL2_MIDR1_MAJOR_SHIFT		(4)
#define SIUL2_MIDR1_MAJOR_MASK		(0xF << SIUL2_MIDR1_MAJOR_SHIFT)

#define TREERUNNER_GENERATION_2_MAJOR   1

/* SIUL2_MIDR2 fields */
#define SIUL2_MIDR2_SPEED_SHIFT		(23)
#define SIUL2_MIDR2_SPEED_GRADING	(0xF << SIUL2_MIDR2_SPEED_SHIFT)

#define SIUL2_MIDR2_SPEED_600MHZ	(0xA)
#define SIUL2_MIDR2_SPEED_800MHZ	(0xB)
#define SIUL2_MIDR2_SPEED_1GHZ		(0xC)

static inline int get_siul2_midr1_minor(void)
{
	return (readl(SIUL2_MIDR1) & SIUL2_MIDR1_MINOR_MASK);
}

static inline int get_siul2_midr1_major(void)
{
	return ((readl(SIUL2_MIDR1) & SIUL2_MIDR1_MAJOR_MASK)
			>> SIUL2_MIDR1_MAJOR_SHIFT);
}

static inline int get_siul2_midr2_speed(void)
{
	return (readl(SIUL2_MIDR2) & SIUL2_MIDR2_SPEED_GRADING)
			>> SIUL2_MIDR2_SPEED_SHIFT;
}

#endif /*__ARCH_ARM_MACH_S32V234_SIUL_H__ */
