#!/bin/sh
# SPDX-License-Identifier: MIT

error() {
	printf "%s\n" "$@" >&2
	exit 1
}

ps() {
	podman ps "$@"
}

start() {
	local name="$1"
	if [ -e "/etc/atmark/containers/$name.conf" ]; then
		podman_start "$name"
	else
		# podman_start restarts container if it was running.
		# stop is noop if already stopped
		podman stop "$name" && podman start "$name"
	fi
}

stop() {
	local name="$1"
	podman stop "$name"
}

logs() {
	local name="$1"
	podman logs "$name"
}

logs_merged() {
	logs "$@" 2>&1
}

logs_follow() {
	local name="$1"
	local pipe found="" abos_pipe podman_pid
	local abos_web_pid

	# for example, even if we use "podman.sh logs_follow <container> | awk ...",
	# stderr remains connected to abos-web, so use fd/2.
	pipe=$(readlink "/proc/$$/fd/2") || exit

	abos_web_pid=$(pgrep -o abos-web)
	for abos_pipe in "/proc/$abos_web_pid/fd/"*; do
		if [ "$(readlink "$abos_pipe")" = "$pipe" ]; then
			found=1
			break
		fi
	done
	# not piped?
	if [ -z "$found" ]; then
		logger -t abos-web "podman.sh: pipe connected to abos-web not found."
		exec podman logs -f "$name"
	fi

	podman logs -f "$name" &
	podman_pid=$!

	while [ "$(readlink "$abos_pipe")" = "$pipe" ]; do
		if ! [ -e "/proc/$podman_pid" ]; then
			wait "$podman_pid"
			exit
		fi
		sleep 5
	done

	# kill podman process if it still exists -- using job hopefully
	# ensures we do not kill a recycled pid
	kill -9 %1
}

mode="$1"
shift

case "$mode" in
ps|start|stop|logs|logs_merged|logs_follow)
	"$mode" "$@"
	;;
*)
	error "Usage: podman.sh [ps options|start/stop/logs/logs_merged/logs_follow CONTAINER]"
	;;
esac
