# SPDX-License-Identifier: MIT

ALLOWLIST="/etc/atmark/usb-filter-allowlist"
DEVICE_LIST="/run/usb-filter-device-list"
CLASS_LIST="/usr/share/abos-ctrl/usb_device_class"
UBOOT_ENV="/boot/uboot_env.d/80_optargs"

usb_filter_help() {
	echo "Usage: abos-ctrl usb-filter [action [arguments]]"
	echo
	echo "Possible actions:"
	echo "  enable: enabling the USB filter function"
	echo "  disable: disabling the USB filter"
	echo "  list-devices [--verbose]: list currently connected USB devices"
	echo "  list-rules [--verbose]: list currently USB device allowlist"
	echo "  reset-rules [--force]: reset USB device allowlist"
	echo "                         with --force option, do not prompt"
	echo "  allow-device [--vendor-id VID] [--product-id PID]"
	echo "               [--model MODEL] [--usb-interfaces INTERFACES]"
	echo "               [--serial SERIAL]: add device to the allowlist"
	echo "                                  Omitted parameters are stored as wildcard"
	echo "                                  parameters can be checked with list-devices and"
	echo "                                  list-rules --verbose flag"
	echo "  allow-device {DEVID}: allow connected USB devices specified by ID to connect,"
	echo "                        and add device's information to the allowlist"
	echo "                        ID can be checked with \`abos-ctrl usb-filter list-devices\`"
	echo "  block-device {DEVID}: block alllowed USB devices specified by ID to connect,"
	echo "                        and delete device's information from the allowlist"
	echo "                        ID can be checked with \`abos-ctrl usb-filter list-devices\`"
	echo "  allow-class [CLASS]...: create allow rule for each USB device class"
	echo "                          the string for 'CLASS' can be found by running"
	echo "                          \`abos-ctrl usb-filter allow-class\`"
	echo "  remove-rule {RULEID}...: remove the allow rule specified by ID"
	echo "                           ID can be checked with \`abos-ctrl usb-filter list-rules\`"
	echo "  help: show this message"
}

log_console() {
	local line console

	log "$@"
	console=$(getty_console --get)
	# if logged in on console only log stderr to avoid duplicate
	[ "$(readlink /proc/self/fd/2)" = "/dev/$console" ] \
		&& console=""
	for line in "$@"; do
		if [ -n "$console" ]; then
			echo "usb-filter: $line" > "/dev/$console"
		fi
		# if stderr is a tty also print to stderr
		if tty > /dev/null <&2; then
			echo "usb-filter: $line" >&2
		fi
	done
}

log() {
	logger -t usb-filter "$@"
}

is_sysfs_ro() {
	# if running from a swu command we can't (and don't want) to write in
	# /sys or run fw_setenv, just write the $UBOOT_ENV file
	# This is imperfect but /sys being ro is a good approximation
	grep -qE '[ ,]ro[, ].*\bsysfs\b' /proc/self/mountinfo;
}

usb_filter_set_auth_default() {
	local value="$1" auth_def optargs

	[ "$value" = "0" ] || [ "$value" = "1" ] \
		|| error "usb_filter_set_auth_default: Invalid argument (got '$value')"

	optargs=$(cat /boot/uboot_env.d/* \
			| sed -n -e 's/ \?usbcore.authorized_default=.//' \
				-e "s/^optargs=.*/& usbcore.authorized_default=$value/p" \
			| tail -n 1)
	if is_sysfs_ro; then
		echo "$optargs" > "$UBOOT_ENV" || exit
		return
	fi

	echo "$optargs" > "$UBOOT_ENV" \
		&& persist_file "$UBOOT_ENV" \
		&& fw_setenv_nowarn -s "$UBOOT_ENV" \
		|| exit 1

	for auth_def in /sys/bus/usb/devices/usb*/authorized_default; do
		echo "$value" > "$auth_def" || exit 1
	done

	usb_filter_refresh_devices
}

usb_filter_enable() {
	usb_filter_set_auth_default "0"
	echo "USB filter enabled."
}

usb_filter_disable() {
	usb_filter_set_auth_default "1"
	echo "USB filter disabled."
}

usb_filter_enabled() {
	grep -q "usbcore.authorized_default=0" "$UBOOT_ENV" 2>/dev/null
}

usb_filter_status() {
	local status="disabled"
	usb_filter_enabled && status="enabled"
	echo "$status"
}

usb_filter_check_device_class() {
	local allowed_classes

	allowed_classes="$(awk '$1 == "class" {print $2}' "$ALLOWLIST")"
	[ -z "$allowed_classes" ] && return 1

	# convert to class ids
	allowed_classes="$(awk -v "classes=$allowed_classes" -F = '
		BEGIN {
			split(classes, a, "\n")
			for (k in a) { cl[a[k]]=1; }
		}
		cl[$1] {
			print $2
		}
		' "$CLASS_LIST")"

	# fail if any of the device's classes is not allowed
	# (all of the devices' classes must be allowed to allow the device)
	echo "$ID_USB_INTERFACES" | \
		awk -F ':' -v "classes=$allowed_classes" '
			BEGIN {
				split(classes, a, "\n")
				for (k in a) { cl[a[k]]=1; }
			}
			{
				for (i=2; i<NF; i++) {
					if (! cl[substr($i, 0, 2)]) {
						exit(1);
					}
				}
			}'
}

usb_filter_authorize_device() {
	if ! usb_filter_enabled; then
		echo 1 > "/sys$DEVPATH/authorized"
		return
	fi
	if ! [ -e "$ALLOWLIST" ]; then
		log_console "denied $ID_VENDOR_ID:$ID_MODEL_ID ($ID_MODEL) (no allow list)"
		echo 0 > "/sys$DEVPATH/authorized"
		return
	fi

	if usb_filter_check_device_class; then
		log "allowed $ID_VENDOR_ID:$ID_MODEL_ID ($ID_MODEL) by class"
		echo 1 > "/sys$DEVPATH/authorized" || exit 1
		return
	fi

	if awk -v "vendor_id=\"$ID_VENDOR_ID\"" -v "model_id=\"$ID_MODEL_ID\"" \
		-v "model=\"$ID_MODEL\"" -v "usb_iface=\"$ID_USB_INTERFACES\"" \
		-v "serial=\"$ID_SERIAL\"" '
			$1 == "device" &&
			($2 == "\"*\"" || $2 == vendor_id) &&
			($3 == "\"*\"" || $3 == model_id) &&
			($4 == "\"*\"" || $4 == model) &&
			($5 == "\"*\"" || $5 == usb_iface) &&
			($6 == "\"*\"" || $6 == serial) {
				exit 1
			}
			' "$ALLOWLIST"; then
		# no match found
		log_console "denied $ID_VENDOR_ID:$ID_MODEL_ID ($ID_MODEL)"
		echo 0 > "/sys$DEVPATH/authorized" || exit 1
		return
	fi

	log "allowed $ID_VENDOR_ID:$ID_MODEL_ID ($ID_MODEL) by device"
	echo 1 > "/sys$DEVPATH/authorized" || exit 1
}

usb_filter_add_device() {
	[ "$ACTION" != "add" ] && \
		error "This option can only be executed from udev rules"

	[ "$(cat "/sys$DEVPATH/devpath")" = "0" ] && return

	[ -n "$BUSNUM" ] || return
	[ -n "$DEVNUM" ] || return

	for param in "$ID_VENDOR_ID" "$ID_MODEL_ID" "$ID_MODEL" \
				 "$ID_USB_INTERFACES" "$ID_SERIAL"; do
		echo "$param" | grep -q " " && \
			error "recognised USB device information contains white space"
	done

	# usbinfo format
	# BUSNUM:DEVNUM,VENDOR_ID,MODEL_ID,MODEL_NAME,USB_INTERFACES,SERIAL,DEVPATH
	local usbinfo=
	usbinfo="$BUSNUM:$DEVNUM"
	usbinfo="$usbinfo $ID_VENDOR_ID"
	usbinfo="$usbinfo $ID_MODEL_ID"
	usbinfo="$usbinfo $ID_MODEL"
	usbinfo="$usbinfo $ID_USB_INTERFACES"
	usbinfo="$usbinfo $ID_SERIAL"
	usbinfo="$usbinfo $DEVPATH"
	echo "$usbinfo" >> $DEVICE_LIST || exit 1

	[ "$(cat "/sys$DEVPATH/authorized")" != 0 ] && exit
	usb_filter_authorize_device
}

usb_filter_refresh_devices() {
	local _BUSDEVNUM ID_VENDOR_ID ID_MODEL_ID ID_MODEL
	local ID_USB_INTERFACES ID_SERIAL DEVPATH

	! [ -e "$DEVICE_LIST" ] && return
	is_sysfs_ro && return

	while read -r _BUSDEVNUM ID_VENDOR_ID ID_MODEL_ID ID_MODEL \
			ID_USB_INTERFACES ID_SERIAL DEVPATH; do
		# can be gone if we removed required hub
		[ -e "/sys/$DEVPATH/authorized" ] || continue
		usb_filter_authorize_device
	done < "$DEVICE_LIST"

	# wait for devices to be updated before returning
	udevadm trigger
	udevadm settle
}

usb_filter_remove_device() {
	[ "$ACTION" != "remove" ] && \
		error "This option can only be executed from udev rules"

	[ -n "$BUSNUM" ] || return
	[ -n "$DEVNUM" ] || return
	sed -i -e "/^$BUSNUM:$DEVNUM/d" $DEVICE_LIST
}

usb_filter_allow_device() {
	# note: the rest of the code calls product_id "model_id", as that is the
	# variable udev uses, but for user-facing argument product id is more common
	local vendor_id="\"*\"" product_id="\"*\"" model="\"*\""
	local usb_interfaces="\"*\"" serial="\"*\""

	# usage is either a single id or a device description
	if [ "$#" = 1 ]; then
		local id="$1" target rule devpath
		target="$(awk -v id="$id" 'NR==id' "$DEVICE_LIST")"
		[ -n "$target" ] \
			|| error "No device with ID '$id' exists"
		# remove busnum/devnum (first word)
		target="${target#* }"
		# remove device path (last word) and add quotes...
		target="${target% *}"
		usb_filter_allow_device_helper "\"${target// /\" \"}\""
		return
	fi
	[ "$#" -gt 0 ] && [ $(( $# % 2 )) = 0 ] \
		|| error "usb-filter allow-device arguments:" \
			"--vendor-id, --product-id, --model, --usb-interfaces, --serial"

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"--vendor-id")
			case "$2" in
			[0-9a-f][0-9a-f][0-9a-f][0-9a-f]|'*') ;; # ok
			*) error "vendor-id '$2' should be a 4 digits lowercase hex string";;
			esac
			vendor_id="\"$2\""
			;;
		"--product-id")
			case "$2" in
			[0-9a-f][0-9a-f][0-9a-f][0-9a-f]|'*') ;; # ok
			*) error "product-id '$2' should be a 4 digits lowercase hex string";;
			esac
			product_id="\"$2\""
			;;
		"--model")
			model="\"$2\""
			;;
		"--usb-interfaces")
			case "$2" in
			:[0-9a-f]*:|'*') ;; # ok
			*) error "usb-interface '$2' should look like ':012345:012345:'";;
			esac
			usb_interfaces="\"$2\""
			;;
		"--serial")
			serial="\"$2\""
			;;
		*)
			error "Unknown option $1" \
				"usb-filter allow-device arguments:" \
				"--vendor-id, --product-id, --model, --usb-interfaces, --serial"
		esac
		shift 2
	done
	local rule="$vendor_id $product_id $model $usb_interfaces $serial"
	usb_filter_allow_device_helper "$rule"
}

usb_filter_allow_device_helper() {
	local rule="$1" devpath="$2"

	local allow_list
	allow_list=$({
			cat "$ALLOWLIST" 2>/dev/null
			echo "device $rule"
		} | sort -u)
	if ! echo "$allow_list" | cmp -s "$ALLOWLIST" - 2>/dev/null; then
		echo "$allow_list" > "$ALLOWLIST.tmp" \
			&& mv "$ALLOWLIST.tmp" "$ALLOWLIST" \
			&& persist_file "$ALLOWLIST" \
			|| exit
	fi

	info "The following rule was added to allow list:"
	info "    $rule"

	usb_filter_refresh_devices
}

usb_filter_block_device() {
	local id="$1"

	if ! usb_filter_enabled; then
		error "USB filter is currently disabled" \
			  "Run \`abos-ctrl usb-filter enable\` first to enable USB filtering"
		return
	fi

	local devinfo
	devinfo="$(awk -v id="$id" 'NR==id' $DEVICE_LIST)"
	[ -z "$devinfo" ] && \
		error "No device with such an ID exists"

	local serial
	serial="$(echo "$devinfo" | awk '{print $6}')"
	# XXX improve this to match other criteria?
	if grep -q "$serial" "$ALLOWLIST" 2>/dev/null; then
		sed -i -e "/$serial/d" "$ALLOWLIST" \
			&& persist_file "$ALLOWLIST" \
			|| error "Could not remove $serial from $ALLOWLIST"
		info "Removed $serial from $ALLOWLIST"
	fi

	local devpath
	devpath="$(echo "$devinfo" | awk '{gsub("^\"|\"$", "", $7); print $7}')"
	[ "$(cat /sys"$devpath"/authorized)" = "0" ] && return

	echo "blocked the following device:"
	echo "    $devinfo"

	usb_filter_refresh_devices
}

usb_filter_allow_class() {
	local class
	if [ "$#" = 0 ]; then
		echo "supported USB device classes:"
		awk -F '=' '{print "\t"$1}' "$CLASS_LIST"
		return
	fi

	local class id class_list="" id_list="" err=""
	for class in "$@"; do
		id="$(awk -F '=' -v class="$class" \
			'BEGIN {
				class = tolower(class)
			}
			tolower($1) == class {
				print $1, $2;
				exit(0)
			}' "$CLASS_LIST")"
		if [ -z "$id" ]; then
			echo "$class is not a supported USB device class" >&2
			err=1
		fi
		class_list="${class_list:+$class_list
}class ${id% *}"
		id_list="${id_list:+$id_list }${id#* }"
	done
	if [ -n "$err" ]; then
		error "supported USB device classes:" \
			  "$(awk -F '=' '{print "\t"$1}' "$CLASS_LIST")"
	fi

	local allow_list
	allow_list=$({
			cat "$ALLOWLIST" 2>/dev/null
			echo "$class_list"
		} | sort -u)
	if ! echo "$allow_list" | cmp -s "$ALLOWLIST" - 2>/dev/null; then
		echo "$allow_list" > "$ALLOWLIST.tmp" \
			&& mv "$ALLOWLIST.tmp" "$ALLOWLIST" \
			&& persist_file "$ALLOWLIST" \
			|| exit
	fi

	usb_filter_refresh_devices
}

usb_filter_list_devices() {
	if [ ! -s "$DEVICE_LIST" ]; then
		echo "No USB devices connected"
		return 0
	fi

	local index=1
	while read -r line; do
		local devpath status="block"
		devpath="$(echo "$line" | awk '{gsub("\"", "", $7); print $7}')"
		if [ ! -e "/sys$devpath/authorized" ]; then
			warning "/sys$devpath/authorized is not found." \
					"This device may not already be connected."
			# sed creates a temporary file then renames, and does not modify
			# the file in place, so this while read loop is fine.
			# shellcheck disable=SC2094
			sed -i -e "/^${line%% *}/d" "$DEVICE_LIST" 2>/dev/null
			continue
		fi
		[ "$(cat /sys"$devpath"/authorized)" = "1" ] && status="allow"
		echo "     $index  $status $line"
		index=$((index + 1))
	done < "$DEVICE_LIST" \
		| if [ "$debug" -le 2 ]; then
			# verbose not passed
			cat
		else
			sed -e 's/\(\(allow\|block\) \S*\) \(\S*\) \(\S*\) \(\S*\) \(\S*\) \(\S*\)/\1 --vendor-id \3 --product-id \4 --model \5 --usb-interfaces \6 --serial \7/'
		fi
}

usb_filter_list_rules() {
	if [ ! -s "$ALLOWLIST" ]; then
		echo "No rules have been set"
		return
	fi
	if [ "$debug" -le 2 ]; then
		# verbose not passed
		cat -n "$ALLOWLIST"
	else
		cat -n "$ALLOWLIST" \
			| sed -e 's/device \(\S*\) \(\S*\) \(\S*\) \(\S*\) \(\S*\)/device --vendor-id \1 --product-id \2 --model \3 --usb-interfaces \4 --serial \5/'
	fi
}

usb_filter_remove_rule() {
	local id target linecount
	linecount="$(wc -l "$ALLOWLIST" 2>/dev/null || echo 0)"
	linecount="${linecount% *}"

	for id in "$@"; do
		[ "$id" -le "$linecount" ] \
			|| error "No rule $id in file (max $linecount)"
		# replace argument with sed delete rule
		shift; set -- "$@" -e "${id}d"
	done

	sed -i "$@" "$ALLOWLIST" && persist_file "$ALLOWLIST" \
		|| exit

	usb_filter_refresh_devices
}

usb_filter_reset_rules() {
	if [ "$1" = "--force" ]; then
		persist_file -d "$ALLOWLIST"
		usb_filter_refresh_devices
		echo "All rules have been removed."
		return
	fi

	local output=""
	output="$(usb_filter_list_rules)"
	echo "$output"

	if [ "$output" = "No rules have been set" ]; then
		echo "so nothing to do."
		return
	fi

	echo ""
	echo "This command will delete all the above configured rules."
	echo "Continue? [y/N]"
	prompt_yesno n || error "Canceled."

	persist_file -d "$ALLOWLIST"
	echo "All rules have been removed."

	usb_filter_refresh_devices
}

ctrl_usb_filter() {
	if [ "$#" -eq 0 ]; then
		echo "Currently USB filter is $(usb_filter_status)."
		return
	fi

	case "$1" in
	"add")
		usb_filter_add_device
		;;
	"remove")
		usb_filter_remove_device
		;;
	"allow-device")
		shift
		usb_filter_allow_device "$@"
		;;
	"block-device")
		[ "$#" -eq 2 ] || error "This command requires one argument."
		usb_filter_block_device "$2"
		;;
	"allow-class")
		shift
		usb_filter_allow_class "$@"
		;;
	"list-devices")
		[ "$2" = "--verbose" ] || [ "$2" = "-v" ] && debug=$((debug+1))
		usb_filter_list_devices
		;;
	"reset-rules")
		usb_filter_reset_rules "$2"
		;;
	"remove-rule")
		[ "$#" -ge 2 ] || error "This command requires at least one argument."
		shift
		usb_filter_remove_rule "$@"
		;;
	"list-rules")
		[ "$2" = "--verbose" ] || [ "$2" = "-v" ] && debug=$((debug+1))
		usb_filter_list_rules
		;;
	"enable")
		usb_filter_enable
		;;
	"disable")
		usb_filter_disable
		;;
	"status")
		usb_filter_status
		;;
	"help")
		usb_filter_help
		;;
	*)
		warning "Invalid option $1"
		usb_filter_help >&2
		exit 1
		;;
	esac
}
