# SPDX-License-Identifier: MIT

rtos_help() {
	echo "Usage: abos-ctrl rtos [options] [action]"
	echo
	echo "Options:"
	echo "  -t: time to wait between each line from rtos"
	echo "  -n: do not read back output from rtos"
	echo
	echo "Actions:"
	echo "  cmd <command>"
	echo "  interact"
}

# split read/write open locks to allow having a logger running
# to syslog all the time and still be able to input command
# for suspend
# Conventions:
#  - lock ttyrpmsg for reading, this rtos.sh for writing
#  - fd 4 for reading, fd 5 for writing (6 for write lock)
rtos_open_read() {
	# avoid mangling input
	stty -echo raw <"$dev"

	exec 4<"$dev"

	if ! flock -n 4; then
		# not interactive
		tty >/dev/null || error "Could not lock rtos console for reading"
		echo "Could not lock rtos console for reading" >&2
		rc-service -q rtos-logger status \
			|| error "rtos-logger service is stopped, stop other console users"
		echo "stop rtos-logger service ? [Y/n]"
		echo "WARNING: It will not be restarted automatically!"
		prompt_yesno y || error "Aborted"
		rc-service rtos-logger stop
		# we apparently need to re-open the device after service stop
		# or nothing can be read from fd 4
		exec 4<"$dev"
		flock -n 4 || error "Still could not get rtos console read lock"
	fi
}

rtos_open_write() {
	exec 5>"$dev" 6<"$scripts_dir/rtos.sh"

	flock -n 6 || error "Could not lock rtos console for writing"
}

rtos_close_write() {
	exec 5<&- 6<&-
}

rtos_close() {
	exec 4<&- 5<&- 6<&-
}

rtos_read_line() {
	# read a line and remove carriage returns..
	# hopefully no-one using this to get binary data.

	# Note this is far from perfect: busybox sh 'read' is bad at
	# non-delimited data and will eat data if no newline is seen
	# before timeout elapses (e.g. 'M33>' prompt)
	read -r ${step_timeout:+-t "$step_timeout"} line <&4
}

rtos_read_until_timeout() {
	while rtos_read_line; do
		echo "$line"
	done <&4
}

rtos_write() {
	printf "\x03%s\n" "$@" >&5
}

rtos_interact() {
	local cmd cat_pid

	# send ^C first to cancel any partial input from earlier interact
	printf "%s" $'\x3' >&5
	if rtos_read_line && [ "$line" != "^C" ]; then
		echo "$line"
	fi

	cat <&4 &
	cat_pid=$!

	echo "Entering console. Press ctrl+D to exit"

	# pass input one letter at a time without local echo
	# use intr instead of eof because eof needs icanon, and min needs -icanon...
	stty -echo intr ^d min 1 time 0 -icanon
	# trap lets script continue after cat exit and does not need special treatment
	trap "true" INT QUIT

	cat >&5

	stty echo intr ^c icanon
	echo
	trap "" INT QUIT
	kill -9 "$cat_pid" 2>/dev/null
	# Silence stderr to hide 'Killed' message
	wait "$cat_pid" 2>/dev/null
}

ctrl_rtos() {
	local dev action="" cmd
	# -n option is processed as dryrun by abos-ctrl option loop
	# shellcheck disable=SC2154
	local noread="$dryrun"
	local log_replay="" log_syslog=""
	# 0.3s was picked experimentally: not too long to feel an odd delay
	# but too short and e.g. 'cmd log' output is truncated.
	local step_timeout=0.3
	local line

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"cmd")
			[ "$#" -ge 2 ] || error "$1 requires at least an argument";
			action=cmd
			shift
			cmd="$*"
			break
			;;
		"interact")
			action=interact
			;;
		"log")
			action=log
			;;
		"-t")
			[ "$#" -ge 2 ] || error "$1 requires an argument";
			step_timeout="$2"
			shift
			;;
		"-n")
			noread=1
			;;
		"--replay")
			log_replay=1
			;;
		"--syslog")
			log_syslog=1
			;;
		"-q"|"--quiet")
			debug=$((debug-1))
			;;
		"-v"|"--verbose")
			debug=$((debug+1))
			;;
		*)
			rtos_help
			error "Unkonwn argument $1"
			;;
		esac
		shift
	done

	if [ -z "$action" ]; then
		rtos_help
		error "No action specified"
	fi

	[ -n "$noread" ] && [ "$action" != cmd ] \
		&& error "-n only usable with cmd"
	[ -n "$log_replay" ] && [ "$action" != log ] \
		&& error "--replay only usable with log"
	[ -n "$log_syslog" ] && [ "$action" != log ] \
		&& error "--syslog only usable with log"

	for dev in /sys/devices/platform/rpmsg-console/tty/ttyrpmsg*; do
		[ -e "$dev" ] || error "rpmsg-console ttyrpmsg device not found"
		break
	done
	dev=/dev/"${dev##*/}"

	# open/get locks first
	case "$action" in
	cmd)
		rtos_open_write
		[ -z "$noread" ] && rtos_open_read
		;;
	interact)
		rtos_open_read
		rtos_open_write
		;;
	log)
		rtos_open_read
		[ -n "$log_replay" ] && rtos_open_write
		;;
	esac

	case "$action" in
	cmd)
		rtos_write "$cmd"
		if [ -z "$noread" ]; then
			# try to consume echo: empty line then cmd with M33 prompt
			if rtos_read_line && [ "$line" != "^C" ]; then
				echo "$line"
			fi
			if rtos_read_line && [ "$line" != "M33> $cmd" ]; then
				echo "$line"
			fi
			rtos_read_until_timeout
		fi
		;;
	interact)
		rtos_interact
		;;
	log)
		if [ -n "$log_replay" ]; then
			rtos_write log
			rtos_close_write
		fi
		if [ -n "$log_syslog" ]; then
			set -- logger -t rtos-logger
		else
			set -- cat
		fi
		exec "$@" <&4
	esac
	rtos_close
}

