# SPDX-License-Identifier: MIT

rollback_help() {
	echo "Usage: abos-ctrl rollback [--reboot] [--allow-downgrade] [--force [target]]"
	echo
	echo "Rollback to other partition."
	echo "If --reboot is passed, continue with reboot without dropping update lock"
	echo "Note reboot is executed even if we could not rollback"
	echo "If --allow-downgrade is passed, accept rolling back to the other partition"
	echo "even if automatic rollback would refuse due to version downgrade"
	echo "If --force is passed, not only allow downgrades but also allow rollback to"
	echo "invalid target. The optional target argument allows being explicit about"
	echo "which side to boot next, which could also be the current side"
}

rollback_cleanup() {
	local last_rc="$?"
	if [ "$last_rc" = 0 ] && [ -n "$rollback_reboot" ]; then
		warning "Rebooting!"
		touch /run/swupdate_rebooting
		reboot
		exit
	fi
	unlock_update
}

get_mmc_name() {
        cat "/sys/class/block/${rootdev#/dev/}/device/name" 2>/dev/null
}

workaround_a600_mmc() {
        # old Armadillo 640 eMMC is broken and issuing a bootpart change without
        # a switch to mmcblk0bootX partition is known to cause corruptions
        # The dd forces the hardware to issue such a switch, and fsfreeze ensures
        # no other writes are done in between
        [ "$(get_mmc_name)" = Q2J55L ] || return

        if command -v fsfreeze >/dev/null; then
                fsfreeze=1
                fsfreeze -f /var/app/volumes
                fsfreeze -f /var/log
        fi

        dd if="${rootdev}boot0" of=/dev/null bs=4k count=1 iflag=direct status=none
}

ctrl_rollback() {
	# not local as used in trap
	rollback_reboot=""
	local force="" allow_downgrade="" fsfreeze="" fail=""

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"--reboot")
			# reboot happens in cleanup
			rollback_reboot=1
			;;
		"--allow-downgrade")
			allow_downgrade=1
			;;
		"--force")
			# optional rollback destination argument
			case "${2:-unset}" in
			unset|-*)
				dev_from_current
				;;
			rootfs_[01]|[01]|/dev/mmcblk*|mmcblk*)
				dev_from_string "$2"
				shift
				;;
			*)
				error "Invalid rollback target $2"
				;;
			esac
			info "Forcing rollback to $target_rootpart"
			force=1
			;;
		"-n"|"--dry-run")
			dryrun=1
			;;
		"-q"|"--quiet")
			debug=$((debug-1))
			;;
		"-v"|"--verbose")
			debug=$((debug+1))
			;;
		*)
			warning "Unrecognized argument $1"
			rollback_help
			exit 1
			;;
		esac
		shift
	done

	# set dev if unset
	dev_from_current
	info "Currently booted on $rootpart"

	# we don't want to set rollback while a swupdate install is in
	# in progress, take update lock...
	lock_update
	trap rollback_cleanup EXIT INT QUIT TERM

	[ -e "$target_rootpart" ] || error "Rollback target $target_rootpart does not exist!"
	if [ -z "$force" ]; then
		case "$(fw_printenv_nowarn upgrade_available)" in
		*upgrade_available=[12]) ;;
		*upgrade_available=0)
			[ -n "$allow_downgrade" ] \
				|| error "other partition is downgrade than the current version, refusing to rollback" \
					"run 'abos-ctrl rollback --allow-downgrade' to rollback anyway"
			;;
		*)
			error "other partition is not usable for rollback, refusing to rollback" \
				"run 'abos-ctrl rollback --force' to skip this check"
			;;
		esac
	fi

	if [ -n "$dryrun" ]; then
		info "Would rollback to $target_rootpart"
		return
	fi

	# actual rollback for mmc, sd card or qemu (extlinux)
	if [ -e "${rootdev}boot0" ]; then
		workaround_a600_mmc
		mmc bootpart enable "$((target_ab+1))" 0 "$rootdev" \
			|| fail=1
		if [ -n "$fsfreeze" ]; then
			# there apparently can still be problems if we write immediately
			# after bootpart as well, 1s seems to be enough to avoid issues.
			sleep 1
			fsfreeze -u /var/app/volumes
			fsfreeze -u /var/log
		fi
		[ -z "$fail" ] || error "Could not flip mmc boot flag"
	elif [ -s /etc/fw_env.config ]; then
		# if uboot env is supported, use it (e.g. sd card)
		fw_setenv_nowarn mmcpart $((target_ab+1)) \
			|| error " Could not setenv mmcpart"
	elif [ -e /target/boot/extlinux.conf ]; then
		# assume gpt boot e.g. extlinux
		sgdisk --attributes=$((target_ab+1)):set:2 --attributes=$((ab+1)):clear:2 "$rootdev" \
			|| error "Could not set boot attribute"

		sed -i -e "s/root=[^ ]*/root=LABEL=rootfs_${target_ab}/" /target/boot/extlinux.conf \
			|| error "Could not update extlinux.conf"
		extlinux -i /target/boot || error "Could not reinstall bootloader"
	else
		error "Do not know how to A/B switch this system"
	fi

	ctrl_atlog_write "Triggered rollback to $target_rootpart"
	info "Switched to $target_rootpart successfully." "Reboot to apply change."
}

