# SPDX-License-Identifier: MIT

. "$scripts_dir/btrfs.sh"

podman_check_running() {
	if [ -n "$(podman ps -q)" ]; then
		echo "This command cannot continue with containers running."
		echo "Stop running containers? Unsaved data will be lost! [y/N]"
		prompt_yesno n || error "Aborting at user request"
		podman stop -a
		podman ps --format '{{.ID}}' \
			| timeout 20s xargs -r podman wait
		[ -z "$(podman ps -q)" ] || error "Containers still running after stop!"
		echo "Containers stopped successfully."
	fi
	# udevfw might stay around in the background,
	# we've checked no containers are up: kill any.
	pkill -9 udevfw || :
}

podman_storage_get_status() {
	if grep -q 'graphroot = "/var/lib/containers/storage' /etc/containers/storage.conf 2>/dev/null; then
		status="disk"
	else
		status="tmpfs"
	fi
}

podman_storage_delete() {
	local target="${1:-$status}"
	# just delete whatever storage we currently use

	case "$target" in
	tmpfs)
		find /run/containers/storage_root -xdev -delete 2>/dev/null
		;;
	disk)
		umount_if_mountpoint /var/lib/containers/storage \
			|| error "Could not unmount container storage: really no container running?"
		unshare -m sh -ec "cd / && abos-ctrl umount /mnt
			mount -t btrfs '$appdev' /mnt
			if [ -d /mnt/containers_storage ]; then
				btrfs subvolume delete /mnt/containers_storage
			fi
			" || error "Could not delete containers storage subvolume"
		;;
	esac
}

podman_storage_create_rw_storage() {
	local CLEANUP="$1"

	# create emmc storage for podman from readonly snapshot
	unshare -m sh -ec "cd / && abos-ctrl umount /mnt
		mount -t btrfs '$appdev' /mnt
		if [ -d /mnt/containers_storage ]; then
			case '$CLEANUP' in
			ask)
				echo 'There appears to already be an old podman storage on eMMC, update it? [Y/n]'
				if abos-ctrl internal '' prompt_yesno y; then
					btrfs subvolume delete /mnt/containers_storage
				fi
				;;
			no)
				;;
			*)
				btrfs subvolume delete /mnt/containers_storage;;
			esac
		fi
		[ -d /mnt/containers_storage ] \
			|| btrfs subvolume snapshot /var/lib/containers/storage_readonly /mnt/containers_storage
		" || error "Could not create containers storage subvolume"

}

podman_storage_replace() {
	# make sure no container is defined as copying leftovers would fail further
	# image manipulation commands
	if [ -n "$(podman ps -qa 2>/dev/null)" ]; then
		info "Removing leftover stopped containers..."
		podman rm -a || error "podman rm -a failed"
	fi

	# replace storage_readonly by current emmc storage
	if [ "$MODE" = "disk" ]; then
		# we need to keep current storage if running from disk to disk
		podman_storage_replace_keep
		return
	fi

	dev_from_current
	dest_storage="/boot_${ab}/containers_storage"

	unshare -m sh -ec "cd / && abos-ctrl umount /mnt
		mount -t btrfs '$appdev' /mnt
		rm -f /mnt/containers_storage/libpod/bolt_state.db \
			/mnt/containers_storage/db.sql 2>/dev/null
		# need writable subvolume to swap
		if [ -d /mnt$dest_storage ]; then
			btrfs property set /mnt$dest_storage ro false
			renameat2 --exchange /mnt/containers_storage /mnt$dest_storage
			btrfs subvolume delete /mnt/containers_storage
		else
			mv /mnt/containers_storage /mnt$dest_storage
		fi
		btrfs property set /mnt$dest_storage ro true
		" || error "Could not replace storage, state might be inconsistent. Rerun abos-ctrl podman-storage to check."

	if ! umount_if_mountpoint /var/lib/containers/storage_readonly \
	    || ! mount /var/lib/containers/storage_readonly; then
		warning "Could not refresh /var/lib/containers/storage_readonly, please reboot to apply replace"
	else
		info "Replacing development images to readonly storage succeeded"
	fi
}

podman_storage_replace_keep_unshared() {
	# like the above but keep containers_storage subvolume as is
	# this is only used by podman-rw
	local appdev
	local dest_storage

	get_appdev
	dev_from_current
	dest_storage="/boot_${ab}/containers_storage"

	# umount /mnt if mounted. we need to leave directory first if inside it
	# that is because remounting appdev to /mnt if it was already there fails
	cd / || error "Could not enter /"
	umount_if_mountpoint /mnt >/dev/null 2>&1

	mount -t btrfs "$appdev" /mnt \
		|| error "Could not mount app"

	# work in new snapshot to avoid problems if something fails
	[ -d /mnt/new_storage ] \
		&& btrfs_subvol_recursive_delete new_storage /mnt
	btrfs subvolume snapshot "/mnt/containers_storage" /mnt/new_storage \
		|| error "Could not snapshot readonly storage"
	trap "btrfs subvolume delete /mnt/new_storage
	      btrfs property set /var/lib/containers/storage_readonly ro true" EXIT INT TERM QUIT

	# Always remove bolt/sql db...
	rm -f "/mnt/new_storage/libpod/bolt_state.db" \
	       "/mnt/new_storage/db.sql" 2>/dev/null

	# We need writable subvolumes to swap.
	btrfs property set "/mnt$dest_storage" ro false
	renameat2 --exchange "/mnt$dest_storage" /mnt/new_storage \
		|| error "Could not swap copied snapshot with regular one"
	btrfs property set "/mnt$dest_storage" ro true

	btrfs subvolume delete /mnt/new_storage
	trap - EXIT INT QUIT TERM
}

podman_storage_replace_keep() {
	unshare -m abos-ctrl internal podman_storage podman_storage_replace_keep_unshared \
		|| exit 1

	if ! umount_if_mountpoint /var/lib/containers/storage_readonly \
	    || ! mount /var/lib/containers/storage_readonly; then
		warning "Could not refresh /var/lib/containers/storage_readonly, please reboot to apply replace"
	else
		info "Replacing development images to readonly storage succeeded"
	fi
}

podman_storage_merge_unshared() {
	local appdev dest_storage

	get_appdev
	dest_storage="$(findmnt -nr -o SOURCE /var/lib/containers/storage_readonly)"
	dest_storage="${dest_storage#*[}"
	dest_storage="${dest_storage%]}"
	[ -n "$dest_storage" ] || error "Could not find readonly storage volume"

	# umount /mnt if mounted. we need to leave directory first if inside it
	# that is because remounting appdev to /mnt if it was already there fails
	cd / || error "Could not enter /"
	umount_if_mountpoint /mnt >/dev/null 2>&1

	mount -t btrfs "$appdev" /mnt \
		|| error "Could not mount app"

	# we need to work in a new snapshot because podman push doesn't allow using
	# the same storage both for source and destination (it gets RO lock first
	# and fails getting RW lock writing)
	[ -d /mnt/new_storage ] \
		&& btrfs_subvol_recursive_delete new_storage /mnt
	btrfs subvolume snapshot "/mnt$dest_storage" /mnt/new_storage \
		|| error "Could not snapshot readonly storage"
	TMPDIR=$(mktemp -d "/var/tmp/abos_ctrl_podman_storage.XXXXXX") \
		|| error "Could not create tmp directory"
	export TMPDIR

	# shellcheck disable=SC2064 # expand TMPDIR now...
	trap "btrfs subvolume delete /mnt/new_storage
	      btrfs property set /var/lib/containers/storage_readonly ro true
	      rm -rf '$TMPDIR'" EXIT INT TERM QUIT

	# Actual copy. Note we don't explicitly copy just ids to save time:
	# swupdate would purge images which only have ids.
	podman image list --filter readonly=false \
			--format '{{range .Names}}{{.}}{{println}}{{end}}' \
		| while read -r tag; do
			# ignore new lines (at end of each image)
			[ -n "$tag" ] || continue
			podman push "$tag" "containers-storage:[overlay@/mnt/new_storage]$tag" \
				|| error "Could not copy $tag"
		done || exit 1

	# as always clean up bolt_state (path information)
	# and swap containers. We need writable subvolumes to swap.
	rm -f "/mnt/new_storage/libpod/bolt_state.db" \
		"/mnt/new_storage/db.sql" 2>/dev/null
	btrfs property set /mnt/new_storage ro false
	btrfs property set "/mnt$dest_storage" ro false
	renameat2 --exchange "/mnt$dest_storage" /mnt/new_storage \
		|| error "Could not swap copied snapshot with regular one"
	btrfs property set "/mnt$dest_storage" ro true

	btrfs subvolume delete /mnt/new_storage
	trap - EXIT INT QUIT TERM
}

podman_storage_merge() {
	unshare -m abos-ctrl internal podman_storage podman_storage_merge_unshared \
		|| error "Rerun abos-ctrl podman-storage if problems arise"

	if ! umount_if_mountpoint /var/lib/containers/storage_readonly \
	    || ! mount /var/lib/containers/storage_readonly; then
		warning "Could not refresh /var/lib/containers/storage_readonly, please reboot to apply merge"
		info "Continuing anyway as merge itself succeeded"
	else
		info "Merging development images to readonly storage succeeded"
		info "Feel free to adjust the result with abos-ctrl podman-rw commands"
	fi

	info
	info "Now freeing up original data..."

	podman_storage_delete

}

podman_storage_has_local() {
	# list all images with names and check if anything is different
	# between local storage and readonly storage
	# use arg array to cleanly escape command
	# shellcheck disable=SC2016 # variable in single quote is ok...
	set -- podman image list --format '{{$id := .Id}}{{range .Names}}{{$id}} {{.}}{{println}}{{end}}{{.Id}}'

	# for tmpfs we just check if anything is present,
	# but for disk we need to compare the list with readonly store
	# Note we use the same check as podman_storage_copy here.
	if grep -q 'containers/storage_readonly' /etc/containers/storage.conf; then
		[ -n "$(podman image list --filter readonly=false -q)" ]
	else
		[ "$("$@" --filter readonly=false 2>/dev/null | sort)" \
			!= "$("$@" --storage-opt \
				additionalimagestore=/var/lib/containers/storage_readonly \
				--filter readonly=true 2>/dev/null | sort)" ]
	fi
}

podman_storage_copy() {
	# copy would normally default to replace for disk and
	# merge for tmpfs, but if a disk storage has been init
	# with an older version of abos-ctrl we still want merge:
	# base the check on storage_readonly additionalimagestore.
	if grep -q 'containers/storage_readonly' /etc/containers/storage.conf; then
		podman_storage_merge
	else
		podman_storage_replace
	fi
}

podman_storage_cleanup() {
	local ask has_local=no

	get_appdev
	podman_storage_has_local && has_local=yes
	case "$CLEANUP,$has_local,$status" in
	# error hard even if there might be nothing to do
	replace,*,tmpfs) error "--cleanup=replace can only be used from disk mode";;
	# explicitly requested no cleanup, or nothing to do on tmpfs: do nothing
	no,*,*|*,no,tmpfs) ;;
	# explicitly requested delete, or nothing to do on disk: remove temporary data
	delete,*,*|*,no,disk) podman_storage_delete;;
	# copy
	copy,yes,*) podman_storage_copy;;
	# replace and merge explicit aliases
	replace,yes,disk) podman_storage_replace;;
	merge,yes,*) podman_storage_merge;;
	# prompt user.
	ask,yes,*)
		echo "List of images configured on development storage:"
		podman image list --filter readonly=false
		echo
		echo "What should we do? ([C]opy (default), [N]othing, [D]elete)"
		while true; do
			read -r ask
			case "$ask" in
			[Dd]|[Dd][Ee][Ll][Ee][Tt][Ee]|[Rr][Ee][Mm][Oo][Vv][Ee])
				podman_storage_delete; break;;
			""|[Cc]|[Cc][Oo][Pp][Yy])
				podman_storage_copy; break;;
			[Rr]|[Rr][Ee][Pp][Ll][Aa][Cc][Ee])
				podman_storage_replace; break;;
			[Mm]|[Mm][Ee][Rr][Gg][Ee])
				podman_storage_merge; break;;
			[Nn]|[Nn][Oo][Tt][Hh][Ii][Nn][Gg])
				break;;
			*)
				echo 'Choice must be empty ("copy"), "nothing", or "delete"';;
			esac
		done
		;;
	*)
		warning "cleanup: Unexpected triplet ($CLEANUP,$has_local,$status), continuing anyway"
		;;
	esac
}

podman_storage_to_tmpfs() {
	podman_check_running
	podman_storage_cleanup

	if [ "$status" = "tmpfs" ]; then
		info "Podman is in tmpfs mode"
		exit 0
	fi

	info "Switching back to tmpfs container storage."
	umount_if_mountpoint /var/lib/containers/storage \
		|| error "Could not unmount container storage: really no container running?"
	sed -i -e 's@^graphroot = .*@graphroot = "/run/containers/storage_root"@' \
			/etc/containers/storage.conf \
		|| error "Could not edit /etc/containers/storage.conf"
	if ! grep -q 'containers/storage_readonly' /etc/containers/storage.conf; then
		sed -i -e 's@^additionalimagestores.*@&\n  "/var/lib/containers/storage_readonly"@' \
				/etc/containers/storage.conf \
			|| error "Could not edit /etc/containers/storage.conf"
	fi
	awk '$2 != "/var/lib/containers/storage"' < /etc/fstab > /etc/fstab.tmp \
		&& mv /etc/fstab.tmp /etc/fstab \
		|| error "Could not edit fstab"
	if command -v persist_file > /dev/null; then
		persist_file /etc/fstab /etc/containers/storage.conf \
			|| error "Could not persist_file fstab or containers/storage.conf"
	fi
	sync

	info "Successfully reverted podman storage to tmpfs"
}

podman_storage_to_disk() {
	local persist_fstab="" persist_storage_conf="" persist_mnt=""

	podman_check_running
	podman_storage_cleanup

	if [ "$status" = "disk" ] && is_mountpoint /var/lib/containers/storage; then
		info "Podman is in disk mode."
		exit 0
	fi

	info "Creating configuration for persistent container storage"

	podman_storage_create_rw_storage "$CLEANUP"
	if awk '$2 == "/var/lib/containers/storage" { exit(1); }' /etc/fstab; then
		local newline
		newline=$(sed -ne 's@/var/tmp\(.*\)subvol=tmp@/var/lib/containers/storage\1subvol=containers_storage@p' \
				< /etc/fstab)
		[ -n "$newline" ] || error "Could not find /var/tmp in fstab to copy from"
		echo "$newline" >> /etc/fstab \
			|| error "Could not write to fstab"
		persist_fstab=1
	fi

	if ! grep -qx 'graphroot = "/var/lib/containers/storage"' /etc/containers/storage.conf; then
		sed -i -e 's@^graphroot = .*@graphroot = "/var/lib/containers/storage"@' \
			-e '/containers\/storage_readonly/d' /etc/containers/storage.conf
		persist_storage_conf=1
	fi

	if ! [ -d /var/lib/containers/storage ]; then
		mkdir /var/lib/containers/storage
		persist_mnt=1
	fi
	if command -v persist_file > /dev/null; then
		[ -n "$persist_fstab" ] \
			&& persist_file /etc/fstab
		[ -n "$persist_storage_conf" ] \
			&& persist_file /etc/containers/storage.conf
		[ -n "$persist_mnt" ] \
			&& persist_file -r /var/lib/containers/storage
	fi
	sync
	is_mountpoint /var/lib/containers/storage \
		|| mount /var/lib/containers/storage

	info "Successfully switched podman-storage to disk"
}

podman_storage_status() {
	if [ "$status" = "disk" ]; then
		is_mountpoint /var/lib/containers/storage \
			|| error "disk storage config but /var/lib/containers/storage is not mounted?" \
				 "Run with --tmpfs once to cleanup first"
		echo "Currently in disk mode, run with --tmpfs to switch"
	else
		echo "Currently in tmpfs mode, run with --disk to switch"
	fi
}


ctrl_podman_storage() {
	local status appdev
	local MODE CLEANUP="" arg

	podman_storage_get_status

	tty >/dev/null && CLEANUP=ask
	MODE="$status"

	for arg in "$@"; do
		case "$arg" in
		--tmpfs|--ram)
			MODE=tmpfs
			;;
		--disk|--mmc)
			MODE=disk
			;;
		--status)
			MODE=status
			;;
		--cleanup|--cleanup=remove|--cleanup=delete)
			CLEANUP=delete
			;;
		--cleanup=copy)
			CLEANUP=copy
			;;
		--cleanup=replace)
			CLEANUP=replace
			;;
		--cleanup=merge)
			CLEANUP=merge
			;;
		--no-cleanup|--cleanup=no)
			CLEANUP=no
			;;
		*)
			echo "Usage: $wrapper [--disk|--tmpfs|--status] [--cleanup=(delete|copy|no)]"
			echo
			echo "Make podman use disk or tmpfs for its storage"
			echo
			echo "Cleanup allows copying development data to readonly storage"
			echo "and can be used without configuration switch"
			[ "$arg" != "-h" ] && [ "$arg" != "--help" ] && exit 1
			exit 0
			;;
		esac
	done

	case "$MODE" in
	tmpfs)
		podman_storage_to_tmpfs
		;;
	disk)
		podman_storage_to_disk
		;;
	status)
		podman_storage_status
		;;
	esac
}
