# SPDX-License-Identifier: MIT

# for certs_check_onetime
. "$scripts_dir/certificates.sh"

# replace numfmt...
numfmt_to_bytes() {
	# remove B suffix if any
	local size="${1%[bB]}" unit

	case "${size%[iI]}" in
	# nondigit then digit, two or more non-digits, starting with non-digit
	*[!0-9]*[0-9]|*[!0-9]*[!0-9]*|[!0-9]*)
		error "Unexpected size format $size (expected e.g. 300M)";;
	0*) error "$size cannot start with 0";;
	*[0-9]) echo "$size"; return;;
	esac

	unit="${size##*[0-9]}"
	size="${size%%[!0-9]*}"
	case "$unit" in
	[Kk]) echo "$((size * 1000))";;
	[Kk][Ii]) echo "$((size * 1024))";;
	[Mm]) echo "$((size * 1000000))";;
	[Mm][Ii]) echo "$((size * 1048576))";;
	[Gg]) echo "$((size * 1000000000))";;
	[Gg][Ii]) echo "$((size * 1073741824))";;
	*) error "unsupported unit in $size (only support G/M/K)";;
	esac
}

roundup() {
	local size="$1" divisor="$2" unit="$3"
	local subdigit

	[ "$size" -ge "$divisor" ] || return 1

	subdigit="$((((size % divisor) + (divisor/10-1))/(divisor/10)))"
	size=$((size / divisor))
	if [ "$size" -ge 10 ]; then
		[ "$subdigit" != 0 ] && size=$((size+1))
		subdigit=""
	fi
	echo "$size${subdigit:+.$subdigit}$unit"
}
numfmt_to_human() {
	local size="$1" suffix="$2"
	roundup "$size" 1000000000 "G$suffix" && return
	roundup "$size" 1000000 "M$suffix" && return
	roundup "$size" 1000 "K$suffix" && return
	echo "$size$suffix"
}

losetup_output() {
	# work around race between losetup -f finding a free device and actually allocating one
	# (safe in util-linux, not in busybox)
	local retry=0

	while true; do
		output=$(losetup -f) || error "Could not find a free loop device"
		losetup -P "$output" "$output_file" && break
		retry=$((retry + 1))
		[ "$retry" -lt 10 ] || error "Could not setup loop device"
		sleep 1
	done
}

compute_verity_size() {
	[ -z "$verity_cert" ] && return

	local part_size="$1"

	# verity requires a bit less than 1/126th of it's target partition.
	# (1/128 for hashes plus some metadata)
	# Just in case add an extra MB, and round to the next sector.
	verity_size=$((part_size / 126 + 1024 * 1024))
	verity_size=$(((verity_size + 511) / 512 * 512))
}

_btrfstune() {
	if command -v btrfstune >/dev/null; then
		btrfstune "$@"
		return
	fi

	info "Trying to install btrfstune (btrfs-progs-extra) in memory from internet"
	if ! apk add btrfs-progs-extra; then
		warning "Could not install btrfs-progs-extra"
		[ -n "$btrfstune_optional" ] && return 0
		return 1
	fi
	btrfstune "$@"
}

# helper from libexec/swupdate-usb
LOCKFILE_swupdateusb=/var/lock/swupdate-usb-lock
lock_swupdateusb() {
	local FD_INO FILE_INO

	# fd cannot be put in variable without eval, keep this simple.
	if [ -e "/proc/$$/fd/8" ]; then
		info "fd 8 already taken, overwriting it: $(ls -l /proc/$$/fd/8)"
	fi

	while :; do
		exec 8<>"$LOCKFILE_swupdateusb" \
			|| error "Could not open $LOCKFILE_swupdateusb"
		flock 8 || error "Could not lock $LOCKFILE_swupdateusb"

		# Check if file has changed since opening
		# if it has, drop lock and try again
		FD_INO=$(stat -c "%i" -L "/proc/self/fd/8")
		FILE_INO=$(stat -c "%i" "$LOCKFILE_swupdateusb" 2>/dev/null)

		[ "$FD_INO" = "$FILE_INO" ] && break

		exec 8<&-
	done
}
unlock_swupdateusb() {
	rm -f "$LOCKFILE_swupdateusb"
	exec 8<&-
}


installer_is_sd_installer() {
	# quick check that we can use fs on sd card as is
	info "Checking if ${output_file:-$output} can be used safely..."
	if ! btrfsck --readonly --check-data-csum "${output}p1" >/dev/null 2>&1 \
	    || ! fstype=btrfs opts=,ro mount_rootfs "${output}p1" mnt; then
		info "It looks like your $installer_what does not contain an installer image"
		return 1
	fi

	# mnt/etc/init.d/install was moved to mnt/installer.sh in 3.17-at.7 (2023/06)
	# can simplify the check in a bit more.
	if ! { [ -e "mnt/lib/rc/sh/functions-atmark.sh" ] \
	    && { [ -e "mnt/etc/init.d/install" ] || [ -e "mnt/installer.sh" ]; }; }; then
		umount mnt
		info "It looks like your $installer_what does not contain an installer image"
		return 1
	fi

	local hwrev
	hwrev=$(cat /etc/hwrevision) || error "Could not get hwrevision"
	if ! grep -qF "${hwrev% *}" mnt/lib/rc/sh/functions-atmark-board.sh; then
		info "It looks like your $installer_what does not support ${hwrev% *}"
		return 1
	fi

	# min versions:
	# - 3.15.4-at.7 appfs.lzo restore
	# - 3.19.1-at.1 regenerate sshd host keys
	# - 3.20.3-at.4 disable JTAG and SD boot
	# - 3.20.3-at.5 use baseos.conf instead of installer.conf for security settings
	local version min_version="3.20.3-at.5"
	version=$(cat mnt/etc/atmark-release 2>/dev/null)
	if [ -z "$version" ] || ! printf "%s\n" "$min_version" "$version" | sort -Vc 2>/dev/null; then
		umount mnt
		info "It looks like your installer is too old (found '$version', need at least '$min_version')"
		return 1
	fi
	umount mnt

	if [ -n "$verity_cert" ] && [ -e "${output}p2" ]; then
		info "verity cannot be used with user partition, refusing to disk as is"
		return 1
	fi

	if [ -z "$noprompt" ]; then
		echo "An installer system ($version) is already available on $installer_what. Use it? [Y/n]"
		prompt_yesno y
	else
		info "Using installer image ($version) on $installer_what."
	fi
}

installer_download_aac() {
	local zip_approx_size="" image_approx_size=""

	if [ -z "$installer_image_url" ]; then
		case "$(cat /etc/hwrevision)" in
		iot-a6e*)
			installer_image_url="https://armadillo.atmark-techno.com/files/downloads/armadillo-iot-a6e/image/baseos-6e-installer-latest.zip"
			zip_approx_size=185M
			image_approx_size=580M
			;;
		a610*|a640*)
			installer_image_url="https://armadillo.atmark-techno.com/files/downloads/armadillo-640/image/baseos-600-installer-latest.zip"
			zip_approx_size=145M
			image_approx_size=440M
			;;
		iot-g4*|x2*|AX2*|AGX4*)
			installer_image_url="https://armadillo.atmark-techno.com/files/downloads/armadillo-iot-g4/image/baseos-x2-installer-latest.zip"
			zip_approx_size=170M
			image_approx_size=440M
			;;
		iot-a9e*|armadillo-900*)
			installer_image_url="https://armadillo.atmark-techno.com/files/downloads/armadillo-iot-a9e/image/baseos-900-installer-latest.zip"
			zip_approx_size=220M
			image_approx_size=450M
			;;
		*)
			error "Unknown hardware revision: $(cat /etc/hwrevision 2>&1)"
			;;
		esac
	fi

	# automatic download for lazy people
	local image="${installer_image_url##*/}"
	local domain="${installer_image_url#http*://}"
	domain="${domain%%/*}"
	# default to y only on empty output
	local dl_default=n dl_prompt="y/N"
	if [ -n "$output_file" ] && [ "$(stat -c %b "$output_file")" = 0 ]; then
		dl_default=y
		dl_prompt="Y/n"
	fi
	echo "Download $image image from $domain${zip_approx_size:+ (~$zip_approx_size)} ? [$dl_prompt]"
	[ "$dl_default" = y ] || echo "WARNING: it will overwrite your $installer_what!!"
	prompt_yesno $dl_default || error "Please write an install disk to mmc card manually"

	if [ -n "$image_approx_size" ] \
	    && [ "$(numfmt_to_bytes "$image_approx_size")" -gt "$output_size" ]; then
		[ -n "$output_file" ] \
			&& [ "${output#/dev/loop}" != "$output" ] \
			|| error "$image requires at least ${image_approx_size}B to fit into $output ($(
					numfmt_to_human "$output_size" B
				))"
		info "Growing $output_file to fit $image ($image_approx_size)"
		losetup -d "$output" \
			|| error "Could not release $output"
		truncate -s "$image_approx_size" "$output_file" \
			|| error "Could not resize $output_file"
		losetup_output
	fi

	info "Downloading and extracting image to $installer_what..."

	# any hash file available?
	local zip_cksum_file=""
	if curl -sf -O "$installer_image_url.sha256"; then
		zip_cksum_file="$image.sha256"
	elif curl -sf -O "$installer_image_url.md5"; then
		zip_cksum_file="$image.md5"
	fi
	[ -e "$zip_cksum_file" ] && sed -i -e 's/baseos.*/-/' "$zip_cksum_file"

	# We're writing the image through many, many pipes as follow:
	# curl | tee (zip_tee) | md5sum (zip_cksum) > /dev/null
	#                      | unzip -l (zip_l) | awk (zip_sz) > zip_sz
	#                      | unzip -p (zip_p) | tee (image_tee) | wc -c (image_wc) > sz
	#                                                           | dd (image_dd) > $output
	# (name) is used for pid, as well as *input* pipe (note curl has no input);
	# command is built in the same order except curl coming last.
	local pid_zip_tee pid_zip_cksum pid_zip_l pid_zip_sz pid_zip_p
	local pid_image_tee pid_image_wc pid_image_dd
	mkfifo pipe_zip_tee pipe_zip_cksum pipe_zip_l pipe_zip_sz pipe_zip_p
	mkfifo pipe_image_tee pipe_image_wc pipe_image_dd

	tee pipe_zip_l ${zip_cksum_file:+pipe_zip_cksum} > pipe_zip_p < pipe_zip_tee &
	pid_zip_tee=$!

	case "$zip_cksum_file" in
	*sha256)
		sha256sum -c "$zip_cksum_file" < pipe_zip_cksum >/dev/null 2>&1 &
		pid_zip_cksum=$!
		;;
	*md5)
		md5sum -c "$zip_cksum_file" < pipe_zip_cksum >/dev/null 2>&1 &
		pid_zip_cksum=$!
		;;
	esac

	( busybox unzip -l - && cat; ) < pipe_zip_l > pipe_zip_sz &
	pid_zip_l=$!

	awk '/.img$/ { print $1, $NF }' < pipe_zip_sz > zip_sz &
	pid_zip_sz=$!

	( busybox unzip -p - '*.img' '*.sig' && cat > /dev/null; ) < pipe_zip_p > pipe_image_tee &
	pid_zip_p=$!

	tee pipe_image_wc > pipe_image_dd < pipe_image_tee &
	pid_image_tee=$!

	wc -c < pipe_image_wc > image_sz &
	pid_image_wc=$!

	dd if=pipe_image_dd of="$output" bs=1M oflag=direct \
		iflag=fullblock status=none &
	pid_image_dd=$!

	# let's rock, and check.
	curl -sSf "$installer_image_url" > pipe_zip_tee \
		|| error "Download of $installer_image_url failed"

	wait "$pid_zip_tee" || error "Could not deduplicate zip stream (zip tee $?)"
	[ -z "$pid_zip_cksum" ] || \
		wait "$pid_zip_cksum" \
		|| error "Downloaded file doesn't match checksum (zip ${zip_cksum_file##*.} $?)"
	wait "$pid_zip_l" || error "Could not list archive content (zip -l $?)"
	wait "$pid_zip_sz" || error "Could not extract image size from zip (zip_sz $?)"
	wait "$pid_zip_p" || error "Could not extract image from archive (zip -p $?)"
	wait "$pid_image_tee" || error "Could not deduplicate image stream (image tee $?)"
	wait "$pid_image_wc" || error "Could not compute image size (wc -c $?)"
	wait "$pid_image_dd" || error "Could not write image to $output (dd $?)"

	# check size in zip and size written match
	# zip_sz contains original image name in second word
	local zip_sz image_name image_sz
	zip_sz=$(cat zip_sz) && [ -n "$zip_sz" ] \
		|| error "Could not read installer size from zip"
	image_name="${zip_sz#* }"
	zip_sz="${zip_sz%% *}"
	image_sz=$(cat image_sz) && [ -n "$image_sz" ] \
		|| error "Could not read installer size"
	# image size contains .sig file as well
	[ "$image_sz" -gt "$zip_sz" ] \
		|| error "Installer size did not match (expected $zip_sz, got $image_sz)"
	[ "$((zip_sz % 512))" = 0 ] \
		|| error "baseos img size should be a multiple of 512"

	rm -f pipe_zip_tee pipe_zip_cksum pipe_zip_l pipe_zip_sz pipe_zip_p \
		 pipe_image_tee pipe_image_wc pipe_image_dd \
		 zip_sz image_sz

	info "Finished writing $image_name, verifying written content..."

	# check sig if it is available
	local sig_sz=$((image_sz - zip_sz))
	if [ "$sig_sz" = 0 ]; then
		echo "Signature was not available, trust downloaded image? [y/N]"
		if [ -z "$zip_cksum_file" ]; then
		       echo "Note there was also no checksum, we recommend creating a .sha256 file next to the archive"
		fi
		prompt_yesno n || error "Aborted"

		# at least check btrfs integrity for sanity
		echo 3 > /proc/sys/vm/drop_caches
		btrfsck --readonly --check-data-csum "${output}p1" >fsck.log 2>&1 \
			|| error "btrfsck failed, problem writing image? $(cat fsck.log)"
		rm -f fsck.log
		return
	fi

	# extract signature...
	dd if="$output" bs=1M count="$sig_sz" skip="$zip_sz" \
			iflag=count_bytes,skip_bytes status=none \
			of=sig \
		|| error "Could not extract signature"
	# and check it
	dd if="$output" bs=1M count="$zip_sz" \
			iflag=direct,count_bytes status=none \
		| openssl cms -verify -no_check_time -inform DER -in sig \
			-content /dev/stdin -nosmimecap -binary \
			-CAfile "$scripts_dir/certificates_atmark.pem" \
			>/dev/null \
		|| error "Signature did not match"
	rm -f sig

}

installer_prompt_custom_partition() {
	# "return" value is stored in $part_end, end of main partition if
	# user partition is created.
	local partition_size

	# verity enabled: user partition is not supported
	[ -n "$verity_cert" ] && return

	echo "Would you like to create a windows partition?"
	echo "That partition would only be used for customization script at the end of"
	echo "install, leave at 0 to skip creating it."
	printf "%s${noprompt:+0\n}" "Custom partition size (MB, [0] or 16 - $leftover_size): "
	while true; do
		if [ -z "$noprompt" ]; then
			read -r partition_size
		fi
		[ -n "$partition_size" ] || partition_size=0
		[ "$partition_size" = 0 ] && break
		case "$partition_size" in
		*[!0-9]*)
			echo "Please enter digits only"
			continue
			;;
		esac
		if [ "$partition_size" -gt "$leftover_size" ]; then
			echo "Please enter a size < $leftover_size"
			continue
		fi
		if [ "$partition_size" -lt "16" ]; then
			echo "Please enter a size > 16"
			continue
		fi
		break
	done

	if [ "$partition_size" != "0" ]; then
		local new_end
		# size in sectors. Last sector is inclusive, so
		# must be one less than a multiple of 2048 for alignment
		partition_size=$((partition_size * 2048))
		new_end=$((disk_end - partition_size - 1))
		[ "$new_end" -ge "$part_end" ] \
			|| error "Sorry, rounding made main partition smaller than it currently is" \
				"run again with a smaller user partition"
		part_end="$new_end"
		user_partition=1
	fi
}

installer_resize_and_mount_sd_root() {
	# installer has no other partition, just make it as big as possible
	# (we need bigger for appfs dump)
	local part_start part_end disk_end
	local leftover_size had_verity=""
	local user_partition=""

	# get current main partition start, end and disk size.
	part_start=$(sgdisk -p "$output" | awk -v disk_end=0 '
			/^Disk \/dev/ { disk_end=$3 }
			/^ *1 .*rootfs_0/ {
				part_start=$2;
				if (!part_end) {
					part_end=$3;
				}
			}
			# 2 partitions: mark end as -1
			/^ *2/ { part_end=-1 }
			# fail on anything except 1, 2 and 127 (verity)
			/^ *[0-9]/ && ! / *(1|2|127) / { exit(1) }
			# fail if sector size is not 512
			/Sector size/ && $4 != "512/512" { exit(2); }
			END {
				if (part_start) {
					print part_start, part_end, disk_end
				}
			}')
	case "$?" in
	0) ;;
	1) error "sd installer should only have p1 and optionally p2 or p127";;
	2) error "$installer_what card should have 512 byte block size";;
	*) error "unknown error while trying to get $installer_what partitions";;
	esac

	[ -n "$part_start" ] || error "Could not find installer main partition"
	# split variable by space in ugly sh...
	disk_end="${part_start##* }"
	part_start="${part_start% *}"
	part_end="${part_start##* }"
	part_start="${part_start% *}"
	[ "$disk_end" != 0 ] || error "Could not parse disk total size"

	# Round down disk_end to a multiple of 1MB for later compuations
	# This should be noop.
	disk_end=$(((disk_end/2048)*2048))
	# sizes in MB. disk size is leftover disk size, rounded down
	leftover_size=$(((disk_end - part_end) / 2048))

	# already resized: mount and stop here if two partitions (end=-1) or
	# trailing space is < 16MB
	if [ -z "$verity_cert" ] \
	    && { [ "$part_end" = "-1" ] || [ "$leftover_size" -lt 16 ]; }; then
		fstype=btrfs mount_rootfs "${output}p1" mnt \
			|| error "Could not mount installer partition"
		return
	fi

	installer_prompt_custom_partition

	if [ -n "$verity_cert" ]; then
		[ "$part_end" = "-1" ] && error "verity does not support user partition"

		compute_verity_size "$(((part_end - part_start) * 512))"

		if [ "$verity_size" -gt "$(((disk_end - part_end + 2048 + 34) * 512))" ]; then
			# cannot fit verity image as is. if it is a file, try to grow it a bit...
			if [ -n "$output_file" ]; then
				disk_end=$((part_end + 2048 + 34 + (verity_size / 512)))
				info "Growing $output_file to fit verity partition"
				losetup -d "$output" \
					|| error "Could not release $output"
				truncate -s "$((disk_end * 512))" "$output_file" \
					|| error "Could not resize $output_file"
				losetup_output
			else
				error "Would need to shrink main installer partition for verity" \
					"Resize disk manually or download a new image."
			fi
		fi
		# also give it 1MB for alignment and 34 sectors for backup gpt
		part_end=-$((verity_size / 512 + 2048 + 34))
	elif [ -z "$user_partition" ]; then
		# no extra partition - extend to the end
		part_end=""
	fi

	# remove pre-existing verity partition if present
	[ -e "$output"p127 ] && had_verity=1
	# unset write-protected flag if set
	if btrfs ins dump-super "$output"p1 | grep -qw SEEDING; then
		_btrfstune -S 0 -f "$output"p1 2>/dev/null \
			|| error "Could not clear write-protect on ${output}p1"
	fi

	info "Growing installer main partition"

	# resize GPT headers first; any command would work here but
	# we cannot specify a $part_end > previous end size without this
	# (note: type 0700 is 'Microsoft basic data', apparently required to recognize
	# the partition on Windows 11...)
	sgdisk -G "$output" >/dev/null \
		&& sgdisk -d 1 ${had_verity:+-d 127} \
			--new "1:$part_start:$part_end" -c 1:rootfs_0 \
			${user_partition:+--new "2:$((part_end +1)):" -c 2:install_data -t 2:0700} \
			"$output" >/dev/null \
		|| error "resizing partitions failed"

	if [ -n "$user_partition" ]; then
		local mkfs=vfat

		# exfat is supposedly better for user partitions, use it if we can
		# but give up quickly: user can always reformat later if they wish
		if command -v mkfs.exfat; then
			mkfs=exfat
		elif ! curl -fs -o /dev/null https://dl-cdn.alpinelinux.org; then
			info "Could not reach dl-cdn.alpinelinux.org, formatting partition with vfat"
		else
			info "Trying to install mkfs.exfat (exfatprogs) in memory from internet"
			if apk add exfatprogs; then
				mkfs=exfat
			else
				echo "Could not install exfatprogs, formatting partition with vfat instead"
			fi
		fi
		case "$mkfs" in
		exfat)
			mkfs.exfat -L INST_DATA "${output}p2";;
		vfat)
			mkfs.vfat -n INST_DATA "${output}p2";;
		esac
	fi

	fstype=btrfs mount_rootfs "${output}p1" mnt \
		|| error "Could not mount installer partition"
	local fstype
	fstype=$(findmnt -nr -o FSTYPE mnt)

	# also copy installer_overrides sample to user partition if set...
	if [ -n "$user_partition" ] && mount -t "$mkfs" "${output}p2" mnt/mnt; then
		if [ -e mnt/installer_overrides.sh.sample ]; then
			cp mnt/installer_overrides.sh.sample mnt/mnt/
		fi
		if [ -e mnt/ip_config.txt.sample ]; then
			cp mnt/ip_config.txt.sample mnt/mnt/
		fi
		umount mnt/mnt
	fi

	case "$fstype" in
	ext4)
		umount mnt \
			|| error "Could not umount installer partition"
		e2fsck -f "${output}p1" \
			|| error "e2fsck failed after growing partition"

		# This requires network for ext4 based installer,
		# but any installer used with this should be btrfs
		command -v resize2fs > /dev/null \
			|| apk add e2fsprogs-extra \
			|| error "Need resize2fs (e2fsprogs-extra) to grow installer"
		resize2fs "${output}p1" \
			|| error "resize2fs failed"

		mount -t ext4 "${output}p1" mnt \
			|| error "Could not mount installer partition after growing"
		;;
	btrfs)
		btrfs filesystem resize max mnt \
			|| error "btrfs filesystem resize failed"
		;;
	*)
		error "Unexpected fstype: $fstype. Only ext4/btrfs supported for installer"
		;;
	esac
}

installer_fix_console() {
	# If the console used isn't the default console
	# also set installer console
	local console
	console=$(grep -oE 'console=\S*' /proc/cmdline)

	# console not found in cmdline
	[ -n "$console" ] || return
	# this is the default value
	grep -qxF "$console" mnt/boot/uboot_env.d/00_defaults && return

	info "Setting console to $console in installer"
	echo "$console" > mnt/boot/uboot_env.d/ZZ_installer \
		|| error "Could not write to mnt/boot/uboot_env.d/ZZ_installer"
}

installer_fix_uboot_env() {
	# we need to fix setenv device for loop images
	# output should be /dev/mmcblk* or /dev/loop* but check to be safe...
	[ "$output" = "${output#*[\\&@]}" ] \
		|| error "$output should not contain @ & or \\"
	sed -e "s@/dev/[^ ]*@$output@" mnt/etc/fw_env.config > fw_env.config \
		|| error "Could not create fw_env.config"
	cat mnt/boot/uboot_env.d/* \
		| sed -e 's/upgrade_available=1/upgrade_available=00/' \
		| fw_setenv --config fw_env.config \
			--script - --defenv /dev/null \
		|| error "Could not set installer uboot env"
}

installer_baseos_conf_settings() {
	if [ "$JTAG_DISABLED" = "yes" ]; then
		info "Installer will disable JTAG access"
	fi

	if [ "$SD_BOOT_DISABLED" = "yes" ]; then
		info "Installer will disable SD boot after installation"
	fi

	# Disable boot prompt for SD boot
	if [ "$BOOTPROMPT_DISABLED" = "yes" ]; then
		echo "bootdelay=-2" > "mnt/boot/uboot_env.d/99_noprompt" \
			|| error "Could not write 99_noprompt (installer)"
		info "Installer will disable uboot prompt"
	else
		rm -f "mnt/boot/uboot_env.d/99_noprompt"
	fi
}

installer_copy_root_pw() {
	# Only copy root password if verity is enabled.
	# If verity is not enabled there is no merit to this as this can
	# easily be circumvented
	# This ensures that in case of error at install debug can only be
	# done with proper credentials
	[ -z "$verity_key" ] && return

	info "Copying armadillo's root password to installer"
	{
		grep ^root: /etc/shadow
		grep -v ^root: mnt/etc/shadow
	} > mnt/etc/shadow.tmp \
		&& mv mnt/etc/shadow.tmp mnt/etc/shadow \
		|| error "Could not write mnt/etc/shadow?"
}

installer_copy_sd() {
	[ -z "$copy_sd" ] && return 0
	# Copy signed kernel to SD boot image.
	cp -a "$copy_sd/." mnt/ \
		|| error "Could not copy kernel image to SD card for secureboot"

}

installer_update_installer_sd() {
	# these functions modify installer's boot env and might require
	# a refresh
	[ -e mnt/boot/uboot_env.d/00_defaults ] \
		|| error "Installer must provide /boot/uboot_env.d/00_defaults"
	installer_fix_console
	installer_fix_uboot_env
	installer_baseos_conf_settings
	installer_copy_root_pw

	# remove any previous installer_swus
	rm -rf mnt/installer_swus

	installer_copy_sd
}

installer_to_lzo_and_csum() {
	# reads piped data, writes lzo and xxh file
	# build-rootfs's build_image.sh's copy_to_lzo_and_csum
	# almost verbatim (sudos removed)
	local dest="$1"
	local sz="$2"
	shift 2
	local xxh
	local pid_xxh pid_wc pid_tee pid_source

	# assume we can read source multiple times
	mkfifo pipe_tee pipe_xxh pipe_lzop pipe_wc
	xxhsum < pipe_xxh > xxh &
	pid_xxh=$!

	wc -c < pipe_wc > sz &
	pid_wc=$!

	tee pipe_lzop pipe_xxh > pipe_wc < pipe_tee &
	pid_tee=$!

	"$@" > pipe_tee &
	pid_source=$!

	lzop > "mnt/$dest.lzo" < pipe_lzop \
		|| error "Failed compressing or writing lzo file $dest"
	wait "$pid_source" \
		|| error "Source command failed for $dest"
	wait "$pid_xxh" \
		|| error "Computing xxh failed for $dest"
	wait "$pid_wc" \
		|| error "wc failed for $dest"
	wait "$pid_tee" \
		|| error "tee failed for $dest"
	if [ -n "$sz" ]; then
		[ "$sz" = "$(cat sz)" ] \
			|| error "Stream was not of the expected size"
	else
		sz="$(cat sz)" \
			|| error "Could not read size"
		# sanity check
		[ -n "$sz" ] && [ "$sz" != 0 ] \
			|| error "Read size was 0, problem with pipes?"
	fi
	xxh=$(cat xxh) \
		|| error "Could not read xxh"
	xxh=${xxh%% *}
	wait "$pid_wc" \
		|| error "$dest file does not have expected size (expected $sz)"
	echo "$sz $xxh" > "mnt/$dest.xxh" \
		|| error "Could not write $dest checksum"
	rm -f pipe_lzop pipe_xxh pipe_wc pipe_tee xxh sz
	local xxhcheck
	xxhcheck="$(lzop -d < "mnt/$dest.lzo" | xxhsum)"
	xxhcheck="${xxhcheck%% *}"
	[ "$xxh" = "$xxhcheck" ] \
		|| error "Sha we just wrote does not match (expected $xxh got $xxhcheck"
}

installer_dump_bootloader() {
	# copy of mkswu's pre_boot.sh copy_boot_imxboot,
	# with extra checks (fw_env.config, encryption)
	local env_offset sz_mb

	# skip if no fw_env.config
	if ! [ -e /etc/fw_env.config ]; then
		return
	fi

	# We copy until env start (assume nothing else than boot image before)
	# Env will be cleared in post_boot
	env_offset=$(awk '/^[^#]/ && $2 > 0 {
			if (!start || $2 < start)
				start = $2;
		}
		END {
			if (!start) exit(1);
			printf("%d\n", start);
		}
		' < /etc/fw_env.config) \
		|| error "Could not get boot env location"

	# use temporary file to check uboot is not encrypted, it's small anyway.
	dd if="${rootdev}boot$ab" of=boot bs=1M count="$env_offset" \
			iflag=count_bytes status=none \
		|| error "Could not read boot image from ${rootdev}boot$ab"

	if ! strings boot | grep -qE 'U-Boot 20[0-9]{2}.[0-1][0-9]'; then
		warning "Could not recognize boot image (encrypted?), keeping installer version"
		rm -f boot
		return
	fi

	info "Copying boot image"

	# round up to next MB (clear env on target)
	sz_mb=$(((env_offset - 1) / 1024 / 1024 + 1))
	truncate -s "${sz_mb}M" boot || error "Could not resize boot image"
	installer_to_lzo_and_csum boot "$((sz_mb * 1024 * 1024))" \
		cat boot
	echo "boot (make-rootfs on $(date))" > mnt/boot.filename \
		|| error "Could not overwrite image filename"
}

check_rootfs_ro() {
	# check there is no writer
	# fun fact: while remount,rw is local to a private mount, remount,ro affects
	# all instances of the fs.
	# the last unshare with a rw mount disappearing also doesn't make the fs back to
	# ro globally (last column of mountinfo)
	# Since we no longer do a block-level copy we can accept a writable fs, but if files
	# get modified something might not be coherent so make the user check.

	local rootopt
	# rootfs is listed as /dev/root (ext4) or $rootfspart (/dev/mmcblk2pX) (btrfs),
	# pick the first match.
	if rootopt=$(awk -v root="$rootfspart" '
			$10 == root || $10 == "/dev/root" {
				print $NF
				exit(1)
			}' /proc/self/mountinfo); then
		# note on mountinfo: $7 is optional (zero or more optional fields), we want the
		# partition that is two fields after '-'... But that is hard to do, so assume
		# rootfs always has exactly one optional field (shared:x) and is $10 and warn
		# if we did not find it.
		echo "WARNING: Did not find '$rootfspart' in /proc/self/mountinfo, skipping rw check"
		echo "(If rootfs changes while we copy it some data may be corrupted)"
		echo "Continue anyway? [y/N]"
		prompt_yesno n || error "Aborted"
		return
	fi
	case ,"$rootopt", in
	*,rw,*) # persist_file or similar was used, needs check.
		;;
	*)      # already ro: remount should be noop and always work.
		mount -o remount,ro /live/rootfs 2>/dev/null && return
		;;
	esac

	# check if still actively rw for more helpful message
	local writers pid pids=""
	writers="$(awk '$6 ~ /(^|,)rw(,|$)/ && $10 == "'"$rootfspart"'" {
				print FILENAME
			}' /proc/*/mountinfo 2>/dev/null)"
	for pid in $writers; do
		pid="${pid#/proc/}"
		pid="${pid%/mountinfo}"
		pids="$pids $pid"
	done

	if [ -n "$pids" ]; then
		# Some processes still have a rw fs - they might not be actually using it,
		# but remounting ro could break later writes so we should not do it.
		echo "WARNING: The following processes are possibly writing in rootfs:"
		ps x | awk -v "list=$pids" '
			BEGIN {
				split(list, array);
				for (i in array)
					pids[array[i]]=1;
			}
			($1 in pids)'
	elif ! mount -o remount,ro /live/rootfs >/dev/null 2>&1; then
		# remount didn't work, something we didn't find is still using the fs
		# (possibly holding deleted files)
		echo "WARNING: Could not remount the rootfs read-only, something might still be modifying it."
	else
		# remount,ro worked, all clear.
		return
	fi
	echo "If rootfs changes while we copy it some data may be corrupted, consider rebooting first."
	echo "Continue anyway? [y/N]"
	prompt_yesno n || error "Aborted"
	rootfs_rw=1
}

dump_linux_mmcboot() {
	# kernel is a FIT image iff it is signed so that is easy to check.
	# Installer (sd card) kernel first
	if [ "$(head -c 4 "mnt/boot/Image" | xxd -p)" != "d00dfeed" ]; then
		error "enabling secure boot requires installer /boot/Image to be signed"
	fi

	# And rootfs kernel (from mmcbootX if valid or /boot/Image)
	local sz
	if grep -qE '^boot_linux ' /etc/sw-versions; then
		sz=$(blockdev --getsize64 "${rootdev}boot$ab") \
			|| error "Could not get size of ${rootdev}boot$ab"
		sz=$((sz - 5 * 1024 * 1024))
		set -- dd if="${rootdev}boot$ab" bs=1M skip=5 status=none
	else
		sz=$(stat -c %s /boot/Image) \
			|| error "Could not get size of /boot/Image"
		set -- cat /boot/Image
	fi
	if [ "$("$@" | head -c 4 | xxd -p)" != "d00dfeed" ]; then
		error "enabling secure boot requires install kernel to be signed"
	fi
	if ! "$@" | grep -q ramdisk; then
		error "disk encryption requires the kernel to be built with an initrd"
	fi
	installer_to_lzo_and_csum boot_linux "$sz" "$@"
}

dump_linux_secboot() {
	# for a900, check kernel starts with a single ahab container
	# Installer (sd card) kernel first
	if [ "$(head -c 4 "mnt/boot/Image" | xxd -p)" != "00c00287" ]; then
		error "enabling secure boot requires installer /boot/Image to be signed"
	fi

	# And rootfs kernel (from mmcbootX if valid or secboot/boot/Image)
	local sz
	local boot_image="/boot/Image"
	if grep -qE '^boot_linux ' /etc/sw-versions; then
		mount "${rootdev}p$((ab + 10))" secboot \
			|| error "Could not mount ${rootdev}p$((ab + 10)) to secboot"
		boot_image="secboot/boot/Image"
	fi
	sz=$(stat -c %s "$boot_image") \
		|| error "Could not get size of $boot_image"
	set -- cat "$boot_image"

	if [ "$("$@" | head -c 4 | xxd -p)" != "00c00287" ]; then
		error "enabling secure boot requires install kernel to be signed"
	fi
	if ! "$@" | grep -q ramdisk; then
		error "disk encryption requires the kernel to be built with an initrd"
	fi
	installer_to_lzo_and_csum secboot_linux "$sz" "$@"
	is_mountpoint "secboot" && umount "secboot"
}

installer_dump_secureboot() {
	if [ "$SECUREBOOT_ENABLED" != "yes" ]; then
		rm -f boot
		return
	fi

	info "Installer will enable secure boot"

	# Copy SRK Hash if Secure boot enabled.
	if device-info --is-secure secureboot; then
		device-info --show-srk > "mnt/secureboot_srk" \
			|| error "Could not write to /etc/atmark/baseos.conf"
	else
		error "Failed to copy SRK Hash because secure boot disabled."
	fi

	if [ -n "$verity_cert" ] && ! strings mnt/boot/Image | grep -q ramdisk; then
		error "verity enabled but installer /boot/Image does not have an initrd"
	fi

	case "$(cat /etc/hwrevision)" in
	iot-g4*|x2*|AX2*|AGX4*)
		dump_linux_mmcboot
		;;
	iot-a9e*|armadillo-900*)
		dump_linux_secboot
		;;
	*)
		error "Unknown hardware revision: $(cat /etc/hwrevision 2>&1)"
		;;
	esac

	if ! [ -e boot ]; then
		error "enabling secure boot requires plain text uboot installer"
	fi
	# Copy signed boot loader to SD boot image.
	dd if=boot of="$output" bs=1k seek=32 status=none \
		|| error "Could not copy boot image to SD card for secureboot"

	rm -f boot
}

installer_dump_rootfs() {
	local avail fstype rootfs_rw=""

	info "Copying rootfs"

	avail=$(findmnt -nr --bytes -o AVAIL /live/rootfs)
	[ -n "$avail" ] || error "Could not find rootfs mount"
	[ "$avail" -gt 1048576 ] \
		|| error "The rootfs is full or almost full ($((avail/1024))KB free)" \
			 "Installer requires some free space to work, please free up at least 1MB and restart"

	check_rootfs_ro

	# try to run fsck as appropriate, no point in saving a broken fs
	fstype=$(findmnt -nr -o FSTYPE /live/rootfs)
	case "$fstype" in
	ext4)   # cannot run fsck if fs is still rw for ext4 (journal not clean)
		if [ -z "$rootfs_rw" ]; then
			e2fsck -f -n "$rootfspart" >fsck.log 2>&1 \
				|| error "e2fsck failed for rootfs, aborting: $(cat fsck.log)"
		fi;;
	btrfs) btrfsck --readonly --force "$rootfspart" >fsck.log 2>&1 \
		|| error "btrfsck failed for rootfs, aborting: $(cat fsck.log)";;
	*) error "Unknown fstype for rootfs, aborting";;
	esac
	rm -f fsck.log

	# Copy into a temporary fs
	truncate -s "$rootpart_size" rootpart

	# XXX share with rollback-clone in later version
	local extlinux=""
	[ -e "/boot/extlinux.conf" ] && extlinux=1
	case "$fstype" in
	ext4)
		mkfs.ext4 -q ${extlinux:+-O "^64bit"} -L "rootfs_0" -F rootpart
		;;
	btrfs)
		mkfs.btrfs -q -L "rootfs_0" -m dup -f rootpart
		;;
	esac || error "Could not create fs for rootfs copy"

	mount_rootfs rootpart rootfs || error "Could not mount rootfs target for copy"

	unshare -m sh -c 'mount --bind /live/rootfs /mnt && cp -a /mnt/. rootfs/' \
		|| error "Could not copy existing fs over"

	# installer will fix that anyway, but might as well make a clean rootfs_0
	if [ -e rootfs/etc/fw_env.config ]; then
		sed -i -e "s/boot[01]/boot0/" rootfs/etc/fw_env.config \
			|| error "Could not update target fw_env.config"
	fi
	sed -i -e "s/boot_[01]/boot_0/" rootfs/etc/fstab \
		|| error "Could not update target fstab"

	if [ -z "$insecure" ]; then
		# remove ssh host keys and mark for regeneration
		if stat rootfs/etc/ssh/ssh_host_*key* >/dev/null 2>&1; then
			rm -f rootfs/etc/ssh/ssh_host_*key* \
				&& touch rootfs/etc/ssh/ssh_host_keys_installer_regenerate \
				|| error "Could not remove ssh host keys"
		fi
		# close node-red admin interface
		if [ -e rootfs/etc/atmark/containers/node-red.conf ]; then
			sed -i -e '/add_ports 1880:1880/d' \
				rootfs/etc/atmark/containers/node-red.conf
		fi
	fi

	# Disable boot prompt - just in case, we don't remove file
	# if unset and it does not exist.
	if [ "$BOOTPROMPT_DISABLED" = "yes" ]; then
		printf "%s\n" "# File created by abos-ctrl make-installer, do not edit" \
				"bootdelay=-2" > "rootfs/boot/uboot_env.d/99_noprompt" \
			|| error "Could not write 99_noprompt (rootfs)"
	fi

	umount rootfs || error "Could not umount rootfs copy"

	installer_to_lzo_and_csum image "$rootpart_size" \
		dd if=rootpart bs=1M status=none

	rm -f rootpart

	echo "rootfs (make-rootfs on $(date))" > mnt/image.filename \
		|| error "Could not overwrite image filename"
}

installer_dump_firmware() {
	local size dev csum

	dev=$(findmnt -nr -o SOURCE /opt/firmware)
	if [ -z "$dev" ]; then
		# only print warning if we are sure there should be one
		# (but we dump if found regardless, we also do not check
		# the value matches what we found...)
		if grep -qE '^HAS_OPT_FIRMWARE' /etc/atmark/baseos.conf 2>/dev/null; then
			warning "/opt/firmware was not mounted, copy skipped"
		fi
		return
	fi
	# we can get the squashfs size through df to not copy too many useless trailing data
	size=$(findmnt -nr --bytes -o SIZE /opt/firmware)
	[ -n "$size" ] || error "Could not get size from /opt/firmware"

	info "Copying /opt/firmware filesystem"
	# this is already squashfs-ified so doesn't use to_lzo_and_csum...
	# do in two steps like build_image, it is small enough.
	dd if="$dev" of=mnt/firm.squashfs bs=1M count="$size" iflag=count_bytes status=none \
		|| error "Could not copy firmware to $installer_what"
	csum=$(xxhsum -q mnt/firm.squashfs) \
		|| error "Could not compute checksum for firmware partition"
	echo "${csum%% *}" > mnt/firm.squashfs.xxh \
		|| error "Could not write firmware checksum"
}

installer_btrfs_del_snap() {
	local snap
	for snap in "$@"; do
		! [ -d "app/snapshots/$snap" ] \
			|| btrfs -q subvolume del "app/snapshots/$snap" \
			|| error "Could not delete snaphsots/$snap subvolume in appfs"
	done
}

installer_btrfs_snap() {
	local src="$1"
	local dst="$2"

	btrfs -q subvolume snapshot -r "$src" "app/snapshots/$dst" \
		|| error "Could not create snapshot for $src"
	snapshots="$snapshots app/snapshots/$dst"
}

installer_dump_appfs() {
	local appdev snapshots=""

	info "Copying appfs"

	get_appdev

	mount -t btrfs "$appdev" app \
		|| error "Could not mount appfs root"

	installer_btrfs_del_snap volumes boot_volumes \
		containers_storage boot_containers_storage

	[ -d app/snapshots ] \
		|| mkdir app/snapshots \
		|| error "Could not create snapshot dir in appfs"

	installer_btrfs_snap /var/app/volumes volumes
	installer_btrfs_snap /var/app/rollback/volumes boot_volumes
	installer_btrfs_snap /var/lib/containers/storage_readonly boot_containers_storage
	is_mountpoint /var/lib/containers/storage \
		&& installer_btrfs_snap /var/lib/containers/storage containers_storage

	# btrfs send outputs both status messages and errors to stderr,
	# so filter manually to keep only real errors on stderr
	mkfifo pipe_btrfs_send
	awk '{ if ($0 ~ /^At subvol/) { print } else { print >"/dev/stderr" } }' < pipe_btrfs_send &
	local awk_pid=$!
	# shellcheck disable=SC2086 ## snapshots is split on purpose
	installer_to_lzo_and_csum appfs "" \
			btrfs send $snapshots > pipe_btrfs_send 2>&1
	wait "$awk_pid" # ignore awk failure
	rm -f pipe_btrfs_send
}

get_file_holes_size() {
	local file="$1"
	# a sparse file has two "sizes":
	# - the apparent size as reported through e.g. ls
	# - the allocated size as reported through e.g. du
	# This computes the difference between the two, to check how much could
	# still be allocated in this file.
	local apparent allocated blocksize
	apparent=$(stat -c "%s %b %B" "$file" 2>/dev/null) || return
	# break the three words by space...
	blocksize=${apparent##* }
	apparent=${apparent% *}
	allocated=${apparent#* }
	apparent=${apparent% *}

	# allocated is number of blocks * block sizes
	allocated=$((allocated * blocksize))

	# bail out if allocated is bigger than apparent size...
	[ "$apparent" -gt "$allocated" ] || return

	echo $((apparent - allocated))
}

file_is_in_appfs() {
	local file="$1"
	[ "$(findmnt -nr -o UUID --target "$file")" = "$(findmnt -nr -o UUID /var/tmp)" ]
}


installer_checks() {
	local freesize holesize reqsize=0

	[ -b "$output" ] || error "Didn't set up loop device properly? Please report this error."
	if ! partprobe "$output"; then
		error "$output seems used (partprobe could not reload partitions), cleanup any existing mount first"
	fi
	if [ -n "$output_file" ]; then
		# image file: check if there's a loop device for it
		if losetup -a | grep -v "$output:" | grep -qw "$output_file"; then
			error "$output_file found in losetup -a output: cleanup any existing mount first"
		fi
		# Check there is enough space to write image:
		# - if the output file is in appfs, we need to add up the size for later check
		# - otherwise just check now.
		if holesize=$(get_file_holes_size "$output_file"); then
			if file_is_in_appfs "$output_file"; then
				reqsize="$holesize"
			else
				freesize=$(findmnt -nr --bytes -o AVAIL --target "$output_file" 2>/dev/null)
				if [ -n "$freesize" ] && [ "$freesize" -lt "$holesize" ]; then
					echo "WARNING: There might not be enough space to write $output_file"
					echo "WARNING: (wanted $(
							numfmt_to_human "$holesize" B
						) more, got $(
							numfmt_to_human "$freesize" B
						))"
					echo "Continue building image anyway? [y/N]"
					prompt_yesno n || error "Aborting at user request"
				fi
			fi
		fi
	fi
	# fail building installer unless magic variable is set
	[ -n "$INSTALLER_ALLOW_ONETIME_CERT$insecure" ] \
		|| ! certs_has_onetime \
		|| error "Refusing to build installer image with public onetime swupdate certificate installed" \
			"Please install initial_setup.swu (from mkswu --init) first"

	if [ -n "$verity_cert" ]; then
		command -v veritysetup >/dev/null \
			|| error "veritysetup command is missing for verity support"
		# check key is not on disk
		local ondisk=""
		case "$verity_key" in
		/var/app/*|/live/rootfs/*) ondisk=1;;
		*) [ -e "/live/rootfs$verity_key" ] && ondisk=1;;
		esac
		if [ -n "$ondisk" ]; then
			error "$verity_key appears to be on disk," \
				"which would allow anyone to re-sign the imager." \
				"Please store it in a temporary directory that will not be copied"
		fi
	fi

	freesize=$(findmnt -nr --bytes -o AVAIL --target /var/tmp 2>/dev/null)
	reqsize=$((reqsize+rootpart_size))
	if [ -n "$freesize" ] && [ "$freesize" -lt "$reqsize" ]; then
		echo "WARNING: You could need up to $(
				numfmt_to_human "$reqsize" B
			) free in /var/tmp to create installer"
		echo "WARNING: Please free some space before running again (currently $(
				numfmt_to_human "$freesize" B
			) available)"
		echo "(You can run \`abos-ctrl rollback-clone\` to free up rollback data if any)"
		echo "Continue building image anyway? [y/N]"
		prompt_yesno n || error "Aborting at user request"
	fi

	output_size=$(blockdev --getsize64 "$output" 2>/dev/null) \
		|| error "You need a SD card plugged in to build installer: nothing found at $output"
	if [ "$output_size" -lt "$used_size" ]; then
		echo "WARNING: Your $installer_what ($(
				numfmt_to_human "$output_size" B
			)) is smaller than estimated max required size ($(
				numfmt_to_human "$used_size" B
			))"
		echo "Continue anyway? [y/N]"
		prompt_yesno n || error "Aborting at user request"
	fi
	if is_mountpoint /var/lib/containers/storage; then
		echo "WARNING: using emmc for podman-storage is a development option and not supported for production"
		echo "WARNING: consider switching back to tmpfs with 'abos-ctrl podman-storage --tmpfs' first"
		echo "Continue building image? [y/N]"
		prompt_yesno n || error "Aborting at user request"
	fi
}

installer_cleanup() {
	if [ "${output#/dev/loop}" != "$output" ]; then
		losetup -d "$output"
		output="$output_file"
	fi
	[ -n "$tmpdir" ] || return

	if [ "$PWD" = "$tmpdir" ]; then
		is_mountpoint "rootfs" && umount "rootfs"
		is_mountpoint "mnt" && umount "mnt"
		is_mountpoint "secboot" && umount "secboot"
		if is_mountpoint "app"; then
			installer_btrfs_del_snap volumes boot_volumes \
				containers_storage boot_containers_storage
			rmdir app/snapshots >/dev/null 2>&1
			umount "app"
		fi
	fi
}
final_cleanup() {
	[ "${output#/dev/loop}" != "$output" ] && losetup -d "$output"
	is_mountpoint "mnt" && umount "mnt"
	find "$tmpdir" -xdev -delete 2>/dev/null
	unlock_swupdateusb
}

installer_trap() {
	info "Terminating, cleaning up..."

	trap - EXIT INT QUIT TERM
	installer_cleanup
	final_cleanup
	exit 1
}

installer_fstrim() {
	# trim data if we think we can: output is a real device or
	# a filesystem that supports punch hole
	# (any other filesystem would display scary errors in dmesg)
	if [ -n "$output_file" ]; then
		case "$(findmnt -nr -o FSTYPE --target "$output_file")" in
		ext4|xfs|btrfs) ;;
		*) return;;
		esac
	fi

	fstrim "$mountpoint"
}

installer_shrink_install_disk() {
	local mountpoint=mnt
	local part_start part_end
	local new_size new_end

	part_start=$(sgdisk -p "$output" 2>/dev/null | awk '
		/^ *1/ { part_start=$2; part_end=$3 }
		/^ *(1[0-9]|[2-9])/ { exit(1) }
		END { print part_start, part_end }
	') || return 0
	part_end="${part_start##* }"
	part_start="${part_start% *}"

	# shrink the btrfs volume
	info "Trying to shrink the installer partition..."
	while btrfs filesystem resize -200M "$mountpoint" > /dev/null 2>&1; do
		true
	done

	# trim after we're done with resize
	installer_fstrim

	new_size="$(btrfs filesystem show --raw "$mountpoint" | \
		awk '/devid/ {print $4}')"
	# round up to next sector if required (just in case)
	new_end=$((part_start + (new_size + 511) / 512))
	if [ "$new_end" -ge "$part_end" ]; then
		info "Installer partition is not shrinkable"
		return
	fi

	info "Shrinking the installer partition..."
	if ! umount "$mountpoint"; then
		info "Sleeping a while and retrying..."
		sleep 3
		if ! umount "$mountpoint"; then
			warning "Could not unmount the $installer_what to shrink the partition." \
				"Shrink manually if required."
			# don't leave partition half-resize behind
			btrfs filesystem resize max "$mountpoint" >/dev/null 2>&1
			return 0
		fi
	fi

	# resize the partition
	sgdisk -d 1 --new "1:$part_start:$new_end" -c 1:rootfs_0 \
		"$output" >/dev/null \
		|| error "resizing partition failed"

	compute_verity_size "$(((new_end - part_start) * 512))"

	# if the output is image file, shrink that file
	if [ -n "$output_file" ]; then
		# interact with file from here on
		losetup -d "$output"
		output="$output_file"

		local gpt_table_offset filesize
		gpt_table_offset=$(sgdisk -p "$output" 2>/dev/null \
			| awk '/Main partition table begins/ { print $NF + 1 }')
		[ -n "$gpt_table_offset" ] || error "Could not get gpt partition table offset"
		filesize=$((new_end * 512 + gpt_table_offset * 512))
		truncate -s "$filesize" "$output_file" \
			|| error "resizing image file failed"

		if [ -n "$verity_cert" ]; then
			# .. and alignment/34 sectors for backup GPT header
			# We truncate twice to ensure that part is zeroed
			filesize=$((filesize + verity_size + (2048 + 34) * 512))
			truncate -s "$filesize" "$output_file" \
				|| error "resizing image file failed"
		fi
		# fix gpt header. This is noisy so always silence output, rerun on error
		sgdisk -e "$output" >/dev/null 2>&1 \
			|| error "fixing GPT header failed: $(sgdisk -e "$output" 2>&1)"
	fi
}

installer_create_verity() {
	[ -n "$verity_cert" ] || return 0

	if [ -n "$output_file" ]; then
		losetup_output
	fi

	if ! [ -e "$output"p127 ]; then
		# create verity partition
		# Default position takes "the largest free area" which could be start of disk,
		# but we reserved room for this partition after rootfs so compute the start
		# sector first.
		local start
		start=$(sgdisk -p "$output" 2>/dev/null \
			| awk '/rootfs_0/ {
					# Round up to the closest MB (2048 sectors boundary)
					print int(($3 + 2048) / 2048) * 2048
				}')
		[ -n "$start" ] || error "Could not find rootfs_0 partition in $output"
		sgdisk -n "127:$start:+$((verity_size/512))" -c 127:verity "$output" >/dev/null \
			|| error "Could not create verity partition (wanted $(
					numfmt_to_human "$verity_size" B
				) from sector $start)"
		mkfs.vfat "$output"p127 \
			|| error "Could not format ${output}p127 (verity)"
	fi

	mount "$output"p127 mnt \
		|| error "Could not mount ${output}p127 (verity)"

	# mark p1 as write-protected if possible
	btrfstune_optional=1 _btrfstune -S 1 "$output"p1 \
		|| error "Could not mark ${output}p1 write-protected"

	info "Computing verity hashes..."

	veritysetup format --root-hash-file=mnt/verity_p1.hashroot \
			"$output"p1 mnt/verity_p1.hashdev >/dev/null \
		|| error "Could not create verity hash device"
	local retries=0
	while ! openssl cms -sign -outform DER -in mnt/verity_p1.hashroot \
			-out mnt/verity_p1.hashroot.sig -nosmimecap -binary \
			-inkey "$verity_key" -signer "$verity_cert" -binary; do
		retries=$((retries + 1))
		[ "$retries" -ge 3 ] && error "Could not sign verity hash root"
		info "verity hash root signature failed ($retries/3), retrying in case password" \
			"was incorrect"
	done

	umount mnt
	if [ -n "$output_file" ]; then
		losetup -d "$output"
		output="$output_file"
	fi
}

installer_make_archive_with_md5() {
	local gzip_md5 image_file

	# if the output isn't image file, do nothing
	[ -n "$output_file" ] || return

	info "Computing install disk image's checksum..."
	image_file="$(basename "$output_file")"
	(
		cd "$(dirname "$output_file")" \
			|| error "Could not enter the image file directory"
		md5sum "$image_file" >"$image_file.md5-in-progress" \
			&& mv "$image_file.md5-in-progress" "$image_file.md5" \
			|| error "Failed to compute the checksum"

		info "Creating the install disk image archive..."
		tar -czf "$image_file.tgz-in-progress" "$image_file" "$image_file.md5" \
			|| error "Failed to create the archive"
		sync "$image_file.tgz-in-progress" \
			&& mv "$image_file.tgz-in-progress" "$image_file.tgz" \
			|| error "Could not rename temporary archive to $image_file.tgz"
	) || exit

	info "Checking archived image's checksum..."
	gzip_md5=$(dd if="$output_file.tgz" bs=1M iflag=direct status=none \
		| tar -zxO "$image_file" | md5sum)
	[ "$(sed -e "s/$image_file/-/g" "$output_file.md5")" = "$gzip_md5" ] \
		|| error "the image checksum does not match!"
}

make_installer_help() {
	echo "Usage: abos-ctrl make-installer [make-installer options]"
	echo
	echo "make-installer options:"
	echo "  --output <image file path>: installer disk image file path,"
	echo "                              defaults to SD card"
	echo "  --image-size <image file size>: specify image file size"
	echo "                                  (in MiB unless unit is specified)"
	echo "  --make-archive: make archive of image file and md5 checksum"
	echo "  --insecure: leave sensitive files in image"
	echo "  --image-url <url>: alternative url to download installer if missing"
	echo "  --verity <cert> <key>: create verity partition"
	echo "                         (secureboot tamper protection)"
	echo "  --copy-sd <directory path>: directory to copy to installer root"
	echo "  --dl-only: stop immediately after downloading image"
	echo
	echo "This command dumps the current system to an installer."
	echo "See also 'abos-ctrl installer-setting' for permanent options."
	echo
	echo "If output has no valid installer, the command can download the latest"
	echo "installer from atmark-techno website."
	echo
	echo "--image-size and --make-archive can only be used with --output."
	echo
	echo "Examples:"
	echo "  abos-ctrl make-installer"
	echo "  abos-ctrl make-installer --output /mnt/installer.img --image-size 600MiB"
	echo "  abos-ctrl make-installer --output /mnt/installer.img --make-archive"
}

ctrl_make_installer() {
	local tmpdir="" used_size existing_size=0
	local output_size output_file=""
	local output="" image_size="" make_archive=""
	local installer_image_url="" insecure=""
	local installer_what="SD card"
	local noprompt="" rootpart_size copy_sd=""
	local verity_cert="" verity_key="" verity_size=0
	local dl_only=""

	if [ -e "/etc/atmark/baseos.conf" ] ; then
		. "/etc/atmark/baseos.conf" \
			|| error "Could not parse baseos.conf"
	fi

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"--image-url")
			[ "$#" -ge 2 ] || error "$1 requires an argument";
			installer_image_url="$2"
			shift 2
			;;
		"--output")
			[ "$#" -ge 2 ] || error "$1 requires an argument";
			output="$2"
			shift 2
			;;
		"--image-size")
			[ -n "$output" ] || error "$1 is only allowed with --output"
			[ "$#" -ge 2 ] || error "$1 requires an argument";
			image_size="$2"
			case "$image_size" in
			*[0-9])
				# assume MiB
				image_size=$((image_size * 1048576))
				;;
			*)
				image_size="$(numfmt_to_bytes "$image_size")" \
					|| error "$2 is not a valid size"
				image_size="${image_size%B}"
				;;
			esac
			# mkfs.btrfs will fail below this size
			[ "$image_size" -ge $((109*1048576)) ] || error "$2 must be at least 109MiB"
			shift 2
			;;
		"--make-archive")
			[ -n "$output" ] || error "$1 is only allowed with --output"
			make_archive="yes"
			shift
			;;
		"--noprompt")
			noprompt=1
			shift
			;;
		"--insecure")
			insecure=1
			shift
			;;
		"--dl-only")
			dl_only=1
			shift
			;;
		"--copy-sd")
			[ "$#" -ge 2 ] || error "$1 requires an argument";
			[ -e "$2" ] || error "$2 did not exist"
			copy_sd="$(realpath "$2")"
			shift 2
			;;
		"--verity")
			[ "$#" -ge 3 ] || error "$1 requires two arguments";
			if [ "$SECUREBOOT_ENABLED" != "yes" ]; then
				error "--verity option can only be used with installer-setting secureboot enabled"
			fi
			if grep -q "BEGIN CERTIFICATE" "$2"; then
				verity_cert="$(realpath "$2")"
				verity_key="$(realpath "$3")"
			elif grep -q "BEGIN CERTIFICATE" "$3"; then
				verity_cert="$(realpath "$3")"
				verity_key="$(realpath "$2")"
			else
				error "verity certificate not found"
			fi
			grep -q "PRIVATE KEY" "$verity_key" \
				|| error "verity key not found"
			shift 3
			;;
		*)
			warning "Invalid option $1"
			make_installer_help >&2
			exit 1
			;;
		esac
	done
	command -v xxhsum > /dev/null \
		|| apk add xxhash \
		|| error "xxhsum command is required and installation from network failed"

	# sets $rootdev and $ab
	dev_from_current
	# rootpart as set from dev_from_current is the underlying mmcblkXpY
	# underlying partition, we want the filesystem partition which can
	# be different in case of encryption
	rootfspart=$(findmnt -nr -o SOURCE /live/rootfs) \
		|| error "Could not get rootfs partition"
	rootpart_size=$(blockdev --getsize64 "$rootfspart") \
		|| error "Could not get size from $rootfspart"

	used_size=$(findmnt -nr --bytes -o USED /var/tmp) \
		|| error "Could not get appfs used size"
	# add 300MB for installer rootfs + rootfs + firm as worst estimate
	used_size=$((used_size + 300 * 1024 * 1024))
	if [ -n "$verity_cert" ]; then
		compute_verity_size "$used_size"
		used_size=$((used_size + verity_size))
	fi
	# round to next sector size
	used_size=$(((used_size + 511) / 512 * 512))

	# forbid swupdate-usb to run while make-installer is running
	# this (saves a bit of time and) avoids fighting races after
	# repartitionning where background mount might happen while we
	# try to access partitions
	lock_swupdateusb
	trap installer_trap EXIT INT QUIT TERM

	if [ -z "$output" ]; then
		for output in /dev/mmcblk[0-9]; do
			# try likely paths without boot partition
			[ -e "$output" ] && ! [ -e "${output}boot0" ] && break
		done
		[ -e "$output" ] && ! [ -e "${output}boot0" ] \
			|| error "Could not find sd card"
	elif ! [ -b "$output" ]; then
		if [ -z "$image_size" ]; then
			image_size="$used_size"
		fi
		# We don't want user to use a partially written installer, so
		# work in a "-partial" file while we operate.
		# We can reuse such a partial file though.
		if [ -e "$output" ]; then
			mv "$output" "$output-in-progress" \
				|| error "Could not rename $output to $output-in-progress"
		fi
		output="$output-in-progress"
		# note what files to cleanup
		if [ -e "$output" ]; then
			existing_size=$(stat -c %s "$output")
		fi
		if [ "$existing_size" -lt "$image_size" ]; then
			truncate -s "$image_size" "$output" \
				|| error "Could not set $output size"
		elif [ "$existing_size" -gt "$image_size" ]; then
			info "$output ($(
					numfmt_to_human "$existing_size" B
				)) was bigger than $(
					numfmt_to_human "$image_size" B
				) and was not truncated."
		fi
		output_file=$(realpath "$output")
		losetup_output
		installer_what="image file"
	fi

	installer_checks

	tmpdir=$(mktemp -d "/var/tmp/make_installer.XXXXXXX") \
		|| error "Could not create tmpdir"
	cd "$tmpdir" || error "Could not enter tmpdir"
	mkdir mnt app rootfs secboot || error "Could not create dirs in tmpdir"

	if [ -n "$dl_only" ]; then
		installer_download_aac
		info "Stopping after download (--dl-only)"
		return
	fi

	printf "%s\n" "${output#/dev/}p1" "${output#/dev/}p2" \
		>> "$LOCKFILE_swupdateusb"
	btrfs device scan --forget
	installer_is_sd_installer || installer_download_aac

	installer_resize_and_mount_sd_root

	installer_update_installer_sd

	installer_dump_bootloader
	installer_dump_secureboot
	installer_dump_rootfs
	installer_dump_firmware
	installer_dump_appfs
	installer_shrink_install_disk

	info "Cleaning up and syncing changes to disk..."

	trap final_cleanup EXIT INT QUIT TERM
	installer_cleanup

	# must be after final umount
	installer_create_verity
	sync

	if [ -n "$output_file" ]; then
		output_file="${output_file%-in-progress}"
		sync "$output_file-in-progress" \
			&& mv "$output_file-in-progress" "$output_file" \
			|| error "Could not rename $output_file-in-progress back to $output_file"
	elif [ -e "${output}p1" ]; then
		info "Checking written media"
		echo 3 > /proc/sys/vm/drop_caches
		btrfsck --readonly --check-data-csum "${output}p1" >fsck.log 2>&1 \
			|| error "btrfsck failed, do not use this card! $(cat fsck.log)"
		rm -f fsck.log
	fi
	[ "$make_archive" != "yes" ] || installer_make_archive_with_md5
	trap - EXIT INT QUIT TERM
	final_cleanup

	info "Installer updated successfully!"
	[ -z "$output_file" ] || ls -lh "${output_file}"*
}
