#!/bin/sh
# SPDX-License-Identifier: MIT

error() {
	printf "%s\n" "$@" >&2
	exit 1
}

enable() {
	abos-ctrl usb-filter enable
}

disable() {
	abos-ctrl usb-filter disable
}

status() {
	abos-ctrl usb-filter status | tr -d "\n"
}

list_devices() {
	abos-ctrl usb-filter list-devices
}

get_rule() {
	abos-ctrl usb-filter list-rules | awk -v list_id="$1" 'NR == list_id'
}

allow_device() {
	vendor_id="$1"
	product_id="$2"
	model="$3"
	usb_interfaces="$4"
	serial="$5"
	abos-ctrl usb-filter allow-device \
		--vendor-id "$vendor_id" --product-id "$product_id" \
		--model "$model" --usb-interfaces "$usb_interfaces" --serial "$serial"
}

list_rules() {
	abos-ctrl usb-filter list-rules
}

remove_rule() {
	abos-ctrl usb-filter remove-rule "$1"
}

list_block_classes() {
	all_classes="$(abos-ctrl usb-filter allow-class | awk 'NR>1 {print}' | tr -d "\t" | tr "\n" " ")"
    # ex: all_classes="Audio CDC HID Physical Image Printer MassStorage Hub CDCdata SmartCard ContentSecurity Video PersonalHealthCare"
	allow_classes="$(abos-ctrl usb-filter list-rules | awk '$2 == "class" {print $3}')"
    # ex: allow_classes="Audio
	# SmartCard"

	if [ -z "$allow_classes" ]; then
		echo "$all_classes" | tr -d "\n"
		return 0
	fi

	allow_classes="$(echo "$allow_classes" | tr "\n" "|" | sed 's/|/ |/g')"
	# ex: allow_classes="Audio |SmartCard |"
	echo "$all_classes" | awk -v exclude="${allow_classes%?}" '{ gsub((exclude), "", $0); print $0 }' | tr -d "\n"
	# ex: "CDC HID Physical Image Printer MassStorage Hub CDCdata ContentSecurity Video PersonalHealthCare "
}

allow_class() {
	abos-ctrl usb-filter allow-class "$1"
}

mode="$1"
shift

case "$mode" in
disable|enable|status \
|list_devices|list_rules \
|list_block_classes|allow_class \
|get_rule|allow_device|remove_rule)
	"$mode" "$@"
	;;
*)
	error "Usage: usb_filter.sh [disable|enable|status|list_devices|list_rules|list_block_classes|allow_class {CLASS}|get_rule/allow_device/remove_rule {ID}]"
	;;
esac
