#!/bin/sh
# SPDX-License-Identifier: MIT

modified_files=""

revert() {
	local IFS="
"

	[ -n "$modified_files" ] || return
	# shellcheck disable=SC2086 ## splitting on purpose
	persist_file -R $modified_files
}

error() {
	printf "%s\n" "$@" >&2
	revert
	exit 1
}

save() {
	local IFS="
"
	# nothing changed
	[ -n "$modified_files" ] || return 0

	chronyd -p >/dev/null || error "New config entry had errors, aborting"
	rc-service chronyd restart \
		|| error "Could not restart chronyd"

	# shellcheck disable=SC2086 ## splitting on purpose
	persist_file $modified_files \
		|| error "Could not persist $modified_files"
}

find_file() {
	# prefer /etc/chrony/conf.d but use atmark one if it exists
	local name="$1.conf"
	if [ -e "/etc/atmark/chrony.conf.d/$name" ]; then
		echo "/etc/atmark/chrony.conf.d/$name"
		return
	fi
	echo "/etc/chrony/conf.d/$name"
}

check_servers() {
	local server IFS="
"
	for server in $1; do
		case "$server" in
		*[\"\']*) error "server $server cannot contain quotes";;
		""|"pool "*|"server "*) ;;
		*) error "server $server should begin with 'pool ' or 'server '";;
		esac
	done
}

set_value() {
	local category="$1" value="$2"
	local file

	file="$(find_file "$category")"

	echo "$value" > "$file" \
		|| error "Could not write to $file"
	modified_files="$modified_files
$file"
}

remove_custom() {
	local name="$1.conf"
	local file="/etc/atmark/chrony.conf.d/$name"
	if [ -e "$file" ]; then
		rm -f "$file" \
			|| error "Could not remove $file"
		modified_files="$modified_files
$file"
	fi
	file="/etc/chrony/conf.d/$name"
	# safety check: only erase /etc file if /lib one is also present
	if [ -e "/lib/chrony.conf.d/$name" ] && [ -e "$file" ]; then
		rm -f "$file" \
			|| error "Could not remove $file"
		modified_files="$modified_files
$file"
	fi
}


# sanity checks
if ! chronyd -p >/dev/null; then
	error "Cannot modify chrony config if it already had errors"
fi

while [ $# -ge 1 ]; do
	case "$1" in
	source)
		[ $# -ge 2 ] || error "source requires an argument"
		case "$2" in
		off|"")
			set_value servers "# servers disabled by ABOS Web"
			;;
		default)
			remove_custom servers
			;;
		*)
			check_servers "$2"
			set_value servers "$2"
			;;
		esac
		shift 2
		;;
	initstepslew)
		[ $# -ge 2 ] || error "initstepslew requires an argument"
		case "$2" in
		off|"")
			set_value initstepslew "# initstepslew disabled by ABOS Web"
			;;
		default)
			remove_custom initstepslew
			;;
		*[\"\'"
"]*)
			error "Cannot have quotes or new lines in initstepslew"
			;;
		*)
			set_value initstepslew "initstepslew $2"
			;;
		esac
		shift 2
		;;
	*)
		error "Unknown option $1"
		;;
	esac
done

save
