# SPDX-License-Identifier: MIT

# required for ctrl_rollback_clone and ctrl_reboot
. "$scripts_dir/rollback_clone.sh"
. "$scripts_dir/reboot.sh"
. "$scripts_dir/btrfs.sh"

# Load custom script to perform additional commands
RESET_DEFAULT_CUSTOM_SCRIPT="/etc/atmark/reset_default_custom.sh"
[ -f "$RESET_DEFAULT_CUSTOM_SCRIPT" ] && . "$RESET_DEFAULT_CUSTOM_SCRIPT"

ctrl_reset_default() {
	local dryrun=1
	local no_btrfs_subvol_sync=""

	while [ "$#" -gt 0 ]; do
		case "$1" in
		-f|--force)
			dryrun="";
			shift;;
		*)
			reset_default_help
			exit 1
			;;
		esac
	done

	if [ -n "$dryrun" ]; then
		info "Run with dry-run mode."
	else
		try_lock_update || error 'Cannot run "abos-ctrl reset-default" while installing a SWU image.'
	fi

	# early custom command if present
	if command -v ${dryrun:+dryrun_}custom_early_commands > /dev/null; then
		(
			${dryrun:+dryrun_}custom_early_commands
		)
	fi

	# abos-web settings
	network_connections_delete
	dhcp_delete
	nat_delete
	vpn_delete
	ntp_delete
	sshd_delete

	# delete user data
	user_data_delete

	# execute additional commands from reset_default_custom.sh
	if command -v ${dryrun:+dryrun_}custom_commands > /dev/null; then
		(
			${dryrun:+dryrun_}custom_commands
		)
	fi

	if [ -n "$ERRORS" ]; then
		warning "There are some soft errors below, but we've done what we can. Please check the error if you care."
		warning "======================"
		warning "$ERRORS"
		warning "======================"
	fi

	if [ "$dryrun" = 1 ]; then
		info ""
		info "Re-run with --force to actually run the above commands"
	else
		# reflected on the B-side
		ctrl_rollback_clone

		# ensure data that can be removed permanently is gone
		# (avoid risk of recovery through recovery utilities)
		# note rollback_clone'd target has been discarded by mkfs
		# Note that for fstrim to work, we need to ensure the filesystem has
		# fully flushed deletions to disk first
		info "Waiting for background data deletion"
		persist_file -c fstrim /target
		if [ -z "$no_btrfs_subvol_sync" ]; then
			btrfs subvolume sync /var/tmp
		fi
		sync -f /var/tmp
		fstrim /var/tmp
		# reboot
		ctrl_reboot
	fi
}

reset_default_help() {
	echo "Usage: abos-ctrl reset-default [reset-default options]"
	echo
	echo "  Delete all of the following settings/files:"
	echo "    Network settings(WWAN, WLAN, LAN, DHCP, NAT, VPN)"
	echo "    NTP settings"
	echo "    /var/app/volumes/*"
	echo "    /var/log/*"
	echo "  and reboot after the process is complete"
	echo
	echo "reset-default options commands:"
	echo "  -f/--force: actually delete the settings/files and reboot"
	echo "              If not set, no file is deleted (dry-run mode)."
}

ERRORS=""
soft_error() {
	warning "$@"
	ERRORS="$ERRORS
$*"
}

network_connections_delete() {
	# delete AP
	if [ -e "/etc/runlevels/default/hostapd" ]; then
		if [ -z "$dryrun" ]; then
			rc-update del hostapd > /dev/null \
				|| soft_error "Could not delete hostapd service"
			persist_file -d "/etc/runlevels/default/hostapd" \
				"/etc/hostapd/hostapd.conf" \
				|| soft_error "Could not delete hostapd setting files"
		fi
		info "rc-update del hostapd"
		info "persist_file -d /etc/runlevels/default/hostapd /etc/hostapd/hostapd.conf"
	fi

	local ignore_if=wlan0
	if [ -e /sys/class/net/uap0 ] && [ -e /sys/class/net/mlan0 ]; then
		ignore_if=mlan0
	fi

	if [ -e /etc/NetworkManager/conf.d/90_abosweb_disable_wlan.conf ]; then
		if [ -z "$dryrun" ]; then
			persist_file -d /etc/NetworkManager/conf.d/90_abosweb_disable_wlan.conf \
				|| soft_error "Could not delete 90_abosweb_disable_wlan.conf"
		fi
		info "persist_file -d /etc/NetworkManager/conf.d/90_abosweb_disable_wlan.conf"
	fi
	if [ -z "$dryrun" ]; then
		if [ -e /sys/class/net/mlan0 ] || [ -e /sys/class/net/wlan0 ]; then
			nmcli d set "$ignore_if" managed yes
			nmcli d wifi list --rescan yes > /dev/null 2>&1
		fi
	fi

	if [ -e /etc/dnsmasq.d/dhcp_ap.conf ]; then
		if [ -z "$dryrun" ]; then
			persist_file -d "/etc/dnsmasq.d/dhcp_ap.conf" > /dev/null \
				|| soft_error "Could not delete dhcp_ap.conf"
		fi
		info "persist_file -d /etc/dnsmasq.d/dhcp_ap.conf"
	fi

	# delete all connections
	if [ -z "$dryrun" ]; then
		rm -f /etc/NetworkManager/system-connections/* \
			|| soft_error "Could not remove system-connections"
		persist_file -r /etc/NetworkManager/system-connections > /dev/null \
			|| soft_error "Could not persist system-connections"
	fi
	info "rm -f /etc/NetworkManager/system-connections/*"
	info "persist_file -r /etc/NetworkManager/system-connections"
}

dhcp_delete() {
	for conf in /etc/dnsmasq.d/abos_web_dhcp_*.conf
	do
		[ -e "$conf" ] || break
		if [ -z "$dryrun" ]; then
			rm -f "$conf" || soft_error "Could not delete $conf"
		fi
		info "rm -f $conf"
	done
	if [ -z "$dryrun" ]; then
		persist_file -r /etc/dnsmasq.d \
			|| soft_error "Could not persist dnsmasq.d"
	fi
	info "persist_file -r /etc/dnsmasq.d"
}

nat_delete() {
	if [ -e /etc/iptables/rules-save ]; then
		if [ -z "$dryrun" ]; then
			# delete all port forwarding & nat settings
			sed -i /etc/iptables/rules-save \
					-e '/\s--comment abos-web-pf\s/d' \
					-e '/\s--comment abos-web-nat\s/d' \
				&& persist_file /etc/iptables/rules-save \
				|| soft_error "Could not reset rules-save"
		fi
		info "sed -i /etc/iptables/rules-save \\
	-e '/\\s--comment abos-web-pf\\s/d' \\
	-e '/\\s--comment abos-web-nat\\s/d'"
		info "persist_file /etc/iptables/rules-save"
	fi

	if [ -e "/etc/sysctl.d/abos_web_ipforward.conf" ]; then
		if [ -z "$dryrun" ]; then
			persist_file -d "/etc/sysctl.d/abos_web_ipforward.conf" \
				|| soft_error "Could not delete abos_web_ipforward.conf"
		fi
		info "persist_file -d /etc/sysctl.d/abos_web_ipforward.conf"
	fi
}

vpn_delete() {
	if [ -e /etc/atmark/containers/abos_web_openvpn.conf ]; then
		if [ -z "$dryrun" ]; then
			podman stop abos_web_openvpn > /dev/null \
				|| soft_error "Could not stop abos_web_openvpn container"
			podman rm abos_web_openvpn > /dev/null \
				|| soft_error "Could not delete abos_web_openvpn container"
			persist_file -d /etc/atmark/containers/abos_web_openvpn.conf \
				|| soft_error "Could not delete abos_web_openvpn.conf"
			persist_file -d /etc/dnsmasq.d/openvpn_dns.conf \
				|| soft_error "Could not delete openvpn_dns.conf"
		fi
		info "podman stop abos_web_openvpn"
		info "podman rm abos_web_openvpn"
		info "persist_file -d /etc/atmark/containers/abos_web_openvpn.conf"
		info "persist_file -d /etc/dnsmasq.d/openvpn_dns.conf"
	fi
	if [ -d /etc/atmark/abos_web/openvpn ]; then
		if [ -z "$dryrun" ]; then
			persist_file -r -d /etc/atmark/abos_web/openvpn \
				|| soft_error "Could not delete openvpn"
		fi
		info "persist_file -r -d /etc/atmark/abos_web/openvpn"
	fi
	if [ -d /etc/atmark/abos_web/wireguard ]; then
		if [ -z "$dryrun" ]; then
			persist_file -r -d /etc/atmark/abos_web/wireguard \
				|| soft_error "Could not delete wireguard"
		fi
		info "persist_file -r -d /etc/atmark/abos_web/wireguard"
	fi
}

ntp_delete() {
	local modified_files=""
	for type in servers initstepslew
	do
		local name="$type.conf"
		local file="/etc/atmark/chrony.conf.d/$name"
		if [ -e "$file" ]; then
			if [ -z "$dryrun" ]; then
				rm -f "$file" || soft_error "Could not remove $file"
			fi
			info "rm -f $file"
			modified_files="$modified_files
$file"
		fi
		file="/etc/chrony/conf.d/$name"
		if [ -e "/lib/chrony.conf.d/$name" ] && [ -e "$file" ]; then
			if [ -z "$dryrun" ]; then
				rm -f "$file" || soft_error "Could not remove $file"
			fi
			info "rm -f $file"
			modified_files="$modified_files
$file"
		fi
	done

	local IFS=$'\n';
	[ -n "$modified_files" ] || return 0

	if [ -z "$dryrun" ]; then
		# shellcheck disable=SC2086 ## split on purpose
		persist_file $modified_files > /dev/null \
			|| soft_error "Could not persist $modified_files"
	fi
	info "persist_file $modified_files"
}

sshd_delete() {
	# skip if not managed by abos-web
	[ -e /etc/atmark/abos_web/manage_root ] || return

	persist_file -drf /etc/ssh/ssh_host_* /etc/runlevels/default/sshd \
		/root/.ssh /etc/ssh/sshd_config.d/50_abos_web.conf
}

user_data_delete() {
	get_appdev

	info "podman stop -a"
	info "reformatting /var/log"
	info "deleting /var/app/volumes/*"

	if [ -z "$dryrun" ]; then
		# stop all containers before remove files
		podman stop -a > /dev/null 2>&1
		# if /var/log is umountable, umount and recreate the fs; otherwise
		# fall back to deletion with a warning than trim did not work
		# (because fstrim apparently is not reliable on ext4...)
		local logdev
		local do_if_fail=:
		rc-service syslog stop > /dev/null 2>&1
		if logdev=$(findmnt -nr -o SOURCE /var/log 2>/dev/null) \
		    && umount /var/log >/dev/null 2>&1; then
			# dd avoids warning from mkfs that another fs existed
			dd if=/dev/zero of="$logdev" bs=32k count=1 status=none
			# mkfs.ext4 has -E discard by default, so will properly wipe previous data
			if mkfs.ext4 -F -q -L logs "$logdev"; then
				mount "$logdev" /var/log
			else
				soft_error "Could not reformat logs in $logdev"
				do_if_fail=
			fi
		else
			do_if_fail=
		fi
		$do_if_fail warning "Could not umount /var/log, wiping it instead (reboot will reformat)"
		$do_if_fail dd if=/dev/zero of="$logdev" bs=1M conv=fsync status=none

		# we do not want to delete external storage mounted
		# under these directories, for example "/var/app/volumes/usb".
		# but we want to delete btrfs subvol,
		# so we use btrfs_subvol_recursive_delete.
		basemount=$(mktemp -d /tmp/btrfs-root.XXXXXX) \
			|| soft_error "Could not create tmpdir"
		mount -t btrfs "$appdev" "$basemount" \
			|| soft_error "Could not mount appfs"
		trap mount_cleanup_basemount EXIT INT QUIT TERM
		btrfs_subvol_recursive_delete "/volumes"
		btrfs_subvol_create "volumes"
		# we need to umount /var/app/volumes for subvol sync to work later
		if ! umount /var/app/volumes >/dev/null 2>&1; then
			warning "Cannot umount /var/app/volumes - will not wait for async deletion"
			no_btrfs_subvol_sync=1
		else
			# remount in case of errors
			mount /var/app/volumes
		fi
		mount_cleanup_basemount
		trap - EXIT INT QUIT TERM
	fi

	if [ -e "/etc/atmark/reset_default_list.txt" ]; then
		while read -r file; do
			[ -z "$file" ] && continue
			if [ -e "$file" ]; then
				if [ -z "$dryrun" ]; then
					persist_file -dr "$file" > /dev/null \
						|| soft_error "Could not delete $file"
				fi
				info "persist_file -dr $file"
			fi
		done < "/etc/atmark/reset_default_list.txt"
	fi
}
