# SPDX-License-Identifier: MIT

# also uses helpers from podman_storage
. "$scripts_dir/podman_storage.sh"

podman_rw_cleanup() {
	if [ -n "$TMPDIR" ] && [ "$TMPDIR" != "/var/tmp" ]; then
		rm -rf "$TMPDIR"
	fi

	# nothing to cleanup or commit if storage is not mounted
	if ! is_mountpoint /var/lib/containers/storage; then
		return
	fi

	if [ -n "$commit" ]; then
		if [ -z "$(podman --root=/var/lib/containers/storage ps -qa)" ]; then
			podman_check_running
			podman_storage_replace_keep
		elif [ "$commit" = 1 ]; then
			error "refusing to commit with containers existing" \
				"kill and remove them with $wrapper rm -fa first"
		else
			commit=""
		fi
	fi

	# also cleanup after commit
	if [ -n "$cleanup$commit" ]; then
		podman_storage_delete disk
	fi
}

podman_rw_help() {
	echo "Usage: abos-ctrl podman-rw [podman-rw options] [--] <podman command>"
	echo
	echo "podman-rw options commands:"
	echo "  --cleanup: remove disk storage (after executing command if any)"
	echo "  --commit: sync storage to main storage (after executing command if any)"
	echo "  --: stop processing options"
	echo
	echo "Furthermore, the following commands automatically resync storage"
	echo "if no container exists:"
	echo "  build, pull, load, rmi, import, image rm, image load,"
	echo "  image build, image import"
	echo "This can be avoided by prefixing a single '--'"
	echo
	echo "Examples:"
	echo "  abos-ctrl podman-rw image list"
	echo "  abos-ctrl podman-rw rmi docker.io/alpine"
	echo "  abos-ctrl podman-rw pull docker.io/alpine"
	echo "  abos-ctrl podman-rw load -i container.tar"
	echo
	echo "See podman --help for more details"
}

# Possible improvements:
# - handle ro mountpoint when not mountpoint root (bind mount first)
# - add ro mode option (e.g. for mount-old) that preserves ro, but
#   podman requires writing to bolt_state.db/sql.db so mount tmpfs there?
ctrl_podman_rw() {
	local rc status appdev tmp

	if [ "$#" = 0 ]; then
		podman_rw_help >&2
		exit 1
	fi
	while [ "$#" -gt 0 ]; do
		case "$1" in
		--cleanup) cleanup=1; shift;;
		--commit) commit=1; shift;;
		build|pull|load|rmi|import) commit=auto; break;;
		image)  case "$2" in
			rm|load|build|import) commit=auto;;
			esac
			break;;
		--) shift; break;;
		*) break;;
		esac
	done

	get_appdev
	podman_storage_get_status
	if [ "$status" = "disk" ]; then
		warning "$wrapper does not make sense with podman-storage as disk, running podman"
		exec podman "$@"
	fi

	if [ "$#" = 0 ]; then
		podman_rw_cleanup
		exit
	fi

	if ! is_mountpoint /var/lib/containers/storage; then
		podman_check_running
		podman_storage_create_rw_storage ask
		local btrfsopts
		btrfsopts=$(findmnt -nr -o OPTIONS /var/tmp) \
			|| error "Could not get mount options for appfs"
		btrfsopts=$(echo "$btrfsopts" \
			| sed -e 's/\(,\?subvol=[^,]*\|$\)/,subvol=containers_storage/' \
			      -e 's/,\?subvolid=[0-9]\+//')
		mount -t btrfs -o "$btrfsopts" "$appdev" /var/lib/containers/storage
	fi

	# podman sometimes leaves lots of data in /var/tmp on errors:
	# force a different TMPDIR and clean up after podman
	tmp=$(mktemp -d "${TMPDIR:-/var/tmp}/abos_ctrl_podman_rw.XXXXXX") \
		|| error "Could not create tmpdir in ${TMPDIR:-/var/tmp}"
	export TMPDIR="$tmp"

	trap podman_rw_cleanup EXIT


	podman --root "/var/lib/containers/storage" \
		--storage-opt additionalimagestore="" \
		"$@"
	rc=$?

	# need to explicitly call cleanup here as variables get out of scope
	trap - EXIT
	podman_rw_cleanup

	return $rc
}

