#!/usr/bin/env sh
# SPDX-License-Identifier: LGPL-2.1
# The license was not clear in original works, and was taken from the lxc
# repo as this is the first version of this file
set -e

EXITCODE=0

# bits of this were adapted from lxc-checkconfig
# see also https://github.com/lxc/lxc/blob/lxc-1.0.2/src/lxc/lxc-checkconfig.in
# then copied from docker and some checks adjusted for ABOS
# https://github.com/moby/moby/blob/abba330bbfe10765822b59bb68af99db439736ba/contrib/check-config.sh

possibleConfigs="
	/proc/config.gz
	/boot/config-$(uname -r)
	/usr/src/linux-$(uname -r)/.config
	/usr/src/linux/.config
"

if [ $# -gt 0 ]; then
	CONFIG="$1"
else
	: "${CONFIG:=/proc/config.gz}"
fi

if ! command -v zgrep > /dev/null 2>&1; then
	zgrep() {
		zcat "$2" | grep "$1"
	}
fi

useColor=true
if [ "$NO_COLOR" = "1" ] || [ ! -t 1 ]; then
	useColor=false
fi
kernelVersion="$(uname -r)"
kernelMajor="${kernelVersion%%.*}"
kernelMinor="${kernelVersion#$kernelMajor.}"
kernelMinor="${kernelMinor%%.*}"

is_set() {
	zgrep "CONFIG_$1=[y|m]" "$CONFIG" > /dev/null
}
is_set_in_kernel() {
	zgrep "CONFIG_$1=y" "$CONFIG" > /dev/null
}
is_set_as_module() {
	zgrep "CONFIG_$1=m" "$CONFIG" > /dev/null
}

color() {
	# if stdout is not a terminal, then don't do color codes.
	if [ "$useColor" = "false" ]; then
		return 0
	fi
	codes=
	if [ "$1" = 'bold' ]; then
		codes='1'
		shift
	fi
	if [ "$#" -gt 0 ]; then
		code=
		case "$1" in
			# see https://en.wikipedia.org/wiki/ANSI_escape_code#Colors
			black) code=30 ;;
			red) code=31 ;;
			green) code=32 ;;
			yellow) code=33 ;;
			blue) code=34 ;;
			magenta) code=35 ;;
			cyan) code=36 ;;
			white) code=37 ;;
		esac
		if [ "$code" ]; then
			codes="${codes:+$codes;}$code"
		fi
	fi
	printf '\033[%sm' "$codes"
}
wrap_color() {
	text="$1"
	shift
	color "$@"
	printf '%s' "$text"
	color reset
	echo
}

wrap_good() {
	echo "$(wrap_color "$1" white): $(wrap_color "$2" green)"
}
wrap_bad() {
	echo "$(wrap_color "$1" bold): $(wrap_color "$2" bold red)"
}
wrap_warning() {
	wrap_color >&2 "$*" red
}

check_flag() {
	if is_set_in_kernel "$1"; then
		wrap_good "CONFIG_$1" 'enabled'
	elif is_set_as_module "$1"; then
		wrap_good "CONFIG_$1" 'enabled (as module)'
	else
		wrap_bad "CONFIG_$1" 'missing'
		EXITCODE=1
	fi
}

check_flags() {
	for flag in "$@"; do
		printf -- '- '
		check_flag "$flag"
	done
}

check_command() {
	if command -v "$1" > /dev/null 2>&1; then
		wrap_good "$1 command" 'available'
	else
		wrap_bad "$1 command" 'missing'
		EXITCODE=1
	fi
}

check_device() {
	if [ -c "$1" ]; then
		wrap_good "$1" 'present'
	else
		wrap_bad "$1" 'missing'
		EXITCODE=1
	fi
}

if [ ! -e "$CONFIG" ]; then
	wrap_warning "warning: $CONFIG does not exist, searching other paths for kernel config ..."
	for tryConfig in $possibleConfigs; do
		if [ -e "$tryConfig" ]; then
			CONFIG="$tryConfig"
			break
		fi
	done
	if [ ! -e "$CONFIG" ]; then
		wrap_warning "error: cannot find kernel config"
		wrap_warning "  try running this script again, specifying the kernel config:"
		wrap_warning "    CONFIG=/path/to/kernel/.config $0 or $0 /path/to/kernel/.config"
		exit 1
	fi
fi

wrap_color "info: reading kernel config from $CONFIG ..." white
echo

printf -- '- '
if [ "$(stat -f -c %t /sys/fs/cgroup 2> /dev/null)" = '63677270' ]; then
	wrap_good 'cgroup hierarchy' 'cgroupv2'
	cgroupv2ControllerFile='/sys/fs/cgroup/cgroup.controllers'
	if [ -f "$cgroupv2ControllerFile" ]; then
		echo '  Controllers:'
		for controller in cpu cpuset io memory pids; do
			if grep -qE '(^| )'"$controller"'($| )' "$cgroupv2ControllerFile"; then
				echo "  - $(wrap_good "$controller" 'available')"
			else
				echo "  - $(wrap_bad "$controller" 'missing')"
			fi
		done
	else
		wrap_bad "$cgroupv2ControllerFile" 'nonexistent??'
	fi
	# TODO find an efficient way to check if cgroup.freeze exists in subdir
else
	cgroupSubsystemDir="$(sed -rne '/^[^ ]+ ([^ ]+) cgroup ([^ ]*,)?(cpu|cpuacct|cpuset|devices|freezer|memory)[, ].*$/ { s//\1/p; q }' /proc/mounts)"
	cgroupDir="$(dirname "$cgroupSubsystemDir")"
	if [ -d "$cgroupDir/cpu" ] || [ -d "$cgroupDir/cpuacct" ] || [ -d "$cgroupDir/cpuset" ] || [ -d "$cgroupDir/devices" ] || [ -d "$cgroupDir/freezer" ] || [ -d "$cgroupDir/memory" ]; then
		echo "$(wrap_good 'cgroup hierarchy' 'properly mounted') [$cgroupDir]"
	else
		if [ "$cgroupSubsystemDir" ]; then
			echo "$(wrap_bad 'cgroup hierarchy' 'single mountpoint!') [$cgroupSubsystemDir]"
		else
			wrap_bad 'cgroup hierarchy' 'nonexistent??'
		fi
		EXITCODE=1
		echo "    $(wrap_color '(see https://github.com/tianon/cgroupfs-mount)' yellow)"
	fi
fi

check_flags \
	NAMESPACES NET_NS PID_NS IPC_NS UTS_NS \
	CGROUPS CGROUP_CPUACCT CGROUP_DEVICE CGROUP_FREEZER CGROUP_SCHED CPUSETS MEMCG \
	KEYS \
	VETH BRIDGE BRIDGE_NETFILTER \
	IP_NF_FILTER IP_NF_MANGLE IP_NF_TARGET_MASQUERADE \
	IP6_NF_FILTER IP6_NF_MANGLE IP6_NF_TARGET_MASQUERADE \
	NETFILTER_XT_MATCH_ADDRTYPE \
	NETFILTER_XT_MATCH_CONNTRACK \
	NETFILTER_XT_MARK \
	IP_NF_RAW IP_NF_NAT NF_NAT \
	IP6_NF_RAW IP6_NF_NAT NF_NAT \
	POSIX_MQUEUE
# (POSIX_MQUEUE is required for bind-mounting /dev/mqueue into containers)

if [ "$kernelMajor" -lt 4 ] || ([ "$kernelMajor" -eq 4 ] && [ "$kernelMinor" -lt 8 ]); then
	check_flags DEVPTS_MULTIPLE_INSTANCES
fi

if [ "$kernelMajor" -lt 5 ] || [ "$kernelMajor" -eq 5 -a "$kernelMinor" -le 1 ]; then
	check_flags NF_NAT_IPV4
fi

if [ "$kernelMajor" -lt 5 ] || [ "$kernelMajor" -eq 5 -a "$kernelMinor" -le 2 ]; then
	check_flags NF_NAT_NEEDED
fi
# check availability of BPF_CGROUP_DEVICE support
if [ "$kernelMajor" -ge 5 ] || ([ "$kernelMajor" -eq 4 ] && [ "$kernelMinor" -ge 15 ]); then
	check_flags CGROUP_BPF
fi

{
	check_flags USER_NS
}
{
	check_flags SECCOMP
	check_flags SECCOMP_FILTER
}
{
	check_flags CGROUP_PIDS
}
{
	check_flags MEMCG_SWAP
	# Kernel v5.8+ removes MEMCG_SWAP_ENABLED.
	if [ "$kernelMajor" -lt 5 ] || [ "$kernelMajor" -eq 5 -a "$kernelMinor" -le 8 ]; then
		CODE=${EXITCODE}
		check_flags MEMCG_SWAP_ENABLED
	else
		# Kernel v5.8+ enables swap accounting by default.
		echo "    $(wrap_color '(cgroup swap accounting is currently enabled)' bold black)"
	fi
}
{
	if is_set LEGACY_VSYSCALL_NATIVE; then
		printf -- '- '
		wrap_bad "CONFIG_LEGACY_VSYSCALL_NATIVE" 'enabled'
		echo "    $(wrap_color '(dangerous, provides an ASLR-bypassing target with usable ROP gadgets.)' bold black)"
	elif is_set LEGACY_VSYSCALL_EMULATE; then
		printf -- '- '
		wrap_good "CONFIG_LEGACY_VSYSCALL_EMULATE" 'enabled'
	elif is_set LEGACY_VSYSCALL_NONE; then
		printf -- '- '
		wrap_bad "CONFIG_LEGACY_VSYSCALL_NONE" 'enabled'
		echo "    $(wrap_color '(containers using eglibc <= 2.13 will not work. Switch to' bold black)"
		echo "    $(wrap_color ' "CONFIG_VSYSCALL_[NATIVE|EMULATE]" or use "vsyscall=[native|emulate]"' bold black)"
		echo "    $(wrap_color ' on kernel command line. Note that this will disable ASLR for the,' bold black)"
		echo "    $(wrap_color ' VDSO which may assist in exploiting security vulnerabilities.)' bold black)"
	# else Older kernels (prior to 3dc33bd30f3e, released in v4.40-rc1) do
	#      not have these LEGACY_VSYSCALL options and are effectively
	#      LEGACY_VSYSCALL_EMULATE. Even older kernels are presumably
	#      effectively LEGACY_VSYSCALL_NATIVE.
	fi
}

if [ "$kernelMajor" -lt 4 ] || ([ "$kernelMajor" -eq 4 ] && [ "$kernelMinor" -le 5 ]); then
	check_flags MEMCG_KMEM
fi

if [ "$kernelMajor" -lt 3 ] || ([ "$kernelMajor" -eq 3 ] && [ "$kernelMinor" -le 18 ]); then
	check_flags RESOURCE_COUNTERS
fi

if [ "$kernelMajor" -lt 5 ]; then
	check_flags IOSCHED_CFQ CFQ_GROUP_IOSCHED
fi

check_flags \
	BLK_CGROUP \
	NET_CLS_CGROUP \
	CFS_BANDWIDTH FAIR_GROUP_SCHED \
	IP_NF_TARGET_REDIRECT

if ! is_set EXT4_USE_FOR_EXT2; then
	check_flags EXT3_FS EXT3_FS_XATTR EXT3_FS_POSIX_ACL EXT3_FS_SECURITY
	if ! is_set EXT3_FS || ! is_set EXT3_FS_XATTR || ! is_set EXT3_FS_POSIX_ACL || ! is_set EXT3_FS_SECURITY; then
		echo "    $(wrap_color '(enable these ext3 configs if you are using ext3 as backing filesystem)' bold black)"
	fi
fi

check_flags EXT4_FS EXT4_FS_POSIX_ACL EXT4_FS_SECURITY
if ! is_set EXT4_FS || ! is_set EXT4_FS_POSIX_ACL || ! is_set EXT4_FS_SECURITY; then
	if is_set EXT4_USE_FOR_EXT2; then
		echo "    $(wrap_color 'enable these ext4 configs if you are using ext3 or ext4 as backing filesystem' bold black)"
	else
		echo "    $(wrap_color 'enable these ext4 configs if you are using ext4 as backing filesystem' bold black)"
	fi
fi

check_flags BTRFS_FS
check_flags BTRFS_FS_POSIX_ACL
check_flags OVERLAY_FS
check_flags VFAT_FS
echo '- Optional FS:'
check_flags EXFAT_FS | sed -e 's/^/  /'

echo '- Optional Network Drivers:'
echo "  - \"$(wrap_color 'overlay' blue)\":"
check_flags VXLAN BRIDGE_VLAN_FILTERING | sed 's/^/    /'
echo '      Optional (for encrypted networks):'
check_flags CRYPTO CRYPTO_AEAD CRYPTO_GCM CRYPTO_SEQIV CRYPTO_GHASH \
	XFRM XFRM_USER XFRM_ALGO INET_ESP NETFILTER_XT_MATCH_BPF | sed 's/^/      /'
if [ "$kernelMajor" -lt 5 ] || [ "$kernelMajor" -eq 5 -a "$kernelMinor" -le 3 ]; then
	check_flags INET_XFRM_MODE_TRANSPORT | sed 's/^/      /'
fi
echo "  - \"$(wrap_color 'ipvlan' blue)\":"
check_flags IPVLAN | sed 's/^/    /'
echo "  - \"$(wrap_color 'macvlan' blue)\":"
check_flags MACVLAN DUMMY | sed 's/^/    /'
echo "  - \"$(wrap_color 'ftp,tftp client in container' blue)\":"
check_flags NF_NAT_FTP NF_CONNTRACK_FTP NF_NAT_TFTP NF_CONNTRACK_TFTP | sed 's/^/    /'

echo

check_limit_over() {
	if [ "$(cat "$1")" -le "$2" ]; then
		wrap_bad "- $1" "$(cat "$1")"
		wrap_color "    This should be set to at least $2, for example set: sysctl -w kernel/keys/root_maxkeys=1000000" bold black
		EXITCODE=1
	else
		wrap_good "- $1" "$(cat "$1")"
	fi
}

echo 'Limits:'
check_limit_over /proc/sys/kernel/keys/root_maxkeys 10000
echo

exit $EXITCODE
