#!/bin/sh
# SPDX-License-Identifier: MIT

usage()
{
    echo "
USAGE:
  $(basename "$0") [OPTIONS] [OUTPUT.img]

  OPTIONS:
    -b|--board <ax2|high-g1-es1|a6e|a600|a900|a9r|qemu>    -- select which output to build
    -B|--boot <boot file to embed>        -- embed boot (sdcard, sw-versions)
    -s|--sign  <key> <cert>               -- produce output.sig signature
    --rootfs <tar archive>                -- rootfs to use (default <OUTPUT>.tar.zst)
    --firmware <image>                    -- add firmware image file (only for main image)
    --installer <image>                   -- produce installer sd card instead of first boot image
    --rootpw <hashed password>            -- password to set for root user in image
    -h|--help

  sbom options:
    --sbom                                -- create sbom (default to not create sbom)
    --sbom-config <config>                -- config used to generate SBOM
                                             (default baseos_sbom.yaml)
    --sbom-external <sbom>                -- add sbom file

  secureboot options:
    --srk <hash>                          -- hash to install into the installer
    --encrypt <userfs|all>                -- have installer produce an encrypted fs.
    --verity <key> <cert>                 -- enable verity for image.
                                             The image will be made read-only and
                                             bootable with a kernel with verity initrd.
    --boot-linux <kernel image>           -- kernel image to embed for encrypted rootfs
                                             Uses /boot/Image from rootfs if unset

  OUTPUT:
    filename of output. (e.g. alpine.img)
    if signing, also output.sig (e.g. alpine.img.sig)
"
}

warning()
{
    printf "===================================\n"
    printf "warning: %s\n" "$@" >&2
    printf "===================================\n"
}

error()
{
    printf "===================================\n"
    printf "error: %s\n" "$@" >&2
    printf "===================================\n"
    exit 1
}

############
# Variables
############
signkey=
signcert=
rootfs=
boot=
output=
installer=
board=
extlinux=
firmware=
fstype=
encrypt=
boot_linux=
boot_linux_sig=
boot_linux_path=
uboot_env=
check_commands="sgdisk:gdisk"
sbom_config="$(realpath baseos_sbom.yaml)"
sbom_external=
verity_key=
verity_cert=
lodev=
srk_hash=
rootpw=

# debian path does not contain /sbin by default, but we
# need it for some programs. Add it at the end if missing
if [ "${PATH#*/sbin}" = "$PATH" ]; then
    PATH="$PATH:/sbin"
fi

# parse args
while [ $# -ne 0 ]
do
    case "$1" in
    "-b"|"--board")
        [ $# -lt 2 ] && error "$1 requires an argument"
        case "$2" in
        "ax2")
            board=ax2
            arch=aarch64
            suffix=x2
            dtb=armadillo_iotg_g4.dtb
            boot_linux_sig=d00dfeed
            boot_linux_path=boot_linux
            ;;
        "high-g1-es1")
            board=high-g1
            arch=aarch64
            suffix=high-g1-es1
            dtb=high-g1-es1.dtb
            ;;
        "a6e")
            board=a6e
            arch=armv7
            suffix=6e
            dtb=armadillo-iotg-a6e.dtb
            ;;
        "a600")
            board=a600
            arch=armv7
            suffix=600
            dtb=armadillo-640.dtb
            ;;
        "a900")
            board=a900
            arch=aarch64
            suffix=900
            dtb=armadillo_900.dtb
            boot_linux_sig=00c00287
            boot_linux_path=secboot_linux
            ;;
        "a9r")
            board=a9r
            arch=aarch64
            suffix=9r
            dtb=armadillo_900.dtb
            boot_linux_sig=00c00287
            boot_linux_path=secboot_linux
            ;;
        "qemu")
            board=qemu
            arch=x86_64
            suffix=qemu
            ;;
        *)
            echo "unknown board($2)"
            usage
            exit 1
            ;;
        esac
        shift
        ;;
    "-B"|"--boot")
        [ $# -lt 2 ] && error "$1 requires an argument"
        boot="$(readlink -e "$2")" \
            || error "boot $2 does not exist"
        shift
        ;;
    "-f"|"--firm"|"--firmware")
        [ $# -lt 2 ] && error "$1 requires an argument"
        firmware="$(readlink -e "$2")" \
            || error "firmware $2 does not exist"
        shift
        ;;
    "--fstype")
        [ $# -lt 2 ] && error "$1 requires an argument"
        fstype="$2"
        case "$fstype" in
        ext4|btrfs) ;;
        *) error "Invalid fstype: $fstype (only support ext4 and btrfs)";;
        esac
        shift
        ;;
    "--encrypt")
        [ $# -lt 2 ] && error "$1 requires an argument (userfs, all)"
        encrypt="$2"
        case "$encrypt" in
        userfs|all) ;;
        *) error "Invalid encrypt option: $encrypt (must be 'userfs' or 'all')";;
        esac
        shift
        ;;
    "--boot-linux")
        [ $# -lt 2 ] && error "$1 requires an argument"
        boot_linux="$(readlink -e "$2")" \
            || error "boot-linux $2 does not exist"
        shift
        ;;
    "--srk")
        [ $# -lt 2 ] && error "$1 requires an argument"
        srk_hash="$2"
        echo "$srk_hash" | grep -qiE '^0x[0-9a-f]{1,8}(,0x[0-9a-f]{1,8}){7}$' \
            || error "$srk_hash not formatted as 8 hex digits separated by commas"
        shift
        ;;
    "-s"|"--sign")
        [ $# -lt 3 ] && error "$1 requires [key cert] arguments"
        signkey="$2"
        signcert="$3"
        [ -r "$signkey" ] || error "key $signkey is not readable"
        [ -r "$signcert" ] || error "cert $signcert is not readable"
        shift 2
        ;;
    "--verity")
        [ $# -lt 3 ] && error "$1 requires [key cert] arguments"
        if grep -q "BEGIN CERTIFICATE" "$2"; then
            verity_cert="$(realpath "$2")"
            verity_key="$(realpath "$3")"
        elif grep -q "BEGIN CERTIFICATE" "$3"; then
            verity_cert="$(realpath "$3")"
            verity_key="$(realpath "$2")"
        else
            error "verity certificate not found"
        fi
        grep -q "PRIVATE KEY" "$verity_key" \
            || error "verity key not found"

        check_commands="$check_commands mkfs.vfat:dosfstools veritysetup:cryptsetup-bin"
        shift 2
        ;;
    "--installer")
        [ $# -lt 2 ] && error "$1 requires an argument"
        installer="$(readlink -e "$2")" \
            || error "install image $2 does not exist"
        [ "$installer" = "${installer%.img}" ] \
            && error "Installer $installer does not end in .img, image type not handled"
        shift
        ;;
    "--rootfs")
        [ $# -lt 2 ] && error "$1 requires an argument"
        rootfs="$(readlink -e "$2")" \
            || error "rootfs $2 does not exist"
        shift
        ;;
    "--rootpw")
        [ $# -lt 2 ] && error "$1 requires an argument"
        rootpw="$2"
        case "$rootpw" in
        '$'*'$'*'$'*) ;;
        *)
            warning "--rootpw argument not in hashed \$x\$salt\$hash form, assuming plain password." \
                "Consider hashing it yourself with 'openssl passwd -6' to avoid handling plain passwords"
            rootpw=$(echo "$rootpw" | openssl passwd -6 -stdin)
            ;;
        esac
        rootpw="$(echo "$rootpw" | sed -e 's:[\\/&]:\\&:g')"
        shift
        ;;
    "--uboot-env")
        [ $# -lt 2 ] && error "$1 requires an argument"
        uboot_env="$2"
        shift
        ;;
    "--sbom")
        sbom=true
        ;;
    "--sbom-config")
        sbom_config="$(readlink -e "$2")" \
            || error "sbom-config $2 does not exist"
        shift
        ;;
    "--sbom-external")
        [ $# -lt 2 ] && error "$1 requires an argument"
        sbom_external="$(readlink -e "$2")" \
            || error "sbom-external $2 does not exist"
        shift
        ;;
    "-h"|"--help")
        usage
        exit 0
        ;;
    -*)
        echo "Invalid option $1" >&2
        usage >&2
        exit 1
        ;;
    *)
        outdir="$(dirname "$(readlink -f "$1")")"
        output="$(basename "$1")"
        [ -z "$rootfs" ] && rootfs="${output%.img}.tar.zst"
        if [ "${output%.img}" = "$output" ]; then
            echo "Output file \"$output\" must end in .img" >&2
            exit 1
        fi
        ;;
    esac
    shift
done

# set default
if [ -z "$board" ]; then
    echo "use default(board=ax2)"
    board=ax2
    arch=aarch64
    suffix=x2
    dtb=armadillo_iotg_g4.dtb
    boot_linux_sig=d00dfeed
    boot_linux_path=boot_linux
fi
if [ -z "$output" ]; then
    if [ -z "$rootfs" ]; then
        # shellcheck disable=SC2010 # ls | grep because too complex to glob
        rootfs=$(ls --sort=time "baseos-$suffix-"*.tar* 2>/dev/null \
                 | grep -vE '\.sig$|\.spdx.json$' | head -n 1)
        # create rootfs if it we didn't find one
        if [ -z "$rootfs" ]; then
            ./build_rootfs.sh -b "$board"
            # shellcheck disable=SC2010 # ls | grep because too complex to glob
            rootfs=$(ls --sort=time "baseos-$suffix-"*.tar* 2>/dev/null \
                     | grep -vE '\.sig$|\.spdx.json$' | head -n 1)
            [ -n "$rootfs" ] || error "Could not find rootfs that was just built"
        fi
    fi
    outdir="$(pwd)"
    output="${rootfs%.tar*}${installer:+-installer}.img"
    output="$(basename "$output")"
    echo "use default(outdir=$outdir)"
    echo "use default(output=$output)"
fi

[ -e "$rootfs" ] \
    || error "rootfs $rootfs does not exist -- specify with --rootfs or match image name to previously built image"
rootfs=$(realpath -e "$rootfs")

[ "$arch" = "x86_64" ] && extlinux=1 && fstype=ext4
if [ -n "$installer" ]; then
    [ -n "$boot" ] || error "--boot must be set for --installer"

    check_commands="$check_commands xxhsum:xxhash"
fi
if [ -z "$installer" ] && [ -n "$verity_cert" ]; then
    [ -n "$boot" ] || error "--boot must be set for image with --verity"
fi

[ -z "$fstype" ] && fstype=btrfs
case "$fstype" in
ext4) check_commands="$check_commands mkfs.ext4:e2fsprogs";;
btrfs) check_commands="$check_commands mkfs.btrfs:btrfs-progs";;
esac
[ -z "$encrypt" ] || [ -n "$installer" ] || error "--encrypt can only be set for --installer"

if [ "${PATH%sbin*}" = "$PATH" ] && ! command -v sgdisk >/dev/null; then
    # debian by default doesn't include sbin in PATH,
    # but only add it if needed
    PATH=/usr/sbin:$PATH
fi

missing_commands=""
for command in $check_commands; do
    command -v "${command%:*}" >/dev/null || missing_commands="$missing_commands ${command#*:}"
done
if [ -n "$missing_commands" ]; then
    sudo=""
    [ "$(id -u)" != "0" ] && sudo="sudo "
    error "Missing required programs: please install with: ${sudo}apt install$missing_commands"
fi

# Installs and configures extlinux.
# from https://github.com/alpinelinux/alpine-make-vm-image
setup_extlinux() {
        local mnt="$1"  # path of directory where is root device currently mounted
        local root_dev="$2"  # root device
        local modules="$3"  # modules which should be loaded before pivot_root
        local kernel_flavor="$4"  # name of default kernel to boot
        local serial_port="$5"  # serial port number for serial console
        local default_kernel="$kernel_flavor"
        local kernel_opts=''

        [ -z "$serial_port" ] || kernel_opts="console=$serial_port"

        if [ "$kernel_flavor" = 'virt' ]; then
                _apk search --root . --exact --quiet linux-lts | grep -q . \
                        && default_kernel='lts' \
                        || default_kernel='vanilla'
        fi

        sudo sed -Ei \
                -e "s|^[# ]*(root)=.*|\1=$root_dev|" \
                -e "s|^[# ]*(default_kernel_opts)=.*|\1=\"$kernel_opts\"|" \
                -e "s|^[# ]*(modules)=.*|\1=\"$modules\"|" \
                -e "s|^[# ]*(default)=.*|\1=$default_kernel|" \
                -e "s|^[# ]*(serial_port)=.*|\1=$serial_port|" \
                "$mnt"/etc/update-extlinux.conf

        sudo chroot "$mnt" extlinux --install /boot
        sudo chroot "$mnt" update-extlinux --warn-only 2>&1 \
                | grep -Fv 'extlinux: cannot open device /dev' >&2
}

#######
# Main
#######
cleanup() {
    [ -n "$workdir" ] || return
    mountpoint -q "$workdir/mnt" && sudo umount "$workdir/mnt"
    mountpoint -q "$workdir/rootfs" && sudo umount "$workdir/rootfs"
    if [ -n "$lodev" ]; then
        sudo losetup -d "$lodev"
    fi
    rm -rf --one-file-system "$workdir"
}
trap cleanup EXIT


scriptdir=$(realpath "$(dirname "$0")")
workdir="$(mktemp -d -t alpine-build-rootfs.XXXXXX)"
cd "$workdir" || error "could not enter workdir"


create_disk() {
    local verity_size=0
    # 'normal' rootfs is fixed at 300MB...
    SIZE=300
    if [ -n "$installer" ]; then
        # ... but installer can be quite bigger with container images:
        # compute size from embedded files + 20% for filesystem overhead
        # with a 390MB base size (installer rootfs and base image)
        # (390 as a workaround to keep the default, almost empty case below
        # 400MB for old abos-ctrl make-installer)
        SIZE=$(du -sm "$outdir"/common/image_common/ \
                "$outdir"/common/image_installer/ \
                "$outdir/$board"/image_common/ \
                "$outdir/$board"/image_installer/ 2>/dev/null \
            | awk '{ tot += $1 } END { print int(tot * 1.2 + 390) }')
    fi
    # x86_64 needs slightly bigger rootfs
    [ "$arch" = "x86_64" ] && SIZE=$((SIZE+50))
    if [ -n "$verity_key" ]; then
        # approx as compute_verity_size in make-installer: size/126 (rounded up) +1
        verity_size=$(( (SIZE+125) / 126 + 1))
    fi
    # truncated size:
    # - partition sizes (p1 SIZE, p127 verity_size)
    # - 10 for space before first partition
    # - 1 for gpt header at the end
    truncate -s $((SIZE + verity_size + 11))M "$output"
    set -- --zap-all --new "1:20480:+$((SIZE))M" -c 1:rootfs_0
    if [ "$verity_size" != 0 ]; then
        set -- "$@" -n "127:$((SIZE+10))M:+$((verity_size))M" -c 127:verity
    fi
    case "$suffix" in
    "6e"|"600")
        set -- "$@" -j $((20480-32))
        ;;
    esac
    sgdisk "$@" "$output"
}

create_mount_partition() {
    local offset=$((20480*512))
    local size=$((SIZE*1024))
    local mountopt=""

    case "$fstype" in
    ext4)
        # extlinux needs the 64bit option to be disabled for some reason
        # (it sometimes works, but is not reliable)
        mkfs.ext4 -F -E "offset=$offset" ${extlinux:+-O "^64bit"} \
                -L rootfs_0 "$output" "$size" \
            || error "mkfs ext4 rootfs"
        ;;
    btrfs)
        # mkfs.btrfs has no offset option: go manual.
        lodev=$(sudo losetup --show -f -o "$offset" \
                --sizelimit "$((size*1024))" "$output") \
            || error "Could not setup loop device for rootfs creation"
        # ATDE9 ships btrfs-progs 5.10, which still requires -m DUP -R free-space-tree
        # (changed in 5.15)
        if ! sudo mkfs.btrfs -L rootfs_0 -m DUP -R free-space-tree --quiet "$lodev"; then
            error "mkfs btrfs failed"
        fi
        sudo losetup -d "$lodev"
        lodev=""
        # loop file is only really removed after flush, so on slow devices
        # mount can fail below with loop device conflict error
        sync
        mountopt=",compress-force=zstd,discard"
        ;;
    *) error "Unknown fstype $fstype";;
    esac
    mkdir mnt \
        && sudo mount -o "offset=$offset,noatime$mountopt" "$output" mnt \
        || error "mount rootfs failed"
}

extract_rootfs() {
    sudo mkdir mnt/boot mnt/mnt mnt/target
    sudo tar -C mnt --xattrs --xattrs-include=security.capability \
        -xf "$rootfs" || error "extract rootfs"
}

offset_bootloader() {
    [ -z "$boot" ] && return

    # need to add the 1KB padding if missing because the installer
    # installs the boot file without 1KB offset
    case "$board" in
    a6e|a600)
        if ! cmp -s -n 1024 "$boot" /dev/zero; then
            local boot_1koffset
            boot_1koffset="$workdir/$(basename "$boot")"

            dd if="$boot" of="$boot_1koffset" bs=1M \
                    seek=1024 oflag=seek_bytes status=none \
                || error "Could not create padded boot file"
            boot="$boot_1koffset"
        fi
        ;;
    esac
}

fix_uboot_env() {
    # fix fw_env.config and write env
    (
        . mnt/lib/rc/sh/functions-atmark-board.sh || exit 1
        sed -e "s:{ENVDISK}:$output:" \
               -e "s:{ENVOFFSET}:${UBOOT_ENVOFFSET}:" \
               -e "s:{ENVREDUND}:${UBOOT_ENVREDUND}:" \
               -e "s:{ENVSIZE}:${UBOOT_ENVSIZE}:" \
               mnt/etc/fw_env.config > build_image_fw_env.config \
            || exit 1
        sudo sed -i -e "s:{ENVDISK}:${UBOOT_ENVSD}:" \
               -e "s:{ENVOFFSET}:${UBOOT_ENVOFFSET}:" \
               -e "s:{ENVREDUND}:${UBOOT_ENVREDUND}:" \
               -e "s:{ENVSIZE}:${UBOOT_ENVSIZE}:" \
               mnt/etc/fw_env.config
    ) || error "Could not create fw_env.config"

    if ! command -v fw_setenv >/dev/null; then
        [ -n "$uboot_env" ] && error "--uboot-env was requested but fw_setenv is not installed"
        echo "fw_setenv was not available, skipping setting env"
    elif ! grep -qE '^[^#]' mnt/boot/uboot_env.d/* 2>/dev/null; then
        [ -n "$uboot_env" ] && error "--uboot-env was requested but no default env available"
        echo "fw_setenv was not available, skipping setting env"
    else
        grep -qE "^bootcmd=" mnt/boot/uboot_env.d/* \
            || error "default env files existed but bootcmd was not set, aborting"

        if [ -n "$uboot_env" ]; then
            echo "$uboot_env" | sudo sh -c "cat > mnt/boot/uboot_env.d/ZZ_installer" \
                || error "Could not write installer uboot env file"
        fi
        cat mnt/boot/uboot_env.d/* \
            | sed -e 's/upgrade_available=1/upgrade_available=00/' \
            | fw_setenv --config build_image_fw_env.config \
                --script - --defenv /dev/null \
            || error "Could not set installer uboot env"
    fi
    rm -f build_image_fw_env.config
}

customize_rootfs_firstboot() {
    # need to remove symlink first
    sudo rm -f mnt/sbin/init
    sudo cp -r "$outdir"/common/image_firstboot/. mnt \
        || error "could not copy first install files"
    if [ -e "$outdir/$board"/image_firstboot ]; then
        sudo cp -r "$outdir/$board"/image_firstboot/. mnt \
            || error "could not copy board first install files"
    fi

    if [ -n "$verity_cert" ]; then
        # Using verity with non-installer image means no init as it cannot modify
        # rootfs. This is meant as a diagnostic image.
        sudo rm -f mnt/sbin/init \
	        && sudo ln -s /bin/busybox mnt/sbin/init \
            || error "Could not update image init"
        if ! sudo awk -F: '$2 == "" { exit 1 }' mnt/etc/shadow; then
            warning "Verity image without password set is not recommended!" \
                "Consider using --rootpw to set password"
        fi
        # update bare minimum to avoid most obvious errors without running
        # firstboot script...
        sudo sed -i -e '/DISKPART/d' mnt/etc/fstab \
            && sudo rm -f mnt/etc/runlevels/default/reset_bootcount \
            && head -c 16 < /dev/urandom | xxd -p \
                | sudo tee mnt/etc/machine-id >/dev/null \
            || error "Could not update firstboot files"

        fix_uboot_env
    fi
}

copy_to_lzo_and_csum() {
    local dest="$1"
    local sz="$2"
    shift 2
    local xxh
    local pid_xxh pid_wc pid_tee pid_source

    # assume we can read source multiple times
    mkfifo pipe_tee pipe_xxh pipe_lzop pipe_wc
    xxhsum < pipe_xxh > xxh &
    pid_xxh=$!

    wc -c < pipe_wc > sz &
    pid_wc=$!

    tee pipe_lzop pipe_xxh > pipe_wc < pipe_tee &
    pid_tee=$!

    "$@" > pipe_tee &
    pid_source=$!

    # shellcheck disable=SC2024 # don't need to read pipe as root
    sudo sh -c "lzop > mnt/$dest.lzo" < pipe_lzop \
        || error "Failed compressing or writing lzo file"
    wait "$pid_source" \
        || error "Source command failed for $dest"
    wait "$pid_xxh" \
        || error "Computing xxh failed for $dest"
    wait "$pid_wc" \
        || error "wc failed for $dest"
    wait "$pid_tee" \
        || error "tee failed for $dest"
    if [ -n "$sz" ]; then
        [ "$sz" = "$(cat sz)" ] \
            || error "Stream was not of the expected size"
    else
        sz="$(cat sz)" \
            || error "Could not read size"
        # sanity check
        [ -n "$sz" ] && [ "$sz" != 0 ] \
            || error "Read size was 0, problem with pipes?"
    fi
    xxh=$(cat xxh) \
        || error "Could not read xxh"
    xxh=${xxh%% *}
    wait "$pid_wc" \
        || error "$dest file does not have expected size (expected $sz)"
    sudo sh -c "echo $sz $xxh > mnt/$dest.xxh" \
        || error "Could not write $dest checksum"
    rm -f pipe_lzop pipe_xxh pipe_wc pipe_tee xxh sz
    local xxhcheck
    xxhcheck="$(lzop -d < "mnt/$dest.lzo" | xxhsum)"
    xxhcheck="${xxhcheck%% *}"
    [ "$xxh" = "$xxhcheck" ] \
        || error "Sha we just wrote does not match (expected $xxh got $xxhcheck"
}

customize_rootfs_installer() {
    local start sz

    # kill autoupdate services
    sudo sh -c 'rm -vf mnt/etc/init.d/swupdate* mnt/lib/udev/rules.d/*swupdate*'


    # copy images to install
    sudo cp -r "$outdir"/common/image_installer/. mnt || error "could not copy installer files"
    if [ -e "$outdir/$board"/image_installer ]; then
        sudo cp -r "$outdir/$board"/image_installer/. mnt \
            || error "could not copy board installer files"
    fi

    # start sector, end sector. End sector in sgdisk is inclusive, add 1...
    start=$(sgdisk -p "$installer" | awk '/rootfs_0/ { print $2, $3 + 1}')
    echo "$start" | grep -qxE '[0-9]+ [0-9]+' \
        || error "Could not find rootfs_0 in image"
    sz=${start#* }
    start=${start% *}
    sz=$(((sz-start)*512))
    start=$((start*512))

    copy_to_lzo_and_csum image "$sz" \
        dd if="$installer" bs=1M iflag=count_bytes,skip_bytes \
            skip="$start" count="$sz" status=none
    basename "$installer" | sudo sh -c 'cat > mnt/image.filename' \
        || error "Could not write image filename"

    sz=$(stat -c %s "$boot") \
            || error "Could not get bootloader size"
    copy_to_lzo_and_csum boot "$sz" \
        cat "$boot"
    basename "$boot" | sudo sh -c 'cat > mnt/boot.filename' \
        || error "Could not write boot filename"

    if [ -n "$srk_hash" ]; then
        echo "$srk_hash" | sudo sh -c 'cat > mnt/secureboot_srk' \
            || error "Could not write srk hash"
    fi
    case "$encrypt" in
    all)
        [ -n "$boot_linux_path" ] || error "Encryption is not supported for $board"
        # if not set, use image from rootfs
        if [ -z "$boot_linux" ]; then
            mkdir rootfs \
                && sudo mount -o offset=$((20480*512)) "$installer" rootfs \
                || error "Could not mount $installer"
            boot_linux=rootfs/boot/Image
            [ -e "$boot_linux" ] || error "image neither in --boot-linux or rootfs"
        fi

        sz=$(stat -c %s "$boot_linux") \
            || error "Could not get kernel boot image size"
        [ "$(head -c 4 "$boot_linux" | xxd -p)" = "$boot_linux_sig" ] \
            || error "$boot_linux is not signed"
        grep -q ramdisk "$boot_linux" \
            || error "rootfs encryption requested but $boot_linux does not have an initrd"
        copy_to_lzo_and_csum "$boot_linux_path" "$sz" \
            cat "$boot_linux"

        if [ -e rootfs ]; then
            sudo umount rootfs
            rmdir rootfs
        fi
        printf "%s\n" "ENCRYPT_ROOTFS=1" "ENCRYPT_USERFS=1" | sudo sh -c 'cat >> mnt/installer.conf'
        ;;
    userfs)
        echo "ENCRYPT_USERFS=1" | sudo sh -c 'cat >> mnt/installer.conf'
        ;;
    esac

    fix_uboot_env
}

customize_rootfs() {
    sudo cp -r "$outdir"/common/image_common/. mnt \
        || error "could not copy installer common files"
    if [ -e "$outdir/$board"/image_common ]; then
        sudo cp -r "$outdir/$board"/image_common/. mnt \
            || error "could not copy board installer common files"
    fi
    if [ -n "$firmware" ]; then
        sudo cp "$firmware" mnt/firm.squashfs \
            || error "Could not copy firmware"
        xxhsum "$firmware" \
                | sudo sh -c 'sed -e "s/ .*//" > mnt/firm.squashfs.xxh' \
            || error "Could not write firmware hash"
    fi
    if [ -n "$rootpw" ]; then
        sudo sed -i -e 's/^root:[^:]*:[0-9]*:/root:'"$rootpw"'::/' mnt/etc/shadow \
            || error "Could not set root pw"
    fi
    if [ -n "$installer" ]; then
        customize_rootfs_installer
    else
        customize_rootfs_firstboot
    fi
    case "$fstype" in
    btrfs)
        sudo sed -i -e 's@^/dev/root.*@/dev/root\t/\t\t\t\tbtrfs\tro,noatime,compress-force=zstd,discard=async\t0 0@' \
                mnt/etc/fstab \
            || error "Could not update fstab rootfs mount options"
    esac
}

link_armadillo_dtb() {
    # armadillo.dtb is normally loaded by uboot, but that is not the case for
    # secureboot or FIT images.
    # There are two possibilities:
    # - Image is present in FIT or signed format
    case "$(xxd -l 4 -p mnt/boot/Image 2>/dev/null)" in
    d00dfeed) return;; # X2, FIT format
    00c00287) return;; # A900, AHAB authentication container
    esac
    # - Image is not present at all because loaded externally (e.g. will encrypt)
    [ -e mnt/boot/Image ] || [ -e mnt/boot/uImage ] \
        || return

    [ -e "mnt/boot/$dtb" ] \
        || error "/boot/$dtb does not exist !!!"
    sudo ln -s "$dtb" mnt/boot/armadillo.dtb \
        || error "Could not add link to dtb"
}

install_bootloader() {
    if [ -n "$extlinux" ]; then
        GPTMBR="mnt/usr/share/syslinux/gptmbr.bin"
        [ -e "$GPTMBR" ] || error "extlinux expected but $GPTMBR not found"
        dd bs=440 count=1 conv=notrunc if="$GPTMBR" of="$output" status=none
        sgdisk --attributes=1:set:2 "$output"
        sudo mount --bind /dev mnt/dev
        sudo mount --bind /proc mnt/proc
        setup_extlinux "$PWD/mnt" "LABEL=rootfs_0" "ext4" lts ttyS0
        sudo umount mnt/dev
        sudo umount mnt/proc
    fi

    link_armadillo_dtb

    # actually write in boot image if provided
    [ -z "$boot" ] && return
    case "$board" in
    ax2|high-g1|a900|a9r)
        dd if="$boot" of="$output" bs=1k seek=32 conv=notrunc status=none \
            || error "Could not write boot file to image"
        ;;
    a6e|a600)
        # skip the 1kB padding for i.MX7/6 (and seek 1kB to avoid corrupting
        # the partition table of the disk)
        dd if="$boot" of="$output" bs=1k skip=1 seek=1 conv=notrunc status=none \
            || error "Could not write boot file to image"
        ;;
    *)
        error "Board does not support --boot: $board"
        ;;
    esac
}

create_verity() {
    [ -n "$verity_cert" ] || return 0

    lodev=$(sudo losetup --show -f -P "$output") \
        || error "Could not setup loop device for rootfs creation"

    sudo mkfs.vfat "$lodev"p127
    sudo mount "$lodev"p127 mnt

    case "$fstype" in
    btrfs)
        sudo btrfstune -S 1 "$lodev"p1 \
            || warning "btrfstune failed -- fs not made read-only, will fail boot if modified"
        ;;
    *)
        warning "$fstype fs not made read-only, will fail boot if modified"
        ;;
    esac

    sudo veritysetup format --root-hash-file=mnt/verity_p1.hashroot \
            "$lodev"p1 mnt/verity_p1.hashdev >/dev/null \
        || error "Could not create verity hash device"

    local retries=0
    while ! sudo openssl cms -sign -outform DER -in mnt/verity_p1.hashroot \
            -out mnt/verity_p1.hashroot.sig -nosmimecap -binary \
            -inkey "$verity_key" -signer "$verity_cert" -binary; do
        retries=$((retries + 1))
        [ "$retries" -ge 3 ] && error "Could not sign verity hash root"
        echo "verity hash root signature failed ($retries/3), retrying in case password"
        echo "was incorrect"
    done

    sudo umount mnt
    sudo losetup -d "$lodev"
    lodev=""
}

board_has_swu() {
    local layer prefix
    for layer in common "$board"; do
        prefix="$scriptdir/$layer/image_installer"
        stat "$prefix/"*.swu >/dev/null 2>&1 && return
        stat "$prefix/installer_swus/"*.swu >/dev/null 2>&1 && return
    done
    return 1
}

build_sbom() {
    echo "Creating SBOM"
    if ! command -v make_sbom.sh >/dev/null; then
        echo "not installed python3-make-sbom, and not created sbom."
        return
    fi
    "make_sbom.sh" -i "$output" -c "$sbom_config" \
            -o "$outdir/$output" \
            ${sbom_external:+-e "$sbom_external"} \
            -e "$rootfs.spdx.json" \
        || error "Could not build sbom"

    if [ -z "$sbom_external" ] && board_has_swu; then
        warning "Note the generated SBOM will not contain license information for software" \
                "added through SWU files. Please append an external SBOM if required."
    fi
}

create_disk
create_mount_partition
extract_rootfs
offset_bootloader
customize_rootfs
install_bootloader

sudo umount mnt || error "could not umount rootfs"

create_verity

rmdir mnt

[ -n "$sbom" ] && build_sbom

# assume VM image for x86_64, in which case create larger sparse file
[ "$arch" = "x86_64" ] && truncate -s 8G "$workdir/$output"

if [ -n "$signkey" ]; then
    openssl cms -sign -in "$workdir/$output" -out "$workdir/$output.sig" \
            -signer "$signcert" -inkey "$signkey" -outform DER \
            -nosmimecap -binary \
        || error "Could not sign image"
fi
! [ -e "$workdir/$output.sig" ] \
    || mv "$workdir/$output.sig" "$outdir" \
    || error "Could not move signature to $outdir"
mv "$workdir/$output" "$outdir/" \
    || error "Could not move image to $outdir"


echo
echo "Successfully built $outdir/$output"
