/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2024 Atmark Techno, Inc.
 */

#include <string.h>
#include <gmodule.h>

#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>

#include "mm-log-object.h"
#include "mm-plugin-common.h"
#include "mm-broadband-modem-simtech-sim7672.h"

#define MM_TYPE_PLUGIN_SIMTECH_SIM7672 mm_plugin_simtech_sim7672_get_type ()
MM_DEFINE_PLUGIN (SIMTECH_SIM7672, simtech_sim7672, SimtechSim7672)

/*****************************************************************************/

static MMBaseModem *
create_modem (MMPlugin *self,
              const gchar *uid,
              const gchar *physdev,
              const gchar **drivers,
              guint16 vendor,
              guint16 product,
              guint16 subsystem_vendor,
              GList *probes,
              GError **error)
{
    return MM_BASE_MODEM (mm_broadband_modem_simtech_sim7672_new (uid,
                                                                  physdev,
                                                                  drivers,
                                                                  mm_plugin_get_name (self),
                                                                  vendor,
                                                                  product));
}

/*****************************************************************************/
/* grab port */

static gboolean
grab_port (MMPlugin     *self,
           MMBaseModem  *modem,
           MMPortProbe  *probe,
           GError      **error)
{
    MMPortSerialAtFlag  pflags = MM_PORT_SERIAL_AT_FLAG_NONE;
    MMKernelDevice     *pdev;
    MMPortType          ptype;
    const gchar        *name = mm_port_probe_get_port_name (probe);

    ptype = mm_port_probe_get_port_type (probe);
    pdev = mm_port_probe_peek_port (probe);

    if (mm_kernel_device_has_property (pdev,
                                       "ID_MM_SIMTECH_SIM7672_TTY_AT_SYMLINK")) {
        name = mm_kernel_device_get_property (pdev,
                                              "ID_MM_SIMTECH_SIM7672_TTY_AT_SYMLINK");
        mm_obj_dbg (self, "[%s] symlink name", name);
        return mm_base_modem_grab_symlink_at_primary_port (modem, pdev, name, error);
    } else if (mm_kernel_device_get_global_property_as_boolean (pdev,
                                                                "ID_MM_SIMTECH_SIM7672_PORT_TYPE_AT")) {
        mm_obj_dbg (self, "(%s/%s)' Port flagged as primary",
                    mm_port_probe_get_port_subsys (probe),
                    mm_port_probe_get_port_name (probe));
        pflags = MM_PORT_SERIAL_AT_FLAG_PRIMARY;
    }

    return mm_base_modem_grab_port (modem, pdev, ptype, pflags, error);
}

/*****************************************************************************/

MM_PLUGIN_NAMED_CREATOR_SCOPE MMPlugin *
mm_plugin_create_simtech_sim7672 (void)
{
    static const gchar *subsystems[] = { "tty", "net", "usbmisc", NULL };
    static const mm_uint16_pair products[] = { { 0x05c6, 0x9330 }, /* SIM7672 */
                                               { 0, 0 } };

    return MM_PLUGIN (
        g_object_new (MM_TYPE_PLUGIN_SIMTECH_SIM7672,
                      MM_PLUGIN_NAME,                MM_MODULE_NAME,
                      MM_PLUGIN_ALLOWED_SUBSYSTEMS,  subsystems,
                      MM_PLUGIN_ALLOWED_PRODUCT_IDS, products,
                      MM_PLUGIN_ALLOWED_AT,          TRUE,
                      NULL));
}

static void
mm_plugin_simtech_sim7672_init (MMPluginSimtechSim7672 *self)
{
}

static void
mm_plugin_simtech_sim7672_class_init (MMPluginSimtechSim7672Class *klass)
{
    MMPluginClass *plugin_class = MM_PLUGIN_CLASS (klass);

    plugin_class->create_modem = create_modem;
    plugin_class->grab_port = grab_port;
}
