/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2023 Atmark Techno, Inc.
 */

#include <glib.h>

#include <ModemManager.h>
#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>

#include "mm-log.h"
#include "mm-modem-helpers.h"
#include "mm-modem-helpers-quectel-ec25.h"
#include "mm-port-serial-at.h"

gboolean
mm_quectel_ec25_parse_qicsgp_read_response (const gchar          *response,
                                            MMBearerAllowedAuth  *out_auth,
                                            MMBearerIpFamily     *out_ip_type,
                                            gchar               **out_apn,
                                            gchar               **out_username,
                                            gchar               **out_password,
                                            GError              **error)
{
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    g_autoptr(GError)      inner_error = NULL;

    r = g_regex_new ("\\+QICSGP:\\s*(\\d),\"(.*)\",\"(.*)\",\"(.*)\",(\\d)",
                     0, 0, NULL);
    g_assert (r != NULL);

    g_regex_match_full (r, response, strlen (response), 0, 0, &match_info, &inner_error);
    if (g_match_info_matches (match_info)) {
        guint qicsgp_ip_type;
        guint qicsgp_auth;

        if (out_ip_type) {
            mm_get_uint_from_match_info (match_info, 1, &qicsgp_ip_type);
            *out_ip_type = mm_quectel_ec25_qicsgp_context_type_to_ip_family (qicsgp_ip_type);
        }

        if (out_apn)
            *out_apn = mm_get_string_unquoted_from_match_info (match_info, 2);

        if (out_username)
            *out_username = mm_get_string_unquoted_from_match_info (match_info, 3);

        if (out_password)
            *out_password = mm_get_string_unquoted_from_match_info (match_info, 4);

        if (out_auth) {
            mm_get_uint_from_match_info (match_info, 5, &qicsgp_auth);
            *out_auth = mm_quectel_ec25_qicsgp_authentication_to_allowed_auth (qicsgp_auth);
        }

        return TRUE;
    }

    g_set_error (error, MM_CORE_ERROR, MM_CORE_ERROR_NOT_FOUND,
                 "settings for context not found");

    return FALSE;
}

void
mm_quectel_ec25_parse_qicsgp_test (gpointer     log_obj,
                                   const gchar *response,
                                   guint       *min_profile_id,
                                   guint       *max_profile_id,
                                   GError      *error)
{
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    gboolean               checked = FALSE;

    if (!response || !min_profile_id || !max_profile_id) {
        mm_obj_err (log_obj,
                    "parse_qicsgp_test: response or min_profile_id or max_profile_id is null");
        error = g_error_new (MM_CORE_ERROR,
                             MM_CORE_ERROR_FAILED,
                             "parse_qicsgp_test: response or min_profile_id or max_profile_id is null");
       return;
    }

    r = g_regex_new ("\\+QICSGP:\\s*\\(\\s*(\\d+)\\s*-?\\s*(\\d+)\\)",
                     0, 0, NULL);
    g_regex_match_full (r, response, strlen (response), 0, 0, &match_info, &error);
    if (g_match_info_matches(match_info)) {
        if (mm_get_uint_from_match_info (match_info, 1, min_profile_id) &&
            mm_get_uint_from_match_info (match_info, 2, max_profile_id))
            checked = TRUE;
    }

    if (!checked) {
        *min_profile_id = 1;
        *max_profile_id = 16;
    }

    mm_obj_dbg (log_obj, "+QICSGP format details for minimum %d, maximum %d",
                *min_profile_id, *max_profile_id);
}

/*
 * +QIACT: <contextID>,<context_state>,<context_type>[,<IP_address>]
 * ex) +QIACT: 7,1,1,"10.175.194.82"
 * context_state: 0: Deactivated
 *                1: Activated
 *  context_type: 1: IPv4
 *                2: IPv6
 *                3: IPv4v6
 */
GList *
mm_quectel_ec25_parse_qiact_read_response (const gchar  *reply,
                                           GError      **error)
{
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    GError                *inner_error = NULL;
    GList                 *list = NULL;

    if (!reply || !reply[0]) {
        /* Nothing configured, all done */
        return NULL;
    }

    r = g_regex_new ("\\+QIACT:\\s*(\\d),(\\d)",
                     0, 0, &inner_error);

    g_regex_match_full (r, reply, strlen (reply), 0, 0, &match_info, &inner_error);
    while (!inner_error && g_match_info_matches (match_info)) {
        MM3gppPdpContextActive *pdp_active;
        guint cid = 0;
        guint aux = 0;

        if (!mm_get_uint_from_match_info (match_info, 1, &cid)) {
            inner_error = g_error_new (MM_CORE_ERROR,
                                       MM_CORE_ERROR_FAILED,
                                       "Couldn't parse CID from reply: '%s'",
                                       reply);
            break;
        }
        if (!mm_get_uint_from_match_info (match_info, 2, &aux) || (aux != 0 && aux != 1)) {
            inner_error = g_error_new (MM_CORE_ERROR,
                                       MM_CORE_ERROR_FAILED,
                                       "Couldn't parse context status from reply: '%s'",
                                       reply);
            break;
        }

        pdp_active = g_slice_new0 (MM3gppPdpContextActive);
        pdp_active->cid = cid;
        pdp_active->active = (gboolean) aux;
        list = g_list_prepend (list, pdp_active);

        g_match_info_next (match_info, &inner_error);
    }

    if (inner_error) {
        mm_3gpp_pdp_context_active_list_free (list);
        g_propagate_error (error, inner_error);
        g_prefix_error (error, "Couldn't properly parse list of active/inactive PDP contexts. ");
        return NULL;
    }

    list = g_list_sort (list, (GCompareFunc) mm_3gpp_pdp_context_active_cmp);

    return list;
}

gchar *
mm_quectel_ec25_build_qicsgp (const guint                   context_id,
                              const MMQicsgpContextType     context_type,
                              const gchar                  *apn,
                              const gchar                  *username,
                              const gchar                  *password,
                              const MMQicsgpAuthentication  authentication)
{
    gchar *cmd;
    g_autofree gchar          *quoted_apn = NULL;
    g_autofree gchar          *quoted_username = NULL;
    g_autofree gchar          *quoted_password = NULL;

    quoted_apn = mm_port_serial_at_quote_string (apn);
    if (authentication == MM_QICSGP_AUTHENTICATION_NONE)
        cmd = g_strdup_printf ("+QICSGP=%d,%d,%s,\"\",\"\",0",
                               context_id,
                               context_type,
                               quoted_apn);
    else {
        quoted_username = mm_port_serial_at_quote_string (username);
        quoted_password = mm_port_serial_at_quote_string (password);
        cmd = g_strdup_printf ("+QICSGP=%d,%d,%s,%s,%s,%d",
                               context_id,
                               context_type,
                               quoted_apn,
                               quoted_username,
                               quoted_password,
                               authentication);
    }
    return cmd;
}

MMQicsgpContextType
mm_quectel_ec25_ip_family_to_qicsgp_context_type (const MMBearerIpFamily family)
{
    if (family == MM_BEARER_IP_FAMILY_IPV4)
        return MM_QICSGP_CONTEXT_TYPE_IPV4;
    else if (family == MM_BEARER_IP_FAMILY_IPV6)
        return MM_QICSGP_CONTEXT_TYPE_IPV6;
    else if (family == MM_BEARER_IP_FAMILY_IPV4V6)
        return MM_QICSGP_CONTEXT_TYPE_IPV4V6;
    else
        return MM_QICSGP_CONTEXT_TYPE_UNKNOWN;
}

MMBearerIpFamily
mm_quectel_ec25_qicsgp_context_type_to_ip_family (const MMQicsgpContextType type)
{
    if (type == MM_QICSGP_CONTEXT_TYPE_IPV4) /* include MM_QICSGP_CONTEXT_TYPE_IPV4V6 */
        return MM_BEARER_IP_FAMILY_IPV4V6;
    else if (type == MM_QICSGP_CONTEXT_TYPE_IPV6)
        return MM_BEARER_IP_FAMILY_IPV6;
    else
        return MM_BEARER_IP_FAMILY_NONE;
}

MMBearerIpFamily
mm_quectel_ec25_get_ip_family_from_qicsgp_context_type (const gchar *str)
{
    if (!str)
        return MM_BEARER_IP_FAMILY_NONE;
    if (g_str_equal (str, "1"))
        return MM_BEARER_IP_FAMILY_IPV4V6;
    if (g_str_equal (str, "2"))
        return MM_BEARER_IP_FAMILY_IPV6;

    return MM_BEARER_IP_FAMILY_NONE;
}

MMQicsgpAuthentication
mm_quectel_ec25_allowed_auth_to_qicsgp_authentication (const MMBearerAllowedAuth auth,
                                                       const gboolean has_user_pass)
{
    if (!has_user_pass) {
        if (auth & MM_BEARER_ALLOWED_AUTH_NONE)
            return MM_QICSGP_AUTHENTICATION_NONE;
        else
            return MM_QICSGP_AUTHENTICATION_UNKNOWN;
    }

    if ((auth & (MM_BEARER_ALLOWED_AUTH_CHAP | MM_BEARER_ALLOWED_AUTH_PAP)) ==
        (MM_BEARER_ALLOWED_AUTH_CHAP | MM_BEARER_ALLOWED_AUTH_PAP))
        return MM_QICSGP_AUTHENTICATION_PAP_OR_CHAP;

    if (auth & MM_BEARER_ALLOWED_AUTH_PAP)
        return MM_QICSGP_AUTHENTICATION_PAP;

    if (auth & MM_BEARER_ALLOWED_AUTH_CHAP)
        return MM_QICSGP_AUTHENTICATION_CHAP;

    if (auth & MM_BEARER_ALLOWED_AUTH_NONE)
        return MM_QICSGP_AUTHENTICATION_NONE;

    return MM_QICSGP_AUTHENTICATION_UNKNOWN;
}

MMBearerAllowedAuth
mm_quectel_ec25_qicsgp_authentication_to_allowed_auth (const MMQicsgpAuthentication authentication)
{
    if (authentication == MM_QICSGP_AUTHENTICATION_NONE)
        return MM_BEARER_ALLOWED_AUTH_NONE;
    else if (authentication == MM_QICSGP_AUTHENTICATION_PAP)
        return MM_BEARER_ALLOWED_AUTH_PAP;
    else if (authentication == MM_QICSGP_AUTHENTICATION_CHAP)
        return MM_BEARER_ALLOWED_AUTH_CHAP;
    else if (authentication == MM_QICSGP_AUTHENTICATION_PAP_OR_CHAP)
        return MM_BEARER_ALLOWED_AUTH_CHAP | MM_BEARER_ALLOWED_AUTH_PAP;
    else
        return MM_BEARER_ALLOWED_AUTH_UNKNOWN;
}
