/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2023 Atmark Techno Inc.
 */

#include <ModemManager.h>
#include "mm-broadband-bearer-quectel-ec25.h"
#include "mm-modem-helpers-quectel-ec25.h"
#include "mm-base-modem-at.h"
#include "mm-log.h"

G_DEFINE_TYPE (MMBroadbandBearerQuectelEc25, mm_broadband_bearer_quectel_ec25, MM_TYPE_BROADBAND_BEARER)

typedef enum {
    CONNECTION_TYPE_NONE,
    CONNECTION_TYPE_3GPP,
    CONNECTION_TYPE_CDMA,
} ConnectionType;

struct _MMBroadbandBearerPrivate {
    /*-- Common stuff --*/
    /* Data port used when modem is connected */
    MMPort *port;
    /* Current connection type */
    ConnectionType connection_type;

    /* PPP specific */
    MMFlowControl flow_control;

    /*-- 3GPP specific --*/
    /* CID of the PDP context */
    gint profile_id;
};

/*****************************************************************************/
/* 3GPP disconnect */
typedef struct {
    MMBaseModem    *modem;
    MMPortSerialAt *primary;
    MMPort         *data;
    /* 3GPP-specific */
    gchar          *qideact_command;
    gboolean        qideact_sent;
} Disconnect3gppContext;

static gboolean
disconnect_3gpp_finish (MMBroadbandBearer  *self,
                        GAsyncResult       *res,
                        GError            **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
disconnect_3gpp_context_free (Disconnect3gppContext *ctx)
{
    g_free (ctx->qideact_command);
    g_object_unref (ctx->data);
    g_object_unref (ctx->primary);
    g_object_unref (ctx->modem);
    g_free (ctx);
}

static Disconnect3gppContext *
disconnect_3gpp_context_new (MMBroadbandModem *modem,
                             MMPortSerialAt   *primary,
                             MMPort           *data)
{
    Disconnect3gppContext *ctx;

    ctx = g_new0 (Disconnect3gppContext, 1);
    ctx->modem = MM_BASE_MODEM (g_object_ref (modem));
    ctx->primary = g_object_ref (primary);
    ctx->data = g_object_ref (data);

    return ctx;
}

static void
qideact_data_ready (MMBaseModem  *modem,
                  GAsyncResult *res,
                  GTask        *task)
{
    MMBroadbandBearer *self;
    GError            *error = NULL;

    self = g_task_get_source_object (task);

    /* Ignore errors for now */
    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_dbg (self, "PDP context deactivation failed (not fatal): %s", error->message);
        g_error_free (error);
    }

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
data_flash_3gpp_ready (MMPortSerial *data,
                       GAsyncResult *res,
                       GTask        *task)
{
    MMBroadbandBearer     *self;
    Disconnect3gppContext *ctx;
    GError                *error = NULL;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    mm_port_serial_flash_finish (data, res, &error);

    /* We kept the serial port open during connection, now we close that open
     * count */
    mm_port_serial_close (data);

    /* Port is disconnected; update the state */
    mm_port_set_connected (MM_PORT (data), FALSE);

    if (error) {
        /* Ignore "NO CARRIER" response when modem disconnects and any flash
         * failures we might encounter. Other errors are hard errors.
         */
        if (!g_error_matches (error,
                              MM_CONNECTION_ERROR,
                              MM_CONNECTION_ERROR_NO_CARRIER) &&
            !g_error_matches (error,
                              MM_SERIAL_ERROR,
                              MM_SERIAL_ERROR_FLASH_FAILED)) {
            /* Fatal */
            g_task_return_error (task, error);
            g_object_unref (task);
            return;
        }

        mm_obj_dbg (self, "port flashing failed (not fatal): %s", error->message);
        g_error_free (error);
    }

    /* Run init port sequence in the data port */
    mm_port_serial_at_run_init_sequence (MM_PORT_SERIAL_AT (data));

    /* Don't bother doing the CGACT again if it was already done on the
     * primary or secondary port */
    if (ctx->qideact_sent) {
        mm_obj_dbg (self, "PDP disconnection already sent");
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
        return;
    }

    /* cannot send command, because no command specified,
     * cid less than zero.
     */
    if (!ctx->qideact_command) {
        mm_obj_dbg (self, "Finish PDP disconnection");
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   ctx->qideact_command,
                                   10,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL,  /* cancellable */
                                   (GAsyncReadyCallback)qideact_data_ready,
                                   task);
}

static void
data_reopen_3gpp_ready (MMPortSerial *data,
                        GAsyncResult *res,
                        GTask        *task)
{
    MMBroadbandBearer     *self;
    Disconnect3gppContext *ctx;
    GError                *error = NULL;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    g_object_set (data, MM_PORT_SERIAL_AT_INIT_SEQUENCE_ENABLED, TRUE, NULL);

    if (!mm_port_serial_reopen_finish (data, res, &error)) {
        /* Fatal */
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* Just flash the data port */
    mm_obj_dbg (self, "flashing data port %s...", mm_port_get_device (MM_PORT (ctx->data)));
    mm_port_serial_flash (MM_PORT_SERIAL (ctx->data),
                          1000,
                          TRUE,
                          (GAsyncReadyCallback)data_flash_3gpp_ready,
                          task);
}

static void
data_reopen_3gpp (GTask *task)
{
    MMBroadbandBearer     *self;
    Disconnect3gppContext *ctx;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    /* We don't want to run init sequence right away during the reopen, as we're
     * going to flash afterwards. */
    g_object_set (ctx->data, MM_PORT_SERIAL_AT_INIT_SEQUENCE_ENABLED, FALSE, NULL);

    /* Fully reopen the port before flashing */
    mm_obj_dbg (self, "reopening data port %s...", mm_port_get_device (MM_PORT (ctx->data)));
    mm_port_serial_reopen (MM_PORT_SERIAL (ctx->data),
                           1000,
                           (GAsyncReadyCallback)data_reopen_3gpp_ready,
                           task);
}

static void
qideact_ready (MMBaseModem  *modem,
               GAsyncResult *res,
               GTask        *task)
{
    MMBroadbandBearer     *self;
    Disconnect3gppContext *ctx;
    GError                *error = NULL;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!error)
        ctx->qideact_sent = TRUE;
    else {
        mm_obj_dbg (self, "PDP context deactivation failed (not fatal): %s", error->message);
        g_error_free (error);
    }

    data_reopen_3gpp (task);
}

static void
disconnect_3gpp (MMBroadbandBearer   *self,
                 MMBroadbandModem    *modem,
                 MMPortSerialAt      *primary,
                 MMPortSerialAt      *secondary,
                 MMPort              *data,
                 guint                cid,
                 GAsyncReadyCallback  callback,
                 gpointer             user_data)
{
    Disconnect3gppContext *ctx;
    GTask                 *task;

    g_assert (primary != NULL);

    ctx = disconnect_3gpp_context_new (modem, primary, data);

    ctx->qideact_command = (cid > 0 ?
                            g_strdup_printf ("+QIDEACT=%d", cid) :
                            NULL);

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, ctx, (GDestroyNotify)disconnect_3gpp_context_free);

    /* If the primary port is NOT connected (doesn't have to be the data port),
     * we'll send CGACT there */
    if (ctx->qideact_command &&
        !mm_port_get_connected (MM_PORT (ctx->primary))) {
        mm_obj_dbg (self, "sending PDP context deactivation in primary port...");
        mm_base_modem_at_command_full (ctx->modem,
                                       ctx->primary,
                                       ctx->qideact_command,
                                       45,
                                       FALSE,
                                       FALSE, /* raw */
                                       NULL,  /* cancellable */
                                       (GAsyncReadyCallback)qideact_ready,
                                       task);
        return;
    }

    data_reopen_3gpp (task);
}

/*****************************************************************************/
/* 3GPP CONNECT */
typedef struct {
    MMBroadbandBearer *self;
    MMBaseModem       *modem;
    MMPortSerialAt    *primary;
    MMPortSerialAt    *secondary;

    MMPort            *data;
    gboolean           close_data_on_exit;

    /* 3GPP-specific */
    guint              cid;
    gboolean           use_existing_cid;
    MMBearerIpFamily   ip_family;

    guint              load_operator_code_retry_max;
    guint              load_operator_code_retry;
    guint              activate_pdp_retry_max;
    guint              activate_pdp_retry;
} DetailedConnectContext;

static const guint LOAD_OPERATOR_CODE_RETRY_MAX = 10;
static const guint ACTIVATE_PDP_RETRY_MAX = 3;

static DetailedConnectContext *
detailed_connect_context_new (MMBroadbandBearer *self,
                              MMBroadbandModem *modem,
                              MMPortSerialAt *primary,
                              MMPortSerialAt *secondary)
{
    DetailedConnectContext *ctx;

    ctx = g_slice_new0 (DetailedConnectContext);
    ctx->self = MM_BROADBAND_BEARER(g_object_ref (self));
    ctx->modem = MM_BASE_MODEM (g_object_ref (modem));
    ctx->primary = g_object_ref (primary);
    ctx->secondary = (secondary ? g_object_ref (secondary) : NULL);
    ctx->close_data_on_exit = FALSE;

    ctx->use_existing_cid = FALSE;
    ctx->ip_family = mm_bearer_properties_get_ip_type (mm_base_bearer_peek_config (MM_BASE_BEARER (self)));
    mm_3gpp_normalize_ip_family (&ctx->ip_family);

    ctx->load_operator_code_retry_max = LOAD_OPERATOR_CODE_RETRY_MAX;
    ctx->activate_pdp_retry_max = ACTIVATE_PDP_RETRY_MAX;

    return ctx;
}

static void
detailed_connect_context_free (DetailedConnectContext *ctx)
{
    g_object_unref (ctx->primary);
    if (ctx->secondary)
        g_object_unref (ctx->secondary);
    if (ctx->data) {
        if (ctx->close_data_on_exit)
            mm_port_serial_close (MM_PORT_SERIAL (ctx->data));
        g_object_unref (ctx->data);
    }
    g_object_unref (ctx->modem);
    g_object_unref (ctx->self);
    g_slice_free (DetailedConnectContext, ctx);
}

static void
get_ip_config_3gpp_ready (MMBroadbandModem *modem,
                          GAsyncResult     *res,
                          GTask            *task)
{
    DetailedConnectContext *ctx;
    MMBearerIpConfig       *ipv4_config = NULL;
    MMBearerIpConfig       *ipv6_config = NULL;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    if (!MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp_finish (ctx->self,
                                                                               res,
                                                                               &ipv4_config,
                                                                               &ipv6_config,
                                                                               &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* Keep port open during connection */
    if (MM_IS_PORT_SERIAL_AT (ctx->data))
        ctx->close_data_on_exit = FALSE;

    g_task_return_pointer (
        task,
        mm_bearer_connect_result_new (ctx->data, ipv4_config, ipv6_config),
        (GDestroyNotify)mm_bearer_connect_result_unref);
    g_object_unref (task);

    if (ipv4_config)
        g_object_unref (ipv4_config);
    if (ipv6_config)
        g_object_unref (ipv6_config);
}

static void
dial_3gpp_ready (MMBroadbandModem *modem,
                 GAsyncResult     *res,
                 GTask            *task)
{
    DetailedConnectContext            *ctx;
    MMBearerIpMethod                  ip_method = MM_BEARER_IP_METHOD_UNKNOWN;
    GError                           *error = NULL;
    g_autoptr(MMBearerConnectResult)  result = NULL;
    g_autoptr(MMBearerIpConfig)       ipv4_config = NULL;
    g_autoptr(MMBearerIpConfig)       ipv6_config = NULL;

    ctx = g_task_get_task_data (task);

    ctx->data = MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->dial_3gpp_finish (ctx->self, res, &error);
    if (!ctx->data) {
        /* Clear CID when it failed to connect. */
        ctx->self->priv->profile_id = MM_3GPP_PROFILE_ID_UNKNOWN;
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* If the dialling operation used an AT port, it is assumed to have an extra
     * open() count. */
    if (MM_IS_PORT_SERIAL_AT (ctx->data))
        ctx->close_data_on_exit = TRUE;

    if (MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp &&
        MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp_finish) {
        /* Launch specific IP config retrieval */
        MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp (
            ctx->self,
            MM_BROADBAND_MODEM (ctx->modem),
            ctx->primary,
            ctx->secondary,
            ctx->data,
            ctx->cid,
            ctx->ip_family,
            (GAsyncReadyCallback)get_ip_config_3gpp_ready,
            task);
        return;
    }

    /* Yuhu! */

    /* Keep port open during connection */
    if (MM_IS_PORT_SERIAL_AT (ctx->data))
        ctx->close_data_on_exit = FALSE;

    /* If no specific IP retrieval requested, set the default implementation
     * (PPP if data port is AT, DHCP otherwise) */
    ip_method = MM_IS_PORT_SERIAL_AT (ctx->data) ?
                    MM_BEARER_IP_METHOD_PPP :
                    MM_BEARER_IP_METHOD_DHCP;

    if (ctx->ip_family & MM_BEARER_IP_FAMILY_IPV4 ||
        ctx->ip_family & MM_BEARER_IP_FAMILY_IPV4V6) {
        ipv4_config = mm_bearer_ip_config_new ();
        mm_bearer_ip_config_set_method (ipv4_config, ip_method);
    }
    if (ctx->ip_family & MM_BEARER_IP_FAMILY_IPV6 ||
        ctx->ip_family & MM_BEARER_IP_FAMILY_IPV4V6) {
        ipv6_config = mm_bearer_ip_config_new ();
        mm_bearer_ip_config_set_method (ipv6_config, ip_method);
    }
    g_assert (ipv4_config || ipv6_config);

    result = mm_bearer_connect_result_new (ctx->data, ipv4_config, ipv6_config);
    mm_bearer_connect_result_set_profile_id (result, ctx->self->priv->profile_id);
    g_task_return_pointer (task, g_steal_pointer (&result), (GDestroyNotify)mm_bearer_connect_result_unref);
    g_object_unref (task);
}

static void
start_3gpp_dial (GTask *task)
{
    DetailedConnectContext *ctx;

    ctx = g_task_get_task_data (task);

    /* Keep CID around after initializing the PDP context in order to
     * handle corresponding unsolicited PDP activation responses. */
    ctx->self->priv->profile_id = ctx->cid;
    MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->dial_3gpp (ctx->self,
                                                          ctx->modem,
                                                          ctx->primary,
                                                          ctx->cid,
                                                          g_task_get_cancellable (task),
                                                          (GAsyncReadyCallback)dial_3gpp_ready,
                                                          task);
}

static gboolean compare_pdp_context (MMBearerProperties  *config,
                                     MMBearerIpFamily     current_ip_family,
                                     const gchar         *current_apn,
                                     MMBearerAllowedAuth  current_auth,
                                     const gchar         *current_username,
                                     const gchar         *current_password,
                                     MMBroadbandBearer   *log_obj)
{
    const gchar         *apn;
    MMBearerAllowedAuth  auth;
    const gchar         *username;
    gboolean             has_username;
    const gchar         *password;
    gboolean             has_password;
    MMBearerIpFamily     ip_family;

    gboolean             current_has_username;
    gboolean             current_has_password;

    apn = mm_bearer_properties_get_apn (config);
    if (g_strcmp0 (current_apn, apn) != 0)
        return FALSE;

    ip_family = mm_bearer_properties_get_ip_type (config);
    if (mm_quectel_ec25_ip_family_to_qicsgp_context_type(current_ip_family) !=
        mm_quectel_ec25_ip_family_to_qicsgp_context_type(ip_family))
        return FALSE;

    auth      = mm_bearer_properties_get_allowed_auth (config);
    username  = mm_bearer_properties_get_user (config);
    password  = mm_bearer_properties_get_password (config);

    has_username = (username && username[0]);
    has_password = (password && password[0]);

    if ((auth == MM_BEARER_ALLOWED_AUTH_UNKNOWN) && (has_username || has_password)) {
        /* If user/passwd given, default to CHAP (more common than PAP) */
        auth = MM_BEARER_ALLOWED_AUTH_CHAP;
    }

    if (has_username || has_password) {
        if ((auth & (MM_BEARER_ALLOWED_AUTH_PAP | MM_BEARER_ALLOWED_AUTH_CHAP)) ==
            (MM_BEARER_ALLOWED_AUTH_PAP | MM_BEARER_ALLOWED_AUTH_CHAP))
            auth = MM_BEARER_ALLOWED_AUTH_PAP | MM_BEARER_ALLOWED_AUTH_CHAP;
        else if (auth & MM_BEARER_ALLOWED_AUTH_CHAP)
            auth = MM_BEARER_ALLOWED_AUTH_CHAP;
        else if (auth & MM_BEARER_ALLOWED_AUTH_PAP)
            auth = MM_BEARER_ALLOWED_AUTH_PAP;
        else if (auth & MM_BEARER_ALLOWED_AUTH_NONE) {
            mm_obj_warn (log_obj, "user/password given but 'none' authentication requested");
            auth = MM_BEARER_ALLOWED_AUTH_NONE;
        }
    } else {
        if ((auth & MM_BEARER_ALLOWED_AUTH_NONE) ||
            (auth == MM_BEARER_ALLOWED_AUTH_UNKNOWN))
            auth = MM_BEARER_ALLOWED_AUTH_NONE;
    }

    if (current_auth != auth)
        return FALSE;

    if (auth == MM_BEARER_ALLOWED_AUTH_NONE) {
        return TRUE;
    }

    current_has_username = (current_username && current_username[0]);
    current_has_password = (current_password && current_password[0]);

    if (current_has_username != has_username || current_has_password != has_password)
        return FALSE;

    if (has_username && (g_strcmp0 (current_username, username) != 0))
        return FALSE;

    if (has_password && (g_strcmp0 (current_password, password) != 0))
        return FALSE;

    return TRUE;
}

static gchar *
build_qicsgp_string (MMBroadbandBearerQuectelEc25 *self,
                     MMBearerProperties           *config,
                     guint                         cid)
{
    MMBearerAllowedAuth      auth;
    MMQicsgpAuthentication   encoded_auth = MM_QICSGP_AUTHENTICATION_UNKNOWN;
    MMQicsgpContextType      encoded_ctx_type;
    gboolean                 has_user;
    gboolean                 has_passwd;
    const gchar             *apn;
    const gchar             *user;
    const gchar             *passwd;

    encoded_ctx_type = mm_quectel_ec25_ip_family_to_qicsgp_context_type (mm_bearer_properties_get_ip_type (config));

    apn    = mm_bearer_properties_get_apn          (config);
    user   = mm_bearer_properties_get_user         (config);
    passwd = mm_bearer_properties_get_password     (config);
    auth   = mm_bearer_properties_get_allowed_auth (config);

    has_user     = (user   && user[0]);
    has_passwd   = (passwd && passwd[0]);
    encoded_auth = mm_quectel_ec25_allowed_auth_to_qicsgp_authentication (auth, has_user || has_passwd);

    /* When 'none' requested, we won't require user/password */
    if (encoded_auth == MM_QICSGP_AUTHENTICATION_NONE && (has_user || has_passwd))
            mm_obj_warn (self, "APN user/password given but 'none' authentication requested");

    /* No explicit auth type requested? */
    if (encoded_auth == MM_QICSGP_AUTHENTICATION_UNKNOWN) {
        if (!has_user && !has_passwd) {
            /* If no user/passwd given, try authenticarion none */
            encoded_auth = MM_QICSGP_AUTHENTICATION_NONE;
        } else {
            /* If user/passwd given, default to CHAP (more common than PAP) */
            mm_obj_dbg (self,
                "APN user/password given but no authentication type explicitly requested: defaulting to 'CHAP'");
            encoded_auth = MM_QICSGP_AUTHENTICATION_CHAP;
        }
    }

    return mm_quectel_ec25_build_qicsgp (cid, encoded_ctx_type, apn, user, passwd, encoded_auth);
}

static void activate_pdp_context (MMBaseModem *modem,
                                  GTask       *task);

static void
activate_pdp_context_ready (MMBaseModem  *modem,
                            GAsyncResult *res,
                            GTask        *task)
{
    DetailedConnectContext *ctx;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        ctx->activate_pdp_retry++;
        if (ctx->activate_pdp_retry >= ctx->activate_pdp_retry_max) {
            mm_obj_warn (ctx->self, "Couldn't Activate PDP context: '%s'", error->message);
            /* Clear CID when it failed to connect. */
            ctx->self->priv->profile_id = MM_3GPP_PROFILE_ID_UNKNOWN;
            g_task_return_error (task, error);
            g_object_unref (task);
            return;
        }
        g_error_free (error);
        mm_obj_dbg (ctx->self, "Retry Activate PDP context");
        activate_pdp_context (modem, task);
        return;
    }

    mm_obj_dbg (ctx->self, "Activated PDP context, start dial.");
    start_3gpp_dial (task);
}

static void activate_pdp_context (MMBaseModem *modem,
                                  GTask       *task)
{
    DetailedConnectContext *ctx;
    g_autofree gchar       *command = NULL;

    ctx = g_task_get_task_data (task);

    command = g_strdup_printf ("+QIACT=%d", ctx->cid);
    mm_base_modem_at_command (
        modem,
        command,
        20,
        FALSE,
        (GAsyncReadyCallback)activate_pdp_context_ready,
        task);
}

static void
query_activate_pdp_context_ready (MMBaseModem  *modem,
                                  GAsyncResult *res,
                                  GTask        *task)
{
    DetailedConnectContext *ctx;
    g_autofree GError      *error = NULL;
    const gchar            *response;
    g_autofree gchar       *command = NULL;
    GList                  *pdp_context_active_list = NULL;
    GList                  *l;
    gboolean                activated = FALSE;

    ctx = g_task_get_task_data (task);

    response = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (response) {
        pdp_context_active_list = mm_quectel_ec25_parse_qiact_read_response (response, &error);
        if (pdp_context_active_list) {
            for (l = pdp_context_active_list; l; l = g_list_next (l)) {
                MM3gppPdpContextActive *iter = l->data;
                if (iter->cid == ctx->cid) {
                    activated = iter->active;
                    break;
                }
            }
        }
    }
    mm_3gpp_pdp_context_active_list_free (pdp_context_active_list);

    if (activated) {
        mm_obj_dbg (ctx->self, "Context %d already activated, start dial.", ctx->cid);
        start_3gpp_dial (task);
        return;
    }

    activate_pdp_context (modem, task);
}

static void send_query_cops (GTask *task);

static void
retry_query_loaded_operator_code (GTask  *task,
                                  GError *error)
{
    DetailedConnectContext *ctx;

    ctx = g_task_get_task_data (task);

    ctx->load_operator_code_retry++;
    if (ctx->load_operator_code_retry >= ctx->load_operator_code_retry_max) {
        GError *inner_error = NULL;
        mm_obj_dbg (ctx->self, "failed to network connection.");
        inner_error = error;
        if (!inner_error)
            inner_error = g_error_new (MM_CORE_ERROR,
                                       MM_CORE_ERROR_FAILED,
                                       "failed to load Operator Code.");
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }
    mm_obj_dbg (ctx->self, "retry check network connection: %d/%d",
                ctx->load_operator_code_retry, ctx->load_operator_code_retry_max);
    g_usleep (3 * G_USEC_PER_SEC);
    g_error_free (error);
    send_query_cops (task);
}

/* check loaded operator code */
static void
query_loaded_operator_code_ready (MMBaseModem  *modem,
                                  GAsyncResult *res,
                                  GTask        *task)
{
    DetailedConnectContext *ctx;
    const gchar            *response;
    GError                 *error = NULL;
    gchar                  *operator_code = NULL;

    ctx = g_task_get_task_data (task);

    response = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!response) {
        retry_query_loaded_operator_code (task, error);
        return;
    }

    if (!mm_3gpp_parse_cops_read_response (response, NULL, NULL, &operator_code, NULL, ctx->self, &error)) {
        retry_query_loaded_operator_code (task, error);
        return;
    }

    mm_obj_dbg (ctx->self, "loaded Operator Code: %s", operator_code);
    g_free(operator_code);

    /* query activated pdp context */
    mm_base_modem_at_command (
        MM_BASE_MODEM (modem),
        "+QIACT?",
        3,
        FALSE,
        (GAsyncReadyCallback)query_activate_pdp_context_ready,
        task);
}


static void
send_query_cops (GTask *task)
{
    DetailedConnectContext *ctx;

    ctx = g_task_get_task_data (task);

    mm_obj_dbg (ctx->self, "load operator code ...");
    mm_base_modem_at_command (ctx->modem, "+COPS=3,2", 3, FALSE, NULL, NULL);
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+COPS?",
                                   10,
                                   FALSE,
                                   FALSE,
                                   NULL,
                                   (GAsyncReadyCallback) query_loaded_operator_code_ready,
                                   task);
}

static void
cfun_one_ready (MMBaseModem  *modem,
                GAsyncResult *res,
                GTask        *task)
{
    DetailedConnectContext *ctx;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_warn (ctx->self, "Couldn't set Phone Functionality Full: %s", error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    send_query_cops (task);
}

static void
cfun_zero_ready (MMBaseModem  *modem,
                 GAsyncResult *res,
                 GTask        *task)
{
    DetailedConnectContext *ctx;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_warn (ctx->self, "Couldn't set Phone Functionality Minimun: %s", error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+CFUN=1",
                                   15,
                                   FALSE,
                                   FALSE,
                                   NULL,
                                   (GAsyncReadyCallback) cfun_one_ready,
                                   task);
}

static void
initialize_pdp_context_ready (MMBaseModem  *modem,
                              GAsyncResult *res,
                              GTask        *task)
{
    DetailedConnectContext *ctx;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    /* If cancelled, complete */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_warn (ctx->self,
                     "Couldn't initialize PDP context with our APN: '%s'",
                     error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+CFUN=0",
                                   15,
                                   FALSE,
                                   FALSE,
                                   NULL,
                                   (GAsyncReadyCallback) cfun_zero_ready,
                                   task);
}

static void
set_pdp_context (MMBaseModem *modem,
                 GTask       *task)
{
    g_autofree gchar             *command;
    MMBearerProperties           *config;
    MMBroadbandBearerQuectelEc25 *self;
    DetailedConnectContext       *ctx;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    config = mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self));

    command = build_qicsgp_string (self, config, ctx->cid);
    mm_base_modem_at_command (
        modem,
        command,
        20,
        FALSE,
        (GAsyncReadyCallback)initialize_pdp_context_ready,
        task);
}

static void
query_qicsgp_ready (MMBaseModem  *modem,
                    GAsyncResult *res,
                    GTask        *task)
{
    DetailedConnectContext *ctx;
    const gchar            *response;
    GError                 *error = NULL;
    MMBearerIpFamily        current_ip_family;
    g_autofree gchar       *current_apn;
    MMBearerAllowedAuth     current_auth;
    g_autofree gchar       *current_username;
    g_autofree gchar       *current_password;
    MMBearerProperties     *config;

    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    ctx = g_task_get_task_data (task);

    response = mm_base_modem_at_command_finish (modem, res, &error);
    if (!response) {
        mm_obj_warn (ctx->self, "failed to get +QISGP response.");
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    if (!mm_quectel_ec25_parse_qicsgp_read_response (response,
                                                     &current_auth,
                                                     &current_ip_family,
                                                     &current_apn,
                                                     &current_username,
                                                     &current_password,
                                                     &error) && ctx->cid == 0) {
        mm_obj_dbg (ctx->self, "query_qicsgp_ready: failed to parse +QISGP:%s", error->message);
        g_error_free (error);
        error = NULL;
    }

    config = mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self));
    if (config) {
        /* The same PDP context exsists */
        if (compare_pdp_context (config,
                                 current_ip_family,
                                 current_apn,
                                 current_auth,
                                 current_username,
                                 current_password,
                                 ctx->self)) {
            ctx->use_existing_cid = TRUE;
            mm_obj_dbg (ctx->self, "using existing cid: %d, start check operator.", ctx->cid);
            send_query_cops (task);
            return;
        }

        /* The APN is the same but there are also different parameters. */
        if (g_strcmp0 (current_apn, mm_bearer_properties_get_apn (config)) == 0) {
            ctx->use_existing_cid = TRUE;
            mm_obj_dbg (ctx->self, "using existing cid: %d, set PDP context.", ctx->cid);
        } else {
            /* The APN is empty, set new settings. */
            ctx->use_existing_cid = FALSE;
            mm_obj_dbg (ctx->self, "using cid: %d, set PDP context.", ctx->cid);
        }
        set_pdp_context (modem, task);
    } else {
        mm_obj_dbg (ctx->self, "query_qicsgp_ready: failed to peek config.");
        error = g_error_new (MM_CORE_ERROR,
                             MM_CORE_ERROR_FAILED,
                             "no bearer settings.");
        g_task_return_error (task, error);
        g_object_unref (task);
    }
}

static void
connect_3gpp (MMBroadbandBearer   *self,
              MMBroadbandModem    *modem,
              MMPortSerialAt      *primary,
              MMPortSerialAt      *secondary,
              GCancellable        *cancellable,
              GAsyncReadyCallback  callback,
              gpointer             user_data)
{
    DetailedConnectContext *ctx;
    GTask                  *task;

    g_assert (primary != NULL);

    ctx = detailed_connect_context_new (self, modem, primary, secondary);

    task = g_task_new (self, cancellable, callback, user_data);
    g_task_set_task_data (task, ctx, (GDestroyNotify)detailed_connect_context_free);

    mm_obj_dbg (ctx->self, "Quectel EC25, We always use cid 1.");
    ctx->cid = 1;

    mm_base_modem_at_command (
        MM_BASE_MODEM (modem),
        "+QICSGP=1",
        3,
        FALSE,
        (GAsyncReadyCallback)query_qicsgp_ready,
        task);
}

/*****************************************************************************/
/* Connection status monitoring */

static MMBearerConnectionStatus
load_connection_status_finish (MMBaseBearer  *bearer,
                               GAsyncResult  *res,
                               GError       **error)
{
    GError *inner_error = NULL;
    gssize value;

    value = g_task_propagate_int (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return MM_BEARER_CONNECTION_STATUS_UNKNOWN;
    }
    return (MMBearerConnectionStatus)value;
}

static void
qiact_periodic_query_ready (MMBaseModem  *modem,
                            GAsyncResult *res,
                            GTask        *task)
{
    MMBroadbandBearer        *self;
    const gchar              *response;
    GError                   *error = NULL;
    GList                    *pdp_active_list = NULL;
    GList                    *l;
    MMBearerConnectionStatus  status = MM_BEARER_CONNECTION_STATUS_UNKNOWN;

    self = MM_BROADBAND_BEARER (g_task_get_source_object (task));

    response = mm_base_modem_at_command_finish (modem, res, &error);
    if (response)
        pdp_active_list = mm_quectel_ec25_parse_qiact_read_response (response, &error);

    if (error) {
        g_assert (!pdp_active_list);
        g_prefix_error (&error, "Couldn't check current list of active PDP contexts: ");
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    for (l = pdp_active_list; l; l = g_list_next (l)) {
        MM3gppPdpContextActive *pdp_active;

        /* Just assume the first active PDP context found is the one we're looking for. */
        pdp_active = (MM3gppPdpContextActive *)(l->data);
        if (pdp_active->cid == (guint)self->priv->profile_id) {
            status = (pdp_active->active ? MM_BEARER_CONNECTION_STATUS_CONNECTED : MM_BEARER_CONNECTION_STATUS_DISCONNECTED);
            break;
        }
    }
    mm_3gpp_pdp_context_active_list_free (pdp_active_list);

    /* PDP context not found? This shouldn't happen, error out */
    if (status == MM_BEARER_CONNECTION_STATUS_UNKNOWN)
        g_task_return_new_error (task, MM_CORE_ERROR, MM_CORE_ERROR_FAILED,
                                 "PDP context not found in the known contexts list");
    else
        g_task_return_int (task, (gssize) status);
    g_object_unref (task);
}

static void
load_connection_status (MMBaseBearer        *self,
                        GAsyncReadyCallback  callback,
                        gpointer             user_data)
{
    GTask          *task;
    MMBaseModem    *modem = NULL;
    MMPortSerialAt *port;

    task = g_task_new (self, NULL, callback, user_data);

    g_object_get (MM_BASE_BEARER (self),
                  MM_BASE_BEARER_MODEM, &modem,
                  NULL);

    /* If CID not defined, error out */
    if (MM_BROADBAND_BEARER (self)->priv->profile_id == MM_3GPP_PROFILE_ID_UNKNOWN) {
        g_task_return_new_error (task, MM_CORE_ERROR, MM_CORE_ERROR_FAILED,
                                 "Couldn't load connection status: cid not defined");
        g_object_unref (task);
        goto out;
    }

    /* If no control port available, error out */
    port = mm_base_modem_peek_best_at_port (modem, NULL);
    if (!port) {
        g_task_return_new_error (task, MM_CORE_ERROR, MM_CORE_ERROR_UNSUPPORTED,
                                 "Couldn't load connection status: no control port available");
        g_object_unref (task);
        goto out;
    }

    mm_base_modem_at_command_full (MM_BASE_MODEM (modem),
                                   port,
                                   "+QIACT?",
                                   3,
                                   FALSE, /* allow cached */
                                   FALSE, /* raw */
                                   NULL, /* cancellable */
                                   (GAsyncReadyCallback) qiact_periodic_query_ready,
                                   task);

out:
    g_clear_object (&modem);
}

/*****************************************************************************/
/* Setup and Init Bearers */

MMBaseBearer *
mm_broadband_bearer_quectel_ec25_new_finish (GAsyncResult  *res,
                                             GError       **error)
{
    GObject *bearer;
    GObject *source;

    source = g_async_result_get_source_object (res);
    bearer = g_async_initable_new_finish (G_ASYNC_INITABLE (source), res, error);
    g_object_unref (source);

    if (!bearer)
        return NULL;

    /* Only export valid bearers */
    mm_base_bearer_export (MM_BASE_BEARER (bearer));

    return MM_BASE_BEARER (bearer);
}

void
mm_broadband_bearer_quectel_ec25_new (MMBroadbandModemQuectelEc25 *modem,
                                      MMBearerProperties          *config,
                                      GCancellable                *cancellable,
                                      GAsyncReadyCallback          callback,
                                      gpointer                     user_data)
{
    g_async_initable_new_async (
        MM_TYPE_BROADBAND_BEARER_QUECTEL_EC25,
        G_PRIORITY_DEFAULT,
        cancellable,
        callback,
        user_data,
        MM_BASE_BEARER_MODEM, modem,
        MM_BASE_BEARER_CONFIG, config,
        NULL);
}

static void
mm_broadband_bearer_quectel_ec25_init (MMBroadbandBearerQuectelEc25 *self)
{
}

static void
mm_broadband_bearer_quectel_ec25_class_init (MMBroadbandBearerQuectelEc25Class *klass)
{
    MMBroadbandBearerClass *broadband_bearer_class = MM_BROADBAND_BEARER_CLASS (klass);
    MMBaseBearerClass *base_bearer_class = &broadband_bearer_class->parent;

    broadband_bearer_class->connect_3gpp               = connect_3gpp;
    broadband_bearer_class->disconnect_3gpp            = disconnect_3gpp;
    broadband_bearer_class->disconnect_3gpp_finish     = disconnect_3gpp_finish;

    base_bearer_class->load_connection_status          = load_connection_status;
    base_bearer_class->load_connection_status_finish   = load_connection_status_finish;
#if defined WITH_SUSPEND_RESUME
    base_bearer_class->reload_connection_status        = load_connection_status;
    base_bearer_class->reload_connection_status_finish = load_connection_status_finish;
#endif

}
