/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Copyright (C) 2022-2023 Atmark Techno, Inc.
 */

#include <string.h>
#include <gmodule.h>

#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>

#include "mm-plugin-common.h"
#include "mm-broadband-modem-cinterion-ems31.h"
#include "mm-serial-parsers.h"

#include "mm-log-object.h"

#if defined WITH_QMI
#include "mm-broadband-modem-qmi-cinterion.h"
#endif

#if defined WITH_MBIM
#include "mm-broadband-modem-mbim-cinterion.h"
#endif

#define MM_TYPE_PLUGIN_CINTERION_EMS31 mm_plugin_cinterion_ems31_get_type ()
MM_DEFINE_PLUGIN (CINTERION_EMS31, cinterion_ems31, CinterionEms31)

/*****************************************************************************/

static MMBaseModem *
create_modem (MMPlugin *self,
              const gchar  *uid,
              const gchar  *physdev,
              const gchar **drivers,
              guint16       vendor,
              guint16       product,
              guint16       subsystem_vendor,
              GList        *probes,
              GError      **error)
{
#if defined WITH_QMI
    if (mm_port_probe_list_has_qmi_port (probes)) {
        mm_obj_dbg (self, "QMI-powered Cinterion modem found...");
        return MM_BASE_MODEM (mm_broadband_modem_qmi_new (uid,
                                                          physdev,
                                                          drivers,
                                                          mm_plugin_get_name (self),
                                                          vendor,
                                                          product));
    }
#endif

#if defined WITH_MBIM
    if (mm_port_probe_list_has_mbim_port (probes)) {
        mm_obj_dbg (self, "MBIM-powered Cinterion modem found...");
        return MM_BASE_MODEM (mm_broadband_modem_mbim_new (uid,
                                                           physdev,
                                                           drivers,
                                                           mm_plugin_get_name (self),
                                                           vendor,
                                                           product));
    }
#endif

    return MM_BASE_MODEM (mm_broadband_modem_cinterion_ems31_new (uid,
                                                                  physdev,
                                                                  drivers,
                                                                  mm_plugin_get_name (self),
                                                                  vendor,
                                                                  product));
}

/*****************************************************************************/
/* grab port */

static gboolean
grab_port (MMPlugin    *self,
           MMBaseModem *modem,
           MMPortProbe *probe,
           GError      **error)
{
    MMPortSerialAtFlag pflags = MM_PORT_SERIAL_AT_FLAG_NONE;
    MMKernelDevice *pdev;
    MMPortType ptype;
    const gchar* name = mm_port_probe_get_port_name (probe);

    ptype = mm_port_probe_get_port_type (probe);
    pdev = mm_port_probe_peek_port (probe);

    if (mm_kernel_device_has_property (pdev,
                                       "ID_MM_CINTERION_EMS31_TTY_AT_SYMLINK")) {
        name = mm_kernel_device_get_property (pdev,
                                              "ID_MM_CINTERION_EMS31_TTY_AT_SYMLINK");
        mm_obj_dbg (self, "[%s] symlink name", name);
        return mm_base_modem_grab_symlink_at_primary_port (modem, pdev, name, error);
    } else if (mm_kernel_device_get_global_property_as_boolean (pdev,
                    "ID_MM_CINTERION_EMS31_PORT_TYPE_AT")) {
        mm_obj_dbg (self, "(%s/%s)' Port flagged as primary",
                    mm_port_probe_get_port_subsys (probe),
                    mm_port_probe_get_port_name (probe));
        pflags = MM_PORT_SERIAL_AT_FLAG_PRIMARY;
    }

    return mm_base_modem_grab_port (modem, pdev, ptype, pflags, error);
}

/*****************************************************************************/
/* Custom init */

static gboolean
cinterion_ems31_custom_init_finish (MMPortProbe   *probe,
                                    GAsyncResult  *result,
                                    GError       **error)
{
    return g_task_propagate_boolean (G_TASK (result), error);
}

static void
scfg_autoattach_disable_ready (MMPortSerialAt *port,
                               GAsyncResult   *res,
                               GTask          *task)
{
    MMPortProbe       *probe;
    g_autoptr(GError)  error = NULL;

    probe = g_task_get_source_object (task);

    mm_port_serial_at_command_finish (port, res, &error);
    if (error)
        mm_obj_warn (probe, "failed to set AutoAttach: %s", error->message);

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}



static void
scfg_autoattach_query_ready (MMPortSerialAt *port,
                             GAsyncResult   *res,
                             GTask          *task)
{
    const gchar       *response;
    MMPortProbe       *probe;
    g_autoptr(GError)  error = NULL;

    probe = g_task_get_source_object (task);

    response = mm_port_serial_at_command_finish (port, res, &error);
    if (error)
        mm_obj_warn (probe, "failed to AT^SCFG=\"GPRS/AutoAttach\": %s", error->message);

    if (response && strstr (response ,"enabled")) {
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
        return;
    }

    mm_port_serial_at_command (port,
                               "AT^SCFG=\"GPRS/AutoAttach\",\"enabled\"",
                               10,
                               FALSE,
                               FALSE,
                               g_task_get_cancellable (task),
                               (GAsyncReadyCallback)scfg_autoattach_disable_ready,
                               task);
}

static void
cinterion_ems31_custom_init (MMPortProbe         *probe,
                             MMPortSerialAt      *port,
                             GCancellable        *cancellable,
                             GAsyncReadyCallback  callback,
                             gpointer             user_data)
{
    GTask *task;

    task = g_task_new (probe, cancellable, callback, user_data);
    g_task_set_check_cancellable (task, FALSE);

    if (mm_kernel_device_get_global_property_as_boolean (mm_port_probe_peek_port (probe),
                                                         "ID_MM_CINTERION_EMS31_PORT_TYPE_AT"))
        mm_port_serial_at_command (port,
                                   "AT^SCFG=\"GPRS/AutoAttach\"",
                                   10,
                                   FALSE,
                                   FALSE,
                                   g_task_get_cancellable (task),
                                   (GAsyncReadyCallback)scfg_autoattach_query_ready,
                                   task);
    else {
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
    }
}

/*****************************************************************************/

MM_PLUGIN_NAMED_CREATOR_SCOPE MMPlugin *
mm_plugin_create_cinterion_ems31 (void)
{
    static const gchar *subsystems[] = { "tty", NULL };
    static const gchar *udev_tags[] = {
        "ID_MM_CINTERION_EMS31",
        NULL };
    static const MMAsyncMethod custom_init = {
        .async  = G_CALLBACK (cinterion_ems31_custom_init),
        .finish = G_CALLBACK (cinterion_ems31_custom_init_finish),
    };

    return MM_PLUGIN (
        g_object_new (MM_TYPE_PLUGIN_CINTERION_EMS31,
                      MM_PLUGIN_NAME,               MM_MODULE_NAME,
                      MM_PLUGIN_ALLOWED_SUBSYSTEMS, subsystems,
                      MM_PLUGIN_ALLOWED_UDEV_TAGS,  udev_tags,
                      MM_PLUGIN_ALLOWED_AT,         TRUE,
                      MM_PLUGIN_ALLOWED_QMI,        TRUE,
                      MM_PLUGIN_ALLOWED_MBIM,       TRUE,
                      MM_PLUGIN_CUSTOM_INIT,        &custom_init,
                      NULL));
}

static void
mm_plugin_cinterion_ems31_init (MMPluginCinterionEms31 *self)
{
}

static void
mm_plugin_cinterion_ems31_class_init (MMPluginCinterionEms31Class *klass)
{
    MMPluginClass *plugin_class = MM_PLUGIN_CLASS (klass);

    plugin_class->create_modem = create_modem;
    plugin_class->grab_port = grab_port;
}
