/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2024 Atmark Techno Inc.
 */

#include "ModemManager.h"
#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>
#include "mm-modem-helpers.h"

static gint
pdp_context_cmp (MM3gppPdpContext *a,
                 MM3gppPdpContext *b)
{
    return (a->cid - b->cid);
}

GList *
mm_simtech_sim7672_parse_cgdcont_read_response (const gchar  *reply,
                                                GError      **error)
{
    GError                *inner_error = NULL;
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    GList                 *list = NULL;

    if (!reply || !reply[0])
        /* No APNs configured, all done */
        return NULL;

    r = g_regex_new ("\\+CGDCONT:\\s*(\\d+)\\s*,([^, \\)]*)\\s*,\"([^, \\)]*)\"\\s*",
                     G_REGEX_DOLLAR_ENDONLY | G_REGEX_RAW,
                     0, NULL);
    g_assert (r);

    g_regex_match_full (r, reply, strlen (reply), 0, 0, &match_info, &inner_error);
    while (!inner_error && g_match_info_matches (match_info)) {
        gchar            *str;
        MMBearerIpFamily  ip_family;

        str = mm_get_string_unquoted_from_match_info (match_info, 2);
        ip_family = mm_3gpp_get_ip_family_from_pdp_type (str);
        if (ip_family != MM_BEARER_IP_FAMILY_NONE) {
            MM3gppPdpContext *pdp;
            guint             cid;

            pdp = g_slice_new0 (MM3gppPdpContext);
            if (!mm_get_uint_from_match_info (match_info, 1, &cid)) {
                inner_error = g_error_new (MM_CORE_ERROR,
                                           MM_CORE_ERROR_FAILED,
                                           "Couldn't parse CID from reply: '%s'",
                                           reply);
                break;
            }

            /* We need only cid=1 */
            if (cid == 1) {
                pdp->cid = cid;
                pdp->pdp_type = ip_family;
                pdp->apn = mm_get_string_unquoted_from_match_info (match_info, 3);

                list = g_list_prepend (list, pdp);
                break;
            }
        }

        g_free (str);
        g_match_info_next (match_info, &inner_error);
    }

    if (inner_error) {
        mm_3gpp_pdp_context_list_free (list);
        g_propagate_error (error, inner_error);
        g_prefix_error (error, "Couldn't properly parse list of PDP contexts. ");
        return NULL;
    }

    return g_list_sort (list, (GCompareFunc)pdp_context_cmp);
}
