/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2024 Atmark Techno, inc.
 */

#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>

#include "ModemManager.h"
#include "mm-modem-helpers.h"
#include "mm-log-object.h"
#include "mm-base-modem-at.h"
#include "mm-iface-modem.h"
#include "mm-iface-modem-3gpp.h"
#include "mm-iface-modem-3gpp-profile-manager.h"
#include "mm-iface-modem-location.h"
#include "mm-iface-modem-time.h"
#include "mm-shared-simtech.h"
#include "mm-sim-simtech-sim7672.h"
#include "mm-broadband-modem-simtech-sim7672.h"
#include "mm-broadband-bearer-simtech-sim7672.h"
#include "mm-modem-helpers-simtech-sim7672.h"

static void iface_modem_init                      (MMIfaceModem           *iface);
static void iface_modem_3gpp_init                 (MMIfaceModem3gpp       *iface);
static void iface_modem_3gpp_profile_manager_init (MMIfaceModem3gppProfileManager *iface);
static void iface_modem_location_init             (MMIfaceModemLocation   *iface);
static void iface_modem_time_init                 (MMIfaceModemTime       *iface);

static MMIfaceModem                   *iface_modem_parent;
static MMIfaceModem3gpp               *iface_modem_3gpp_parent;
static MMIfaceModem3gppProfileManager *iface_modem_3gpp_profile_manager_parent;
static MMIfaceModemLocation           *iface_modem_location_parent;

G_DEFINE_TYPE_EXTENDED (MMBroadbandModemSimtechSim7672, mm_broadband_modem_simtech_sim7672, MM_TYPE_BROADBAND_MODEM, 0,
        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM, iface_modem_init)
        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_3GPP, iface_modem_3gpp_init)
        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_3GPP_PROFILE_MANAGER, iface_modem_3gpp_profile_manager_init)
        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_LOCATION, iface_modem_location_init)
        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_TIME, iface_modem_time_init))

typedef enum {
    FEATURE_SUPPORT_UNKNOWN,
    FEATURE_NOT_SUPPORTED,
    FEATURE_SUPPORTED
} FeatureSupport;

struct _MMBroadbandModemSimtechSim7672Private {
    FeatureSupport  cnsmod_support;
    GRegex         *cnsmod_regex;
    GRegex         *csq_regex;
};

/*****************************************************************************/
/* Setup/Cleanup unsolicited events (3GPP interface) */

static MMModemAccessTechnology
simtech_act_to_mm_act (guint nsmod)
{
    static const MMModemAccessTechnology simtech_act_to_mm_act_map[] = {
        [0] = MM_MODEM_ACCESS_TECHNOLOGY_UNKNOWN,
        [1] = MM_MODEM_ACCESS_TECHNOLOGY_GSM,
        [2] = MM_MODEM_ACCESS_TECHNOLOGY_GPRS,
        [3] = MM_MODEM_ACCESS_TECHNOLOGY_EDGE,
        [4] = MM_MODEM_ACCESS_TECHNOLOGY_UMTS,
        [5] = MM_MODEM_ACCESS_TECHNOLOGY_HSDPA,
        [6] = MM_MODEM_ACCESS_TECHNOLOGY_HSUPA,
        [7] = MM_MODEM_ACCESS_TECHNOLOGY_HSPA,
        [8] = MM_MODEM_ACCESS_TECHNOLOGY_LTE,
    };

    return (nsmod < G_N_ELEMENTS (simtech_act_to_mm_act_map) ? simtech_act_to_mm_act_map[nsmod] : MM_MODEM_ACCESS_TECHNOLOGY_UNKNOWN);
}

static void
simtech_tech_changed (MMPortSerialAt *port,
                      GMatchInfo *match_info,
                      MMBroadbandModemSimtechSim7672 *self)
{
    guint simtech_act = 0;

    if (!mm_get_uint_from_match_info (match_info, 1, &simtech_act))
        return;

    mm_iface_modem_update_access_technologies (
        MM_IFACE_MODEM (self),
        simtech_act_to_mm_act (simtech_act),
        MM_IFACE_MODEM_3GPP_ALL_ACCESS_TECHNOLOGIES_MASK);
}

static void
simtech_signal_changed (MMPortSerialAt *port,
                        GMatchInfo *match_info,
                        MMBroadbandModemSimtechSim7672 *self)
{
    guint quality = 0;

    if (!mm_get_uint_from_match_info (match_info, 1, &quality))
        return;

    if (quality != 99)
        quality = MM_CLAMP_HIGH (quality, 31) * 100 / 31;
    else
        quality = 0;

    mm_iface_modem_update_signal_quality (MM_IFACE_MODEM (self), quality);
}

static void
set_unsolicited_events_handlers (MMBroadbandModemSimtechSim7672 *self,
                                 gboolean enable)
{
    MMPortSerialAt *ports[2];
    guint i;

    ports[0] = mm_base_modem_peek_port_primary (MM_BASE_MODEM (self));
    ports[1] = mm_base_modem_peek_port_secondary (MM_BASE_MODEM (self));

    /* Enable unsolicited events in given port */
    for (i = 0; i < G_N_ELEMENTS (ports); i++) {
        if (!ports[i])
            continue;

        /* Access technology related */
        mm_port_serial_at_add_unsolicited_msg_handler (
            ports[i],
            self->priv->cnsmod_regex,
            enable ? (MMPortSerialAtUnsolicitedMsgFn)simtech_tech_changed : NULL,
            enable ? self : NULL,
            NULL);

        /* Signal quality related */
        mm_port_serial_at_add_unsolicited_msg_handler (
            ports[i],
            self->priv->csq_regex,
            enable ? (MMPortSerialAtUnsolicitedMsgFn)simtech_signal_changed : NULL,
            enable ? self : NULL,
            NULL);
    }
}

static gboolean
modem_3gpp_setup_cleanup_unsolicited_events_finish (MMIfaceModem3gpp  *self,
                                                    GAsyncResult      *res,
                                                    GError           **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
parent_setup_unsolicited_events_ready (MMIfaceModem3gpp *self,
                                       GAsyncResult     *res,
                                       GTask            *task)
{
    GError *error = NULL;

    if (!iface_modem_3gpp_parent->setup_unsolicited_events_finish (self, res, &error))
        g_task_return_error (task, error);
    else {
        /* Our own setup now */
        set_unsolicited_events_handlers (MM_BROADBAND_MODEM_SIMTECH_SIM7672 (self), TRUE);
        g_task_return_boolean (task, TRUE);
    }
    g_object_unref (task);
}

static void
modem_3gpp_setup_unsolicited_events (MMIfaceModem3gpp *self,
                                     GAsyncReadyCallback callback,
                                     gpointer user_data)
{
    /* Chain up parent's setup */
    iface_modem_3gpp_parent->setup_unsolicited_events (
        self,
        (GAsyncReadyCallback)parent_setup_unsolicited_events_ready,
        g_task_new (self, NULL, callback, user_data));
}

static void
parent_cleanup_unsolicited_events_ready (MMIfaceModem3gpp *self,
                                         GAsyncResult     *res,
                                         GTask            *task)
{
    GError *error = NULL;

    if (!iface_modem_3gpp_parent->cleanup_unsolicited_events_finish (self, res, &error))
        g_task_return_error (task, error);
    else
        g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_cleanup_unsolicited_events (MMIfaceModem3gpp    *self,
                                       GAsyncReadyCallback  callback,
                                       gpointer             user_data)
{
    /* Our own cleanup first */
    set_unsolicited_events_handlers (MM_BROADBAND_MODEM_SIMTECH_SIM7672 (self), FALSE);

    /* And now chain up parent's cleanup */
    iface_modem_3gpp_parent->cleanup_unsolicited_events (
        self,
        (GAsyncReadyCallback)parent_cleanup_unsolicited_events_ready,
        g_task_new (self, NULL, callback, user_data));
}

/*****************************************************************************/
/* Enable unsolicited events (3GPP interface) */

typedef enum {
    ENABLE_UNSOLICITED_EVENTS_STEP_FIRST,
    ENABLE_UNSOLICITED_EVENTS_STEP_PARENT,
    ENABLE_UNSOLICITED_EVENTS_STEP_CHECK_SUPPORT_CNSMOD,
    ENABLE_UNSOLICITED_EVENTS_STEP_ENABLE_CNSMOD,
    ENABLE_UNSOLICITED_EVENTS_STEP_LAST,
} EnableUnsolicitedEventsStep;

typedef struct {
    EnableUnsolicitedEventsStep step;
} EnableUnsolicitedEventsContext;

static gboolean
modem_3gpp_enable_unsolicited_events_finish (MMIfaceModem3gpp  *self,
                                             GAsyncResult      *res,
                                             GError           **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void enable_unsolicited_events_context_step (GTask *task);

static void
cnsmod_set_enabled_ready (MMBaseModem  *self,
                          GAsyncResult *res,
                          GTask        *task)
{
    EnableUnsolicitedEventsContext *ctx;
    GError                         *error = NULL;
    gboolean                        cnsmod_urcs_enabled = FALSE;

    ctx = g_task_get_task_data (task);

    if (!mm_base_modem_at_command_finish (self, res, &error)) {
        mm_obj_dbg (self, "couldn't enable automatic access technology reporting: %s", error->message);
        g_error_free (error);
    } else
        cnsmod_urcs_enabled = TRUE;

    /* Disable access technology polling if we can use the +CNSMOD URCs */
    g_object_set (self,
                  MM_IFACE_MODEM_PERIODIC_ACCESS_TECH_CHECK_DISABLED, cnsmod_urcs_enabled,
                  NULL);

    /* go to next step */
    ctx->step++;
    enable_unsolicited_events_context_step (task);
}

static void
cnsmod_test_ready (MMBaseModem  *_self,
                   GAsyncResult *res,
                   GTask        *task)
{
    MMBroadbandModemSimtechSim7672        *self;
    EnableUnsolicitedEventsContext *ctx;

    self = MM_BROADBAND_MODEM_SIMTECH_SIM7672 (_self);
    ctx  = g_task_get_task_data (task);

    if (!mm_base_modem_at_command_finish (_self, res, NULL))
        self->priv->cnsmod_support = FEATURE_NOT_SUPPORTED;
    else
        self->priv->cnsmod_support = FEATURE_SUPPORTED;

    /* go to next step */
    ctx->step++;
    enable_unsolicited_events_context_step (task);
}

static void
parent_enable_unsolicited_events_ready (MMIfaceModem3gpp *self,
                                        GAsyncResult     *res,
                                        GTask            *task)
{
    EnableUnsolicitedEventsContext *ctx;
    GError                         *error = NULL;

    ctx = g_task_get_task_data (task);

    if (!iface_modem_3gpp_parent->enable_unsolicited_events_finish (self, res, &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* go to next step */
    ctx->step++;
    enable_unsolicited_events_context_step (task);
}

static void
enable_unsolicited_events_context_step (GTask *task)
{
    MMBroadbandModemSimtechSim7672        *self;
    EnableUnsolicitedEventsContext *ctx;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    switch (ctx->step) {
    case ENABLE_UNSOLICITED_EVENTS_STEP_FIRST:
        ctx->step++;
        /* fall through */

    case ENABLE_UNSOLICITED_EVENTS_STEP_PARENT:
        iface_modem_3gpp_parent->enable_unsolicited_events (
            MM_IFACE_MODEM_3GPP (self),
            (GAsyncReadyCallback)parent_enable_unsolicited_events_ready,
            task);
        return;

    case ENABLE_UNSOLICITED_EVENTS_STEP_CHECK_SUPPORT_CNSMOD:
        if (self->priv->cnsmod_support == FEATURE_SUPPORT_UNKNOWN) {
            mm_base_modem_at_command (MM_BASE_MODEM (self),
                                      "+CNSMOD=?",
                                      3,
                                      TRUE,
                                      (GAsyncReadyCallback)cnsmod_test_ready,
                                      task);
            return;
        }
        ctx->step++;
        /* fall through */

    case ENABLE_UNSOLICITED_EVENTS_STEP_ENABLE_CNSMOD:
        if (self->priv->cnsmod_support == FEATURE_SUPPORTED) {
            mm_base_modem_at_command (MM_BASE_MODEM (self),
                                      /* Autoreport +CNSMOD when it changes */
                                      "+CNSMOD=1",
                                      20,
                                      FALSE,
                                      (GAsyncReadyCallback)cnsmod_set_enabled_ready,
                                      task);
            return;
        }
        ctx->step++;
        /* fall through */

    case ENABLE_UNSOLICITED_EVENTS_STEP_LAST:
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
        return;

    default:
        g_assert_not_reached ();
    }
}

static void
modem_3gpp_enable_unsolicited_events (MMIfaceModem3gpp    *self,
                                      GAsyncReadyCallback  callback,
                                      gpointer             user_data)
{
    EnableUnsolicitedEventsContext *ctx;
    GTask                          *task;

    task = g_task_new (self, NULL, callback, user_data);

    ctx = g_new (EnableUnsolicitedEventsContext, 1);
    ctx->step = ENABLE_UNSOLICITED_EVENTS_STEP_FIRST;
    g_task_set_task_data (task, ctx, g_free);

    enable_unsolicited_events_context_step (task);
}

/*****************************************************************************/
/* Disable unsolicited events (3GPP interface) */

typedef enum {
    DISABLE_UNSOLICITED_EVENTS_STEP_FIRST,
    DISABLE_UNSOLICITED_EVENTS_STEP_DISABLE_CNSMOD,
    DISABLE_UNSOLICITED_EVENTS_STEP_PARENT,
    DISABLE_UNSOLICITED_EVENTS_STEP_LAST,
} DisableUnsolicitedEventsStep;

typedef struct {
    DisableUnsolicitedEventsStep step;
} DisableUnsolicitedEventsContext;

static gboolean
modem_3gpp_disable_unsolicited_events_finish (MMIfaceModem3gpp  *self,
                                              GAsyncResult      *res,
                                              GError           **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void disable_unsolicited_events_context_step (GTask *task);

static void
parent_disable_unsolicited_events_ready (MMIfaceModem3gpp *self,
                                         GAsyncResult     *res,
                                         GTask            *task)
{
    DisableUnsolicitedEventsContext *ctx;
    GError                         *error = NULL;

    ctx = g_task_get_task_data (task);

    if (!iface_modem_3gpp_parent->disable_unsolicited_events_finish (self, res, &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* go to next step */
    ctx->step++;
    disable_unsolicited_events_context_step (task);
}

static void
cnsmod_set_disabled_ready (MMBaseModem  *self,
                           GAsyncResult *res,
                           GTask        *task)
{
    DisableUnsolicitedEventsContext *ctx;
    GError                          *error = NULL;

    ctx = g_task_get_task_data (task);

    if (!mm_base_modem_at_command_finish (self, res, &error)) {
        mm_obj_dbg (self, "couldn't disable automatic access technology reporting: %s", error->message);
        g_error_free (error);
    }

    /* go to next step */
    ctx->step++;
    disable_unsolicited_events_context_step (task);
}

static void
disable_unsolicited_events_context_step (GTask *task)
{
    MMBroadbandModemSimtechSim7672         *self;
    DisableUnsolicitedEventsContext *ctx;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    switch (ctx->step) {
    case DISABLE_UNSOLICITED_EVENTS_STEP_FIRST:
        ctx->step++;
        /* fall through */

    case DISABLE_UNSOLICITED_EVENTS_STEP_DISABLE_CNSMOD:
        if (self->priv->cnsmod_support == FEATURE_SUPPORTED) {
            mm_base_modem_at_command (MM_BASE_MODEM (self),
                                      "+CNSMOD=0",
                                      20,
                                      FALSE,
                                      (GAsyncReadyCallback)cnsmod_set_disabled_ready,
                                      task);
            return;
        }
        ctx->step++;
        /* fall through */

    case DISABLE_UNSOLICITED_EVENTS_STEP_PARENT:
        iface_modem_3gpp_parent->disable_unsolicited_events (
            MM_IFACE_MODEM_3GPP (self),
            (GAsyncReadyCallback)parent_disable_unsolicited_events_ready,
            task);
        return;

    case DISABLE_UNSOLICITED_EVENTS_STEP_LAST:
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
        return;

    default:
        g_assert_not_reached ();
    }
}

static void
modem_3gpp_disable_unsolicited_events (MMIfaceModem3gpp    *self,
                                      GAsyncReadyCallback  callback,
                                      gpointer             user_data)
{
    DisableUnsolicitedEventsContext *ctx;
    GTask                          *task;

    task = g_task_new (self, NULL, callback, user_data);

    ctx = g_new (DisableUnsolicitedEventsContext, 1);
    ctx->step = DISABLE_UNSOLICITED_EVENTS_STEP_FIRST;
    g_task_set_task_data (task, ctx, g_free);

    disable_unsolicited_events_context_step (task);
}

/*****************************************************************************/
/* Registration checks (3GPP interface) */

static gboolean
modem_3gpp_run_registration_checks_finish (MMIfaceModem3gpp  *self,
                                           GAsyncResult      *res,
                                           GError           **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
modem_3gpp_run_registration_checks (MMIfaceModem3gpp    *self,
                                    gboolean             is_cs_supported,
                                    gboolean             is_ps_supported,
                                    gboolean             is_eps_supported,
                                    gboolean             is_5gs_supported,
                                    GAsyncReadyCallback  callback,
                                    gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);

    mm_iface_modem_3gpp_update_eps_registration_state (MM_IFACE_MODEM_3GPP (self),
                                                       MM_MODEM_3GPP_REGISTRATION_STATE_HOME,
                                                       TRUE);
    mm_iface_modem_3gpp_update_access_technologies (MM_IFACE_MODEM_3GPP (self),
                                                    MM_MODEM_ACCESS_TECHNOLOGY_LTE);
    mm_iface_modem_3gpp_update_location (MM_IFACE_MODEM_3GPP (self), 0xffff, 0xfffffffe, 0);
    mm_iface_modem_update_state (MM_IFACE_MODEM (self),
                                 MM_MODEM_STATE_REGISTERED,
                                 MM_MODEM_STATE_CHANGE_REASON_UNKNOWN);

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

/*****************************************************************************/
/* Register in network (3GPP interface) */

static gboolean
modem_3gpp_register_in_network_finish (MMIfaceModem3gpp  *self,
                                       GAsyncResult      *res,
                                       GError           **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
set_operator_ready (MMIfaceModem3gpp *self,
                    GAsyncResult     *res,
                    GTask            *task)
{
    GError *error = NULL;

    if (!mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error)) {
        if (error)
            g_error_free (error);
    }

    /* set fake state */
    mm_iface_modem_3gpp_update_eps_registration_state (self,
                                                       MM_MODEM_3GPP_REGISTRATION_STATE_HOME,
                                                       FALSE);
    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_register_in_network (MMIfaceModem3gpp    *self,
                                const gchar         *operator_id,
                                GCancellable        *cancellable,
                                GAsyncReadyCallback  callback,
                                gpointer             user_data)
{
    GTask            *task;
    g_autofree gchar *cmd = NULL;

    task = g_task_new (self, cancellable, callback, user_data);

    if (operator_id) {
        cmd = g_strdup_printf ("+COPS=1,2,%s", operator_id);
        mm_base_modem_at_command (
            MM_BASE_MODEM (self),
            cmd,
            20,
            FALSE,
            (GAsyncReadyCallback)set_operator_ready,
            task);
        return;
    }

    /* set fake state */
    mm_iface_modem_3gpp_update_eps_registration_state (self,
                                                       MM_MODEM_3GPP_REGISTRATION_STATE_HOME,
                                                       FALSE);

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

/*****************************************************************************/
/* Set packet service state (3GPP interface) */

static gboolean
modem_3gpp_set_packet_service_state_finish (MMIfaceModem3gpp  *self,
                                            GAsyncResult      *res,
                                            GError          **error)
{
    return !!mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, error);
}

static void
modem_3gpp_set_packet_service_state (MMIfaceModem3gpp              *self,
                                     MMModem3gppPacketServiceState  state,
                                     GAsyncReadyCallback            callback,
                                     gpointer                       user_data)
{
    g_autofree gchar *cmd = NULL;

    cmd = mm_3gpp_build_cgatt_set_request (state);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              cmd,
                              30,
                              FALSE,
                              callback,
                              user_data);
}

/*****************************************************************************/
/* Check format (3GPP profile management interface) */

typedef struct {
    MMBearerIpFamily ip_type;
    guint            min_profile_id;
    guint            max_profile_id;
} CheckFormatContext;

static void
check_format_context_free (CheckFormatContext *ctx)
{
    g_slice_free (CheckFormatContext, ctx);
}

static gboolean
modem_3gpp_profile_manager_check_format_finish (MMIfaceModem3gppProfileManager  *self,
                                                GAsyncResult                    *res,
                                                gboolean                        *new_id,
                                                gint                            *min_profile_id,
                                                gint                            *max_profile_id,
                                                GEqualFunc                      *apn_cmp,
                                                MM3gppProfileCmpFlags           *profile_cmp_flags,
                                                GError                         **error)
{
    CheckFormatContext *ctx;

    if (!g_task_propagate_boolean (G_TASK (res), error))
        return FALSE;

    ctx = g_task_get_task_data (G_TASK (res));
    if (new_id)
        *new_id = TRUE;
    if (min_profile_id)
        *min_profile_id = (gint) ctx->min_profile_id;
    if (max_profile_id)
        *max_profile_id = (gint) ctx->max_profile_id;
    if (apn_cmp)
        *apn_cmp = (GEqualFunc) mm_3gpp_cmp_apn_name;
    if (profile_cmp_flags)
        *profile_cmp_flags = MM_3GPP_PROFILE_CMP_FLAGS_NO_AUTH;

    return TRUE;
}

static void
modem_3gpp_profile_manager_check_format (MMIfaceModem3gppProfileManager *self,
                                         MMBearerIpFamily                ip_type,
                                         GAsyncReadyCallback             callback,
                                         gpointer                        user_data)
{
    GTask              *task;
    CheckFormatContext *ctx;

    task = g_task_new (self, NULL, callback, user_data);
    ctx = g_slice_new0 (CheckFormatContext);
    ctx->ip_type = ip_type;
    g_task_set_task_data (task, ctx, (GDestroyNotify)check_format_context_free);

    /* set range only 1, sim7672 must use cid 1 */
    ctx->min_profile_id = 1;
    ctx->max_profile_id = 1;

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

/*****************************************************************************/
/* List profiles (3GPP profile management interface) */

typedef struct {
    GList *profiles;
} ListProfilesContext;

static void
list_profiles_context_free (ListProfilesContext *ctx)
{
    mm_3gpp_profile_list_free (ctx->profiles);
    g_slice_free (ListProfilesContext, ctx);
}

static gboolean
modem_3gpp_profile_manager_list_profiles_finish (MMIfaceModem3gppProfileManager  *self,
                                                 GAsyncResult                    *res,
                                                 GList                          **out_profiles,
                                                 GError                         **error)
{
    ListProfilesContext *ctx;

    if (!g_task_propagate_boolean (G_TASK (res), error))
        return FALSE;

    ctx = g_task_get_task_data (G_TASK (res));
    if (out_profiles)
        *out_profiles = g_steal_pointer (&ctx->profiles);
    return TRUE;
}

static void
profile_manager_cgdcont_query_ready (MMBaseModem  *self,
                                     GAsyncResult *res,
                                     GTask        *task)
{
    ListProfilesContext *ctx;
    const gchar         *response;
    GError              *error = NULL;
    GList               *pdp_context_list;

    response = mm_base_modem_at_command_finish (self, res, &error);
    if (!response) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* may return NULL without error if response is empty */
    pdp_context_list = mm_simtech_sim7672_parse_cgdcont_read_response (response, &error);
    if (error) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    ctx = g_slice_new0 (ListProfilesContext);
    g_task_set_task_data (task, ctx, (GDestroyNotify) list_profiles_context_free);
    ctx->profiles = mm_3gpp_profile_list_new_from_pdp_context_list (pdp_context_list);
    mm_3gpp_pdp_context_list_free (pdp_context_list);

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_list_profiles (MMIfaceModem3gppProfileManager  *self,
                                          GAsyncReadyCallback              callback,
                                          gpointer                         user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);

    /* Query with CGDCONT? */
    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "+CGDCONT?",
        3,
        FALSE,
        (GAsyncReadyCallback)profile_manager_cgdcont_query_ready,
        task);
}

/*****************************************************************************/
/* Store profile (3GPP profile management interface) */

static gboolean
modem_3gpp_profile_manager_store_profile_finish (MMIfaceModem3gppProfileManager  *self,
                                                 GAsyncResult                    *res,
                                                 gint                            *out_profile_id,
                                                 MMBearerApnType                 *out_apn_type,
                                                 GError                         **error)
{
    if (!g_task_propagate_boolean (G_TASK (res), error))
        return FALSE;

    if (out_profile_id)
        *out_profile_id = 1;
    if (out_apn_type)
        *out_apn_type = MM_BEARER_APN_TYPE_NONE;
    return TRUE;
}

static void
store_profile_activate_context_ready (MMBaseModem  *self,
                                      GAsyncResult *res,
                                      GTask        *task)
{
    g_autoptr(GError) error = NULL;

    mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
store_profile_cgdcont_set_ready (MMBaseModem  *self,
                                 GAsyncResult *res,
                                 GTask        *task)
{
    g_autoptr(GError) error = NULL;

    mm_base_modem_at_command_finish (self, res, &error);

    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+CGACT=1,1",
                              60,
                              FALSE,
                              (GAsyncReadyCallback) store_profile_activate_context_ready,
                              task);
}

static void
store_profile_deactivate_context_ready (MMBaseModem  *self,
                                        GAsyncResult *res,
                                        GTask        *task)
{
    g_autoptr(GError)  error = NULL;
    MM3gppProfile     *profile;
    gint               profile_id;
    const gchar       *pdp_type;
    g_autofree gchar  *quoted_apn = NULL;
    g_autofree gchar  *cmd = NULL;

    mm_base_modem_at_command_finish (self, res, &error);

    profile = (MM3gppProfile *) g_task_get_task_data (task);

    profile_id = mm_3gpp_profile_get_profile_id (profile);
    pdp_type = mm_3gpp_get_pdp_type_from_ip_family (mm_3gpp_profile_get_ip_type (profile));
    quoted_apn = mm_port_serial_at_quote_string (mm_3gpp_profile_get_apn (profile));

    cmd = g_strdup_printf ("+CGDCONT=%d,\"%s\",%s", profile_id, pdp_type, quoted_apn);

    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              cmd,
                              5,
                              FALSE,
                              (GAsyncReadyCallback) store_profile_cgdcont_set_ready,
                              task);
}

static void
store_profile_cgdcont_query_ready (MMBaseModem  *self,
                                   GAsyncResult *res,
                                   GTask        *task)
{
    const gchar       *response;
    g_autoptr(GError)  error = NULL;
    GList             *pdp_list = NULL;
    GList             *l;

    response = mm_base_modem_at_command_full_finish (self, res, &error);
    if (!response)
        goto deactivate_context;

    pdp_list = mm_simtech_sim7672_parse_cgdcont_read_response (response, &error);
    if (!pdp_list)
        goto deactivate_context;

    /* Look for the exact PDP context we want */
    for (l = pdp_list; l; l = g_list_next (l)) {
        MM3gppPdpContext *pdp = l->data;
        if (pdp->cid != 1)
            continue;
        else {
            MM3gppProfile *profile;
            const gchar *apn;

            profile = (MM3gppProfile *) g_task_get_task_data (task);

            apn = mm_3gpp_profile_get_apn (profile);
            if (g_ascii_strcasecmp (pdp->apn, apn))
                goto deactivate_context;

            if (pdp->pdp_type != mm_3gpp_profile_get_ip_type (profile))
                goto deactivate_context;

            /* no need to update */
            mm_3gpp_pdp_context_list_free (pdp_list);
            g_task_return_boolean (task, TRUE);
            g_object_unref (task);
            return;
        }
    }

deactivate_context:
    if (pdp_list)
        mm_3gpp_pdp_context_list_free (pdp_list);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+CGACT=0,1",
                              5,
                              FALSE,
                              (GAsyncReadyCallback) store_profile_deactivate_context_ready,
                              task);
}

static void
modem_3gpp_profile_manager_store_profile (MMIfaceModem3gppProfileManager *self,
                                          MM3gppProfile                  *profile,
                                          const gchar                    *index_field,
                                          GAsyncReadyCallback             callback,
                                          gpointer                        user_data)
{
    GTask             *task;
    gint               profile_id;
    MMBearerIpFamily   ip_type;
    const gchar       *apn;
    g_autofree gchar  *ip_type_str = NULL;

    g_assert (g_strcmp0 (index_field, "profile-id") == 0);

    task = g_task_new (self, NULL, callback, user_data);

    profile_id = mm_3gpp_profile_get_profile_id (profile);
    if (profile_id != 1) {
        profile_id = 1;
        mm_3gpp_profile_set_profile_id (profile, profile_id);
    }

    ip_type = mm_3gpp_profile_get_ip_type (profile);
    if (ip_type == MM_BEARER_IP_FAMILY_NONE ||
        ip_type == MM_BEARER_IP_FAMILY_ANY) {
        ip_type = MM_BEARER_IP_FAMILY_IPV4;
        mm_3gpp_profile_set_ip_type (profile, ip_type);
    }
    ip_type_str = mm_bearer_ip_family_build_string_from_mask (ip_type);

    apn = mm_3gpp_profile_get_apn (profile);

    mm_obj_dbg (self, "storing profile '%d': apn '%s', ip type '%s'",
                profile_id, apn, ip_type_str);

    g_task_set_task_data (task, profile, NULL);

    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+CGDCONT?",
                              5,
                              FALSE,
                              (GAsyncReadyCallback) store_profile_cgdcont_query_ready,
                              task);
}

/*****************************************************************************/
/* Load access technologies (Modem interface) */

static gboolean
load_access_technologies_finish (MMIfaceModem             *self,
                                 GAsyncResult             *res,
                                 MMModemAccessTechnology  *access_technologies,
                                 guint                    *mask,
                                 GError                  **error)
{
    GError *inner_error = NULL;
    gssize  act;

    act = g_task_propagate_int (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return FALSE;
    }

    *access_technologies = (MMModemAccessTechnology) act;
    *mask = MM_MODEM_ACCESS_TECHNOLOGY_ANY;
    return TRUE;
}

static void
cnsmod_query_ready (MMBaseModem  *self,
                    GAsyncResult *res,
                    GTask        *task)
{
    const gchar *response, *p;
    GError      *error = NULL;

    response = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (!response) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    p = mm_strip_tag (response, "+CNSMOD:");
    if (p)
        p = strchr (p, ',');

    if (!p || !isdigit (*(p + 1)))
        g_task_return_new_error (
            task,
            MM_CORE_ERROR,
            MM_CORE_ERROR_FAILED,
            "Failed to parse the +CNSMOD response: '%s'",
            response);
    else
        g_task_return_int (task, simtech_act_to_mm_act (atoi (p + 1)));
    g_object_unref (task);
}

static void
load_access_technologies (MMIfaceModem        *_self,
                          GAsyncReadyCallback  callback,
                          gpointer             user_data)
{
    MMBroadbandModemSimtechSim7672 *self;
    GTask                          *task;

    self = MM_BROADBAND_MODEM_SIMTECH_SIM7672 (_self);
    task = g_task_new (self, NULL, callback, user_data);

    /* Launch query only for 3GPP modems */
    if (!mm_iface_modem_is_3gpp (_self)) {
        g_task_return_int (task, MM_MODEM_ACCESS_TECHNOLOGY_UNKNOWN);
        g_object_unref (task);
        return;
    }

    g_assert (self->priv->cnsmod_support != FEATURE_SUPPORT_UNKNOWN);
    if (self->priv->cnsmod_support == FEATURE_NOT_SUPPORTED) {
        g_task_return_new_error (task, MM_CORE_ERROR, MM_CORE_ERROR_UNSUPPORTED,
                                 "Loading access technologies with +CNSMOD is not supported");
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "AT+CNSMOD?",
        5,
        FALSE,
        (GAsyncReadyCallback)cnsmod_query_ready,
        task);
}

/*****************************************************************************/
/* Load signal quality (Modem interface) */

static guint
load_signal_quality_finish (MMIfaceModem  *self,
                            GAsyncResult  *res,
                            GError       **error)
{
    gssize value;

    value = g_task_propagate_int (G_TASK (res), error);
    return value < 0 ? 0 : value;
}

static void
csq_query_ready (MMBaseModem  *self,
                 GAsyncResult *res,
                 GTask        *task)
{
    const gchar *response, *p;
    GError      *error = NULL;
    gint         quality;
    gint         ber;

    response = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (!response) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    if (!response[0]) {
        g_task_return_new_error (task, MM_CORE_ERROR, MM_CORE_ERROR_IN_PROGRESS,
                                 "already refreshed via URCs");
        g_object_unref (task);
        return;
    }

    p = mm_strip_tag (response, "+CSQ:");
    if (sscanf (p, "%d, %d", &quality, &ber)) {
        if (quality != 99)
            quality = CLAMP (quality, 0, 31) * 100 / 31;
        else
            quality = 0;
        g_task_return_int (task, quality);
        g_object_unref (task);
        return;
    }

    g_task_return_new_error (task,
                             MM_CORE_ERROR,
                             MM_CORE_ERROR_FAILED,
                             "Could not parse signal quality results");
    g_object_unref (task);
}

static void
load_signal_quality (MMIfaceModem        *self,
                     GAsyncReadyCallback  callback,
                     gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);

    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "+CSQ",
        5,
        FALSE,
        (GAsyncReadyCallback)csq_query_ready,
        task);
}

/*****************************************************************************/
/* Load supported modes (Modem interface) */

static GArray *
load_supported_modes_finish (MMIfaceModem  *self,
                             GAsyncResult  *res,
                             GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

static void
load_supported_modes (MMIfaceModem        *self,
                      GAsyncReadyCallback  callback,
                      gpointer             user_data)
{
    GTask                  *task;
    GArray                 *combinations;
    MMModemModeCombination  mode;

    task = g_task_new (self, NULL, callback, user_data);

    /* 4G only */
    combinations = g_array_sized_new (FALSE, FALSE, sizeof (MMModemModeCombination), 1);
    mode.allowed = MM_MODEM_MODE_4G;
    mode.preferred = MM_MODEM_MODE_NONE;

    g_array_append_val (combinations, mode);
    g_task_return_pointer (task, combinations, (GDestroyNotify) g_array_unref);
    g_object_unref (task);
}

/*****************************************************************************/
/* Load initial allowed/preferred modes (Modem interface) */

typedef struct {
    MMModemMode allowed;
    MMModemMode preferred;
} LoadCurrentModesResult;

static gboolean
load_current_modes_finish (MMIfaceModem  *self,
                           GAsyncResult  *res,
                           MMModemMode   *allowed,
                           MMModemMode   *preferred,
                           GError       **error)
{
    LoadCurrentModesResult *result;

    result = g_task_propagate_pointer (G_TASK (res), error);
    if (result) {
        *allowed   = result->allowed;
        *preferred = result->preferred;
        g_free (result);
    } else {
        /* 4G only */
        *allowed   = MM_MODEM_MODE_4G;
        *preferred = MM_MODEM_MODE_NONE;
    }
    return TRUE;
}

static void
load_current_modes (MMIfaceModem        *self,
                    GAsyncReadyCallback  callback,
                    gpointer             user_data)
{
    GTask                  *task;
    LoadCurrentModesResult *result;

    task = g_task_new (self, NULL, callback, user_data);

    result = g_new (LoadCurrentModesResult, 1);
    result->allowed   = MM_MODEM_MODE_4G;
    result->preferred = MM_MODEM_MODE_NONE;
    g_task_return_pointer (task, result, g_free);
    g_object_unref (task);
}

/*****************************************************************************/
/* Set allowed modes (Modem interface) */
static gboolean
set_current_modes_finish (MMIfaceModem  *self,
                          GAsyncResult  *res,
                          GError       **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
set_current_modes (MMIfaceModem        *self,
                   MMModemMode          allowed,
                   MMModemMode          preferred,
                   GAsyncReadyCallback  callback,
                   gpointer             user_data)
{
    GTask *task;

    /* cannot set, only 4G */
    task = g_task_new (self, NULL, callback, user_data);
    g_task_return_boolean (task, TRUE);
}

/*****************************************************************************/
/* Supported bands (Modem interface) */

static GArray *
load_supported_bands_finish (MMIfaceModem  *self,
                             GAsyncResult  *res,
                             GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

/* AT+CNBP=?
 * +CNBP: (1,2,3,4,5,7,8,12,13,18,19,20,25,26,28,38,40,41,66,71)
 */
static gboolean
cnbp_build_bands (const gchar  *response,
                  GArray      **bands,
                  GError      **error)
{
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    g_autoptr(GArray)      tmp_bands = NULL;
    GError                *inner_error = NULL;
    MMModemBand            band;
    guint                  i;

    r = g_regex_new ("\\+CNBP:\\s*\\((.*)\\)",
                     G_REGEX_DOLLAR_ENDONLY | G_REGEX_RAW,
                     0, NULL);
    g_assert (r != NULL);

    g_regex_match_full (r, response, strlen (response), 0, 0, &match_info, &inner_error);
    if (!inner_error && g_match_info_matches (match_info)) {
        g_autofree gchar *str = NULL;
        str = mm_get_string_unquoted_from_match_info (match_info, 1);
        tmp_bands = mm_parse_uint_list (str, &inner_error);
        if (inner_error)
            goto out;
    }

    for (i = 0; i < tmp_bands->len; i++) {
        band = MM_MODEM_BAND_EUTRAN_1 - 1 + g_array_index (tmp_bands, guint, i);
        if (band >= MM_MODEM_BAND_EUTRAN_1 && band <= MM_MODEM_BAND_EUTRAN_71) {
            if (!*bands)
                *bands = g_array_sized_new (FALSE, FALSE, sizeof (MMModemBand), 22);
            g_array_append_val (*bands, band);
        }
    }

out:
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return FALSE;
    }
    return TRUE;
}

static void
cnbp_test_ready (MMBaseModem  *self,
                 GAsyncResult *res,
                 GTask        *task)
{
    const gchar *response;
    GError      *error = NULL;
    GArray      *bands = NULL;

    response = mm_base_modem_at_command_finish (self, res, &error);
    if (!response ||
        !cnbp_build_bands (response, &bands, &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    g_task_return_pointer (task, bands, (GDestroyNotify)g_array_unref);
    g_object_unref (task);
}

static void
load_supported_bands (MMIfaceModem        *self,
                      GAsyncReadyCallback  callback,
                      gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "AT+CNBP=?",
                              5,
                              FALSE,
                              (GAsyncReadyCallback)cnbp_test_ready,
                              task);
}

/*****************************************************************************/
/* Load current bands (Modem interface) */

static GArray *
load_current_bands_finish (MMIfaceModem  *self,
                           GAsyncResult  *res,
                           GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

/*
 * +CPSI: LTE,Online,440-10,0x30B0,197210628,460,EUTRAN-BAND1,276,4,4,-14,-115,-101,0
 * +CPSI: <System Mode>,<Operation Mode>[,<MCC>-<MNC>,<TAC>,<SCellID>,<PCellID>,<Frequency Band>,
 *        <earfcn>,<dlbw>,<ulbw>,<RSRQ>,<RSRP>,<RSSI>,<RSSNR>]
 */
static gboolean
cpsi_response_to_band (const gchar  *response,
                       MMModemBand  *out_band,
                       GError      **error)
{
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    GError                *inner_error = NULL;
    guint                  band;
    gboolean               success = FALSE;

    r = g_regex_new ("\\+CPSI: LTE,[^,]+,[^,]+,[^,]+,[^,]+,[^,]+,EUTRAN-BAND(\\d+)", 0, 0, NULL);
    g_regex_match_full (r, response, strlen (response), 0, 0, &match_info, &inner_error);
    if (!inner_error && g_match_info_matches (match_info)) {
        if (!mm_get_uint_from_match_info (match_info, 1, &band)) {
            inner_error = g_error_new (MM_CORE_ERROR, MM_CORE_ERROR_FAILED, "Couldn't read Band");
            goto out;
        }
        *out_band = MM_MODEM_BAND_EUTRAN_1 + band - 1;
        success = TRUE;
    }

out:
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return FALSE;
    }

    if (!success) {
        g_set_error (error, MM_CORE_ERROR, MM_CORE_ERROR_FAILED,
                     "Couldn't parse +CPSI response: %s", response);
        return FALSE;
    }

    return TRUE;
}

static void
get_band_ready (MMBaseModem  *self,
                GAsyncResult *res,
                GTask        *task)
{
    const gchar *response;
    GError      *error = NULL;
    GArray      *bands = NULL;
    MMModemBand  band;

    response = mm_base_modem_at_command_finish (self, res, &error);
    if (!response ||
        !cpsi_response_to_band (response, &band, &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    bands = g_array_sized_new (FALSE, FALSE, sizeof (MMModemBand), 1);
    g_array_append_val (bands, band);
    g_task_return_pointer (task, bands, (GDestroyNotify) g_array_unref);
    g_object_unref (task);
}

static void
load_current_bands (MMIfaceModem        *self,
                    GAsyncReadyCallback  callback,
                    gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+CPSI?",
                              5,
                              FALSE,
                              (GAsyncReadyCallback)get_band_ready,
                              task);
}

/*****************************************************************************/
/* Set current bands (Modem interface) */

static gboolean
set_current_bands_finish (MMIfaceModem  *self,
                          GAsyncResult  *res,
                          GError       **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
set_band_ready (MMBaseModem  *self,
                GAsyncResult *res,
                GTask        *task)
{
    GError *error = NULL;

    mm_base_modem_at_command_finish (self, res, &error);
    if (error)
        g_task_return_error (task, error);
    else
        g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
set_current_bands (MMIfaceModem        *self,
                   GArray              *bands_array,
                   GAsyncReadyCallback  callback,
                   gpointer             user_data)
{
    GTask            *task;
    guint64           bands = 0;
    g_autofree gchar *command = NULL;
    guint             i;

    task = g_task_new (self, NULL, callback, user_data);

    for (i = 0; i < bands_array->len; i++) {
        bands |= (guint64)(1 << (g_array_index (bands_array, guint, i) - MM_MODEM_BAND_EUTRAN_1));
    }

    command = g_strdup_printf ("+CNBP=0X\"%lX\"", bands);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              command,
                              5,
                              FALSE,
                              (GAsyncReadyCallback)set_band_ready,
                              task);
}

/*****************************************************************************/
/* Supported modes loading (Modem interface) */

static MMUnlockRetries *
load_unlock_retries_finish (MMIfaceModem  *self,
                            GAsyncResult  *res,
                            GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

static void
spic_ready (MMBaseModem  *self,
            GAsyncResult *res,
            GTask        *task)
{
    const gchar     *response;
    GError          *error = NULL;
    gint             pin, puk, pin2, puk2;
    MMUnlockRetries *retries;

    response = mm_base_modem_at_command_finish (self, res, &error);
    if (!response) {
        mm_obj_dbg (self, "couldn't load retry count for lock. %s", error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    sscanf (response, "+SPIC: %d,%d,%d,%d", &pin, &puk, &pin2, &puk2);
    retries = mm_unlock_retries_new ();
    mm_unlock_retries_set (retries, MM_MODEM_LOCK_SIM_PIN, pin);
    mm_unlock_retries_set (retries, MM_MODEM_LOCK_SIM_PUK, puk);
    mm_unlock_retries_set (retries, MM_MODEM_LOCK_SIM_PIN2, pin2);
    mm_unlock_retries_set (retries, MM_MODEM_LOCK_SIM_PUK2, puk2);

    g_task_return_pointer (task, g_object_ref (retries), g_object_unref);
    g_object_unref (task);
}

static void
load_unlock_retries (MMIfaceModem        *self,
                     GAsyncReadyCallback  callback,
                     gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);
    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "+SPIC",
        5,
        FALSE,
        (GAsyncReadyCallback)spic_ready,
        task);
}

/*****************************************************************************/
/* Create Bearer (Modem interface) */

static MMBaseBearer *
create_bearer_finish (MMIfaceModem  *self,
                      GAsyncResult  *res,
                      GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

static void
broadband_bearer_simtech_sim7672_new_ready (GObject      *unused,
                                            GAsyncResult *res,
                                            GTask        *task)
{
    MMBaseBearer *bearer;
    GError       *error = NULL;

    bearer = mm_broadband_bearer_simtech_sim7672_new_finish (res, &error);
    if (!bearer)
        g_task_return_error (task, error);
    else
        g_task_return_pointer (task, bearer, g_object_unref);
    g_object_unref (task);
}

static void
create_bearer (MMIfaceModem        *_self,
               MMBearerProperties  *properties,
               GAsyncReadyCallback  callback,
               gpointer             user_data)
{
    MMBroadbandModemSimtechSim7672 *self = MM_BROADBAND_MODEM_SIMTECH_SIM7672 (_self);
    GTask                          *task;

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, g_object_ref (properties), g_object_unref);

    mm_broadband_bearer_simtech_sim7672_new (MM_BROADBAND_MODEM_SIMTECH_SIM7672 (self),
                                             g_task_get_task_data (task),
                                             NULL, /* cancellable */
                                             (GAsyncReadyCallback)broadband_bearer_simtech_sim7672_new_ready,
                                             task);
}

/*****************************************************************************/
/* Load current capabilities (Modem interface) */

static MMModemCapability
load_current_capabilities_finish (MMIfaceModem  *self,
                                  GAsyncResult  *res,
                                  GError       **error)
{
    GError *inner_error = NULL;
    gssize  value;

    value = g_task_propagate_int (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return MM_MODEM_CAPABILITY_NONE;
    }
    return (MMModemCapability)value;
}

static void
load_current_capabilities (MMIfaceModem        *self,
                           GAsyncReadyCallback  callback,
                           gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);
    /* This modem is LTE only.*/
    g_task_return_int (task, MM_MODEM_CAPABILITY_LTE);
    g_object_unref (task);
}

/*****************************************************************************/
/* Create SIM (Modem interface) */

static MMBaseSim *
create_sim_finish (MMIfaceModem  *self,
                   GAsyncResult  *res,
                   GError       **error)
{
    return mm_sim_simtech_sim7672_new_finish (res, error);
}

static void
create_sim (MMIfaceModem        *self,
            GAsyncReadyCallback  callback,
            gpointer             user_data)
{
    mm_sim_simtech_sim7672_new (MM_BASE_MODEM (self),
                                NULL, /* cancellable */
                                callback,
                                user_data);
}

/*****************************************************************************/
/* Setup ports (Broadband modem class) */

static void
setup_ports (MMBroadbandModem *self)
{
    /* Call parent's setup ports first always */
    MM_BROADBAND_MODEM_CLASS (mm_broadband_modem_simtech_sim7672_parent_class)->setup_ports (self);

    /* Now reset the unsolicited messages we'll handle when enabled */
    set_unsolicited_events_handlers (MM_BROADBAND_MODEM_SIMTECH_SIM7672 (self), FALSE);
}

/*****************************************************************************/

MMBroadbandModemSimtechSim7672 *
mm_broadband_modem_simtech_sim7672_new (const gchar *device,
                                const gchar *physdev,
                                const gchar **drivers,
                                const gchar *plugin,
                                guint16 vendor_id,
                                guint16 product_id)
{
    return g_object_new (MM_TYPE_BROADBAND_MODEM_SIMTECH_SIM7672,
                         MM_BASE_MODEM_DEVICE, device,
                         MM_BASE_MODEM_PHYSDEV, physdev,
                         MM_BASE_MODEM_DRIVERS, drivers,
                         MM_BASE_MODEM_PLUGIN, plugin,
                         MM_BASE_MODEM_VENDOR_ID, vendor_id,
                         MM_BASE_MODEM_PRODUCT_ID, product_id,
                         /* Generic bearer supports TTY only */
                         MM_BASE_MODEM_DATA_NET_SUPPORTED, FALSE,
                         MM_BASE_MODEM_DATA_TTY_SUPPORTED, TRUE,
                         MM_BROADBAND_MODEM_INDICATORS_DISABLED, TRUE,
                         NULL);
}

static void
mm_broadband_modem_simtech_sim7672_init (MMBroadbandModemSimtechSim7672 *self)
{
    /* Initialize private data */
    self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
                                              MM_TYPE_BROADBAND_MODEM_SIMTECH_SIM7672,
                                              MMBroadbandModemSimtechSim7672Private);

    self->priv->cnsmod_support = FEATURE_SUPPORT_UNKNOWN;

    self->priv->cnsmod_regex = g_regex_new ("\\r\\n\\+CNSMOD:\\s*(\\d+)\\r\\n",
                                            G_REGEX_RAW | G_REGEX_OPTIMIZE, 0, NULL);
    self->priv->csq_regex    = g_regex_new ("\\r\\n\\+CSQ:\\s*(\\d+),(\\d+)\\r\\n",
                                            G_REGEX_RAW | G_REGEX_OPTIMIZE, 0, NULL);
}

static void
finalize (GObject *object)
{
    MMBroadbandModemSimtechSim7672 *self = MM_BROADBAND_MODEM_SIMTECH_SIM7672 (object);

    g_regex_unref (self->priv->cnsmod_regex);
    g_regex_unref (self->priv->csq_regex);

    G_OBJECT_CLASS (mm_broadband_modem_simtech_sim7672_parent_class)->finalize (object);
}

static void
iface_modem_init (MMIfaceModem *iface)
{
    iface_modem_parent = g_type_interface_peek_parent (iface);


    iface->create_sim                       = create_sim;
    iface->create_sim_finish                = create_sim_finish;
    iface->create_bearer                    = create_bearer;
    iface->create_bearer_finish             = create_bearer_finish;
    iface->load_current_capabilities        = load_current_capabilities;
    iface->load_current_capabilities_finish = load_current_capabilities_finish;
    iface->load_signal_quality              = load_signal_quality;
    iface->load_signal_quality_finish       = load_signal_quality_finish;
    iface->load_access_technologies         = load_access_technologies;
    iface->load_access_technologies_finish  = load_access_technologies_finish;
    iface->load_supported_modes             = load_supported_modes;
    iface->load_supported_modes_finish      = load_supported_modes_finish;
    iface->load_current_modes               = load_current_modes;
    iface->load_current_modes_finish        = load_current_modes_finish;
    iface->set_current_modes                = set_current_modes;
    iface->set_current_modes_finish         = set_current_modes_finish;
    iface->load_supported_bands             = load_supported_bands;
    iface->load_supported_bands_finish      = load_supported_bands_finish;
    iface->load_current_bands               = load_current_bands;
    iface->load_current_bands_finish        = load_current_bands_finish;
    iface->set_current_bands                = set_current_bands;
    iface->set_current_bands_finish         = set_current_bands_finish;
    iface->load_unlock_retries              = load_unlock_retries;
    iface->load_unlock_retries_finish       = load_unlock_retries_finish;
}

static void
iface_modem_3gpp_init (MMIfaceModem3gpp *iface)
{
    iface_modem_3gpp_parent = g_type_interface_peek_parent (iface);

    iface->setup_unsolicited_events          = modem_3gpp_setup_unsolicited_events;
    iface->setup_unsolicited_events_finish   = modem_3gpp_setup_cleanup_unsolicited_events_finish;
    iface->cleanup_unsolicited_events        = modem_3gpp_cleanup_unsolicited_events;
    iface->cleanup_unsolicited_events_finish = modem_3gpp_setup_cleanup_unsolicited_events_finish;

    iface->enable_unsolicited_events         = modem_3gpp_enable_unsolicited_events;
    iface->enable_unsolicited_events_finish  = modem_3gpp_enable_unsolicited_events_finish;
    iface->disable_unsolicited_events        = modem_3gpp_disable_unsolicited_events;
    iface->disable_unsolicited_events_finish = modem_3gpp_disable_unsolicited_events_finish;

    iface->run_registration_checks           = modem_3gpp_run_registration_checks;
    iface->run_registration_checks_finish    = modem_3gpp_run_registration_checks_finish;
    iface->register_in_network               = modem_3gpp_register_in_network;
    iface->register_in_network_finish        = modem_3gpp_register_in_network_finish;
    iface->set_packet_service_state          = modem_3gpp_set_packet_service_state;
    iface->set_packet_service_state_finish   = modem_3gpp_set_packet_service_state_finish;
}

static void
iface_modem_3gpp_profile_manager_init (MMIfaceModem3gppProfileManager *iface)
{
    iface_modem_3gpp_profile_manager_parent = g_type_interface_peek_parent (iface);

    iface->list_profiles = modem_3gpp_profile_manager_list_profiles;
    iface->list_profiles_finish = modem_3gpp_profile_manager_list_profiles_finish;
    iface->check_format = modem_3gpp_profile_manager_check_format;
    iface->check_format_finish = modem_3gpp_profile_manager_check_format_finish;
    iface->store_profile = modem_3gpp_profile_manager_store_profile;
    iface->store_profile_finish = modem_3gpp_profile_manager_store_profile_finish;
}

static void
iface_modem_location_init (MMIfaceModemLocation *iface)
{
    iface_modem_location_parent = g_type_interface_peek_parent (iface);
}

static void
iface_modem_time_init (MMIfaceModemTime *iface)
{
    iface->check_support        = mm_shared_simtech_time_check_support;
    iface->check_support_finish = mm_shared_simtech_time_check_support_finish;
}

static void
mm_broadband_modem_simtech_sim7672_class_init (MMBroadbandModemSimtechSim7672Class *klass)
{
    GObjectClass          *object_class = G_OBJECT_CLASS (klass);
    MMBroadbandModemClass *broadband_modem_class = MM_BROADBAND_MODEM_CLASS (klass);

    g_type_class_add_private (object_class, sizeof (MMBroadbandModemSimtechSim7672Private));

    object_class->finalize = finalize;

    broadband_modem_class->setup_ports = setup_ports;
}
