/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2018-2023 Atmark Techno, Inc.
 */

#include <stdlib.h>
#include <gmodule.h>

#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>

#include "mm-log-object.h"
#include "mm-plugin-common.h"
#include "mm-broadband-modem-quectel-ec25.h"

#if defined WITH_QMI
#include "mm-broadband-modem-qmi-quectel.h"
#endif

#if defined WITH_MBIM
#include "mm-broadband-modem-mbim.h"
#include "mm-broadband-modem-mbim-quectel.h"
#endif

#define MM_TYPE_PLUGIN_QUECTEL_EC25 mm_plugin_quectel_ec25_get_type ()
MM_DEFINE_PLUGIN (QUECTEL_EC25, quectel_ec25, QuectelEc25)

/*****************************************************************************/
/* grab port */
static gboolean
grab_port (MMPlugin    *self,
           MMBaseModem *modem,
           MMPortProbe *probe,
           GError      **error)
{
    MMPortSerialAtFlag pflags = MM_PORT_SERIAL_AT_FLAG_NONE;
    MMKernelDevice *pdev;
    MMPortType ptype;
    const gchar* name = mm_port_probe_get_port_name (probe);

    ptype = mm_port_probe_get_port_type (probe);
    pdev = mm_port_probe_peek_port (probe);

    if (mm_kernel_device_get_global_property_as_boolean (pdev,
                                                         "ID_MM_QUECTEL_EC25_PORT_TYPE_PPP")) {
        mm_obj_dbg (self, "(%s/%s)' Port flagged as PPP",
                    mm_port_probe_get_port_subsys (probe), name);
          pflags = MM_PORT_SERIAL_AT_FLAG_PPP;
    } else if (mm_kernel_device_has_property (pdev,
                                              "ID_MM_QUECTEL_EC25_TTY_AT_SYMLINK")) {
        mm_obj_dbg (self, "(%s/%s)' Port flagged as primary",
                    mm_port_probe_get_port_subsys (probe),
                    mm_port_probe_get_port_name (probe));
        name = mm_kernel_device_get_property (pdev,
                                              "ID_MM_QUECTEL_EC25_TTY_AT_SYMLINK");
        mm_obj_dbg (self, "[%s] symlink name", name);
        return mm_base_modem_grab_symlink_at_primary_port (modem, pdev, name, error);
    } else if (mm_kernel_device_get_global_property_as_boolean (pdev,
                                                                "ID_MM_QUECTEL_EC25_PORT_TYPE_AT")) {
        mm_obj_dbg (self, "(%s/%s)' Port flagged as primary",
                    mm_port_probe_get_port_subsys (probe),
                    mm_port_probe_get_port_name (probe));
        pflags = MM_PORT_SERIAL_AT_FLAG_PRIMARY;
    }

    return mm_base_modem_grab_port (modem, pdev, ptype, pflags, error);
}

/*****************************************************************************/
/* Custom init */

static gboolean
quectel_ec25_custom_init_finish (MMPortProbe   *probe,
                                 GAsyncResult  *result,
                                 GError        **error)
{
    return g_task_propagate_boolean (G_TASK (result), error);
}


static void
custom_init_final(MMPortSerialAt *port,
                  GAsyncResult   *res,
                  GTask          *task)
{

    MMPortProbe *probe;
    const gchar *response;

    probe = g_task_get_source_object (task);

    response = mm_port_serial_at_command_finish (port, res, NULL);
    if (response)
        mm_port_probe_set_result_at(probe, TRUE);

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
send_command (MMPortSerialAt      *port,
              GTask               *task,
              const gchar         *command,
              GAsyncReadyCallback callback)
{
    mm_port_serial_at_command (
        port,
        command,
        6,
        FALSE, /* raw */
        FALSE, /* allow cached */
        g_task_get_cancellable (task),
        callback,
        task);
}

static void
send_next_command (MMPortSerialAt      *port,
                   GAsyncResult        *res,
                   GTask               *task,
                   const gchar         *command,
                   GAsyncReadyCallback callback)
{
    GError *error = NULL;

    mm_port_serial_at_command_finish (port, res, &error);
    if (error)
        g_error_free (error);

    send_command (port, task, command, callback);
}

static void
query_and_next_command (MMPortSerialAt      *port,
                        GAsyncResult        *res,
                        GTask               *task,
                        const gchar         *query_str,
                        const gchar         *set_command,
                        GAsyncReadyCallback set_callback,
                        const gchar         *next_command,
                        GAsyncReadyCallback next_callback)
{
    const gchar *response;
    GError *error = NULL;

    response = mm_port_serial_at_command_finish (port, res, &error);
    if (!error && response && strstr (response, query_str)) {
        if (!next_command || !next_callback) {
            MMPortProbe *probe = g_task_get_source_object (task);
            /* finalize */
            mm_port_probe_set_result_at(probe, TRUE);
            g_task_return_boolean (task, TRUE);
            g_object_unref (task);
            return;
        }
        send_command (port, task, next_command, next_callback);
        return;
    }

    if (error)
        g_error_free (error);

    send_command (port, task, set_command, set_callback);
}

static const gchar *QUERY_QCFG_URC_RI_OTHER="AT+QCFG=\"urc/ri/other\"";
static const gchar *QCFG_URC_RI_OTHER_OFF="AT+QCFG=\"urc/ri/other\",\"off\"";
static const gchar *RI_OTHER_OFF="off";

/* received AT+QCFG="urc/ri/other" response */
static void
query_qcfg_urc_ri_other (MMPortSerialAt *port,
                         GAsyncResult   *res,
                         GTask          *task)
{
    query_and_next_command (port, res, task,
                            RI_OTHER_OFF,
                            QCFG_URC_RI_OTHER_OFF,
                            (GAsyncReadyCallback)custom_init_final,
                            NULL,
                            (GAsyncReadyCallback)NULL);
}

static const gchar *QUERY_QCFG_URC_RI_SMSINCOMING="AT+QCFG=\"urc/ri/smsincoming\"";
static const gchar *QCFG_URC_RI_SMSINCOMING_PULSE="AT+QCFG=\"urc/ri/smsincoming\",\"pulse\"";
static const gchar *URC_RI_SMSINCOMING_PULSE="pulse";

/* received AT+QCFG="urc/ri/smsincoming","pulse" response */
static void
qcfg_urc_ri_smsincoming (MMPortSerialAt *port,
                         GAsyncResult   *res,
                         GTask          *task)
{
    send_next_command (port, res, task,
                       QUERY_QCFG_URC_RI_OTHER,
                       (GAsyncReadyCallback)query_qcfg_urc_ri_other);
}

/* received AT+QCFG="urc/ri/smsincoming" response */
static void
query_qcfg_urc_ri_smsincoming (MMPortSerialAt *port,
                               GAsyncResult   *res,
                               GTask          *task)
{
    query_and_next_command (port, res, task,
                            URC_RI_SMSINCOMING_PULSE,
                            QCFG_URC_RI_SMSINCOMING_PULSE,
                            (GAsyncReadyCallback)qcfg_urc_ri_smsincoming,
                            QUERY_QCFG_URC_RI_OTHER,
                            (GAsyncReadyCallback)query_qcfg_urc_ri_other);
}

static const gchar *QUERY_QCFG_URC_RI_RING="AT+QCFG=\"urc/ri/ring\"";
static const gchar *QCFG_URC_RI_RING_PULSE="AT+QCFG=\"urc/ri/ring\",\"pulse\"";
static const gchar *URC_RI_RING_PULSE="pulse";

/* received AT+QCFG="urc/ri/ring","pulse" response */
static void
qcfg_urc_ri_ring (MMPortSerialAt *port,
                  GAsyncResult   *res,
                  GTask          *task)
{
    send_next_command (port, res, task,
                       QUERY_QCFG_URC_RI_SMSINCOMING,
                       (GAsyncReadyCallback)query_qcfg_urc_ri_smsincoming);
}

/* received AT+QCFG="urc/ri/ring" response */
static void
query_qcfg_urc_ri_ring (MMPortSerialAt *port,
                        GAsyncResult   *res,
                        GTask          *task)
{
    query_and_next_command (port, res, task,
                            URC_RI_RING_PULSE,
                            QCFG_URC_RI_RING_PULSE,
                            (GAsyncReadyCallback)qcfg_urc_ri_ring,
                            QUERY_QCFG_URC_RI_SMSINCOMING,
                            (GAsyncReadyCallback)query_qcfg_urc_ri_smsincoming);
}

static const gchar *QUERY_QCFG_RISIGNALTYPE="AT+QCFG=\"risignaltype\"";
static const gchar *QCFG_RISIGNALTYPE_PHYSICAL="AT+QCFG=\"risignaltype\",\"physical\"";
static const gchar *RISIGNALTYPE_PHYSICAL="physical";

/* received AT+QCFG="risignaltype","physical" response */
static void
qcfg_risignaltype (MMPortSerialAt *port,
                   GAsyncResult   *res,
                   GTask          *task)
{
    send_next_command (port, res, task,
                       QUERY_QCFG_URC_RI_RING,
                       (GAsyncReadyCallback)query_qcfg_urc_ri_ring);
}

/* received AT+QCFG="risignaltype" response */
static void
query_qcfg_risignaltype (MMPortSerialAt *port,
                         GAsyncResult   *res,
                         GTask          *task)
{
    query_and_next_command (port, res, task,
                            RISIGNALTYPE_PHYSICAL,
                            QCFG_RISIGNALTYPE_PHYSICAL,
                            (GAsyncReadyCallback)qcfg_risignaltype,
                            QUERY_QCFG_URC_RI_RING,
                            (GAsyncReadyCallback)query_qcfg_urc_ri_ring);
}

static const gchar *QUERY_QCFG_URCPORT="AT+QURCCFG=\"urcport\"";
static const gchar *QCFG_URCPORT_USBAT="AT+QURCCFG=\"urcport\",\"usbat\"";
static const gchar *URCPORT_USBAT="usbat";

/* received AT+QURCCFG="urcport","usbat" response */
static void
qcfg_urcport (MMPortSerialAt *port,
              GAsyncResult   *res,
              GTask          *task)
{
    send_next_command (port, res, task,
                       QUERY_QCFG_RISIGNALTYPE,
                       (GAsyncReadyCallback)query_qcfg_risignaltype);
}

/* received AT+QURCCFG="urcport" response */
static void
query_qcfg_urcport (MMPortSerialAt *port,
                    GAsyncResult   *res,
                    GTask          *task)
{
    query_and_next_command (port, res, task,
                            URCPORT_USBAT,
                            QCFG_URCPORT_USBAT,
                            (GAsyncReadyCallback)qcfg_urcport,
                            QUERY_QCFG_RISIGNALTYPE,
                            (GAsyncReadyCallback)query_qcfg_risignaltype);
}

static const gchar *QUERY_DIAL_CLEAR="AT+QNVR=930,0";
static const gchar *DIAL_CLEAR_CMD="AT+QNVW=930,0,\"2A39380000030003002A3939000003000300233737370004000400000000000000000000\"";
static const gchar *DIAL_CLEAR_VALUE="2A39380000030003002A3939000003000300233737370004000400000000000000000000";

/* received AT+QNVW=930,0,"2A....." response */
static void
qnv_dial_clear (MMPortSerialAt *port,
                GAsyncResult   *res,
                GTask          *task)
{
    send_next_command (port, res, task,
                       QUERY_QCFG_URCPORT,
                       (GAsyncReadyCallback)query_qcfg_urcport);
}

/* received AT+QNVR=930,0 response */
static void
query_qnv_dial_clear (MMPortSerialAt *port,
                      GAsyncResult   *res,
                      GTask          *task)
{
    query_and_next_command (port, res, task,
                            DIAL_CLEAR_VALUE,
                            DIAL_CLEAR_CMD,
                            (GAsyncReadyCallback)qnv_dial_clear,
                            QUERY_QCFG_URCPORT,
                            (GAsyncReadyCallback)query_qcfg_urcport);
}

static const gchar *QUERY_SIM_CLEAR="AT+QNVR=4548,0";
static const gchar *SIM_CLEAR_CMD="AT+QNVW=4548,0,\"0000400C00000210\"";
static const gchar *SIM_CLEAR_VALUE="0000400C00000210";

/* received AT+QNVW=4548,0,"00....." response */
static void
qnv_sim_info_clear (MMPortSerialAt *port,
                    GAsyncResult   *res,
                    GTask          *task)
{
    send_next_command (port, res, task,
                       QUERY_DIAL_CLEAR,
                       (GAsyncReadyCallback)query_qnv_dial_clear);
}

/* received AT+QNVR=4548,0 response */
static void
query_qnv_sim_info_clear (MMPortSerialAt *port,
                          GAsyncResult   *res,
                          GTask          *task)
{
    query_and_next_command (port, res, task,
                            SIM_CLEAR_VALUE,
                            SIM_CLEAR_CMD,
                            (GAsyncReadyCallback)qnv_sim_info_clear,
                            QUERY_DIAL_CLEAR,
                            (GAsyncReadyCallback)query_qnv_dial_clear);
}

static void
quectel_ec25_custom_init (MMPortProbe         *probe,
                          MMPortSerialAt      *port,
                          GCancellable        *cancellable,
                          GAsyncReadyCallback  callback,
                          gpointer             user_data)
{
    GTask          *task;
    MMKernelDevice *pdev;

    task = g_task_new (probe, cancellable, callback, user_data);

    pdev = mm_port_probe_peek_port (probe);
    if (mm_kernel_device_get_global_property_as_boolean (
                                             pdev,
                                             "ID_MM_QUECTEL_EC25_PORT_TYPE_AT"))
        send_command (port, task,
                      QUERY_SIM_CLEAR,
                      (GAsyncReadyCallback)query_qnv_sim_info_clear);
    else {
        mm_port_probe_set_result_at(probe, TRUE);
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
    }
}

/*****************************************************************************/

static MMBaseModem *
create_modem (MMPlugin     *self,
              const gchar  *uid,
              const gchar  *physdev,
              const gchar **drivers,
              guint16       vendor,
              guint16       product,
              guint16       subsystem_vendor,
              GList        *probes,
              GError      **error)
{
#if defined WITH_QMI
    if (mm_port_probe_list_has_qmi_port (probes)) {
        mm_obj_dbg (self, "QMI-powered Quectel modem found...");
        return MM_BASE_MODEM (mm_broadband_modem_qmi_new (uid,
                                                          physdev,
                                                          drivers,
                                                          mm_plugin_get_name (self),
                                                          vendor,
                                                          product));
    }
#endif

#if defined WITH_MBIM
    if (mm_port_probe_list_has_mbim_port (probes)) {
        mm_obj_dbg (self, "MBIM-powered Quectel modem found...");
        return MM_BASE_MODEM (mm_broadband_modem_mbim_new (uid,
                                                           physdev,
                                                           drivers,
                                                           mm_plugin_get_name (self),
                                                           vendor,
                                                           product));
    }
#endif

    return MM_BASE_MODEM (mm_broadband_modem_quectel_ec25_new (uid,
                                                               physdev,
                                                               drivers,
                                                               mm_plugin_get_name (self),
                                                               vendor,
                                                               product));
}

/*****************************************************************************/

G_MODULE_EXPORT MMPlugin *
mm_plugin_create_quectel_ec25 (void)
{
    static const gchar *subsystems[] = { "tty", "net", "usbmisc", "wwan", NULL };
    static const mm_uint16_pair product_ids[] = {
        { 0x2c7c, 0x0125 }, /* EC25 */
        { 0x2c7c, 0x030e }, /* EM05-G */
        { 0, 0 }
    };

    static const MMAsyncMethod custom_init = {
        .async  = G_CALLBACK (quectel_ec25_custom_init),
        .finish = G_CALLBACK (quectel_ec25_custom_init_finish),
    };

    return MM_PLUGIN (
        g_object_new (MM_TYPE_PLUGIN_QUECTEL_EC25,
                      MM_PLUGIN_NAME,                   MM_MODULE_NAME,
                      MM_PLUGIN_ALLOWED_SUBSYSTEMS,     subsystems,
                      MM_PLUGIN_ALLOWED_PRODUCT_IDS,    product_ids,
                      MM_PLUGIN_ALLOWED_AT,             TRUE,
                      MM_PLUGIN_ALLOWED_QCDM,           TRUE,
                      MM_PLUGIN_ALLOWED_QMI,            TRUE,
                      MM_PLUGIN_ALLOWED_MBIM,           TRUE,
                      MM_PLUGIN_CUSTOM_INIT,            &custom_init,
                      NULL));
}

static void
mm_plugin_quectel_ec25_init (MMPluginQuectelEc25 *self)
{
}

static void
mm_plugin_quectel_ec25_class_init (MMPluginQuectelEc25Class *klass)
{
    MMPluginClass *plugin_class = MM_PLUGIN_CLASS (klass);

    plugin_class->create_modem = create_modem;
    plugin_class->grab_port = grab_port;
}
