/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2019-2023 Atmark Techno, Inc.
 */

#include <config.h>
#include <stdio.h>

#include "mm-broadband-modem-quectel-ec25.h"
#include "mm-iface-modem-3gpp.h"
#include "mm-iface-modem-3gpp-profile-manager.h"
#include "mm-iface-modem-firmware.h"
#include "mm-iface-modem-location.h"
#include "mm-iface-modem-time.h"
#include "mm-shared-quectel.h"
#include "mm-base-modem-at.h"
#include "mm-broadband-bearer-quectel-ec25.h"
#include "mm-modem-helpers.h"
#include "mm-modem-helpers-quectel-ec25.h"
#include "mm-base-sms.h"
#include "mm-sms-list.h"
#include "mm-call-list.h"
#include "mm-log.h"

static void iface_modem_init                      (MMIfaceModem                   *iface);
static void iface_modem_3gpp_init                 (MMIfaceModem3gpp               *iface);
static void iface_modem_3gpp_profile_manager_init (MMIfaceModem3gppProfileManager *iface);
static void iface_modem_firmware_init             (MMIfaceModemFirmware           *iface);
static void iface_modem_location_init             (MMIfaceModemLocation           *iface);
static void iface_modem_time_init                 (MMIfaceModemTime               *iface);
static void shared_quectel_init                   (MMSharedQuectel                *iface);

static MMIfaceModem         *iface_modem_parent;
static MMIfaceModem3gpp     *iface_modem_3gpp_parent;
static MMIfaceModemLocation *iface_modem_location_parent;

G_DEFINE_TYPE_EXTENDED (MMBroadbandModemQuectelEc25, mm_broadband_modem_quectel_ec25, MM_TYPE_BROADBAND_MODEM, 0,
                        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM, iface_modem_init)
                        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_3GPP, iface_modem_3gpp_init)
                        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_3GPP_PROFILE_MANAGER, iface_modem_3gpp_profile_manager_init)
                        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_FIRMWARE, iface_modem_firmware_init)
                        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_LOCATION, iface_modem_location_init)
                        G_IMPLEMENT_INTERFACE (MM_TYPE_IFACE_MODEM_TIME, iface_modem_time_init)
                        G_IMPLEMENT_INTERFACE (MM_TYPE_SHARED_QUECTEL, shared_quectel_init))


typedef struct _PortsContext PortsContext;

struct _MMBroadbandModemPrivate {
    /* Broadband modem specific implementation */
    PortsContext *enabled_ports_ctx;
    PortsContext *sim_hot_swap_ports_ctx;
    PortsContext *in_call_ports_ctx;
    gboolean modem_init_run;
    gboolean sim_hot_swap_supported;
    gboolean periodic_signal_check_disabled;
    gboolean periodic_access_tech_check_disabled;

    /*<--- Modem interface --->*/
    /* Properties */
    GObject *modem_dbus_skeleton;
    MMBaseSim *modem_sim;
    GPtrArray *modem_sim_slots;
    MMBearerList *modem_bearer_list;
    MMModemState modem_state;
    gchar *carrier_config_mapping;
    /* Implementation helpers */
    MMModemCharset modem_current_charset;
    gboolean modem_cind_disabled;
    gboolean modem_cind_support_checked;
    gboolean modem_cind_supported;
    guint modem_cind_indicator_signal_quality;
    guint modem_cind_min_signal_quality;
    guint modem_cind_max_signal_quality;
    guint modem_cind_indicator_roaming;
    guint modem_cind_indicator_service;
    MM3gppCmerMode modem_cmer_enable_mode;
    MM3gppCmerMode modem_cmer_disable_mode;
    MM3gppCmerInd modem_cmer_ind;
    gboolean modem_cgerep_support_checked;
    gboolean modem_cgerep_supported;
    MMFlowControl flow_control;

    /*<--- Modem 3GPP interface --->*/
    /* Properties */
    GObject *modem_3gpp_dbus_skeleton;
    MMModem3gppRegistrationState modem_3gpp_registration_state;
    gboolean modem_3gpp_cs_network_supported;
    gboolean modem_3gpp_ps_network_supported;
    gboolean modem_3gpp_eps_network_supported;
    gboolean modem_3gpp_5gs_network_supported;
    /* Implementation helpers */
    GPtrArray *modem_3gpp_registration_regex;
    MMModem3gppFacility modem_3gpp_ignored_facility_locks;
    MMBaseBearer *modem_3gpp_initial_eps_bearer;
    MMModem3gppPacketServiceState modem_3gpp_packet_service_state;

    /*<--- Modem 3GPP Profile Manager interface --->*/
    /* Properties */
    GObject *modem_3gpp_profile_manager_dbus_skeleton;

    /*<--- Modem 3GPP USSD interface --->*/
    /* Properties */
    GObject *modem_3gpp_ussd_dbus_skeleton;
    /* Implementation helpers */
    gboolean use_unencoded_ussd;
    GTask *pending_ussd_action;

    /*<--- Modem CDMA interface --->*/
    /* Properties */
    GObject *modem_cdma_dbus_skeleton;
    MMModemCdmaRegistrationState modem_cdma_cdma1x_registration_state;
    MMModemCdmaRegistrationState modem_cdma_evdo_registration_state;
    gboolean modem_cdma_cdma1x_network_supported;
    gboolean modem_cdma_evdo_network_supported;
    GCancellable *modem_cdma_pending_registration_cancellable;
    /* Implementation helpers */
    gboolean checked_sprint_support;
    gboolean has_spservice;
    gboolean has_speri;
    gint evdo_pilot_rssi;

    /*<--- Modem Simple interface --->*/
    /* Properties */
    GObject *modem_simple_dbus_skeleton;
    MMSimpleStatus *modem_simple_status;

    /*<--- Modem Location interface --->*/
    /* Properties */
    GObject *modem_location_dbus_skeleton;
    gboolean modem_location_allow_gps_unmanaged_always;

    /*<--- Modem Messaging interface --->*/
    /* Properties */
    GObject *modem_messaging_dbus_skeleton;
    MMSmsList *modem_messaging_sms_list;
    gboolean modem_messaging_sms_pdu_mode;
    MMSmsStorage modem_messaging_sms_default_storage;
    /* Implementation helpers */
    gboolean sms_supported_modes_checked;
    gboolean mem1_storage_locked;
    MMSmsStorage current_sms_mem1_storage;
    gboolean mem2_storage_locked;
    MMSmsStorage current_sms_mem2_storage;

    /*<--- Modem Voice interface --->*/
    /* Properties */
    GObject    *modem_voice_dbus_skeleton;
    MMCallList *modem_voice_call_list;
    gboolean    periodic_call_list_check_disabled;
    gboolean    indication_call_list_reload_enabled;
    gboolean    clcc_supported;

    /*<--- Modem Time interface --->*/
    /* Properties */
    GObject *modem_time_dbus_skeleton;

    /*<--- Modem Signal interface --->*/
    /* Properties */
    GObject *modem_signal_dbus_skeleton;

    /*<--- Modem OMA interface --->*/
    /* Properties */
    GObject *modem_oma_dbus_skeleton;

    /*<--- Modem Firmware interface --->*/
    /* Properties */
    GObject  *modem_firmware_dbus_skeleton;

    /*<--- Modem Sar interface --->*/
    /* Properties */
    GObject  *modem_sar_dbus_skeleton;

    gboolean  modem_firmware_ignore_carrier;
};

/*****************************************************************************/
/* Generic ports open/close context */

struct _PortsContext {
    volatile gint     ref_count;
    MMPortSerialAt   *primary;
    gboolean          primary_open;
    MMPortSerialAt   *secondary;
    gboolean          secondary_open;
    MMPortSerialQcdm *qcdm;
    gboolean          qcdm_open;
};


/*****************************************************************************/
/* Check if unlock required (Modem interface) */

typedef struct {
    const gchar *result;
    MMModemLock code;
} CPinResult;

static CPinResult unlock_results[] = {
    /* Longer entries first so we catch the correct one with strcmp() */
    { "READY",         MM_MODEM_LOCK_NONE           },
    { "SIM PIN2",      MM_MODEM_LOCK_SIM_PIN2       },
    { "SIM PUK2",      MM_MODEM_LOCK_SIM_PUK2       },
    { "SIM PIN",       MM_MODEM_LOCK_SIM_PIN        },
    { "SIM PUK",       MM_MODEM_LOCK_SIM_PUK        },
    { "PH-NETSUB PIN", MM_MODEM_LOCK_PH_NETSUB_PIN  },
    { "PH-NETSUB PUK", MM_MODEM_LOCK_PH_NETSUB_PUK  },
    { "PH-FSIM PIN",   MM_MODEM_LOCK_PH_FSIM_PIN    },
    { "PH-FSIM PUK",   MM_MODEM_LOCK_PH_FSIM_PUK    },
    { "PH-CORP PIN",   MM_MODEM_LOCK_PH_CORP_PIN    },
    { "PH-CORP PUK",   MM_MODEM_LOCK_PH_CORP_PUK    },
    { "PH-SIM PIN",    MM_MODEM_LOCK_PH_SIM_PIN     },
    { "PH-NET PIN",    MM_MODEM_LOCK_PH_NET_PIN     },
    { "PH-NET PUK",    MM_MODEM_LOCK_PH_NET_PUK     },
    { "PH-SP PIN",     MM_MODEM_LOCK_PH_SP_PIN      },
    { "PH-SP PUK",     MM_MODEM_LOCK_PH_SP_PUK      },
    { NULL }
};

static MMModemLock
modem_load_unlock_required_finish (MMIfaceModem *self,
                                   GAsyncResult *res,
                                   GError       **error)
{
    GError *inner_error = NULL;
    gssize value;

    value = g_task_propagate_int (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return MM_MODEM_LOCK_UNKNOWN;
    }
    return (MMModemLock)value;
}

static MMModemLock
str_to_MMModemLock(const gchar *result)
{
    MMModemLock lock = MM_MODEM_LOCK_UNKNOWN;
    CPinResult *iter = &unlock_results[0];
    const gchar *str;

    str = strstr (result, "+CPIN:") + 6;
    /* Skip possible whitespaces after '+CPIN:' and before the response */
    while (*str == ' ')
        str++;

    /* Some phones (Motorola EZX models) seem to quote the response */
    if (str[0] == '"')
        str++;

    /* Translate the reply */
    while (iter->result) {
        if (g_str_has_prefix (str, iter->result)) {
            lock = iter->code;
            break;
        }
        iter++;
    }
    return lock;
}

static void
error_finish (GTask  *task,
              GError *error)
{
    g_task_return_error (task, error);
    g_object_unref (task);
}

static void
cpin_query_ready_again (MMIfaceModem *self,
                        GAsyncResult *res,
                        GTask        *task)
{

    MMModemLock lock = MM_MODEM_LOCK_UNKNOWN;
    const gchar *result;
    GError *error = NULL;

    result = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (error) {
        error_finish (task, error);
        return;
    }

    if (result && strstr (result, "+CPIN:")) {
        lock = str_to_MMModemLock (result);
    }

    g_task_return_int (task, lock);
    g_object_unref (task);
}

static void
clck_pu_unlock_ready (MMIfaceModem *self,
                      GAsyncResult *res,
                      GTask        *task)
{
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+CPIN?",
                              3,
                              FALSE,
                              (GAsyncReadyCallback)cpin_query_ready_again,
                              task);
}

static void
cpin_query_ready (MMIfaceModem *self,
                  GAsyncResult *res,
                  GTask        *task)
{
    MMModemLock lock = MM_MODEM_LOCK_UNKNOWN;
    const gchar *result;
    GError *error = NULL;

    result = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (error) {
        error_finish (task, error);
        return;
    }

    if (result && strstr (result, "+CPIN:")) {
        lock = str_to_MMModemLock (result);
    }

    if (lock != MM_MODEM_LOCK_NONE) {
        mm_obj_dbg (self, "quectel ec25 try unlock");
        mm_base_modem_at_command (MM_BASE_MODEM (self),
                                  "+CLCK=\"PU\",0,\"12341234\"",
                                  3,
                                  FALSE,
                                  (GAsyncReadyCallback)clck_pu_unlock_ready,
                                  task);
        return;
    }

    g_task_return_int (task, lock);
    g_object_unref (task);
}

static void
modem_load_unlock_required (MMIfaceModem        *self,
                            gboolean            last_attempt,
                            GAsyncReadyCallback callback,
                            gpointer            user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);

    mm_obj_dbg (self, "checking if unlock required...");
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+CPIN?",
                              10,
                              FALSE,
                              (GAsyncReadyCallback)cpin_query_ready,
                              task);
}

/*****************************************************************************/
/* Supported IP families loading (Modem interface) */

static MMBearerIpFamily
modem_load_supported_ip_families_finish (MMIfaceModem *self,
                                         GAsyncResult *res,
                                         GError **error)
{
    GError *inner_error = NULL;
    gssize value;

    value = g_task_propagate_int (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return MM_BEARER_IP_FAMILY_NONE;
    }
    return (MMBearerIpFamily)value;
}

static void
supported_ip_families_qicsgp_test_ready (MMBaseModem *self,
                                         GAsyncResult *res,
                                         GTask *task)
{
    const gchar           *response;
    MMBearerIpFamily       mask = MM_BEARER_IP_FAMILY_IPV4 | MM_BEARER_IP_FAMILY_IPV4V6 | MM_BEARER_IP_FAMILY_IPV6;
    g_autoptr(GError)      error = NULL;
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    guint                  ec25_min_ip_type;
    guint                  ec25_max_ip_type;

    response = mm_base_modem_at_command_full_finish (self, res, &error);
    if (!response || !g_str_has_prefix (response, "+QICSGP:"))
        mm_obj_dbg (self, "failed checking context definition format: %s", error->message);
    else {
        r = g_regex_new ("\\+QICSGP:\\s*\\(\\s*(\\d+)\\s*-?\\s*(\\d+)\\),\\(\\s*(\\d+)\\s*-?\\s*(\\d+)\\)",
                         0, 0, NULL);
        g_assert (r != NULL);

        g_regex_match_full (r, response, strlen (response), 0, 0, &match_info, &error);
        if (g_match_info_matches(match_info)) {
            if (mm_get_uint_from_match_info (match_info, 3, &ec25_min_ip_type) &&
                mm_get_uint_from_match_info (match_info, 4, &ec25_max_ip_type)) {
                mask = MM_BEARER_IP_FAMILY_NONE;
                if (ec25_min_ip_type <= MM_QICSGP_CONTEXT_TYPE_IPV4) {
                    mask |= MM_BEARER_IP_FAMILY_IPV4;
                    mask |= MM_BEARER_IP_FAMILY_IPV4V6; /* QICSGP IPV4V6 is the same as IPV4 */
                }
                if (ec25_min_ip_type <= MM_QICSGP_CONTEXT_TYPE_IPV6 || ec25_max_ip_type >= MM_QICSGP_CONTEXT_TYPE_IPV6)
                    mask |= MM_BEARER_IP_FAMILY_IPV6;
            } else
                mm_obj_dbg (self, "AT+QISGP=? response has not ip family: %s", response);
        } else
            mm_obj_dbg (self, "AT+QISGP=? response is not match to regex: %s", response);
    }
    g_task_return_int (task, mask);
    g_object_unref (task);
}

static void
modem_load_supported_ip_families (MMIfaceModem *self,
                                  GAsyncReadyCallback callback,
                                  gpointer user_data)
{
    GTask *task;

    mm_obj_dbg (self, "loading supported IP families...");
    task = g_task_new (self, NULL, callback, user_data);

    if (mm_iface_modem_is_cdma_only (self)) {
        g_task_return_int (task, MM_BEARER_IP_FAMILY_IPV4);
        g_object_unref (task);
        return;
    }

    /* Query with QICSGP=? */
    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "+QICSGP=?",
        3,
        TRUE, /* allow caching, it's a test command */
        (GAsyncReadyCallback)supported_ip_families_qicsgp_test_ready,
        task);
}

/*****************************************************************************/
/* Check support (3GPP profile management interface) */

static gboolean
modem_3gpp_profile_manager_check_support_finish (MMIfaceModem3gppProfileManager  *self,
                                                 GAsyncResult                    *res,
                                                 gchar                          **index_field,
                                                 GError                         **error)
{
    if (g_task_propagate_boolean (G_TASK (res), error)) {
        *index_field = g_strdup ("profile-id");;
        return TRUE;
    }

    return FALSE;
}

static void
profile_manager_qicsgp_test_ready (MMBaseModem  *self,
                                   GAsyncResult *res,
                                   GTask        *task)
{
    GError *error = NULL;

    mm_base_modem_at_command_finish (self, res, &error);
    if (error)
        g_task_return_error (task, error);
    else
        g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_check_support (MMIfaceModem3gppProfileManager *self,
                                          GAsyncReadyCallback             callback,
                                          gpointer                        user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);
    if (!mm_iface_modem_is_3gpp (MM_IFACE_MODEM (self))) {
        g_task_return_boolean (task, FALSE);
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "+QICSGP=?",
        3,
        TRUE, /* allow caching, it's a test command */
        (GAsyncReadyCallback)profile_manager_qicsgp_test_ready,
        task);
}

/*****************************************************************************/
/* List profiles (3GPP profile management interface) */

typedef struct {
    GList *profiles;
    guint  next_pid;
} ListProfilesContext;

static void
list_profiles_context_free (ListProfilesContext *ctx)
{
    mm_3gpp_profile_list_free (ctx->profiles);
    g_slice_free (ListProfilesContext, ctx);
}

static gboolean
modem_3gpp_profile_manager_list_profiles_finish (MMIfaceModem3gppProfileManager  *self,
                                                 GAsyncResult                    *res,
                                                 GList                          **out_profiles,
                                                 GError                         **error)
{
    ListProfilesContext *ctx;

    if (!g_task_propagate_boolean (G_TASK (res), error))
        return FALSE;

    ctx = g_task_get_task_data (G_TASK (res));
    if (out_profiles)
        *out_profiles = g_steal_pointer (&ctx->profiles);
    return TRUE;
}

static void
send_query_qicsgp (MMIfaceModem3gppProfileManager *self, GTask *task);

static void
list_profiles_query_qicsgp_ready (MMIfaceModem3gppProfileManager *self,
                                  GAsyncResult                   *res,
                                  GTask                          *task)
{
    ListProfilesContext *ctx;
    const gchar         *response;
    GError              *error = NULL;
    MM3gppPdpContext    *pdp = NULL;
    MMBearerAllowedAuth  auth;
    MMBearerIpFamily     ip_family;
    gchar               *apn = NULL;
    g_autofree gchar    *username = NULL;
    g_autofree gchar    *password = NULL;

    response = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (!response || error) {
        g_error_free (error);
    } else {
        ctx = g_task_get_task_data (task);
        if (mm_quectel_ec25_parse_qicsgp_read_response (response,
                                                        &auth,
                                                        &ip_family,
                                                        &apn,
                                                        &username,
                                                        &password,
                                                        &error)) {
            pdp = g_slice_new0 (MM3gppPdpContext);
            pdp->pdp_type = ip_family;
            pdp->apn = apn;
        } else
            g_error_free (error);

        if (pdp)
            ctx->profiles = g_list_prepend (ctx->profiles, pdp);

        ctx->next_pid++;

        if (ctx->next_pid > 16) {
            g_task_return_boolean (task, TRUE);
            g_object_unref (task);
            return;
        }
    }
    send_query_qicsgp (self, task);
}

static void
send_query_qicsgp (MMIfaceModem3gppProfileManager *self,
                   GTask                          *task)
{
    ListProfilesContext *ctx;
    g_autofree gchar    *command = NULL;

    ctx = g_task_get_task_data (task);

    command = g_strdup_printf ("+QICSGP=%u", ctx->next_pid);
    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        command,
        3,
        FALSE,
        (GAsyncReadyCallback)list_profiles_query_qicsgp_ready,
        task);
}

static void
modem_3gpp_profile_manager_list_profiles (MMIfaceModem3gppProfileManager *self,
                                          GAsyncReadyCallback             callback,
                                          gpointer                        user_data)
{
    GTask               *task;
    ListProfilesContext *ctx;

    task = g_task_new (self, NULL, callback, user_data);
    ctx = g_slice_new0 (ListProfilesContext);
    g_task_set_task_data (task, ctx, (GDestroyNotify) list_profiles_context_free);
    ctx->next_pid = 1;

    send_query_qicsgp (self, task);
}

/*****************************************************************************/
/* Check format (3GPP profile management interface) */

typedef struct {
    MMBearerIpFamily ip_type;
    guint            min_profile_id;
    guint            max_profile_id;
} CheckFormatContext;

static void
check_format_context_free (CheckFormatContext *ctx)
{
    g_slice_free (CheckFormatContext, ctx);
}

static gboolean
modem_3gpp_profile_manager_check_format_finish (MMIfaceModem3gppProfileManager  *self,
                                                GAsyncResult                    *res,
                                                gboolean                        *new_id,
                                                gint                            *min_profile_id,
                                                gint                            *max_profile_id,
                                                GEqualFunc                      *apn_cmp,
                                                MM3gppProfileCmpFlags           *profile_cmp_flags,
                                                GError                         **error)
{
    CheckFormatContext *ctx;

    if (!g_task_propagate_boolean (G_TASK (res), error))
        return FALSE;

    ctx = g_task_get_task_data (G_TASK (res));
    if (new_id)
        *new_id = TRUE;
    if (min_profile_id)
        *min_profile_id = (gint) ctx->min_profile_id;
    if (max_profile_id)
        *max_profile_id = (gint) ctx->max_profile_id;
    if (apn_cmp)
        *apn_cmp = (GEqualFunc) mm_3gpp_cmp_apn_name;
    if (profile_cmp_flags)
        *profile_cmp_flags = (MM_3GPP_PROFILE_CMP_FLAGS_NO_AUTH |
                              MM_3GPP_PROFILE_CMP_FLAGS_NO_APN_TYPE |
                              MM_3GPP_PROFILE_CMP_FLAGS_NO_ACCESS_TYPE_PREFERENCE |
                              MM_3GPP_PROFILE_CMP_FLAGS_NO_ROAMING_ALLOWANCE |
                              MM_3GPP_PROFILE_CMP_FLAGS_NO_PROFILE_SOURCE);
    return TRUE;
}

static void
check_format_qicsgp_test_ready (MMBaseModem  *self,
                                GAsyncResult *res,
                                GTask        *task)
{
    CheckFormatContext    *ctx;
    const gchar           *response;
    g_autoptr(GError)      error = NULL;
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;

    ctx = g_task_get_task_data (task);

    response = mm_base_modem_at_command_full_finish (self, res, &error);
    if (!response || !g_str_has_prefix (response, "+QICSGP:")) {
        mm_obj_dbg (self, "failed checking context definition format: %s", error->message);
        ctx->min_profile_id = 1;
        ctx->max_profile_id = 16;
    } else {
        mm_quectel_ec25_parse_qicsgp_test (self, response, &ctx->min_profile_id, &ctx->max_profile_id, error);
        if (error) {
            ctx->min_profile_id = 1;
            ctx->max_profile_id = 16;
        }
    }

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_check_format (MMIfaceModem3gppProfileManager *self,
                                         MMBearerIpFamily                ip_type,
                                         GAsyncReadyCallback             callback,
                                         gpointer                        user_data)
{
    GTask              *task;
    CheckFormatContext *ctx;

    task = g_task_new (self, NULL, callback, user_data);
    ctx = g_slice_new0 (CheckFormatContext);
    ctx->ip_type = ip_type;
    g_task_set_task_data (task, ctx, (GDestroyNotify)check_format_context_free);

    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+QICSGP=?",
                              3,
                              TRUE, /* cached */
                              (GAsyncReadyCallback)check_format_qicsgp_test_ready,
                              task);
}

/*****************************************************************************/
/* Delete profile (3GPP profile management interface) */

static gboolean
modem_3gpp_profile_manager_delete_profile_finish (MMIfaceModem3gppProfileManager  *self,
                                                  GAsyncResult                    *res,
                                                  GError                         **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
profile_manager_qicsgp_reset_ready (MMBaseModem  *self,
                                    GAsyncResult *res,
                                    GTask        *task)
{
    GError *error = NULL;
    gint    profile_id;

    profile_id = GPOINTER_TO_INT (g_task_get_task_data (task));

    if (!mm_base_modem_at_command_finish (self, res, &error)) {
        mm_obj_dbg (self, "attempting to reset context with id '%d' failed: %s", profile_id, error->message);
        g_task_return_error (task, error);
    } else {
        mm_obj_dbg (self, "reseted context with profile id '%d'", profile_id);
        g_task_return_boolean (task, TRUE);
    }
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_delete_profile (MMIfaceModem3gppProfileManager *self,
                                           MM3gppProfile                  *profile,
                                           const gchar                    *index_field,
                                           GAsyncReadyCallback             callback,
                                           gpointer                        user_data)
{
    g_autofree gchar *cmd = NULL;
    GTask            *task;
    gint              profile_id;

    g_assert (g_strcmp0 (index_field, "profile-id") == 0);

    task = g_task_new (self, NULL, callback, user_data);

    profile_id = mm_3gpp_profile_get_profile_id (profile);

    if (profile_id == MM_3GPP_PROFILE_ID_UNKNOWN) {
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_INVALID_ARGS,
                                 "invalid profile id:%d",
                                 profile_id);
        g_object_unref (task);
        return;
    }

    g_task_set_task_data (task, GINT_TO_POINTER (profile_id), NULL);

    cmd = g_strdup_printf ("+QICSGP=%d,1,\"\",\"\",\"\",0", profile_id);

    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        cmd,
        3,
        FALSE,
        (GAsyncReadyCallback)profile_manager_qicsgp_reset_ready,
        task);
}

/*****************************************************************************/
/* Check Activated profile (3GPP profile management interface) */

static gboolean
modem_3gpp_profile_manager_check_activated_profile_finish (MMIfaceModem3gppProfileManager  *self,
                                                           GAsyncResult                    *res,
                                                           gboolean                        *out_activated,
                                                           GError                         **error)
{
    GError   *inner_error = NULL;
    gboolean  result;

    result = g_task_propagate_boolean (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return FALSE;
    }

    if (out_activated)
        *out_activated = result;
    return TRUE;
}

static void
check_activated_profile_qiact_query_ready (MMBaseModem  *self,
                                           GAsyncResult *res,
                                           GTask        *task)
{
    MM3gppProfile *profile;
    const gchar   *response;
    GError        *error = NULL;
    GList         *pdp_context_active_list = NULL;
    GList         *l;
    gint           profile_id;
    gboolean       activated = FALSE;

    response = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (response)
        pdp_context_active_list = mm_quectel_ec25_parse_qiact_read_response (response, &error);

    if (error) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    profile = g_task_get_task_data (task);
    profile_id = mm_3gpp_profile_get_profile_id (profile);

    for (l = pdp_context_active_list; l; l = g_list_next (l)) {
        MM3gppPdpContextActive *iter = l->data;

        if ((gint)iter->cid == profile_id) {
            activated = iter->active;
            break;
        }
    }
    mm_3gpp_pdp_context_active_list_free (pdp_context_active_list);

    if (!l)
        g_task_return_new_error (task, MM_CORE_ERROR, MM_CORE_ERROR_NOT_FOUND,
                                 "Profile '%d' not found in QIACT? response",
                                 profile_id);
    else
        g_task_return_boolean (task, activated);
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_check_activated_profile (MMIfaceModem3gppProfileManager *self,
                                                    MM3gppProfile                  *profile,
                                                    GAsyncReadyCallback             callback,
                                                    gpointer                        user_data)
{
    GTask *task;
    gint   profile_id;

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, g_object_ref (profile), g_object_unref);

    profile_id = mm_3gpp_profile_get_profile_id (profile);

    mm_obj_dbg (self, "checking if profile with id '%d' is already activated...", profile_id);
    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        "+QIACT?",
        3,
        FALSE,
        (GAsyncReadyCallback)check_activated_profile_qiact_query_ready,
        task);
}

/*****************************************************************************/
/* Deactivate profile (3GPP profile management interface) */

static gboolean
modem_3gpp_profile_manager_deactivate_profile_finish (MMIfaceModem3gppProfileManager  *self,
                                                      GAsyncResult                    *res,
                                                      GError                         **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
deactivate_profile_qideact_set_ready (MMBaseModem  *self,
                                      GAsyncResult *res,
                                      GTask        *task)
{
    GError *error = NULL;

    if (!mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error))
        g_task_return_error (task, error);
    else
        g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_deactivate_profile (MMIfaceModem3gppProfileManager *self,
                                               MM3gppProfile                  *profile,
                                               GAsyncReadyCallback             callback,
                                               gpointer                        user_data)
{
    GTask            *task;
    gint              profile_id;
    g_autofree gchar *cmd = NULL;

    task = g_task_new (self, NULL, callback, user_data);

    profile_id = mm_3gpp_profile_get_profile_id (profile);
    mm_obj_dbg (self, "deactivating profile with id '%d'...", profile_id);

    cmd = g_strdup_printf ("+QIDEACT=%d", profile_id);
    mm_base_modem_at_command (
        MM_BASE_MODEM (self),
        cmd,
        MM_BASE_BEARER_DEFAULT_DISCONNECTION_TIMEOUT,
        FALSE,
        (GAsyncReadyCallback)deactivate_profile_qideact_set_ready,
        task);
}

/*****************************************************************************/
/* Store profile (3GPP profile management interface) */

static gboolean
modem_3gpp_profile_manager_store_profile_finish (MMIfaceModem3gppProfileManager  *self,
                                                 GAsyncResult                    *res,
                                                 gint                            *out_profile_id,
                                                 MMBearerApnType                 *out_apn_type,
                                                 GError                         **error)
{
    if (!g_task_propagate_boolean (G_TASK (res), error))
        return FALSE;

    if (out_profile_id)
        *out_profile_id = GPOINTER_TO_INT (g_task_get_task_data (G_TASK (res)));
    if (out_apn_type)
        *out_apn_type = MM_BEARER_APN_TYPE_NONE;
    return TRUE;
}

static void
store_profile_qicsgp_set_ready (MMBaseModem  *self,
                                GAsyncResult *res,
                                GTask        *task)
{
    GError *error = NULL;

    if (!mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error))
        g_task_return_error (task, error);
    else
        g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
modem_3gpp_profile_manager_store_profile (MMIfaceModem3gppProfileManager *self,
                                          MM3gppProfile                  *profile,
                                          const gchar                    *index_field,
                                          GAsyncReadyCallback             callback,
                                          gpointer                        user_data)
{
    GTask                     *task;
    gint                       context_id;
    MMBearerIpFamily           ip_type;
    const gchar               *apn;
    MMBearerAllowedAuth        allowed_auth;
    const gchar               *username;
    const gchar               *password;
    gboolean                   has_username;
    gboolean                   has_password;
    g_autofree gchar          *ip_type_str = NULL;
    guint                      qicsgp_context_type;
    guint                      qicsgp_auth;
    g_autofree gchar          *str = NULL;
    g_autofree gchar          *cmd = NULL;

    g_assert (g_strcmp0 (index_field, "profile-id") == 0);

    task = g_task_new (self, NULL, callback, user_data);

    /* contextID */
    context_id = mm_3gpp_profile_get_profile_id (profile);
    if (context_id == MM_3GPP_PROFILE_ID_UNKNOWN) {
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_INVALID_ARGS,
                                 "Profile ID is unknown.");
        g_object_unref (task);
        return;
    }

    g_task_set_task_data (task, GINT_TO_POINTER (context_id), NULL);

    /* context_type */
    ip_type = mm_3gpp_profile_get_ip_type (profile);
    qicsgp_context_type = mm_quectel_ec25_ip_family_to_qicsgp_context_type (ip_type);
    if (qicsgp_context_type == MM_BEARER_IP_FAMILY_NONE) {
        str = mm_bearer_ip_family_build_string_from_mask (ip_type);
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_UNSUPPORTED,
                                 "Cannot use any of the specified IP type (%s)",
                                 str);
        g_object_unref (task);
        return;
    }

    /* APM */
    apn = mm_3gpp_profile_get_apn (profile);

    /* authentication, username, password */
    allowed_auth = mm_3gpp_profile_get_allowed_auth (profile);
    username = mm_3gpp_profile_get_user (profile);
    password = mm_3gpp_profile_get_password (profile);
    has_username = (username && username[0]);
    has_password = (password && password[0]);

    if (allowed_auth == MM_BEARER_ALLOWED_AUTH_UNKNOWN) {
        if (has_username && has_password) {
            /* If user/passwd given, default to CHAP (more common than PAP) */
            allowed_auth = MM_BEARER_ALLOWED_AUTH_CHAP;
        } else {
            /* If user/passwd not given, default to none */
            allowed_auth = MM_BEARER_ALLOWED_AUTH_NONE;
        }
    }

    qicsgp_auth = mm_quectel_ec25_allowed_auth_to_qicsgp_authentication (allowed_auth,
                                                                         has_username || has_password);
    if (qicsgp_auth == MM_QICSGP_AUTHENTICATION_UNKNOWN) {
        str = mm_bearer_allowed_auth_build_string_from_mask (allowed_auth);
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_UNSUPPORTED,
                                 "Cannot use any of the specified authentication methods (%s)",
                                 str);
        g_object_unref (task);
        return;
    }

    mm_obj_dbg (self, "storing profile '%d': apn '%s', ip type '%s'",
                context_id,
                apn,
                mm_bearer_ip_family_build_string_from_mask (ip_type));

    cmd = mm_quectel_ec25_build_qicsgp (context_id,
                                        qicsgp_context_type,
                                        apn,
                                        username,
                                        password,
                                        qicsgp_auth);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              cmd,
                              20,
                              FALSE,
                              (GAsyncReadyCallback) store_profile_qicsgp_set_ready,
                              task);
}

/*****************************************************************************/
/* Register in network (3GPP interface) */

static gboolean
register_in_network_finish (MMIfaceModem3gpp  *self,
                            GAsyncResult      *res,
                            GError           **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
cops_set_ready (MMBaseModem  *self,
                GAsyncResult *res,
                GTask        *task)
{
    GError *error = NULL;

    if (!mm_base_modem_at_command_full_finish (MM_BASE_MODEM (self), res, &error)) {
        if (error)
            g_error_free (error);

        /* set fake state */
        mm_iface_modem_3gpp_update_eps_registration_state (MM_IFACE_MODEM_3GPP (self),
                                                           MM_MODEM_3GPP_REGISTRATION_STATE_HOME,
                                                           FALSE);
    }
    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
cops_ascii_set_ready (MMBaseModem  *_self,
                      GAsyncResult *res,
                      GTask        *task)
{
    MMBroadbandModem  *self = MM_BROADBAND_MODEM (_self);
    g_autoptr(GError)  error = NULL;

    if (!mm_base_modem_at_command_full_finish (_self, res, &error)) {
        /* If it failed with an unsupported error, retry with current modem charset */
        if (g_error_matches (error, MM_MOBILE_EQUIPMENT_ERROR, MM_MOBILE_EQUIPMENT_ERROR_NOT_SUPPORTED)) {
            g_autoptr(GError)  enc_error = NULL;
            g_autofree gchar  *operator_id_enc = NULL;
            gchar             *operator_id;

            /* try to encode to current charset */
            operator_id = g_task_get_task_data (task);
            operator_id_enc = mm_modem_charset_str_from_utf8 (operator_id, self->priv->modem_current_charset, FALSE, &enc_error);
            if (!operator_id_enc) {
                mm_obj_dbg (self, "couldn't convert operator id to current charset: %s", enc_error->message);
                g_task_return_error (task, g_steal_pointer (&error));
                g_object_unref (task);
                return;
            }

            /* retry only if encoded string is different to the non-encoded one */
            if (g_strcmp0 (operator_id, operator_id_enc) != 0) {
                g_autofree gchar *command = NULL;

                command = g_strdup_printf ("+COPS=1,2,\"%s\"", operator_id_enc);
                mm_base_modem_at_command_full (_self,
                                               mm_base_modem_peek_best_at_port (_self, NULL),
                                               command,
                                               120,
                                               FALSE,
                                               FALSE, /* raw */
                                               g_task_get_cancellable (task),
                                               (GAsyncReadyCallback)cops_set_ready,
                                               task);
                return;
            }
        }
        g_task_return_error (task, g_steal_pointer (&error));
    } else
        g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
register_in_network (MMIfaceModem3gpp    *self,
                     const gchar         *operator_id,
                     GCancellable        *cancellable,
                     GAsyncReadyCallback  callback,
                     gpointer             user_data)
{
    GTask *task;
    gchar *command;

    task = g_task_new (self, cancellable, callback, user_data);

    /* Trigger automatic network registration if no explicit operator id given */
    if (!operator_id) {
        /* Note that '+COPS=0,,' (same but with commas) won't work in some Nokia
         * phones */
        mm_base_modem_at_command_full (MM_BASE_MODEM (self),
                                       mm_base_modem_peek_best_at_port (MM_BASE_MODEM (self), NULL),
                                       "+COPS=0",
                                       120,
                                       FALSE,
                                       FALSE, /* raw */
                                       cancellable,
                                       (GAsyncReadyCallback)cops_set_ready,
                                       task);
        return;
    }

    /* Store operator id in context, in case we need to retry with the current
     * modem charset */
    g_task_set_task_data (task, g_strdup (operator_id), g_free);

    /* Use the operator id given in ASCII initially */
    command = g_strdup_printf ("+COPS=1,2,\"%s\"", operator_id);
    mm_base_modem_at_command_full (MM_BASE_MODEM (self),
                                   mm_base_modem_peek_best_at_port (MM_BASE_MODEM (self), NULL),
                                   command,
                                   120,
                                   FALSE,
                                   FALSE, /* raw */
                                   cancellable,
                                   (GAsyncReadyCallback)cops_ascii_set_ready,
                                   task);
    g_free (command);
}

/*****************************************************************************/
/* Registration checks (3GPP interface) */

typedef struct _RunRegistrationChecksContext RunRegistrationChecksContext;

struct _RunRegistrationChecksContext {
    gboolean is_cs_supported;
    gboolean is_ps_supported;
    gboolean is_eps_supported;
    gboolean run_cs;
    gboolean run_ps;
    gboolean run_eps;
    gboolean running_cs;
    gboolean running_ps;
    gboolean running_eps;
    GError *error_cs;
    GError *error_ps;
    GError *error_eps;
    gboolean is_registerd;
};

static void
run_registration_checks_context_free (RunRegistrationChecksContext *ctx)
{
    g_clear_error (&ctx->error_cs);
    g_clear_error (&ctx->error_ps);
    g_clear_error (&ctx->error_eps);
    g_free (ctx);
}

static gboolean
run_registration_checks_finish (MMIfaceModem3gpp  *self,
                                GAsyncResult      *res,
                                GError           **error)
{
    GError *inner_error = NULL;
    gint    result;

    result = g_task_propagate_int (G_TASK (res), &inner_error);
    if (inner_error) {
        g_propagate_error (error, inner_error);
        return FALSE;
    }

    if (result == 1) {
        mm_iface_modem_3gpp_apply_deferred_registration_state (self);
        return TRUE;
    } else
        return (result == 0)?TRUE:FALSE;
}

static void run_registration_checks_context_step (GTask *task);

static void
run_registration_checks_context_set_error (RunRegistrationChecksContext *ctx,
                                           GError                       *error)
{
    g_assert (error != NULL);
    if (ctx->running_cs)
        ctx->error_cs = error;
    else if (ctx->running_ps)
        ctx->error_ps = error;
    else if (ctx->running_eps)
        ctx->error_eps = error;
    else
        g_assert_not_reached ();
}

static void
registration_status_check_ready (MMBroadbandModem *self,
                                 GAsyncResult     *res,
                                 GTask            *task)
{
    g_autoptr(GMatchInfo)         match_info = NULL;
    RunRegistrationChecksContext *ctx;
    const gchar                  *response;
    GError                       *error = NULL;
    guint                         i;
    gboolean                      parsed;
    gboolean                      cgreg = FALSE;
    gboolean                      cereg = FALSE;
    gboolean                      c5greg = FALSE;
    MMModem3gppRegistrationState  state = MM_MODEM_3GPP_REGISTRATION_STATE_UNKNOWN;
    MMModemAccessTechnology       act = MM_MODEM_ACCESS_TECHNOLOGY_UNKNOWN;
    gulong                        lac = 0;
    gulong                        tac = 0;
    gulong                        cid = 0;

    ctx = g_task_get_task_data (task);

    /* Only one must be running */
    g_assert ((ctx->running_cs + ctx->running_ps + ctx->running_eps) == 1);

    response = mm_base_modem_at_command_finish (MM_BASE_MODEM (self), res, &error);
    if (!response) {
        run_registration_checks_context_set_error (ctx, error);
        run_registration_checks_context_step (task);
        return;
    }

    /* Unsolicited registration status handlers will usually process the
     * response for us, but just in case they don't, do that here.
     */
    if (!response[0]) {
        /* Done */
        run_registration_checks_context_step (task);
        return;
    }

    /* Try to match the response */
    for (i = 0;
         i < self->priv->modem_3gpp_registration_regex->len;
         i++) {
        if (g_regex_match ((GRegex *)g_ptr_array_index (self->priv->modem_3gpp_registration_regex, i),
                           response,
                           0,
                           &match_info))
            break;
        g_clear_pointer (&match_info, g_match_info_free);
    }

    if (!match_info) {
        error = g_error_new (MM_CORE_ERROR,
                             MM_CORE_ERROR_FAILED,
                             "Unknown registration status response: '%s'",
                             response);
        run_registration_checks_context_set_error (ctx, error);
        run_registration_checks_context_step (task);
        return;
    }

    parsed = mm_3gpp_parse_creg_response (match_info,
                                          self,
                                          &state,
                                          &lac,
                                          &cid,
                                          &act,
                                          &cgreg,
                                          &cereg,
                                          &c5greg,
                                          &error);

    if (!parsed) {
        if (!error)
            error = g_error_new (MM_CORE_ERROR,
                                 MM_CORE_ERROR_FAILED,
                                 "Error parsing registration response: '%s'",
                                 response);
        run_registration_checks_context_set_error (ctx, error);
        run_registration_checks_context_step (task);
        return;
    }

    if ((state == MM_MODEM_3GPP_REGISTRATION_STATE_HOME) ||
        (state == MM_MODEM_3GPP_REGISTRATION_STATE_ROAMING))
        ctx->is_registerd = TRUE;

    /* Report new registration state and fix LAC/TAC.
     * According to 3GPP TS 27.007:
     *  - If CREG reports <AcT> 7 (LTE) then the <lac> field contains TAC
     *  - CEREG always reports TAC
     */
    if (cgreg) {
        if (ctx->running_cs)
            mm_obj_dbg (self, "got PS registration state when checking CS registration state");
        else if (ctx->running_eps)
            mm_obj_dbg (self, "got PS registration state when checking EPS registration state");
        mm_iface_modem_3gpp_update_ps_registration_state (MM_IFACE_MODEM_3GPP (self), state, FALSE);
    } else if (cereg) {
        tac = lac;
        lac = 0;
        if (ctx->running_cs)
            mm_obj_dbg (self, "got EPS registration state when checking CS registration state");
        else if (ctx->running_ps)
            mm_obj_dbg (self, "got EPS registration state when checking PS registration state");
        mm_iface_modem_3gpp_update_eps_registration_state (MM_IFACE_MODEM_3GPP (self), state, FALSE);
    } else {
        if (act == MM_MODEM_ACCESS_TECHNOLOGY_LTE) {
            tac = lac;
            lac = 0;
        }
        if (ctx->running_ps)
            mm_obj_dbg (self, "got CS registration state when checking PS registration state");
        else if (ctx->running_eps)
            mm_obj_dbg (self, "got CS registration state when checking EPS registration state");
        mm_iface_modem_3gpp_update_cs_registration_state (MM_IFACE_MODEM_3GPP (self), state, FALSE);
    }

    mm_iface_modem_3gpp_update_access_technologies (MM_IFACE_MODEM_3GPP (self), act);
    mm_iface_modem_3gpp_update_location (MM_IFACE_MODEM_3GPP (self), lac, tac, cid);

    run_registration_checks_context_step (task);
}

static void
run_registration_checks_context_step (GTask *task)
{
    MMBroadbandModem *self;
    RunRegistrationChecksContext *ctx;
    GError *error = NULL;

    self = g_task_get_source_object (task);
    ctx = g_task_get_task_data (task);

    ctx->running_cs = FALSE;
    ctx->running_ps = FALSE;
    ctx->running_eps = FALSE;

    if (ctx->run_cs) {
        ctx->running_cs = TRUE;
        ctx->run_cs = FALSE;
        /* Check current CS-registration state. */
        mm_base_modem_at_command (MM_BASE_MODEM (self),
                                  "+CREG?",
                                  10,
                                  FALSE,
                                  (GAsyncReadyCallback)registration_status_check_ready,
                                  task);
        return;
    }

    if (ctx->run_ps) {
        ctx->running_ps = TRUE;
        ctx->run_ps = FALSE;
        /* Check current PS-registration state. */
        mm_base_modem_at_command (MM_BASE_MODEM (self),
                                  "+CGREG?",
                                  10,
                                  FALSE,
                                  (GAsyncReadyCallback)registration_status_check_ready,
                                  task);
        return;
    }

    if (ctx->run_eps) {
        ctx->running_eps = TRUE;
        ctx->run_eps = FALSE;
        /* Check current EPS-registration state. */
        mm_base_modem_at_command (MM_BASE_MODEM (self),
                                  "+CEREG?",
                                  10,
                                  FALSE,
                                  (GAsyncReadyCallback)registration_status_check_ready,
                                  task);
        return;
    }

    /* If all run checks returned errors we fail */
    if ((ctx->is_cs_supported || ctx->is_ps_supported || ctx->is_eps_supported) &&
        (!ctx->is_cs_supported || ctx->error_cs) &&
        (!ctx->is_ps_supported || ctx->error_ps) &&
        (!ctx->is_eps_supported || ctx->error_eps)) {
        /* When reporting errors, prefer the EPS, then PS, then CS */
        if (ctx->error_eps)
            error = g_steal_pointer (&ctx->error_eps);
        else if (ctx->error_ps)
            error = g_steal_pointer (&ctx->error_ps);
        else if (ctx->error_cs)
            error = g_steal_pointer (&ctx->error_cs);
        else
            g_assert_not_reached ();
    }

    if (error || !ctx->is_registerd) {
        /* in LTE network only, set home, LTE, and dummy location
           To register only LTE network needs apn, username, password. */
        mm_iface_modem_3gpp_update_eps_registration_state (MM_IFACE_MODEM_3GPP (self),
                                                           MM_MODEM_3GPP_REGISTRATION_STATE_HOME,
                                                           TRUE);
        mm_iface_modem_3gpp_update_access_technologies (MM_IFACE_MODEM_3GPP (self),
                                                        MM_MODEM_ACCESS_TECHNOLOGY_LTE);
        mm_iface_modem_3gpp_update_location (MM_IFACE_MODEM_3GPP (self), 0xffff, 0xfffffffe, 0);
        mm_iface_modem_update_state (MM_IFACE_MODEM (self),
                                     MM_MODEM_STATE_REGISTERED,
                                     MM_MODEM_STATE_CHANGE_REASON_UNKNOWN);
        g_error_free (error);
        g_task_return_int (task, 1);
    } else
        g_task_return_int (task, 0);

    g_object_unref (task);
}

static void
run_registration_checks (MMIfaceModem3gpp    *self,
                         gboolean             is_cs_supported,
                         gboolean             is_ps_supported,
                         gboolean             is_eps_supported,
                         gboolean             is_5gs_supported,
                         GAsyncReadyCallback  callback,
                         gpointer             user_data)
{
    RunRegistrationChecksContext *ctx;
    GTask *task;

    ctx = g_new0 (RunRegistrationChecksContext, 1);
    ctx->is_cs_supported = is_cs_supported;
    ctx->is_ps_supported = is_ps_supported;
    ctx->is_eps_supported = is_eps_supported;
    ctx->run_cs = is_cs_supported;
    ctx->run_ps = is_ps_supported;
    ctx->run_eps = is_eps_supported;

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, ctx, (GDestroyNotify)run_registration_checks_context_free);

    run_registration_checks_context_step (task);
}

/*****************************************************************************/
/* Create Bearer (Modem interface) */

static MMBaseBearer *
create_bearer_finish (MMIfaceModem  *self,
                      GAsyncResult  *res,
                      GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

static void
broadband_bearer_quectel_ec25_new_ready (GObject      *unused,
                                         GAsyncResult *res,
                                         GTask        *task)
{
    MMBaseBearer *bearer;
    GError       *error = NULL;

    bearer = mm_broadband_bearer_quectel_ec25_new_finish (res, &error);
    if (!bearer)
        g_task_return_error (task, error);
    else
        g_task_return_pointer (task, bearer, g_object_unref);
    g_object_unref (task);
}

static void
create_bearer (MMIfaceModem        *_self,
               MMBearerProperties  *properties,
               GAsyncReadyCallback  callback,
               gpointer             user_data)
{
    MMBroadbandModemQuectelEc25 *self = MM_BROADBAND_MODEM_QUECTEL_EC25 (_self);
    GTask                       *task;

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, g_object_ref (properties), g_object_unref);

    mm_broadband_bearer_quectel_ec25_new (MM_BROADBAND_MODEM_QUECTEL_EC25 (self),
                                          g_task_get_task_data (task),
                                          NULL, /* cancellable */
                                          (GAsyncReadyCallback)broadband_bearer_quectel_ec25_new_ready,
                                          task);
}

/*****************************************************************************/
/* Supported bands (Modem interface) */

static GArray *
load_supported_bands_finish (MMIfaceModem  *self,
                             GAsyncResult  *res,
                             GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

typedef struct {
    guint64      flag;
    char        *num_str;
    MMModemBand  mm_band;
} QuectelEc25Band;

/* WCDMA */
static const QuectelEc25Band quectel_ec25_wcdma_bands[] = {
    { 0x00000010, "2100", MM_MODEM_BAND_UTRAN_1  },
    { 0x00000100,  "800", MM_MODEM_BAND_UTRAN_6  },
    { 0x00000080,  "900", MM_MODEM_BAND_UTRAN_8  },
    { 0x00000800,  "800", MM_MODEM_BAND_UTRAN_19 },
};

/* LTE */
static const QuectelEc25Band quectel_ec25_lte_bands[] = {
    { 0x00000000001,  "1", MM_MODEM_BAND_EUTRAN_1  },
    { 0x00000000004,  "3", MM_MODEM_BAND_EUTRAN_3  },
    { 0x00000000080,  "8", MM_MODEM_BAND_EUTRAN_8  },
    { 0x00000020000, "18", MM_MODEM_BAND_EUTRAN_18 },
    { 0x00000040000, "19", MM_MODEM_BAND_EUTRAN_19 },
    { 0x00002000000, "26", MM_MODEM_BAND_EUTRAN_26 },
    { 0x10000000000, "41", MM_MODEM_BAND_EUTRAN_41 },
};

static void
qcfg_band_ready (MMBaseModem  *self,
                 GAsyncResult *res,
                 GTask        *task)
{
    const gchar *response;
    GError      *error = NULL;
    GArray      *bands = NULL;
    guint64      bands_wcdma;
    guint64      bands_lte;
    guint64      bands_tds;
    guint        i;

    response = mm_base_modem_at_command_finish (self, res, &error);
    if (!response) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    sscanf (response, "+QCFG: \"band\",%llx,%llx,%llx",&bands_wcdma, &bands_lte, &bands_tds);

    /* WCDMA */
    for (i = 0; i < G_N_ELEMENTS (quectel_ec25_wcdma_bands); i++) {
        if (bands_wcdma & quectel_ec25_wcdma_bands[i].flag) {
            if (!bands)
                bands = g_array_sized_new (FALSE, FALSE, sizeof (MMModemBand), 11);
            g_array_append_val (bands, quectel_ec25_wcdma_bands[i].mm_band);
        }
    }

    /* LTE */
    for (i = 0; i < G_N_ELEMENTS (quectel_ec25_lte_bands); i++) {
        if (bands_lte & quectel_ec25_lte_bands[i].flag) {
            if (!bands)
                bands = g_array_sized_new (FALSE, FALSE, sizeof (MMModemBand), 7);
            g_array_append_val (bands, quectel_ec25_lte_bands[i].mm_band);
        }
    }

    if (!bands) {
        /* No bands */
        g_task_return_error (task,
                             g_error_new (MM_CORE_ERROR, MM_CORE_ERROR_FAILED, "No Bands"));
    } else
        g_task_return_pointer (task, bands, (GDestroyNotify)g_array_unref);

    g_object_unref (task);
}

static void
load_supported_bands (MMIfaceModem        *self,
                      GAsyncReadyCallback  callback,
                      gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);
    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "AT+QCFG=\"band\"",
                              3,
                              FALSE,
                              (GAsyncReadyCallback)qcfg_band_ready,
                              task);
}

/*****************************************************************************/
/* Load current bands (Modem interface) */

static GArray *
load_current_bands_finish (MMIfaceModem  *self,
                           GAsyncResult  *res,
                           GError       **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

static void
get_band_ready (MMBaseModem  *self,
                GAsyncResult *res,
                GTask        *task)
{
    const gchar *response;
    g_autoptr(GRegex)      r = NULL;
    g_autoptr(GMatchInfo)  match_info = NULL;
    g_autoptr(GError)      inner_error = NULL;
    g_autofree gchar      *band_str = NULL;
    GError                *error = NULL;
    GArray                *bands = NULL;
    guint                  i;

    response = mm_base_modem_at_command_finish (self, res, &error);
    if (!response) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }


    r = g_regex_new ("\\+QNWINFO:\\s*\"([A-Za-z0-9 ]+)\",\"([A-Za-z0-9 ]+)\",\"([A-Za-z0-9 ]+)\",(\\d+)",
                     0,
                     0,
                     NULL);

    if (!g_regex_match_full (r, response, strlen (response), 0, 0, &match_info, &inner_error))
        goto no_bands_error;

    if (!g_match_info_matches (match_info))
        goto no_bands_error;

    band_str = g_match_info_fetch (match_info ,3);

    if (!band_str)
        goto no_bands_error;

    if (strstr (band_str, "WCDMA")) {
        for (i = 0; i < G_N_ELEMENTS (quectel_ec25_wcdma_bands); i++) {
            if (strstr (band_str, quectel_ec25_wcdma_bands[i].num_str)) {
                if (!bands)
                    bands = g_array_sized_new (FALSE, FALSE, sizeof (MMModemBand), 2);
                g_array_append_val (bands, quectel_ec25_wcdma_bands[i].mm_band);
            }
        }
    } else if (strstr (band_str, "LTE BAND")) {
        i = G_N_ELEMENTS (quectel_ec25_lte_bands) - 1;
        while (1) {
            if (strstr (band_str, quectel_ec25_lte_bands[i].num_str)) {
                if (!bands)
                    bands = g_array_sized_new (FALSE, FALSE, sizeof (MMModemBand), 1);
                g_array_append_val (bands, quectel_ec25_lte_bands[i].mm_band);
                break;
            }
            if (i == 0)
                break;
            i--;
        }
    }

    if (!bands)
        goto no_bands_error;

    g_task_return_pointer (task, bands, (GDestroyNotify) g_array_unref);
    g_object_unref (task);
    return;

no_bands_error:
    g_task_return_error (task,
                         g_error_new (MM_CORE_ERROR, MM_CORE_ERROR_FAILED, "No Bands"));
    g_object_unref (task);
}

static void
load_current_bands (MMIfaceModem        *self,
                    GAsyncReadyCallback  callback,
                    gpointer             user_data)
{
    GTask *task;

    task = g_task_new (self, NULL, callback, user_data);

    mm_base_modem_at_command (MM_BASE_MODEM (self),
                              "+QNWINFO",
                              10,
                              FALSE,
                              (GAsyncReadyCallback)get_band_ready,
                              task);
}

/*****************************************************************************/
MMBroadbandModemQuectelEc25 *
mm_broadband_modem_quectel_ec25_new (const gchar  *device,
                                     const gchar *physdev,
                                     const gchar **drivers,
                                     const gchar  *plugin,
                                     guint16       vendor_id,
                                     guint16       product_id)
{
    return g_object_new (MM_TYPE_BROADBAND_MODEM_QUECTEL_EC25,
                         MM_BASE_MODEM_DEVICE, device,
                         MM_BASE_MODEM_PHYSDEV, physdev,
                         MM_BASE_MODEM_DRIVERS, drivers,
                         MM_BASE_MODEM_PLUGIN, plugin,
                         MM_BASE_MODEM_VENDOR_ID, vendor_id,
                         MM_BASE_MODEM_PRODUCT_ID, product_id,
                         /* Generic bearer supports TTY only */
                         MM_BASE_MODEM_DATA_NET_SUPPORTED, FALSE,
                         MM_BASE_MODEM_DATA_TTY_SUPPORTED, TRUE,
                         MM_IFACE_MODEM_SIM_HOT_SWAP_SUPPORTED,  TRUE,
                         NULL);
}

static void
mm_broadband_modem_quectel_ec25_init (MMBroadbandModemQuectelEc25 *self)
{
}

static void
iface_modem_init (MMIfaceModem *iface)
{
    iface_modem_parent = g_type_interface_peek_parent (iface);

    iface->create_bearer = create_bearer;
    iface->create_bearer_finish = create_bearer_finish;
    iface->load_supported_bands = load_supported_bands;
    iface->load_supported_bands_finish = load_supported_bands_finish;
    iface->load_current_bands = load_current_bands;
    iface->load_current_bands_finish = load_current_bands_finish;
    iface->setup_sim_hot_swap = mm_shared_quectel_setup_sim_hot_swap;
    iface->setup_sim_hot_swap_finish = mm_shared_quectel_setup_sim_hot_swap_finish;
    iface->cleanup_sim_hot_swap = mm_shared_quectel_cleanup_sim_hot_swap;
    iface->load_unlock_required = modem_load_unlock_required;
    iface->load_unlock_required_finish = modem_load_unlock_required_finish;
    iface->load_supported_ip_families = modem_load_supported_ip_families;
    iface->load_supported_ip_families_finish = modem_load_supported_ip_families_finish;
}

static MMIfaceModem *
peek_parent_modem_interface (MMSharedQuectel *self)
{
    return iface_modem_parent;
}

static MMBroadbandModemClass *
peek_parent_broadband_modem_class (MMSharedQuectel *self)
{
    return MM_BROADBAND_MODEM_CLASS (mm_broadband_modem_quectel_ec25_parent_class);
}

static void
iface_modem_3gpp_init (MMIfaceModem3gpp *iface)
{
    iface_modem_3gpp_parent = g_type_interface_peek_parent (iface);
    iface->run_registration_checks = run_registration_checks;
    iface->run_registration_checks_finish = run_registration_checks_finish;
    iface->register_in_network = register_in_network;
    iface->register_in_network_finish = register_in_network_finish;
}

static void
iface_modem_3gpp_profile_manager_init (MMIfaceModem3gppProfileManager *iface)
{
    /* Initialization steps */
    iface->check_support = modem_3gpp_profile_manager_check_support;
    iface->check_support_finish = modem_3gpp_profile_manager_check_support_finish;

    /* User actions */
    iface->list_profiles = modem_3gpp_profile_manager_list_profiles;
    iface->list_profiles_finish = modem_3gpp_profile_manager_list_profiles_finish;
    iface->delete_profile = modem_3gpp_profile_manager_delete_profile;
    iface->delete_profile_finish = modem_3gpp_profile_manager_delete_profile_finish;
    iface->check_format = modem_3gpp_profile_manager_check_format;
    iface->check_format_finish = modem_3gpp_profile_manager_check_format_finish;
    iface->check_activated_profile = modem_3gpp_profile_manager_check_activated_profile;
    iface->check_activated_profile_finish = modem_3gpp_profile_manager_check_activated_profile_finish;
    iface->deactivate_profile = modem_3gpp_profile_manager_deactivate_profile;
    iface->deactivate_profile_finish = modem_3gpp_profile_manager_deactivate_profile_finish;
    iface->store_profile = modem_3gpp_profile_manager_store_profile;
    iface->store_profile_finish = modem_3gpp_profile_manager_store_profile_finish;
}

static void
iface_modem_firmware_init (MMIfaceModemFirmware *iface)
{
    iface->load_update_settings = mm_shared_quectel_firmware_load_update_settings;
    iface->load_update_settings_finish = mm_shared_quectel_firmware_load_update_settings_finish;
}

static void
iface_modem_location_init (MMIfaceModemLocation *iface)
{
    iface_modem_location_parent = g_type_interface_peek_parent (iface);

    iface->load_capabilities                 = mm_shared_quectel_location_load_capabilities;
    iface->load_capabilities_finish          = mm_shared_quectel_location_load_capabilities_finish;
    iface->enable_location_gathering         = mm_shared_quectel_enable_location_gathering;
    iface->enable_location_gathering_finish  = mm_shared_quectel_enable_location_gathering_finish;
    iface->disable_location_gathering        = mm_shared_quectel_disable_location_gathering;
    iface->disable_location_gathering_finish = mm_shared_quectel_disable_location_gathering_finish;
}

static MMIfaceModemLocation *
peek_parent_modem_location_interface (MMSharedQuectel *self)
{
    return iface_modem_location_parent;
}

static void
iface_modem_time_init (MMIfaceModemTime *iface)
{
    iface->check_support        = mm_shared_quectel_time_check_support;
    iface->check_support_finish = mm_shared_quectel_time_check_support_finish;
}

static void
shared_quectel_init (MMSharedQuectel *iface)
{
    iface->peek_parent_modem_interface          = peek_parent_modem_interface;
    iface->peek_parent_modem_location_interface = peek_parent_modem_location_interface;
    iface->peek_parent_broadband_modem_class    = peek_parent_broadband_modem_class;
}

static void
mm_broadband_modem_quectel_ec25_class_init (MMBroadbandModemQuectelEc25Class *klass)
{
    MMBroadbandModemClass *broadband_modem_class = MM_BROADBAND_MODEM_CLASS (klass);

    broadband_modem_class->setup_ports = mm_shared_quectel_setup_ports;
}
