/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Copyright (C) 2022 Atmark Techno, Inc.
 */

#include <string.h>
#include <gmodule.h>

#define _LIBMM_INSIDE_MM
#include <libmm-glib.h>

#include "mm-plugin-common.h"
#include "mm-broadband-modem-cinterion-els31.h"
#include "mm-serial-parsers.h"

#include "mm-log-object.h"

#if defined WITH_QMI
#include "mm-broadband-modem-qmi-cinterion.h"
#endif

#if defined WITH_MBIM
#include "mm-broadband-modem-mbim-cinterion.h"
#endif

#define MM_TYPE_PLUGIN_CINTERION_ELS31 mm_plugin_cinterion_els31_get_type ()
MM_DEFINE_PLUGIN (CINTERION_ELS31, cinterion_els31, CinterionEls31)

/*****************************************************************************/

static MMBaseModem *
create_modem (MMPlugin     *self,
              const gchar  *uid,
              const gchar  *physdev,
              const gchar **drivers,
              guint16       vendor,
              guint16       product,
              guint16       subsystem_vendor,
              GList        *probes,
              GError      **error)
{
#if defined WITH_QMI
    if (mm_port_probe_list_has_qmi_port (probes)) {
        mm_obj_dbg (self, "QMI-powered Cinterion modem found...");
        return MM_BASE_MODEM (mm_broadband_modem_qmi_new (uid,
                                                          physdev,
                                                          drivers,
                                                          mm_plugin_get_name (self),
                                                          vendor,
                                                          product));
    }
#endif

#if defined WITH_MBIM
    if (mm_port_probe_list_has_mbim_port (probes)) {
        mm_obj_dbg (self, "MBIM-powered Cinterion modem found...");
        return MM_BASE_MODEM (mm_broadband_modem_mbim_new (uid,
                                                           physdev,
                                                           drivers,
                                                           mm_plugin_get_name (self),
                                                           vendor,
                                                           product));
    }
#endif

    return MM_BASE_MODEM (mm_broadband_modem_cinterion_els31_new (uid,
                                                                  physdev,
                                                                  drivers,
                                                                  mm_plugin_get_name (self),
                                                                  vendor,
                                                                  product));
}

/*****************************************************************************/
/* grab port */

static gboolean
grab_port (MMPlugin     *self,
           MMBaseModem  *modem,
           MMPortProbe  *probe,
           GError      **error)
{
    MMPortSerialAtFlag  pflags = MM_PORT_SERIAL_AT_FLAG_NONE;
    MMKernelDevice     *pdev;
    MMPortType          ptype;
    const gchar        *name = mm_port_probe_get_port_name (probe);

    ptype = mm_port_probe_get_port_type (probe);
    pdev = mm_port_probe_peek_port (probe);

    if (mm_kernel_device_has_property (pdev,
                                       "ID_MM_CINTERION_ELS31_TTY_AT_SYMLINK")) {
        name = mm_kernel_device_get_property (pdev,
                                              "ID_MM_CINTERION_ELS31_TTY_AT_SYMLINK");
        mm_obj_dbg (self, "[%s] symlink name", name);
        return mm_base_modem_grab_symlink_at_primary_port (modem, pdev, name, error);
    } else if (mm_kernel_device_get_global_property_as_boolean (pdev,
                                                                "ID_MM_CINTERION_ELS31_PORT_TYPE_AT")) {
        mm_obj_dbg (self, "(%s/%s)' Port flagged as primary",
                    mm_port_probe_get_port_subsys (probe),
                    mm_port_probe_get_port_name (probe));
        pflags = MM_PORT_SERIAL_AT_FLAG_PRIMARY;
    }

    return mm_base_modem_grab_port (modem, pdev, ptype, pflags, error);
}

/*****************************************************************************/
/* Custom init */

typedef struct {
    guint cscs_query_retries;
} CustomInitContext;

static const guint CSCS_QUERY_RETRY_MAX = 10;

static void
custom_init_context_free (CustomInitContext *ctx)
{
    g_slice_free (CustomInitContext, ctx);
}

static gboolean
cinterion_els31_custom_init_finish (MMPortProbe   *probe,
                                    GAsyncResult  *result,
                                    GError       **error)
{
    return g_task_propagate_boolean (G_TASK (result), error);
}

static void send_query_cscs (MMPortSerialAt *port, GTask *task);

static void
cscs_ready (MMPortSerialAt *port,
            GAsyncResult   *res,
            GTask          *task)
{
    MMPortProbe       *probe;
    g_autoptr(GError)  error = NULL;

    probe = g_task_get_source_object (task);

    mm_port_serial_at_command_finish (port, res, &error);
    if (error)
        mm_obj_warn (probe, "failed to AT+CSCS=\"GSM\": %s", error->message);

    mm_port_probe_set_result_at (probe, TRUE);
    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
query_cscs_ready (MMPortSerialAt *port,
                  GAsyncResult   *res,
                  GTask          *task)
{
    MMPortProbe       *probe;
    g_autoptr(GError)  error = NULL;
    const gchar       *response = NULL;
    CustomInitContext *ctx;

    probe = g_task_get_source_object (task);
    ctx = g_task_get_task_data (task);

    response = mm_port_serial_at_command_finish (port, res, &error);
    if (error) {
        mm_obj_dbg (probe, "failed to AT+CSCS?: %s", error->message);
        if ( ctx->cscs_query_retries < CSCS_QUERY_RETRY_MAX) {
            /* Retry to collect rubbish URCs at start-up ELS31 */
            mm_obj_dbg (probe, "retry send AT+CSCS?");
            ctx->cscs_query_retries++;
            g_usleep (500000); /* 500 msec */
            send_query_cscs (port, task);
            return;
        }
    } else if (response && strstr (response, "GSM")) {
        /* already set GSM */
        mm_port_probe_set_result_at (probe, TRUE);
        g_task_return_boolean (task, TRUE);
        g_object_unref (task);
        return;
    }

    mm_port_serial_at_command (port,
                               "AT+CSCS=\"GSM\"",
                               6,
                               FALSE,
                               FALSE,
                               g_task_get_cancellable (task),
                               (GAsyncReadyCallback)cscs_ready,
                               task);
}

static void
send_query_cscs (MMPortSerialAt *port,
                 GTask          *task)
{
    mm_port_serial_at_command (port,
                               "AT+CSCS?",
                               6,
                               FALSE,
                               FALSE,
                               g_task_get_cancellable (task),
                               (GAsyncReadyCallback)query_cscs_ready,
                               task);
}

static void
cinterion_els31_custom_init (MMPortProbe         *probe,
                             MMPortSerialAt      *port,
                             GCancellable        *cancellable,
                             GAsyncReadyCallback  callback,
                             gpointer             user_data)
{
    GTask             *task;
    CustomInitContext *ctx;

    task = g_task_new (probe, cancellable, callback, user_data);
    g_task_set_check_cancellable (task, FALSE);
    ctx = g_slice_new0 (CustomInitContext);
    g_task_set_task_data (task, ctx, (GDestroyNotify) custom_init_context_free);

    send_query_cscs (port, task);
}

/*****************************************************************************/

G_MODULE_EXPORT MMPlugin *
mm_plugin_create_cinterion_els31 (void)
{
    static const gchar *subsystems[] = { "tty", "net", "usb", NULL };
    static const mm_uint16_pair products[] = { { 0x1e2d, 0x00a0 }, /* ELS31-J */
                                               { 0, 0 } };
    static const MMAsyncMethod custom_init = {
        .async  = G_CALLBACK (cinterion_els31_custom_init),
        .finish = G_CALLBACK (cinterion_els31_custom_init_finish),
    };

    return MM_PLUGIN (
        g_object_new (MM_TYPE_PLUGIN_CINTERION_ELS31,
                      MM_PLUGIN_NAME,                MM_MODULE_NAME,
                      MM_PLUGIN_ALLOWED_SUBSYSTEMS,  subsystems,
                      MM_PLUGIN_ALLOWED_PRODUCT_IDS, products,
                      MM_PLUGIN_ALLOWED_AT,          TRUE,
                      MM_PLUGIN_CUSTOM_INIT,         &custom_init,
                      NULL));
}

static void
mm_plugin_cinterion_els31_init (MMPluginCinterionEls31 *self)
{
}

static void
mm_plugin_cinterion_els31_class_init (MMPluginCinterionEls31Class *klass)
{
    MMPluginClass *plugin_class = MM_PLUGIN_CLASS (klass);

    plugin_class->create_modem = create_modem;
    plugin_class->grab_port = grab_port;
}
