/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2022 Atmark Techno Inc.
 */

#include <ModemManager.h>
#include "mm-broadband-bearer-cinterion-ems31.h"
#include "mm-base-modem-at.h"
#include "mm-log.h"

G_DEFINE_TYPE (MMBroadbandBearerCinterionEms31, mm_broadband_bearer_cinterion_ems31, MM_TYPE_BROADBAND_BEARER)

/*****************************************************************************/
/* Generic implementations (both 3GPP and CDMA) are always AT-port based */

static MMPortSerialAt *
common_get_at_data_port (MMBroadbandBearer *self,
                         MMBaseModem       *modem,
                         GError           **error)
{
    MMPort *data;

    /* Look for best data port, NULL if none available. */
    data = mm_base_modem_peek_best_data_port (modem, MM_PORT_TYPE_AT);
    if (!data) {
        /* It may happen that the desired data port grabbed during probing was
         * actually a 'net' port, which the generic logic cannot handle, so if
         * that is the case, and we have no AT data ports specified, just
         fallback to the primary AT port. */
        data = (MMPort *) mm_base_modem_peek_port_primary (modem);
    }

    if (!mm_port_serial_open (MM_PORT_SERIAL (data), error)) {
        g_prefix_error (error, "Couldn't connect: cannot keep data port open.");
        return NULL;
    }

    mm_obj_dbg (self, "connection through a plain serial AT port: %s", mm_port_get_device (data));

    return MM_PORT_SERIAL_AT (g_object_ref (data));
}

/*****************************************************************************/
/* set profile id unknown, to reuse the same bearer */

static void
set_profile_id_unknown (MMBaseBearer *base_bearer)
{
    MMBearerProperties *properties;
    MM3gppProfile      *profile;

    properties = mm_base_bearer_peek_config (base_bearer);
    if (properties) {
        profile = mm_bearer_properties_peek_3gpp_profile (properties);
        if (profile) {
            mm_3gpp_profile_set_profile_id (profile, MM_3GPP_PROFILE_ID_UNKNOWN);
        }
    }
}

/*****************************************************************************/
/* 3GPP disconnect */
typedef struct {
    MMBaseModem    *modem;
    MMPortSerialAt *primary;
    MMPort         *data;
} Disconnect3gppContext;

static gboolean
disconnect_3gpp_finish (MMBroadbandBearer  *self,
                        GAsyncResult       *res,
                        GError            **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
disconnect_3gpp_context_free (Disconnect3gppContext *ctx)
{
    g_object_unref (ctx->data);
    g_object_unref (ctx->primary);
    g_object_unref (ctx->modem);
    g_free (ctx);
}

static Disconnect3gppContext *
disconnect_3gpp_context_new (MMBroadbandModem *modem,
                             MMPortSerialAt   *primary,
                             MMPort           *data)
{
    Disconnect3gppContext *ctx;

    ctx = g_new0 (Disconnect3gppContext, 1);
    ctx->modem = MM_BASE_MODEM (g_object_ref (modem));
    ctx->primary = g_object_ref (primary);
    ctx->data = g_object_ref (data);

    return ctx;
}

static void
data_flash_3gpp_ready (MMPortSerial *data,
                       GAsyncResult *res,
                       GTask        *task)
{
    MMBroadbandBearer *self;
    GError            *error = NULL;

    self = g_task_get_source_object (task);

    mm_port_serial_flash_finish (data, res, &error);

    /* We kept the serial port open during connection, now we close that open
     * count */
    mm_port_serial_close (data);

    /* Port is disconnected; update the state */
    mm_port_set_connected (MM_PORT (data), FALSE);

    if (error) {
        /* Ignore "NO CARRIER" response when modem disconnects and any flash
         * failures we might encounter. Other errors are hard errors.
         */
        if (!g_error_matches (error,
                              MM_CONNECTION_ERROR,
                              MM_CONNECTION_ERROR_NO_CARRIER) &&
            !g_error_matches (error,
                              MM_SERIAL_ERROR,
                              MM_SERIAL_ERROR_FLASH_FAILED)) {
            /* Fatal */
            g_task_return_error (task, error);
            g_object_unref (task);
            return;
        }

        mm_obj_dbg (self, "port flashing failed (not fatal): %s", error->message);
        g_error_free (error);
    }

    /* Run init port sequence in the data port */
    mm_port_serial_at_run_init_sequence (MM_PORT_SERIAL_AT (data));

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
data_reopen_3gpp_ready (MMPortSerial *data,
                        GAsyncResult *res,
                        GTask        *task)
{
    MMBroadbandBearer     *self;
    Disconnect3gppContext *ctx;
    GError                *error = NULL;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    g_object_set (data, MM_PORT_SERIAL_AT_INIT_SEQUENCE_ENABLED, TRUE, NULL);

    if (!mm_port_serial_reopen_finish (data, res, &error)) {
        /* Fatal */
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    set_profile_id_unknown (MM_BASE_BEARER (self));

    /* Just flash the data port */
    mm_obj_dbg (self, "flashing data port %s...", mm_port_get_device (MM_PORT (ctx->data)));
    mm_port_serial_flash (MM_PORT_SERIAL (ctx->data),
                          1000,
                          TRUE,
                          (GAsyncReadyCallback)data_flash_3gpp_ready,
                          task);
}

static void
data_reopen_3gpp (GTask *task)
{
    MMBroadbandBearer     *self;
    Disconnect3gppContext *ctx;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    /* We don't want to run init sequence right away during the reopen, as we're
     * going to flash afterwards. */
    g_object_set (ctx->data, MM_PORT_SERIAL_AT_INIT_SEQUENCE_ENABLED, FALSE, NULL);

    /* Fully reopen the port before flashing */
    mm_obj_dbg (self, "reopening data port %s...", mm_port_get_device (MM_PORT (ctx->data)));
    mm_port_serial_reopen (MM_PORT_SERIAL (ctx->data),
                           1000,
                           (GAsyncReadyCallback)data_reopen_3gpp_ready,
                           task);
}

static void
disconnect_3gpp (MMBroadbandBearer   *self,
                 MMBroadbandModem    *modem,
                 MMPortSerialAt      *primary,
                 MMPortSerialAt      *secondary,
                 MMPort              *data,
                 guint                cid,
                 GAsyncReadyCallback  callback,
                 gpointer             user_data)
{
    Disconnect3gppContext *ctx;
    GTask                 *task;

    g_assert (primary != NULL);

    ctx = disconnect_3gpp_context_new (modem, primary, data);

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, ctx, (GDestroyNotify)disconnect_3gpp_context_free);

    data_reopen_3gpp (task);
}

/******************************************************************************/
/* Dial 3GPP */

typedef enum {
    DIAL_3GPP_CONTEXT_STEP_FIRST = 0,
    DIAL_3GPP_CONTEXT_STEP_QUERY_PDP_CONTEXT,
    DIAL_3GPP_CONTEXT_STEP_QUERY_AUTH,
    DIAL_3GPP_CONTEXT_STEP_CFUN_ZERO,
    DIAL_3GPP_CONTEXT_STEP_PDP_CONTEXT,
    DIAL_3GPP_CONTEXT_STEP_AUTH,
    DIAL_3GPP_CONTEXT_STEP_CFUN_ONE,
    DIAL_3GPP_CONTEXT_STEP_OPERATOR_SELECTION,
    DIAL_3GPP_CONTEXT_STEP_LAST,
} Dial3gppContextStep;

static const guint OP_RETRY_MAX = 5;

typedef struct {
    MMBroadbandBearerCinterionEms31 *self;
    MMBaseModem                     *modem;
    MMPortSerialAt                  *primary;
    guint                            cid;
    guint                            op_retry; /* operator selection retry */
    MMPort                          *data;
    Dial3gppContextStep              step;
    GError                          *saved_error;
    MMPortSerialAt                  *dial_port;
    gboolean                         is_needed_update_pdp_context;
    gboolean                         is_needed_update_user_password;
} Dial3gppContext;

static void
dial_3gpp_context_free (Dial3gppContext *ctx)
{
    g_object_unref (ctx->modem);
    g_object_unref (ctx->self);
    g_object_unref (ctx->primary);
    if (ctx->saved_error)
        g_error_free (ctx->saved_error);
    if (ctx->dial_port)
        g_object_unref (ctx->dial_port);
    g_slice_free (Dial3gppContext, ctx);
}

static MMPort *
dial_3gpp_finish (MMBroadbandBearer  *self,
                  GAsyncResult       *res,
                  GError            **error)
{
    return MM_PORT (g_task_propagate_pointer (G_TASK (res), error));
}

static void
dial_3gpp_error (GTask *task, GError *error)
{
    Dial3gppContext    *ctx;

    ctx = g_task_get_task_data (task);

    set_profile_id_unknown (MM_BASE_BEARER (ctx->self));

    g_task_return_error (task, error);
    g_object_unref (task);
}

static void dial_3gpp_context_step (GTask *task);

static void
extended_error_ready (MMBaseModem *modem,
                      GAsyncResult *res,
                      GTask *task)
{
    Dial3gppContext *ctx;
    const gchar *result;
    GError *error = NULL;

    ctx = g_task_get_task_data (task);

    /* Close the dialling port as we got an error */
    mm_port_serial_close (MM_PORT_SERIAL (ctx->dial_port));

    /* If cancelled, complete */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    result = mm_base_modem_at_command_full_finish (modem, res, NULL);
    if (result &&
        g_str_has_prefix (result, "+CEER: ") &&
        strlen (result) > 7) {
        error = g_error_new (ctx->saved_error->domain,
                             ctx->saved_error->code,
                             "%s",
                             &result[7]);
        g_error_free (ctx->saved_error);
    } else
        g_propagate_error (&error, ctx->saved_error);

    ctx->saved_error = NULL;

    /* Done with errors */
    dial_3gpp_error (task, error);
}

static void
atd_ready (MMBaseModem *modem,
           GAsyncResult *res,
           GTask *task)
{
    MMBroadbandBearerCinterionEms31 *self;
    Dial3gppContext                 *ctx;
    GError                          *error = NULL;

    self = g_task_get_source_object (task);
    ctx = g_task_get_task_data (task);

    /* DO NOT check for cancellable here. If we got here without errors, the
     * bearer is really connected and therefore we need to reflect that in
     * the state machine. */
    mm_base_modem_at_command_full_finish (modem, res, &ctx->saved_error);

    if (ctx->saved_error) {
        mm_obj_warn (self, "ATD response has error %s: %s",
                     mm_port_get_device (MM_PORT (ctx->dial_port)),
                     ctx->saved_error->message);

        /* Try to get more information why it failed */
        mm_base_modem_at_command_full (ctx->modem,
                                       ctx->primary,
                                       "+CEER",
                                       3,
                                       FALSE,
                                       FALSE, /* raw */
                                       NULL, /* cancellable */
                                       (GAsyncReadyCallback)extended_error_ready,
                                       task);
        return;
    }

    /* Configure flow control to use while connected */
    if (!mm_port_serial_set_flow_control (MM_PORT_SERIAL (ctx->dial_port),
                                          MM_FLOW_CONTROL_RTS_CTS,
                                          &error)) {
        mm_obj_warn (self, "couldn't set flow control settings in %s: %s",
                     mm_port_get_device (MM_PORT (ctx->dial_port)),
                     error->message);
        g_clear_error (&error);
    }

    /* The ATD command has succeeded, and therefore the TTY is in data mode now.
     * Instead of waiting for setting the port as connected later in
     * connect_succeeded(), we do it right away so that we stop our polling. */
    mm_port_set_connected (MM_PORT (ctx->dial_port), TRUE);

    g_task_return_pointer (task,
                           g_object_ref (ctx->dial_port),
                           g_object_unref);
    g_object_unref (task);
}

static void
atd (GTask *task)
{
    gchar           *command;
    Dial3gppContext *ctx;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    /* Use default *99 to connect */
    command = g_strdup_printf ("ATD*99***%d#", ctx->cid);
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->dial_port,
                                   command,
                                   MM_BASE_BEARER_DEFAULT_CONNECTION_TIMEOUT,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL, /* cancellable */
                                   (GAsyncReadyCallback)atd_ready,
                                   task);
    g_free (command);
}

static void
cfun_one_and_out_ready (MMBaseModem  *modem,
                        GAsyncResult *res,
                        GTask        *task)
{
    Dial3gppContext *ctx;
    GError          *error = NULL;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    if (!mm_base_modem_at_command_full_finish (modem, res, &error)) {
        dial_3gpp_error (task, error);
        return;
    }

    dial_3gpp_error (task, g_steal_pointer(&ctx->saved_error));
}

static void
cfun_one_and_out (GTask *task)
{
    Dial3gppContext *ctx;

    ctx  = g_task_get_task_data (task);

    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+CFUN=1",
                                   5,
                                   FALSE,
                                   FALSE,
                                   NULL,
                                   (GAsyncReadyCallback) cfun_one_and_out_ready,
                                   task);
}

static void
common_dial_goto_next_step (GTask *task)
{
    Dial3gppContext *ctx;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    ctx->step++;
    dial_3gpp_context_step (task);
}

static void
common_dial_operation_ready (MMBaseModem  *modem,
                             GAsyncResult *res,
                             GTask        *task)
{
    GError *error = NULL;

    if (!mm_base_modem_at_command_full_finish (modem, res, &error)) {
        dial_3gpp_error (task, error);
        return;
    }

    common_dial_goto_next_step (task);
}

static void
common_dial_operation_ready_needs_cfun_one (MMBaseModem  *modem,
                                            GAsyncResult *res,
                                            GTask        *task)
{
    Dial3gppContext *ctx;
    GError          *error = NULL;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    if (!mm_base_modem_at_command_full_finish (modem, res, &error)) {
        ctx->saved_error = error;
        cfun_one_and_out (task);
        return;
    }

    g_usleep (5 * G_USEC_PER_SEC);

    common_dial_goto_next_step (task);
}

static void
pdp_context_query_ready (MMBaseModem  *modem,
                         GAsyncResult *res,
                         GTask        *task)
{
    Dial3gppContext   *ctx;
    GList             *pdp_list;
    g_autoptr(GError)  error = NULL;
    const gchar       *response;
    GList             *l;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    ctx->is_needed_update_pdp_context = TRUE;

    response = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!response) {
        common_dial_goto_next_step (task);
        return;
    }

    pdp_list = mm_3gpp_parse_cgdcont_read_response (response, &error);
    if (!pdp_list) {
        common_dial_goto_next_step (task);
        return;
    }

    /* Look for the exact PDP context we want */
    for (l = pdp_list; l; l = g_list_next (l)) {
        MM3gppPdpContext *pdp = l->data;
        /* We can use only cid=1 */
        if (pdp->cid == 1) {
            /* PDP with no APN set? we may use that one if not exact match found */
            if (!pdp->apn || !pdp->apn[0])
                mm_obj_dbg (ctx->self, "Found PDP context with CID %u and no APN", pdp->cid);
            else {
                const gchar *apn;

                apn = mm_bearer_properties_get_apn (mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self)));
                if (apn && !g_ascii_strcasecmp (pdp->apn, apn)) {
                    if (pdp->pdp_type ==
                            mm_bearer_properties_get_ip_type(
                                mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self)))) {
                        g_autofree gchar *ip_family_str = NULL;
                        /* Found a PDP context with the same CID and PDP type, we'll use it. */
                        ip_family_str = mm_bearer_ip_family_build_string_from_mask (pdp->pdp_type);
                        mm_obj_dbg (ctx->self, "Found PDP context with CID %u and PDP type %s for APN '%s'",
                                    pdp->cid, ip_family_str, pdp->apn);
                        ctx->is_needed_update_pdp_context = FALSE;
                    }
                }
            }
            break;
        }
    }
    mm_3gpp_pdp_context_list_free (pdp_list);
    common_dial_goto_next_step (task);
}

static void
cgauth_query_ready (MMBaseModem  *modem,
                    GAsyncResult *res,
                    GTask        *task)
{
    Dial3gppContext     *ctx;
    g_autoptr(GError)    error = NULL;
    const gchar         *response;
    MMBearerAllowedAuth  current_auth;
    gchar               *current_user;
    gchar               *current_password;
    MMBearerProperties  *config;
    MMBearerAllowedAuth  auth;
    const gchar         *user;
    gboolean             has_user;
    const gchar         *password;
    gboolean             has_password;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    ctx->is_needed_update_user_password = TRUE;

    response = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!response) {
        common_dial_goto_next_step (task);
        return;
    }

    if (!mm_cinterion_parse_cgauth_response (response, ctx->cid, &current_auth,
                                              &current_user, &current_password, &error)) {
        common_dial_goto_next_step (task);
        return;
    }

    config = mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self));
    if (!config) {
        common_dial_goto_next_step (task);
        return;
    }

    auth     = mm_bearer_properties_get_allowed_auth (config);
    user     = mm_bearer_properties_get_user (config);
    password = mm_bearer_properties_get_password (config);

    has_user     = (user     && user[0]);
    has_password = (password && password[0]);

    if (auth == MM_BEARER_ALLOWED_AUTH_UNKNOWN) {
        if (has_user && has_password) {
            /* If user/password given, default to CHAP (more common than PAP) */
            auth = MM_BEARER_ALLOWED_AUTH_CHAP;
        } else {
            /* If user/password not given, default to none */
            auth = MM_BEARER_ALLOWED_AUTH_NONE;
        }
    }

    if (has_user && has_password) {
        if (auth & MM_BEARER_ALLOWED_AUTH_CHAP)
            auth = MM_BEARER_ALLOWED_AUTH_CHAP;
        else if (auth & MM_BEARER_ALLOWED_AUTH_PAP)
            auth = MM_BEARER_ALLOWED_AUTH_PAP;
    } else {
        if (auth & MM_BEARER_ALLOWED_AUTH_NONE)
            auth = MM_BEARER_ALLOWED_AUTH_NONE;
    }

    if (current_auth != auth) {
        common_dial_goto_next_step (task);
        return;
    }

    if (auth == MM_BEARER_ALLOWED_AUTH_NONE) {
        /* skip set AUTH */
        ctx->is_needed_update_user_password = FALSE;
        common_dial_goto_next_step (task);
        return;
    }

    if (g_strcmp0 (current_user, user) != 0) {
        common_dial_goto_next_step (task);
        return;
    }

    if (g_strcmp0 (current_password, password) != 0) {
        common_dial_goto_next_step (task);
        return;
    }

    /* skip set AUTH */
    ctx->is_needed_update_user_password = FALSE;
    common_dial_goto_next_step (task);
    return;
}

static void
cops_response_ready (MMBaseModem  *modem,
                     GAsyncResult *res,
                     GTask        *task)
{
    Dial3gppContext *ctx;
    GError          *error = NULL;
    const gchar     *response;
    gchar           *operator_code = NULL;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    response = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!response) {
        ctx->op_retry++;
        if (ctx->op_retry >= OP_RETRY_MAX) {
            mm_obj_dbg (modem, "not selected operator.");
            dial_3gpp_error (task, error);
            return;
        }
        mm_obj_dbg (modem, "retry %d/%d", ctx->op_retry, OP_RETRY_MAX);
        g_usleep (5 * G_USEC_PER_SEC);
        /* retry */
        dial_3gpp_context_step (task);
        return;
    }

    if (!mm_3gpp_parse_cops_read_response (response, NULL, NULL, &operator_code, NULL, modem, &error)) {
        ctx->op_retry++;
        if (ctx->op_retry >= OP_RETRY_MAX) {
            mm_obj_dbg (modem, "not selected operator.");
            dial_3gpp_error (task, error);
            return;
        }
        mm_obj_dbg (modem, "retry %d/%d", ctx->op_retry, OP_RETRY_MAX);
        g_usleep (5 * G_USEC_PER_SEC);
        /* retry */
        dial_3gpp_context_step (task);
        return;
    }

    mm_obj_dbg (modem, "loaded Operator Code: %s", operator_code);
    g_free(operator_code);
    /* Go to next step */
    ctx->step++;
    dial_3gpp_context_step (task);
}

static void
dial_3gpp_context_step (GTask *task)
{
    MMBroadbandBearerCinterionEms31 *self;
    Dial3gppContext                 *ctx;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    /* Check for cancellation */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    switch (ctx->step) {
    case DIAL_3GPP_CONTEXT_STEP_FIRST:
        ctx->step++;
        /* fall through */

    case DIAL_3GPP_CONTEXT_STEP_QUERY_PDP_CONTEXT:
        mm_obj_dbg (self, "dial step %u/%u: query pdp context...", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
        mm_base_modem_at_command_full (ctx->modem, ctx->primary,
                                       "+CGDCONT?", 3, FALSE, FALSE, NULL,
                                       (GAsyncReadyCallback) pdp_context_query_ready,
                                       task);
        break;

    case DIAL_3GPP_CONTEXT_STEP_QUERY_AUTH:
        mm_obj_dbg (self, "dial step %u/%u: query auth...", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
        mm_base_modem_at_command_full (ctx->modem, ctx->primary,
                                       "+CGAUTH?", 10, FALSE, FALSE, NULL,
                                       (GAsyncReadyCallback) cgauth_query_ready,
                                       task);
        break;

    case DIAL_3GPP_CONTEXT_STEP_CFUN_ZERO:
        if (ctx->is_needed_update_pdp_context || ctx->is_needed_update_user_password) {
            mm_obj_dbg (self, "dial step %u/%u: set cfun zero...", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
            mm_base_modem_at_command_full (ctx->modem, ctx->primary,
                                           "+CFUN=0", 5, FALSE, FALSE, NULL,
                                           (GAsyncReadyCallback) common_dial_operation_ready,
                                           task);
            return;
        }
        ctx->step++;
        /* fall through */

    case DIAL_3GPP_CONTEXT_STEP_PDP_CONTEXT:
        if (ctx->is_needed_update_pdp_context) {
            const gchar      *pdp_type = NULL;
            g_autofree gchar *command = NULL;
            g_autofree gchar *quoted_apn = NULL;

            pdp_type = mm_3gpp_get_pdp_type_from_ip_family (
                                    mm_bearer_properties_get_ip_type (
                                            mm_base_bearer_peek_config (MM_BASE_BEARER (self))));
            if (!pdp_type) {
                g_autofree gchar *str = NULL;

                str = mm_bearer_ip_family_build_string_from_mask (
                                    mm_bearer_properties_get_ip_type(
                                            mm_base_bearer_peek_config (MM_BASE_BEARER (self))));
                ctx->saved_error = g_error_new (MM_CORE_ERROR,
                                                MM_CORE_ERROR_INVALID_ARGS,
                                                "Unsupported IP type requested: '%s'",
                                                str);
                cfun_one_and_out (task);
                return;
            }

            quoted_apn = mm_port_serial_at_quote_string (
                            mm_bearer_properties_get_apn (
                                mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self))));

            command = g_strdup_printf ("+CGDCONT=1,\"%s\",%s", pdp_type, quoted_apn);
            if (command) {
                mm_obj_dbg (self,
                            "dial step %u/%u: set pdp context...",
                            ctx->step,
                            DIAL_3GPP_CONTEXT_STEP_LAST);
                mm_base_modem_at_command_full (ctx->modem, ctx->primary,
                                               command, 10, FALSE, FALSE, NULL,
                                               (GAsyncReadyCallback) common_dial_operation_ready_needs_cfun_one,
                                               task);
                return;
            }
        }
        mm_obj_dbg (self,
                    "dial step %u/%u: set pdp context not required",
                    ctx->step,
                    DIAL_3GPP_CONTEXT_STEP_LAST);
        ctx->step++;
        /* fall through */

    case DIAL_3GPP_CONTEXT_STEP_AUTH:
        if (ctx->is_needed_update_user_password) {
            g_autofree gchar *command = NULL;

            command = mm_cinterion_build_auth_string (self,
                                                      mm_broadband_modem_cinterion_ems31_get_family (
                                                                  MM_BROADBAND_MODEM_CINTERION_EMS31 (ctx->modem)),
                                                      mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self)),
                                                      ctx->cid);
            if (command) {
                mm_obj_dbg (self, "dial step %u/%u: authenticating...", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
                /* Send CGAUTH write, if User & Pass are provided.
                 * advance to next state by callback */
                mm_base_modem_at_command_full (ctx->modem, ctx->primary,
                                               command, 10, FALSE, FALSE, NULL,
                                               (GAsyncReadyCallback) common_dial_operation_ready_needs_cfun_one,
                                               task);
                return;
            }
        }
        mm_obj_dbg (self,
                    "dial step %u/%u: authentication not required",
                    ctx->step,
                    DIAL_3GPP_CONTEXT_STEP_LAST);
        ctx->step++;
        /* fall through */

    case DIAL_3GPP_CONTEXT_STEP_CFUN_ONE:
        if (ctx->is_needed_update_pdp_context || ctx->is_needed_update_user_password) {
            mm_obj_dbg (self, "dial step %u/%u: set cfun one...", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
            mm_base_modem_at_command_full (ctx->modem,
                                           ctx->primary,
                                           "+CFUN=1",
                                           5,
                                           FALSE,
                                           FALSE,
                                           NULL,
                                           (GAsyncReadyCallback) common_dial_operation_ready,
                                           task);
            return;
        }
        ctx->step++;
        /* fall through */

    case DIAL_3GPP_CONTEXT_STEP_OPERATOR_SELECTION:
        /* check operator selection with AT+COPS */
        mm_obj_dbg (self, "dial step %u/%u: operator selection...", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
        mm_base_modem_at_command (ctx->modem, "+COPS=3,2", 3, FALSE, NULL, NULL);
        mm_base_modem_at_command_full (ctx->modem,
                                       ctx->primary,
                                       "+COPS?",
                                       10,
                                       FALSE,
                                       FALSE,
                                       NULL,
                                       (GAsyncReadyCallback) cops_response_ready,
                                       task);
        break;

    case DIAL_3GPP_CONTEXT_STEP_LAST:
        mm_obj_dbg (self, "dial step %u/%u: try to connect ppp", ctx->step, DIAL_3GPP_CONTEXT_STEP_LAST);
        atd (task);
        return;

    default:
        g_assert_not_reached ();
    }
}

static void
dial_3gpp (MMBroadbandBearer   *self,
           MMBaseModem         *modem,
           MMPortSerialAt      *primary,
           guint                cid,
           GCancellable        *cancellable,
           GAsyncReadyCallback  callback,
           gpointer             user_data)
{
    GTask           *task;
    Dial3gppContext *ctx;
    GError          *error = NULL;

    g_assert (primary != NULL);

    /* Setup task and create connection context */
    task = g_task_new (self, cancellable, callback, user_data);
    ctx = g_slice_new0 (Dial3gppContext);
    g_task_set_task_data (task, ctx, (GDestroyNotify) dial_3gpp_context_free);

    /* Setup context */
    ctx->self     = MM_BROADBAND_BEARER_CINTERION_EMS31 (g_object_ref (self));
    ctx->modem    = g_object_ref (modem);
    ctx->primary  = g_object_ref (primary);
    ctx->cid      = cid;
    ctx->op_retry = 0;
    ctx->step     = DIAL_3GPP_CONTEXT_STEP_FIRST;

    /* Grab dial port. This gets a reference to the dial port and OPENs it.
     * If we fail, we'll need to close it ourselves. */
    ctx->dial_port = common_get_at_data_port (self, ctx->modem, &error);
    if (!ctx->dial_port) {
        dial_3gpp_error (task, error);
        return;
    }

    /* Run! */
    dial_3gpp_context_step (task);
}

/*****************************************************************************/
/* Setup and Init Bearers */

MMBaseBearer *
mm_broadband_bearer_cinterion_ems31_new_finish (GAsyncResult  *res,
                                                GError       **error)
{
    GObject *bearer;
    GObject *source;

    source = g_async_result_get_source_object (res);
    bearer = g_async_initable_new_finish (G_ASYNC_INITABLE (source), res, error);
    g_object_unref (source);

    if (!bearer)
        return NULL;

    /* Only export valid bearers */
    mm_base_bearer_export (MM_BASE_BEARER (bearer));

    return MM_BASE_BEARER (bearer);
}

void
mm_broadband_bearer_cinterion_ems31_new (MMBroadbandModemCinterionEms31 *modem,
                                         MMBearerProperties             *config,
                                         GCancellable                   *cancellable,
                                         GAsyncReadyCallback             callback,
                                         gpointer                        user_data)
{
    g_async_initable_new_async (
        MM_TYPE_BROADBAND_BEARER_CINTERION_EMS31,
        G_PRIORITY_DEFAULT,
        cancellable,
        callback,
        user_data,
        MM_BASE_BEARER_MODEM, modem,
        MM_BASE_BEARER_CONFIG, config,
        NULL);
}

static void
mm_broadband_bearer_cinterion_ems31_init (MMBroadbandBearerCinterionEms31 *self)
{
}

static void
mm_broadband_bearer_cinterion_ems31_class_init (MMBroadbandBearerCinterionEms31Class *klass)
{
    MMBaseBearerClass      *base_bearer_class      = MM_BASE_BEARER_CLASS      (klass);
    MMBroadbandBearerClass *broadband_bearer_class = MM_BROADBAND_BEARER_CLASS (klass);

    base_bearer_class->load_connection_status        = NULL;
    base_bearer_class->load_connection_status_finish = NULL;

    broadband_bearer_class->dial_3gpp              = dial_3gpp;
    broadband_bearer_class->dial_3gpp_finish       = dial_3gpp_finish;
    broadband_bearer_class->disconnect_3gpp        = disconnect_3gpp;
    broadband_bearer_class->disconnect_3gpp_finish = disconnect_3gpp_finish;


}
