/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details:
 *
 * Copyright (C) 2022 Atmark Techno Inc.
 */

#include <ModemManager.h>
#include "mm-broadband-bearer-cinterion-els31.h"
#include "mm-base-modem-at.h"
#include "mm-log.h"

G_DEFINE_TYPE (MMBroadbandBearerCinterionEls31, mm_broadband_bearer_cinterion_els31, MM_TYPE_BROADBAND_BEARER)

typedef enum {
    CONNECTION_TYPE_NONE,
    CONNECTION_TYPE_3GPP,
    CONNECTION_TYPE_CDMA,
} ConnectionType;

struct _MMBroadbandBearerPrivate {
    /*-- Common stuff --*/
    /* Data port used when modem is connected */
    MMPort *port;
    /* Current connection type */
    ConnectionType connection_type;

    /* PPP specific */
    MMFlowControl flow_control;

    /*-- 3GPP specific --*/
    /* CID of the PDP context */
    gint profile_id;
};

/*****************************************************************************/
/* 3GPP disconnect */
typedef struct {
    MMBaseModem    *modem;
    MMPortSerialAt *primary;
    MMPort         *data;
    guint           cid;
} Disconnect3gppContext;

static gboolean
disconnect_3gpp_finish (MMBroadbandBearer  *self,
                        GAsyncResult       *res,
                        GError            **error)
{
    return g_task_propagate_boolean (G_TASK (res), error);
}

static void
disconnect_3gpp_context_free (Disconnect3gppContext *ctx)
{
    g_object_unref (ctx->data);
    g_object_unref (ctx->primary);
    g_object_unref (ctx->modem);
    g_free (ctx);
}

static Disconnect3gppContext *
disconnect_3gpp_context_new (MMBroadbandModem *modem,
                             MMPortSerialAt   *primary,
                             MMPort           *data,
                             guint             cid)
{
    Disconnect3gppContext *ctx;

    ctx = g_new0 (Disconnect3gppContext, 1);
    ctx->modem = MM_BASE_MODEM (g_object_ref (modem));
    ctx->primary = g_object_ref (primary);
    ctx->data = g_object_ref (data);
    ctx->cid = cid;

    return ctx;
}

static void
disconnect_ready (MMBaseModem  *modem,
                  GAsyncResult *res,
                  GTask        *task)
{
    MMBroadbandBearer *self;
    GError            *error = NULL;

    self = g_task_get_source_object (task);

    /* Ignore errors for now */
    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_dbg (self, "Deactivation failed (not fatal): %s", error->message);
        g_error_free (error);
    }

    g_task_return_boolean (task, TRUE);
    g_object_unref (task);
}

static void
ath_ready (MMBaseModem  *modem,
           GAsyncResult *res,
           GTask        *task)
{
    Disconnect3gppContext *ctx;
    MMBroadbandBearer     *self;
    GError                *error = NULL;
    gchar                 *command;

    self = g_task_get_source_object (task);
    ctx  = g_task_get_task_data (task);

    /* Ignore errors for now */
    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_dbg (self, "Disconnection failed (not fatal): %s", error->message);
        g_error_free (error);
    }

    /* Deactivate context */
    command = g_strdup_printf ("+CGACT=0,%u", ctx->cid);
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   command,
                                   3,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL, /* cancellable */
                                   (GAsyncReadyCallback)disconnect_ready,
                                   task);
    g_free (command);
}

static void
disconnect_3gpp (MMBroadbandBearer   *self,
                 MMBroadbandModem    *modem,
                 MMPortSerialAt      *primary,
                 MMPortSerialAt      *secondary,
                 MMPort              *data,
                 guint                cid,
                 GAsyncReadyCallback  callback,
                 gpointer             user_data)
{
    Disconnect3gppContext *ctx;
    GTask                 *task;

    g_assert (primary != NULL);

    ctx = disconnect_3gpp_context_new (modem, primary, data, cid);

    task = g_task_new (self, NULL, callback, user_data);
    g_task_set_task_data (task, ctx, (GDestroyNotify)disconnect_3gpp_context_free);

    /* Disconnect existing connection */
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "ATH",
                                   3,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL, /* cancellable */
                                   (GAsyncReadyCallback)ath_ready,
                                   task);
}

/*****************************************************************************/
/* 3GPP Dialing (sub-step of the 3GPP Connection sequence) */

typedef struct {
    MMBroadbandBearerCinterionEls31 *self;
    MMBaseModem                     *modem;
    MMPortSerialAt                  *primary;
    guint                            cid;
    GError                          *saved_error;
    MMPort                          *data;
    gint                             retries;
} Dial3gppContext;

static void
dial_3gpp_context_free (Dial3gppContext *ctx)
{
    if (ctx->saved_error)
        g_error_free (ctx->saved_error);
    if (ctx->data)
        g_object_unref (ctx->data);
    g_object_unref (ctx->primary);
    g_object_unref (ctx->modem);
    g_object_unref (ctx->self);
    g_slice_free (Dial3gppContext, ctx);
}

static MMPort *
dial_3gpp_finish (MMBroadbandBearer  *self,
                  GAsyncResult       *res,
                  GError            **error)
{
    return g_task_propagate_pointer (G_TASK (res), error);
}

static gboolean
is_els31_connected (const gchar *str,
                    guint        cid)
{
    gchar *resp;

    resp = g_strdup_printf ("CGPADDR: %u,\"", cid);
    if (g_strrstr (str, resp) == NULL) {
        g_free (resp);
        return FALSE;
    }

    g_free (resp);
    return TRUE;
}

static gboolean
els31_connect_status (GTask *task);

static void
els31_connect_status_ready (MMBaseModem  *modem,
                            GAsyncResult *res,
                            GTask        *task)
{
    Dial3gppContext *ctx;
    const gchar     *result;
    GError          *error = NULL;

    ctx = g_task_get_task_data (task);

    result = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!result) {
        mm_obj_warn (ctx->self, "ELS31-J connection status failed: %s", error->message);
        if (!g_error_matches (error, MM_MOBILE_EQUIPMENT_ERROR, MM_MOBILE_EQUIPMENT_ERROR_UNKNOWN)) {
            g_task_return_error (task, error);
            g_object_unref (task);
            return;
        }
    } else if (is_els31_connected (result, ctx->cid)) {
        mm_obj_dbg(ctx->self, "ELS31-J Connected");
        g_task_return_pointer (task,
                               g_object_ref (ctx->data),
                               g_object_unref);
        g_object_unref (task);
        return;
    }

    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    if (ctx->retries > 0) {
        ctx->retries--;
        mm_obj_dbg (ctx->self, "Retrying status check in a second. %d retries left.",
                    ctx->retries);
        g_timeout_add_seconds (1, (GSourceFunc)els31_connect_status, task);
        return;
    }

    g_task_return_new_error (task,
                             MM_CORE_ERROR,
                             MM_CORE_ERROR_FAILED,
                             "ELS31 connect failed");
    g_object_unref (task);
}

static gboolean
els31_connect_status (GTask *task)
{
    Dial3gppContext *ctx;

    ctx = g_task_get_task_data (task);

    mm_base_modem_at_command_full (
        ctx->modem,
        ctx->primary,
        "+CGPADDR",
        10, /* timeout */
        FALSE, /* allow_cached */
        FALSE, /* raw */
        g_task_get_cancellable (task),
        (GAsyncReadyCallback)els31_connect_status_ready, /* callback */
        task); /* user_data */

    return FALSE;
}

static void
els31_connect_ready (MMBaseModem  *modem,
                     GAsyncResult *res,
                     GTask        *task)
{
    Dial3gppContext *ctx;
    const gchar     *result;
    GError          *error = NULL;

    ctx = g_task_get_task_data (task);

    result = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!result) {
        mm_obj_warn (ctx->self, "ELS31-J connection failed: %s", error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /*
     * The connection takes a bit of time to set up, but there's no
     * asynchronous notification from the modem when this has
     * happened. Instead, we need to poll the modem to see if it's
     * ready.
     */
    g_timeout_add_seconds (1, (GSourceFunc)els31_connect_status, task);
}

static void
start_modem_rf (MMBaseModem  *modem,
                GAsyncResult *res,
                GTask        *task)
{
    Dial3gppContext *ctx;
    GError          *error = NULL;

    ctx = g_task_get_task_data (task);

    if (!mm_base_modem_at_command_full_finish (modem, res, &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* Success, Stop Modem RF */
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+CFUN=1",
                                   5,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL,  /* cancellable */ 
                                   (GAsyncReadyCallback)els31_connect_ready,
                                   task);
}

static void
stop_modem_rf (GTask *task)
{
    Dial3gppContext *ctx;

    ctx = g_task_get_task_data (task);

    /* Success, Stop Modem RF */
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+CFUN=4",
                                   5,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL,  /* cancellable */
                                   (GAsyncReadyCallback)start_modem_rf,
                                   task);
}

static void
cgauth_ready (MMBaseModem  *modem,
              GAsyncResult *res,
              GTask        *task)
{
    GError *error = NULL;

    /* If cancelled, complete */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    if (!mm_base_modem_at_command_full_finish (modem, res, &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* Success, Stop Modem RF */
    stop_modem_rf (task);
}

static void
authenticate (GTask *task)
{
    Dial3gppContext     *ctx;
    MMBearerProperties  *config;
    gchar               *command;
    const gchar         *user;
    const gchar         *password;
    MMBearerAllowedAuth  allowed_auth;
    gboolean             has_user;
    gboolean             has_password;
    gchar               *quoted_user;
    gchar               *quoted_password;
    guint                cinterion_els_auth;
    gchar               *str;

    ctx = g_task_get_task_data (task);

    config = mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self));

    user         = mm_bearer_properties_get_user (config);
    password     = mm_bearer_properties_get_password (config);
    allowed_auth = mm_bearer_properties_get_allowed_auth (config);

    has_user     = (user     && user[0]);
    has_password = (password && password[0]);

    if (allowed_auth == MM_BEARER_ALLOWED_AUTH_UNKNOWN) {
        if (has_user && has_password) {
            /* If user/passwd given, default to CHAP (more common than PAP) */
            allowed_auth = MM_BEARER_ALLOWED_AUTH_CHAP;
        } else {
            /* If user/passwd not given, default to none */
            allowed_auth = MM_BEARER_ALLOWED_AUTH_NONE;
        }
    }

    if (allowed_auth & MM_BEARER_ALLOWED_AUTH_PAP) {
        mm_obj_dbg (ctx->self, "Using PAP authentication method");
        cinterion_els_auth = 1;
    } else if (allowed_auth & MM_BEARER_ALLOWED_AUTH_CHAP) {
        mm_obj_dbg (ctx->self, "Using CHAP authentication method");
        cinterion_els_auth = 2;
    } else if (allowed_auth & MM_BEARER_ALLOWED_AUTH_NONE) {
        mm_obj_dbg (ctx->self, "Using NONE authentication method");
        cinterion_els_auth = 0;
    } else {
        str = mm_bearer_allowed_auth_build_string_from_mask (allowed_auth);
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_UNSUPPORTED,
                                 "Cannot use any of the specified authentication methods (%s)",
                                 str);
        g_object_unref (task);
        g_free (str);
        return;
    }

    if (cinterion_els_auth == 0) {
        command = g_strdup_printf ("+CGAUTH=%u,0", ctx->cid);
    } else {
        quoted_user = mm_port_serial_at_quote_string (user);
        quoted_password = mm_port_serial_at_quote_string (password);
        command = g_strdup_printf ("+CGAUTH=%u,%u,%s,%s",
                                   ctx->cid,
                                   cinterion_els_auth,
                                   quoted_user,
                                   quoted_password);
        g_free (quoted_user);
        g_free (quoted_password);
    }

    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   command,
                                   3,
                                   FALSE,
                                   FALSE, /* raw */
                                   g_task_get_cancellable (task),
                                   (GAsyncReadyCallback)cgauth_ready,
                                   task);
    g_free (command);
}

static void
cgauth_query_ready (MMBaseModem  *modem,
                    GAsyncResult *res,
                    GTask        *task)
{
    Dial3gppContext     *ctx;
    GError              *error = NULL;
    const gchar         *response;
    MMBearerAllowedAuth  current_auth;
    gchar               *current_user;
    gchar               *current_password;
    MMBearerProperties  *config;
    MMBearerAllowedAuth  auth;
    const gchar         *user;
    gboolean             has_user;
    const gchar         *password;
    gboolean             has_password;

    ctx = (Dial3gppContext *) g_task_get_task_data (task);

    response = mm_base_modem_at_command_full_finish (modem, res, &error);
    if (!response) {
        g_error_free (error);
        goto do_authenticate;
    }

    if (!mm_cinterion_parse_cgauth_response (response, ctx->cid, &current_auth,
                                              &current_user, &current_password, &error)) {
        g_error_free (error);
        goto do_authenticate;
    }

    config = mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self));
    if (!config)
        goto do_authenticate;

    auth     = mm_bearer_properties_get_allowed_auth (config);
    user     = mm_bearer_properties_get_user (config);
    password = mm_bearer_properties_get_password (config);

    has_user     = (user     && user[0]);
    has_password = (password && password[0]);

    if (auth == MM_BEARER_ALLOWED_AUTH_UNKNOWN) {
        if (has_user && has_password) {
            /* If user/passwd given, default to CHAP (more common than PAP) */
            auth = MM_BEARER_ALLOWED_AUTH_CHAP;
        } else {
            /* If user/passwd not given, default to none */
            auth = MM_BEARER_ALLOWED_AUTH_NONE;
        }
    }

    if (current_auth != auth)
        goto do_authenticate;

    if (auth == MM_BEARER_ALLOWED_AUTH_NONE)
        goto skip_authenticate;

    if (g_strcmp0 (current_user, user) != 0)
        goto do_authenticate;

    if (g_strcmp0 (current_password, password) != 0)
        goto do_authenticate;

skip_authenticate: /* skip to set CGAUTH */
    stop_modem_rf (task);
    return;

do_authenticate: /* set CGATUTH */
    authenticate (task);
}

static void
dial_3gpp (MMBroadbandBearer   *self,
           MMBaseModem         *modem,
           MMPortSerialAt      *primary,
           guint                cid,
           GCancellable        *cancellable,
           GAsyncReadyCallback  callback,
           gpointer             user_data)
{
    GTask           *task;
    Dial3gppContext *ctx;

    g_assert (primary != NULL);

    /* Setup task and create connection context */
    task = g_task_new (self, cancellable, callback, user_data);
    ctx = g_slice_new0 (Dial3gppContext);
    g_task_set_task_data (task, ctx, (GDestroyNotify) dial_3gpp_context_free);

    /* Setup context */
    ctx->self     = MM_BROADBAND_BEARER_CINTERION_ELS31 (g_object_ref (self));
    ctx->modem    = g_object_ref (modem);
    ctx->primary  = g_object_ref (primary);
    ctx->cid      = cid;
    ctx->retries  = 70;

    /* We need a net data port */
    ctx->data = mm_base_modem_get_best_data_port (modem, MM_PORT_TYPE_NET);
    if (!ctx->data) {
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_NOT_FOUND,
                                 "Data port not found");
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   "+CGAUTH?",
                                   10,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL,
                                   (GAsyncReadyCallback)cgauth_query_ready,
                                   task);
}

/*****************************************************************************/
/* 3GPP CONNECT */
typedef struct {
    MMBroadbandBearer *self;
    MMBaseModem       *modem;
    MMPortSerialAt    *primary;
    MMPortSerialAt    *secondary;

    MMPort            *data;
    gboolean           close_data_on_exit;

    /* 3GPP-specific */
    guint              cid;
    guint              max_cid;
    gboolean           use_existing_cid;
    MMBearerIpFamily   ip_family;
} DetailedConnectContext;

static DetailedConnectContext *
detailed_connect_context_new (MMBroadbandBearer *self,
                              MMBroadbandModem  *modem,
                              MMPortSerialAt    *primary,
                              MMPortSerialAt    *secondary)
{
    DetailedConnectContext *ctx;

    ctx = g_slice_new0 (DetailedConnectContext);
    ctx->self = MM_BROADBAND_BEARER(g_object_ref (self));
    ctx->modem = MM_BASE_MODEM (g_object_ref (modem));
    ctx->primary = g_object_ref (primary);
    ctx->secondary = (secondary ? g_object_ref (secondary) : NULL);
    ctx->close_data_on_exit = FALSE;

    ctx->cid = MM_3GPP_PROFILE_ID_UNKNOWN;
    ctx->max_cid = MM_3GPP_PROFILE_ID_UNKNOWN;
    ctx->use_existing_cid = FALSE;
    ctx->ip_family = mm_bearer_properties_get_ip_type (mm_base_bearer_peek_config (MM_BASE_BEARER (self)));
    mm_3gpp_normalize_ip_family (&ctx->ip_family);

    return ctx;
}

static void
detailed_connect_context_free (DetailedConnectContext *ctx)
{
    g_object_unref (ctx->primary);
    if (ctx->secondary)
        g_object_unref (ctx->secondary);
    if (ctx->data) {
        if (ctx->close_data_on_exit)
            mm_port_serial_close (MM_PORT_SERIAL (ctx->data));
        g_object_unref (ctx->data);
    }
    g_object_unref (ctx->modem);
    g_object_unref (ctx->self);
    g_slice_free (DetailedConnectContext, ctx);
}

static void
get_ip_config_3gpp_ready (MMBroadbandModem *modem,
                          GAsyncResult     *res,
                          GTask            *task)
{
    DetailedConnectContext *ctx;
    MMBearerIpConfig       *ipv4_config = NULL;
    MMBearerIpConfig       *ipv6_config = NULL;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    if (!MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp_finish (ctx->self,
                                                                               res,
                                                                               &ipv4_config,
                                                                               &ipv6_config,
                                                                               &error)) {
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* Keep port open during connection */
    if (MM_IS_PORT_SERIAL_AT (ctx->data))
        ctx->close_data_on_exit = FALSE;

    g_task_return_pointer (
        task,
        mm_bearer_connect_result_new (ctx->data, ipv4_config, ipv6_config),
        (GDestroyNotify)mm_bearer_connect_result_unref);
    g_object_unref (task);

    if (ipv4_config)
        g_object_unref (ipv4_config);
    if (ipv6_config)
        g_object_unref (ipv6_config);
}

static void
dial_3gpp_ready (MMBroadbandModem *modem,
                 GAsyncResult     *res,
                 GTask            *task)
{
    DetailedConnectContext           *ctx;
    MMBearerIpMethod                  ip_method = MM_BEARER_IP_METHOD_UNKNOWN;
    GError                           *error = NULL;
    g_autoptr(MMBearerConnectResult)  result = NULL;
    g_autoptr(MMBearerIpConfig)       ipv4_config = NULL;
    g_autoptr(MMBearerIpConfig)       ipv6_config = NULL;

    ctx = g_task_get_task_data (task);

    ctx->data = MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->dial_3gpp_finish (ctx->self, res, &error);
    if (!ctx->data) {
        /* Clear CID when it failed to connect. */
        ctx->self->priv->profile_id = MM_3GPP_PROFILE_ID_UNKNOWN;
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* If the dialling operation used an AT port, it is assumed to have an extra
     * open() count. */
    if (MM_IS_PORT_SERIAL_AT (ctx->data))
        ctx->close_data_on_exit = TRUE;

    if (MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp &&
        MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp_finish) {
        /* Launch specific IP config retrieval */
        MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->get_ip_config_3gpp (
            ctx->self,
            MM_BROADBAND_MODEM (ctx->modem),
            ctx->primary,
            ctx->secondary,
            ctx->data,
            ctx->cid,
            ctx->ip_family,
            (GAsyncReadyCallback)get_ip_config_3gpp_ready,
            task);
        return;
    }

    /* Yuhu! */

    /* Keep port open during connection */
    if (MM_IS_PORT_SERIAL_AT (ctx->data))
        ctx->close_data_on_exit = FALSE;

    /* If no specific IP retrieval requested, set the default implementation
     * (PPP if data port is AT, DHCP otherwise) */
    ip_method = MM_IS_PORT_SERIAL_AT (ctx->data) ?
                    MM_BEARER_IP_METHOD_PPP :
                    MM_BEARER_IP_METHOD_DHCP;

    if (ctx->ip_family & MM_BEARER_IP_FAMILY_IPV4 ||
            ctx->ip_family & MM_BEARER_IP_FAMILY_IPV4V6) {
        ipv4_config = mm_bearer_ip_config_new ();
        mm_bearer_ip_config_set_method (ipv4_config, ip_method);
    }
    if (ctx->ip_family & MM_BEARER_IP_FAMILY_IPV6 ||
            ctx->ip_family & MM_BEARER_IP_FAMILY_IPV4V6) {
        ipv6_config = mm_bearer_ip_config_new ();
        mm_bearer_ip_config_set_method (ipv6_config, ip_method);
    }
    g_assert (ipv4_config || ipv6_config);

    result = mm_bearer_connect_result_new (ctx->data, ipv4_config, ipv6_config);
    mm_bearer_connect_result_set_profile_id (result, ctx->self->priv->profile_id);
    g_task_return_pointer (task, g_steal_pointer (&result), (GDestroyNotify)mm_bearer_connect_result_unref);
    g_object_unref (task);
}

static void
start_3gpp_dial (GTask *task)
{
    DetailedConnectContext *ctx;

    ctx = g_task_get_task_data (task);

    /* Keep CID around after initializing the PDP context in order to
     * handle corresponding unsolicited PDP activation responses. */
    ctx->self->priv->profile_id = ctx->cid;
    MM_BROADBAND_BEARER_GET_CLASS (ctx->self)->dial_3gpp (ctx->self,
                                                          ctx->modem,
                                                          ctx->primary,
                                                          ctx->cid,
                                                          g_task_get_cancellable (task),
                                                          (GAsyncReadyCallback)dial_3gpp_ready,
                                                          task);
}

static void
initialize_pdp_context_ready (MMBaseModem  *modem,
                              GAsyncResult *res,
                              GTask        *task)
{
    DetailedConnectContext *ctx;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    /* If cancelled, complete */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_warn (ctx->self,
                     "Couldn't initialize PDP context with our APN: '%s'",
                     error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    start_3gpp_dial (task);
}

static void
find_cid_ready (MMBaseModem  *modem,
                GAsyncResult *res,
                GTask        *task)
{
    DetailedConnectContext *ctx;
    GVariant               *result;
    gchar                  *apn, *command;
    GError                 *error = NULL;
    const gchar            *pdp_type;

    ctx = g_task_get_task_data (task);

    result = mm_base_modem_at_sequence_full_finish (modem, res, NULL, &error);
    if (!result) {
        mm_obj_warn (ctx->self, "Couldn't find best CID to use: '%s'", error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    /* If cancelled, complete. Normally, we would get the cancellation error
     * already when finishing the sequence, but we may still get cancelled
     * between last command result parsing in the sequence and the ready(). */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    pdp_type = mm_3gpp_get_pdp_type_from_ip_family (ctx->ip_family);
    if (!pdp_type) {
        gchar * str;

        str = mm_bearer_ip_family_build_string_from_mask (ctx->ip_family);
        g_task_return_new_error (task,
                                 MM_CORE_ERROR,
                                 MM_CORE_ERROR_INVALID_ARGS,
                                 "Unsupported IP type requested: '%s'",
                                 str);
        g_free (str);
        g_object_unref (task);
        return;
    }
    ctx->cid = g_variant_get_uint32 (result);

    /* If there's already a PDP context defined, just use it */
    if (ctx->use_existing_cid) {
        start_3gpp_dial (task);
        return;
    }

    /* Otherwise, initialize a new PDP context with our APN */
    apn = mm_port_serial_at_quote_string (
                mm_bearer_properties_get_apn (
                    mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self))));
    command = g_strdup_printf ("+CGDCONT=%u,\"%s\",%s",
                               ctx->cid,
                               pdp_type,
                               apn);
    g_free (apn);
    mm_base_modem_at_command_full (ctx->modem,
                                   ctx->primary,
                                   command,
                                   3,
                                   FALSE,
                                   FALSE, /* raw */
                                   g_task_get_cancellable (task),
                                   (GAsyncReadyCallback)initialize_pdp_context_ready,
                                   task);
    g_free (command);
}

static gboolean
parse_cid_range (MMBaseModem   *modem,
                 GTask         *task,
                 const gchar   *command,
                 const gchar   *response,
                 gboolean       last_command,
                 const GError  *error,
                 GVariant     **result,
                 GError       **result_error)
{
    DetailedConnectContext *ctx;
    GError                 *inner_error = NULL;
    GList                  *formats, *l;
    guint                   cid;

    ctx = g_task_get_task_data (task);

    /* If cancelled, set result error */
    if (g_task_return_error_if_cancelled (task)) 
        return FALSE;

    if (error) {
        mm_obj_dbg (ctx->self, "Unexpected +CGDCONT error: '%s'", error->message);
        mm_obj_dbg (ctx->self, "Defaulting to CID=1");
        *result = g_variant_new_uint32 (1);
        return TRUE;
    }

    formats = mm_3gpp_parse_cgdcont_test_response (response, ctx->self, &inner_error);
    if (inner_error) {
        mm_obj_dbg (ctx->self, "Error parsing +CGDCONT test response: '%s'", inner_error->message);
        mm_obj_dbg (ctx->self, "Defaulting to CID=1");
        g_error_free (inner_error);
        *result = g_variant_new_uint32 (1);
        return TRUE;
    }

    cid = 0;
    for (l = formats; l; l = g_list_next (l)) {
        MM3gppPdpContextFormat *format = l->data;

        /* Found exact PDP type? */
        if (format->pdp_type == ctx->ip_family) {
            if (ctx->max_cid < format->max_cid)
                cid = ctx->max_cid + 1;
            else
                cid = ctx->max_cid;
            break;
        }
    }

    mm_3gpp_pdp_context_format_list_free (formats);

    if (cid == 0) {
        mm_obj_dbg (ctx->self, "Defaulting to CID=1");
        cid = 1;
    } else
        mm_obj_dbg (ctx->self, "Using CID %u", cid);

    *result = g_variant_new_uint32 (cid);
    return TRUE;
}

static gboolean
parse_pdp_list (MMBaseModem   *modem,
                GTask         *task,
                const gchar   *command,
                const gchar   *response,
                gboolean       last_command,
                const GError  *error,
                GVariant     **result,
                GError       **result_error)
{
    DetailedConnectContext *ctx;
    GError                 *inner_error = NULL;
    GList                  *pdp_list;
    GList                  *l;
    guint                   cid;

    ctx = g_task_get_task_data (task);

    /* If cancelled, set result error */
    if (g_task_return_error_if_cancelled (task)) {
        mm_obj_dbg (ctx->self, "cancelled: parse_pdp_list");
        return FALSE;
    }

    ctx->max_cid = 0;

    /* Some Android phones don't support querying existing PDP contexts,
     * but will accept setting the APN.  So if CGDCONT? isn't supported,
     * just ignore that error and hope for the best. (bgo #637327)
     */
    if (g_error_matches (error,
                         MM_MOBILE_EQUIPMENT_ERROR,
                         MM_MOBILE_EQUIPMENT_ERROR_NOT_SUPPORTED)) {
        mm_obj_dbg (ctx->self, "Querying PDP context list is unsupported");
        return FALSE;
    }

    if (error) {
        mm_obj_dbg (ctx->self, "Unexpected +CGDCONT? error: '%s'", error->message);
        return FALSE;
    }

    pdp_list = mm_3gpp_parse_cgdcont_read_response (response, &inner_error);
    if (!pdp_list) {
        mm_obj_dbg (ctx->self, "(%d)%s: no list.", __LINE__, __func__);
        if (inner_error) {
            mm_obj_dbg (ctx->self, "%s", inner_error->message);
            g_error_free (inner_error);
        } else {
            /* No predefined PDP contexts found */
            mm_obj_dbg (ctx->self, "No PDP contexts found");
        }
        return FALSE;
    }

    cid = 0;

    /* Show all found PDP contexts in debug log */
    mm_obj_dbg (ctx->self, "Found '%u' PDP contexts", g_list_length (pdp_list));
    for (l = pdp_list; l; l = g_list_next (l)) {
        MM3gppPdpContext *pdp = l->data;
        gchar *ip_family_str;

        ip_family_str = mm_bearer_ip_family_build_string_from_mask (pdp->pdp_type);
        mm_obj_dbg (ctx->self, "PDP context [cid=%u] [type='%s'] [apn='%s']",
                pdp->cid,
                ip_family_str,
                pdp->apn ? pdp->apn : "");
        g_free (ip_family_str);
    }

    /* Look for the exact PDP context we want */
    for (l = pdp_list; l; l = g_list_next (l)) {
        MM3gppPdpContext *pdp = l->data;

        if (pdp->pdp_type == ctx->ip_family) {
            /* PDP with no APN set? we may use that one if not exact match found */
            if (!pdp->apn || !pdp->apn[0]) {
                mm_obj_dbg (ctx->self, "Found PDP context with CID %u and no APN",
                            pdp->cid);
                cid = pdp->cid;
            } else {
                const gchar *apn;

                apn = mm_bearer_properties_get_apn (mm_base_bearer_peek_config (MM_BASE_BEARER (ctx->self)));
                if (apn && !g_ascii_strcasecmp (pdp->apn, apn)) {
                    gchar *ip_family_str;

                    /* Found a PDP context with the same CID and PDP type, we'll use it. */
                    ip_family_str = mm_bearer_ip_family_build_string_from_mask (pdp->pdp_type);
                    mm_obj_dbg (ctx->self, "Found PDP context with CID %u and PDP type %s for APN '%s'",
                                pdp->cid, ip_family_str, pdp->apn);
                    cid = pdp->cid;
                    ctx->use_existing_cid = TRUE;
                    g_free (ip_family_str);
                    /* In this case, stop searching */
                    break;
                }
            }
        }

        if (ctx->max_cid < pdp->cid)
            ctx->max_cid = pdp->cid;
    }
    mm_3gpp_pdp_context_list_free (pdp_list);

    if (cid > 0) {
        *result = g_variant_new_uint32 (cid);
        return TRUE;
    }
    return FALSE;
}

static const MMBaseModemAtCommand find_cid_sequence[] = {
    { "+CGDCONT?",  3, FALSE, (MMBaseModemAtResponseProcessor)parse_pdp_list  },
    { "+CGDCONT=?", 3, TRUE,  (MMBaseModemAtResponseProcessor)parse_cid_range },
    { NULL }
};

static void
cfun_zero_ready (MMBaseModem  *modem,
                 GAsyncResult *res,
                 GTask        *task)
{
    DetailedConnectContext *ctx;
    GError                 *error = NULL;

    ctx = g_task_get_task_data (task);

    /* If cancelled, complete */
    if (g_task_return_error_if_cancelled (task)) {
        g_object_unref (task);
        return;
    }

    mm_base_modem_at_command_full_finish (modem, res, &error);
    if (error) {
        mm_obj_warn (ctx->self, "ELS31-J AT+CFUN=0 failed: %s", error->message);
        g_task_return_error (task, error);
        g_object_unref (task);
        return;
    }

    mm_obj_dbg (ctx->self, "Looking for best CID...");
    mm_base_modem_at_sequence_full (MM_BASE_MODEM (ctx->modem),
                                    ctx->primary,
                                    find_cid_sequence,
                                    task,  /* also passed as response processor context */
                                    NULL, /* response_processor_context_free */
                                    NULL, /* cancellable */
                                    (GAsyncReadyCallback)find_cid_ready,
                                    task);
}

static void
connect_3gpp (MMBroadbandBearer   *self,
              MMBroadbandModem    *modem,
              MMPortSerialAt      *primary,
              MMPortSerialAt      *secondary,
              GCancellable        *cancellable,
              GAsyncReadyCallback  callback,
              gpointer             user_data)
{
    DetailedConnectContext *ctx;
    GTask                  *task;

    g_assert (primary != NULL);

    ctx = detailed_connect_context_new (self, modem, primary, secondary);

    task = g_task_new (self, cancellable, callback, user_data);
    g_task_set_task_data (task, ctx, (GDestroyNotify)detailed_connect_context_free);

    mm_base_modem_at_command_full (MM_BASE_MODEM (ctx->modem),
                                   ctx->primary,
                                   "+CFUN=0",
                                   10,
                                   FALSE,
                                   FALSE, /* raw */
                                   NULL, /* cancellable */
                                   (GAsyncReadyCallback)cfun_zero_ready,
                                   task);
}

/*****************************************************************************/
/* Setup and Init Bearers */

MMBaseBearer *
mm_broadband_bearer_cinterion_els31_new_finish (GAsyncResult  *res,
                                                GError       **error)
{
    GObject *bearer;
    GObject *source;

    source = g_async_result_get_source_object (res);
    bearer = g_async_initable_new_finish (G_ASYNC_INITABLE (source), res, error);
    g_object_unref (source);

    if (!bearer)
        return NULL;

    /* Only export valid bearers */
    mm_base_bearer_export (MM_BASE_BEARER (bearer));

    return MM_BASE_BEARER (bearer);
}

void
mm_broadband_bearer_cinterion_els31_new (MMBroadbandModemCinterionEls31 *modem,
                                         MMBearerProperties             *config,
                                         GCancellable                   *cancellable,
                                         GAsyncReadyCallback             callback,
                                         gpointer                        user_data)
{
    g_async_initable_new_async (
        MM_TYPE_BROADBAND_BEARER_CINTERION_ELS31,
        G_PRIORITY_DEFAULT,
        cancellable,
        callback,
        user_data,
        MM_BASE_BEARER_MODEM, modem,
        MM_BASE_BEARER_CONFIG, config,
        NULL);
}

static void
mm_broadband_bearer_cinterion_els31_init (MMBroadbandBearerCinterionEls31 *self)
{
}

static void
mm_broadband_bearer_cinterion_els31_class_init (MMBroadbandBearerCinterionEls31Class *klass)
{
    MMBroadbandBearerClass *broadband_bearer_class = MM_BROADBAND_BEARER_CLASS (klass);

    broadband_bearer_class->connect_3gpp           = connect_3gpp;
    broadband_bearer_class->dial_3gpp              = dial_3gpp;
    broadband_bearer_class->dial_3gpp_finish       = dial_3gpp_finish;
    broadband_bearer_class->disconnect_3gpp        = disconnect_3gpp;
    broadband_bearer_class->disconnect_3gpp_finish = disconnect_3gpp_finish;
}
