/**
* @file       og01h1b.c
* @brief      OG01H1B sensor driver
* @date       2025/02/27
* @version    1.01
* @par        Copyright
*             2025- Shikino High-Tech Co.,LTD. All rights reserved.
*
*             This program is free software; you can redistribute it and/or modify it
*             under the terms and conditions of the GNU General Public License,
*             version 2, as published by the Free Software Foundation.
*
*             This program is distributed in the hope it will be useful, but WITHOUT
*             ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
*             FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
*             more details.
*
*             You should have received a copy of the GNU General Public License
*             along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <asm/unaligned.h>

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/pm_runtime.h>
#include <linux/regulator/consumer.h>
#include <linux/regmap.h>

#include <media/v4l2-ctrls.h>
#include <media/v4l2-event.h>
#include <media/v4l2-fwnode.h>
#include <media/v4l2-subdev.h>

/* Streaming Mode */
#define OG01H1B_REG_MODE_SELECT	0x0100
#define OG01H1B_MODE_STANDBY		0x00
#define OG01H1B_MODE_STREAMING	0x01

#define OG01H1B_REG_PLL_CTRL_0D	0x431E
#define OG01H1B_PLL_CTRL_0D_RAW8	0x08
#define OG01H1B_PLL_CTRL_0D_RAW10	0x00

/* Chip ID */
#define OG01H1B_REG_ID		0x300a
#define OG01H1B_ID			0x5801

/* AEAGC */
#define OG01H1B_REG_AUTO_AEAGC	0x3503
#define OG01H1B_AEAGC_BIT		0x88

/* Exposure control */
#define OG01H1B_REG_EXPOSURE_H		0x3500
#define OG01H1B_REG_EXPOSURE_M		0x3501
#define OG01H1B_REG_EXPOSURE_L		0x3502
#define OG01H1B_EXPOSURE_MIN		1
#define OG01H1B_EXPOSURE_OFFSET		12
#define OG01H1B_EXPOSURE_STEP		1
#define OG01H1B_EXPOSURE_DEFAULT	0x18

/* Analog gain control */
#define OG01H1B_REG_AGAIN	0x3508
#define OG01H1B_AGAIN_MIN	1
#define OG01H1B_AGAIN_MAX	15
#define OG01H1B_AGAIN_STEP	1
#define OG01H1B_AGAIN_DEFAULT	1

/* Group hold register */
#define OG01H1B_REG_HOLD		0x3208

#define OG01H1B_REG_TIMING_FORMAT_1	0x3820
#define OG01H1B_REG_TIMING_FORMAT_2	0x3821
#define OG01H1B_FLIP_BIT		BIT(2)

#define OG01H1B_REG_MIPI_CTRL00	0x4800
#define OG01H1B_GATED_CLOCK		BIT(5)

/* Input clock rate */
#define OG01H1B_INCLK_RATE	24000000

/* CSI2 HW configuration */
#define OG01H1B_LINK_FREQ	400000000

/* Pixel rate */
#define OG01H1B_PIXEL_RATE_10BIT_LANE_4		(OG01H1B_LINK_FREQ / 10 * 2 * 4)
#define OG01H1B_PIXEL_RATE_8BIT_LANE_4		(OG01H1B_LINK_FREQ / 8 * 2 * 4)
#define OG01H1B_PIXEL_RATE_10BIT_LANE_2		(OG01H1B_LINK_FREQ / 10 * 2 * 2)
#define OG01H1B_PIXEL_RATE_8BIT_LANE_2		(OG01H1B_LINK_FREQ / 8 * 2 * 2)
#define OG01H1B_PIXEL_RATE_10BIT_LANE_1		(OG01H1B_LINK_FREQ / 10 * 2 * 1)
#define OG01H1B_PIXEL_RATE_8BIT_LANE_1		(OG01H1B_LINK_FREQ / 8 * 2 * 1)

#define MODE_MIPI_LANE_NUM 	3
#define MODE_MIPI_LANE_4 	0
#define MODE_MIPI_LANE_2 	1
#define MODE_MIPI_LANE_1 	2

#define MODE_NUM 			3
#define MODE_1440_1080 		0
#define MODE_1280_960 		1
#define MODE_640_480 		2

#define OG01H1B_TABLE_WAIT_MS	0
#define OG01H1B_TABLE_END		1
#define OG01H1B_WAIT_MS			20

static const char * const og01h1b_supply_names[] = {
	"avdd",		/* Analog power */
	"dovdd",	/* Digital I/O power */
	"dvdd",		/* Digital core power */
};

#define OG01H1B_NUM_SUPPLIES ARRAY_SIZE(og01h1b_supply_names)

/**
 * struct og01h1b_reg - og01h1b sensor register
 * @address: Register address
 * @val: Register value
 */
struct og01h1b_reg {
	u16 address;
	u8 val;
};

/**
 * struct og01h1b_reg_list - og01h1b sensor register list
 * @num_of_regs: Number of registers in the list
 * @regs: Pointer to register list
 */
struct og01h1b_reg_list {
	u32 num_of_regs;
	const struct og01h1b_reg *regs;
};

/**
 * struct og01h1b_mode - og01h1b sensor mode structure
 * @width: Frame width
 * @height: Frame height
 * @hblank_min: Minimum horizontal blanking in lines for non-continuous[0] and
 *		continuous[1] clock modes
 * @vblank: Vertical blanking in lines
 * @vblank_min: Minimum vertical blanking in lines
 * @vblank_max: Maximum vertical blanking in lines
 * @link_freq_idx: Link frequency index
 * @crop: on-sensor cropping for this mode
 * @reg_list: Register list for sensor mode
 */
struct og01h1b_mode {
	u32 width;
	u32 height;
	u32 link_freq_idx;
	struct v4l2_rect crop;
	struct og01h1b_reg_list reg_list;
};

/**
 * struct og01h1b - og01h1b sensor device structure
 * @dev: Pointer to generic device
 * @sd: V4L2 sub-device
 * @pad: Media pad. Only one pad supported
 * @reset_gpio: Sensor reset gpio
 * @inclk: Sensor input clock
 * @supplies: Regulator supplies for the sensor
 * @ctrl_handler: V4L2 control handler
 * @link_freq_ctrl: Pointer to link frequency control
 * @hblank_ctrl: Pointer to horizontal blanking control
 * @vblank_ctrl: Pointer to vertical blanking control
 * @exp_ctrl: Pointer to exposure control
 * @again_ctrl: Pointer to analog gain control
 * @pixel_rate: Pointer to pixel rate control
 * @vblank: Vertical blanking in lines
 * @noncontinuous_clock: Selection of CSI2 noncontinuous clock mode
 * @cur_mode: Pointer to current selected sensor mode
 * @code: Mbus code currently selected
 * @mutex: Mutex for serializing sensor controls
 * @streaming: Flag indicating streaming state
 */
struct og01h1b {
	struct device *dev;
	struct v4l2_subdev sd;
	struct media_pad pad;
	struct gpio_desc *pwdn;
	struct clk *inclk;
	struct regmap *regmap;
	struct regulator_bulk_data supplies[OG01H1B_NUM_SUPPLIES];
	struct v4l2_ctrl_handler ctrl_handler;
	struct v4l2_ctrl *link_freq_ctrl;
	struct {
		struct v4l2_ctrl *exp_ctrl;
		struct v4l2_ctrl *again_ctrl;
	};
	struct v4l2_ctrl *pixel_rate;
	bool noncontinuous_clock;
	const struct og01h1b_mode *cur_mode;
	u32 code;
	struct mutex mutex;
	bool streaming;
	u32 mode;
};

static const s64 link_freq[] = {
	OG01H1B_LINK_FREQ,
};

static const struct regmap_config sensor_regmap_config = {
	.reg_bits = 16,
	.val_bits = 8,
};

/*
 * Common registers
 *
 * Note: Do NOT include a software reset (0x0103, 0x01) in any of these
 * and the reset will clear them.
 */
 
static const struct og01h1b_reg common_regs[] = {
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS},
	{ 0x3503, 0x00 }, //AEC ON
	{ 0x3501, 0x01 }, //ManualEC default
	{ 0x3502, 0x01 },
	{ 0x3508, 0x01 }, //ManualGC default
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x4004, 0x00 }, //BLC high
	{ 0x4005, 0x40 }, //BLC low
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS},
	{ 0x0100, 0x00 },
	{OG01H1B_TABLE_END, 0x00}
};


static struct og01h1b_reg_list common_regs_list = {
	.num_of_regs = ARRAY_SIZE(common_regs),
	.regs = common_regs,
};

//1440*1080 120fps
static const struct og01h1b_reg mode_1440x1080_120fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x41 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0x20 },
	{ 0x3610, 0x40 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x00 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x05 },
	{ 0x3809, 0xa0 },
	{ 0x380a, 0x04 },
	{ 0x380b, 0x38 },
	{ 0x380c, 0x01 },
	{ 0x380d, 0x28 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x37 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x08 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x06 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x40 },
	{ 0x3821, 0x04 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0x40 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x01 },
	{ 0x3869, 0x01 },
	{ 0x386a, 0x01 },
	{ 0x386b, 0x01 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x01 },
	{ 0x3872, 0x01 },
	{ 0x3873, 0x01 },
	{ 0x3874, 0x01 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x11 },
	{ 0x390c, 0x15 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x10 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x00 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x00 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5186, 0x02 },
	{ 0x5188, 0x02 },
	{ 0x518d, 0x08 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x04 },
	{ 0x4009, 0x1b },
	{ 0x4050, 0x02 },
	{ 0x4051, 0x0d },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x2b },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x05 },
	{ 0x4c05, 0xa0 },
	{ 0x4c06, 0x04 },
	{ 0x4c07, 0x38 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x18 },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x18 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};


//1440*1080 60fps
static const struct og01h1b_reg mode_1440x1080_60fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x21 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0x20 },
	{ 0x3610, 0x40 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x00 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x05 },
	{ 0x3809, 0xa0 },
	{ 0x380a, 0x04 },
	{ 0x380b, 0x38 },
	{ 0x380c, 0x02 },
	{ 0x380d, 0x50 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x37 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x08 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x06 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x40 },
	{ 0x3821, 0x04 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0x40 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x01 },
	{ 0x3869, 0x01 },
	{ 0x386a, 0x01 },
	{ 0x386b, 0x01 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x01 },
	{ 0x3872, 0x01 },
	{ 0x3873, 0x01 },
	{ 0x3874, 0x01 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x11 },
	{ 0x390c, 0x15 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x10 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x00 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x00 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5186, 0x02 },
	{ 0x5188, 0x02 },
	{ 0x518d, 0x08 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x04 },
	{ 0x4009, 0x1b },
	{ 0x4050, 0x02 },
	{ 0x4051, 0x0d },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x2b },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x05 },
	{ 0x4c05, 0xa0 },
	{ 0x4c06, 0x04 },
	{ 0x4c07, 0x38 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x18 },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x18 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

//1440*1080 30fps 1lane
static const struct og01h1b_reg mode_1440x1080_30fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x11 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0x20 },
	{ 0x3610, 0x40 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x00 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x05 },
	{ 0x3809, 0xa0 },
	{ 0x380a, 0x04 },
	{ 0x380b, 0x38 },
	{ 0x380c, 0x04 },
	{ 0x380d, 0xa0 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x37 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x08 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x06 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x40 },
	{ 0x3821, 0x04 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0x40 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x01 },
	{ 0x3869, 0x01 },
	{ 0x386a, 0x01 },
	{ 0x386b, 0x01 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x01 },
	{ 0x3872, 0x01 },
	{ 0x3873, 0x01 },
	{ 0x3874, 0x01 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x11 },
	{ 0x390c, 0x15 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x10 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x00 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x00 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5186, 0x02 },
	{ 0x5188, 0x02 },
	{ 0x518d, 0x08 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x04 },
	{ 0x4009, 0x1b },
	{ 0x4050, 0x02 },
	{ 0x4051, 0x0d },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x2b },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x05 },
	{ 0x4c05, 0xa0 },
	{ 0x4c06, 0x04 },
	{ 0x4c07, 0x38 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x18 },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x18 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

//1280*960 120fps
static const struct og01h1b_reg mode_1280x960_120fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x41 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0x20 },
	{ 0x3610, 0x40 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x00 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x05 },
	{ 0x3809, 0x00 },
	{ 0x380a, 0x03 },
	{ 0x380b, 0xc0 },
	{ 0x380c, 0x01 },
	{ 0x380d, 0x28 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x37 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x58 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x42 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x40 },
	{ 0x3821, 0x04 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0x40 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x01 },
	{ 0x3869, 0x01 },
	{ 0x386a, 0x01 },
	{ 0x386b, 0x01 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x01 },
	{ 0x3872, 0x01 },
	{ 0x3873, 0x01 },
	{ 0x3874, 0x01 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x11 },
	{ 0x390c, 0x15 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x10 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x00 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x00 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5186, 0x02 },
	{ 0x5188, 0x02 },
	{ 0x518d, 0x08 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x04 },
	{ 0x4009, 0x1b },
	{ 0x4050, 0x02 },
	{ 0x4051, 0x0d },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x2b },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x05 },
	{ 0x4c05, 0x00 },
	{ 0x4c06, 0x03 },
	{ 0x4c07, 0xc0 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x18 },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x18 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

//1280*960 60fps
static const struct og01h1b_reg mode_1280x960_60fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x21 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0x20 },
	{ 0x3610, 0x40 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x00 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x05 },
	{ 0x3809, 0x00 },
	{ 0x380a, 0x03 },
	{ 0x380b, 0xc0 },
	{ 0x380c, 0x02 },
	{ 0x380d, 0x50 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x37 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x58 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x42 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x40 },
	{ 0x3821, 0x04 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0x40 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x01 },
	{ 0x3869, 0x01 },
	{ 0x386a, 0x01 },
	{ 0x386b, 0x01 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x01 },
	{ 0x3872, 0x01 },
	{ 0x3873, 0x01 },
	{ 0x3874, 0x01 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x11 },
	{ 0x390c, 0x15 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x10 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x00 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x00 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5186, 0x02 },
	{ 0x5188, 0x02 },
	{ 0x518d, 0x08 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x04 },
	{ 0x4009, 0x1b },
	{ 0x4050, 0x02 },
	{ 0x4051, 0x0d },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x2b },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x05 },
	{ 0x4c05, 0x00 },
	{ 0x4c06, 0x03 },
	{ 0x4c07, 0xc0 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x18 },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x18 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

//1280*960 30fps 1lane
static const struct og01h1b_reg mode_1280x960_30fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x11 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0x20 },
	{ 0x3610, 0x40 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x00 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x05 },
	{ 0x3809, 0x00 },
	{ 0x380a, 0x03 },
	{ 0x380b, 0xc0 },
	{ 0x380c, 0x04 },
	{ 0x380d, 0xa0 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x37 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x58 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x42 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x40 },
	{ 0x3821, 0x04 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0x40 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x01 },
	{ 0x3869, 0x01 },
	{ 0x386a, 0x01 },
	{ 0x386b, 0x01 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x01 },
	{ 0x3872, 0x01 },
	{ 0x3873, 0x01 },
	{ 0x3874, 0x01 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x11 },
	{ 0x390c, 0x15 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x10 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x00 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x00 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5186, 0x02 },
	{ 0x5188, 0x02 },
	{ 0x518d, 0x08 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x04 },
	{ 0x4009, 0x1b },
	{ 0x4050, 0x02 },
	{ 0x4051, 0x0d },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x2b },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x05 },
	{ 0x4c05, 0x00 },
	{ 0x4c06, 0x03 },
	{ 0x4c07, 0xc0 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x18 },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x18 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

static const struct og01h1b_reg mode_640x480_180fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x41 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0xa8 },
	{ 0x3610, 0xd0 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x10 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x02 },
	{ 0x3809, 0x80 },
	{ 0x380a, 0x01 },
	{ 0x380b, 0xe0 },
	{ 0x380c, 0x01 },
	{ 0x380d, 0x28 },
	{ 0x380e, 0x03 },
	{ 0x380f, 0x74 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x2c },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x21 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x61 },
	{ 0x3821, 0x05 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0xc0 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x02 },
	{ 0x3869, 0x02 },
	{ 0x386a, 0x02 },
	{ 0x386b, 0x02 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x02 },
	{ 0x3872, 0x02 },
	{ 0x3873, 0x02 },
	{ 0x3874, 0x02 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x22 },
	{ 0x390c, 0x22 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x90 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x80 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x80 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5185, 0x00 },
	{ 0x5186, 0x01 },
	{ 0x5188, 0x01 },
	{ 0x518d, 0x08 },
	{ 0x5252, 0x82 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x02 },
	{ 0x4009, 0x0d },
	{ 0x4050, 0x01 },
	{ 0x4051, 0x06 },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x60 },
	{ 0x5140, 0x04 },
	{ 0x5141, 0x00 },
	{ 0x5142, 0x04 },
	{ 0x5143, 0x00 },
	{ 0x5144, 0x04 },
	{ 0x5145, 0x00 },
	{ 0x5146, 0x04 },
	{ 0x5147, 0x00 },
	{ 0x3501, 0x01 },
	{ 0x3502, 0x18 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x02 },
	{ 0x3809, 0x80 },
	{ 0x380a, 0x01 },
	{ 0x380b, 0xe0 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x2c },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x21 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x02 },
	{ 0x4c05, 0x80 },
	{ 0x4c06, 0x01 },
	{ 0x4c07, 0xe0 },
	{ 0x46a6, 0x01 },
	{ 0x46a7, 0x38 },
	{ 0x46a8, 0x01 },
	{ 0x46a9, 0x38 },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

static const struct og01h1b_reg mode_640x480_60fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x41 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0xa8 },
	{ 0x3610, 0xd0 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x10 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x02 },
	{ 0x3809, 0xd0 },
	{ 0x380a, 0x02 },
	{ 0x380b, 0x1c },
	{ 0x380c, 0x01 },
	{ 0x380d, 0x28 },
	{ 0x380e, 0x03 },
	{ 0x380f, 0x74 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x04 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x03 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x61 },
	{ 0x3821, 0x05 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0xc0 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x02 },
	{ 0x3869, 0x02 },
	{ 0x386a, 0x02 },
	{ 0x386b, 0x02 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x02 },
	{ 0x3872, 0x02 },
	{ 0x3873, 0x02 },
	{ 0x3874, 0x02 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x22 },
	{ 0x390c, 0x22 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x90 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x80 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x80 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5185, 0x00 },
	{ 0x5186, 0x01 },
	{ 0x5188, 0x01 },
	{ 0x518d, 0x08 },
	{ 0x5252, 0x82 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x02 },
	{ 0x4009, 0x0d },
	{ 0x4050, 0x01 },
	{ 0x4051, 0x06 },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x60 },
	{ 0x5140, 0x04 },
	{ 0x5141, 0x00 },
	{ 0x5142, 0x04 },
	{ 0x5143, 0x00 },
	{ 0x5144, 0x04 },
	{ 0x5145, 0x00 },
	{ 0x5146, 0x04 },
	{ 0x5147, 0x00 },
	{ 0x3501, 0x01 },
	{ 0x3502, 0x18 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x3010, 0x21 },
	{ 0x380c, 0x02 },
	{ 0x380d, 0x50 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x2e },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x02 },
	{ 0x3809, 0x80 },
	{ 0x380a, 0x01 },
	{ 0x380b, 0xe0 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x2c },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x21 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x02 },
	{ 0x4c05, 0x80 },
	{ 0x4c06, 0x01 },
	{ 0x4c07, 0xe0 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x0f },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x0f },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

static const struct og01h1b_reg mode_640x480_30fps[] = {
	{ 0x0107, 0x01 },
	{OG01H1B_TABLE_WAIT_MS, OG01H1B_WAIT_MS },
	{ 0x0105, 0x04 },
	{ 0x0104, 0x00 },
	{ 0x0301, 0x1a },
	{ 0x0304, 0x01 },
	{ 0x0305, 0xe0 },
	{ 0x0306, 0x04 },
	{ 0x0307, 0x01 },
	{ 0x0360, 0x80 },
	{ 0x040b, 0x5c },
	{ 0x040c, 0xcd },
	{ 0x3000, 0x00 },
	{ 0x3001, 0x00 },
	{ 0x3002, 0x10 },
	{ 0x3009, 0x06 },
	{ 0x3010, 0x41 },
	{ 0x301f, 0xe9 },
	{ 0x3182, 0x10 },
	{ 0x3215, 0xcc },
	{ 0x3218, 0x06 },
	{ 0x3400, 0x0c },
	{ 0x3401, 0x01 },
	{ 0x3500, 0x00 },
	{ 0x3501, 0x00 },
	{ 0x3502, 0x40 },
	{ 0x3506, 0x01 },
	{ 0x3507, 0x10 },
	{ 0x3508, 0x01 },
	{ 0x3509, 0x00 },
	{ 0x350a, 0x01 },
	{ 0x350b, 0x00 },
	{ 0x350c, 0x00 },
	{ 0x351e, 0x00 },
	{ 0x351f, 0x00 },
	{ 0x3541, 0x00 },
	{ 0x3542, 0x40 },
	{ 0x3503, 0xa8 },
	{ 0x3610, 0xd0 },
	{ 0x3674, 0x00 },
	{ 0x36a5, 0x2d },
	{ 0x370f, 0x80 },
	{ 0x3710, 0x10 },
	{ 0x3778, 0x10 },
	{ 0x37e7, 0x68 },
	{ 0x37e8, 0x07 },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3822, 0x10 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x02 },
	{ 0x3809, 0xd0 },
	{ 0x380a, 0x02 },
	{ 0x380b, 0x1c },
	{ 0x380c, 0x01 },
	{ 0x380d, 0x28 },
	{ 0x380e, 0x03 },
	{ 0x380f, 0x74 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x04 },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x03 },
	{ 0x3814, 0x11 },
	{ 0x3815, 0x11 },
	{ 0x3820, 0x61 },
	{ 0x3821, 0x05 },
	{ 0x3823, 0x00 },
	{ 0x3826, 0x00 },
	{ 0x3827, 0x00 },
	{ 0x382c, 0x09 },
	{ 0x382d, 0x8c },
	{ 0x382e, 0x01 },
	{ 0x3838, 0x00 },
	{ 0x3840, 0xc0 },
	{ 0x384a, 0xa2 },
	{ 0x3858, 0x00 },
	{ 0x3859, 0x00 },
	{ 0x3860, 0x00 },
	{ 0x3861, 0x00 },
	{ 0x3866, 0x10 },
	{ 0x3867, 0x07 },
	{ 0x3868, 0x02 },
	{ 0x3869, 0x02 },
	{ 0x386a, 0x02 },
	{ 0x386b, 0x02 },
	{ 0x386d, 0x08 },
	{ 0x386e, 0x74 },
	{ 0x3871, 0x02 },
	{ 0x3872, 0x02 },
	{ 0x3873, 0x02 },
	{ 0x3874, 0x02 },
	{ 0x3908, 0x00 },
	{ 0x3909, 0x14 },
	{ 0x390a, 0x00 },
	{ 0x390b, 0x22 },
	{ 0x390c, 0x22 },
	{ 0x390d, 0x84 },
	{ 0x390f, 0x88 },
	{ 0x3a7f, 0x04 },
	{ 0x3a80, 0x01 },
	{ 0x3a84, 0x00 },
	{ 0x3a85, 0x08 },
	{ 0x3ab2, 0x00 },
	{ 0x3ab3, 0x08 },
	{ 0x3a93, 0x60 },
	{ 0x3a98, 0x00 },
	{ 0x3b1e, 0x00 },
	{ 0x3b20, 0x00 },
	{ 0x3b21, 0x00 },
	{ 0x3b22, 0x00 },
	{ 0x3b23, 0x00 },
	{ 0x3b24, 0x00 },
	{ 0x3b25, 0x00 },
	{ 0x3b26, 0x00 },
	{ 0x3b27, 0x00 },
	{ 0x3b28, 0x1a },
	{ 0x3b2f, 0x40 },
	{ 0x3b9d, 0x00 },
	{ 0x3b56, 0x00 },
	{ 0x40f9, 0x00 },
	{ 0x4001, 0x2b },
	{ 0x400a, 0x02 },
	{ 0x400b, 0x00 },
	{ 0x4032, 0x3e },
	{ 0x4300, 0x00 },
	{ 0x4308, 0x00 },
	{ 0x430a, 0x90 },
	{ 0x430b, 0x11 },
	{ 0x4310, 0x00 },
	{ 0x4313, 0x00 },
	{ 0x4315, 0x00 },
	{ 0x4316, 0x14 },
	{ 0x431e, 0x00 },
	{ 0x4342, 0x00 },
	{ 0x4410, 0x09 },
	{ 0x4503, 0x80 },
	{ 0x4508, 0x80 },
	{ 0x4509, 0x80 },
	{ 0x450b, 0x83 },
	{ 0x4514, 0x00 },
	{ 0x458c, 0x00 },
	{ 0x4681, 0x04 },
	{ 0x4682, 0x10 },
	{ 0x4683, 0xa0 },
	{ 0x4698, 0x00 },
	{ 0x4699, 0xf8 },
	{ 0x4837, 0x10 },
	{ 0x4860, 0x00 },
	{ 0x4883, 0x02 },
	{ 0x4885, 0x8f },
	{ 0x4888, 0x10 },
	{ 0x488c, 0x10 },
	{ 0x4980, 0x03 },
	{ 0x4984, 0x00 },
	{ 0x4f01, 0xff },
	{ 0x4f05, 0x13 },
	{ 0x5000, 0x3e },
	{ 0x5001, 0x80 },
	{ 0x500a, 0x00 },
	{ 0x5080, 0x00 },
	{ 0x5081, 0x00 },
	{ 0x5082, 0x00 },
	{ 0x5083, 0x00 },
	{ 0x2805, 0xff },
	{ 0x2806, 0x0f },
	{ 0x4301, 0xff },
	{ 0x4302, 0xf0 },
	{ 0x4303, 0x00 },
	{ 0x4304, 0xff },
	{ 0x4305, 0xf0 },
	{ 0x4306, 0x00 },
	{ 0x4434, 0xf8 },
	{ 0x602e, 0x10 },
	{ 0x602f, 0x00 },
	{ 0x8801, 0x0a },
	{ 0x8802, 0x2f },
	{ 0x8804, 0x0b },
	{ 0x8807, 0x0c },
	{ 0x880a, 0x0d },
	{ 0x880b, 0x4a },
	{ 0x880d, 0x0e },
	{ 0x880e, 0x1e },
	{ 0x8810, 0x0f },
	{ 0x8828, 0x1f },
	{ 0x8829, 0x30 },
	{ 0x882c, 0x29 },
	{ 0x882f, 0x86 },
	{ 0x8832, 0x16 },
	{ 0x8835, 0x16 },
	{ 0x8838, 0x2f },
	{ 0x883b, 0x83 },
	{ 0x883e, 0x8c },
	{ 0x8840, 0x3b },
	{ 0x8841, 0x16 },
	{ 0x8843, 0x3c },
	{ 0x8844, 0x70 },
	{ 0x8845, 0x00 },
	{ 0x8846, 0x3d },
	{ 0x8847, 0xdd },
	{ 0x8849, 0x3e },
	{ 0x884a, 0x03 },
	{ 0x884c, 0x3f },
	{ 0x884d, 0x68 },
	{ 0x8850, 0x03 },
	{ 0x8853, 0x68 },
	{ 0x8856, 0x0e },
	{ 0x8859, 0x0e },
	{ 0x885c, 0x0e },
	{ 0x885f, 0x0e },
	{ 0x8862, 0x0e },
	{ 0x8865, 0x07 },
	{ 0x8868, 0x07 },
	{ 0x886b, 0x07 },
	{ 0x886d, 0x4a },
	{ 0x886e, 0x07 },
	{ 0x886f, 0x00 },
	{ 0x8870, 0xd7 },
	{ 0x8871, 0xb8 },
	{ 0x8873, 0xd8 },
	{ 0x8876, 0xd9 },
	{ 0x8879, 0xda },
	{ 0x887c, 0xdb },
	{ 0x887f, 0xdc },
	{ 0x8882, 0xdd },
	{ 0x8885, 0xe0 },
	{ 0x8886, 0xe7 },
	{ 0x8888, 0xe2 },
	{ 0x888b, 0xe3 },
	{ 0x888e, 0xe4 },
	{ 0x8891, 0xe5 },
	{ 0x89d3, 0x00 },
	{ 0x89d6, 0x00 },
	{ 0x89db, 0x00 },
	{ 0x89df, 0x02 },
	{ 0x89e2, 0x00 },
	{ 0x89e4, 0x00 },
	{ 0x89e6, 0x00 },
	{ 0x89f0, 0x19 },
	{ 0x8a04, 0x19 },
	{ 0x8a08, 0x19 },
	{ 0x8a0a, 0x19 },
	{ 0x8a0c, 0x19 },
	{ 0x8a0e, 0x19 },
	{ 0x89f8, 0x1a },
	{ 0x8960, 0x19 },
	{ 0x8964, 0x19 },
	{ 0x895a, 0x21 },
	{ 0x895b, 0xe6 },
	{ 0x8952, 0x3d },
	{ 0x8953, 0x19 },
	{ 0x8954, 0xb3 },
	{ 0x8955, 0x40 },
	{ 0x8956, 0xe6 },
	{ 0x8957, 0xda },
	{ 0x89e2, 0x00 },
	{ 0x89e3, 0x00 },
	{ 0x89e4, 0x3b },
	{ 0x89e5, 0x40 },
	{ 0x89e6, 0x47 },
	{ 0x89e7, 0xda },
	{ 0x89e8, 0x00 },
	{ 0x89e9, 0x00 },
	{ 0x89d2, 0x00 },
	{ 0x89d7, 0x00 },
	{ 0x89da, 0x00 },
	{ 0x8001, 0x0c },
	{ 0x8002, 0x08 },
	{ 0x8004, 0x16 },
	{ 0x8005, 0x08 },
	{ 0x8007, 0x20 },
	{ 0x8008, 0x01 },
	{ 0x800a, 0x29 },
	{ 0x800b, 0x01 },
	{ 0x800d, 0x34 },
	{ 0x8010, 0x35 },
	{ 0x8011, 0x01 },
	{ 0x8013, 0x37 },
	{ 0x8014, 0x01 },
	{ 0x8016, 0x3a },
	{ 0x8017, 0x01 },
	{ 0x8019, 0x3f },
	{ 0x801a, 0x01 },
	{ 0x801c, 0x48 },
	{ 0x801d, 0x1d },
	{ 0x801f, 0x4b },
	{ 0x8020, 0x01 },
	{ 0x8022, 0x4f },
	{ 0x8025, 0x58 },
	{ 0x8026, 0x01 },
	{ 0x8028, 0x5f },
	{ 0x8029, 0x0f },
	{ 0x802b, 0x6e },
	{ 0x802c, 0x01 },
	{ 0x802e, 0x73 },
	{ 0x802f, 0x01 },
	{ 0x8031, 0x7a },
	{ 0x8032, 0x3f },
	{ 0x8034, 0x7d },
	{ 0x8037, 0x7e },
	{ 0x8038, 0x0a },
	{ 0x8062, 0x4d },
	{ 0x8063, 0x50 },
	{ 0x8066, 0x88 },
	{ 0x8067, 0x16 },
	{ 0x8068, 0x88 },
	{ 0x8069, 0x16 },
	{ 0x806a, 0x88 },
	{ 0x806b, 0x35 },
	{ 0x806c, 0x88 },
	{ 0x806f, 0x5f },
	{ 0x8071, 0x7d },
	{ 0x8073, 0x29 },
	{ 0x8075, 0x29 },
	{ 0x8077, 0x20 },
	{ 0x8078, 0x34 },
	{ 0x8079, 0x88 },
	{ 0x807c, 0x32 },
	{ 0x807d, 0x88 },
	{ 0x807e, 0x32 },
	{ 0x807f, 0x00 },
	{ 0x8081, 0x31 },
	{ 0x8082, 0x00 },
	{ 0x8083, 0x31 },
	{ 0x8084, 0x31 },
	{ 0x8087, 0x31 },
	{ 0x8088, 0x34 },
	{ 0x8089, 0x39 },
	{ 0x808a, 0x4b },
	{ 0x808c, 0x88 },
	{ 0x808e, 0x88 },
	{ 0x8090, 0x88 },
	{ 0x8092, 0x88 },
	{ 0x8094, 0x88 },
	{ 0x8098, 0x46 },
	{ 0x8099, 0x49 },
	{ 0x809a, 0x7a },
	{ 0x809b, 0x7b },
	{ 0x809d, 0x4b },
	{ 0x809e, 0x51 },
	{ 0x809f, 0x7c },
	{ 0x80a0, 0x4b },
	{ 0x80a1, 0x7c },
	{ 0x80a7, 0x88 },
	{ 0x80a9, 0x88 },
	{ 0x80ac, 0x05 },
	{ 0x80ad, 0x88 },
	{ 0x80b8, 0x4a },
	{ 0x80b9, 0x88 },
	{ 0x80bc, 0x4a },
	{ 0x80bd, 0x88 },
	{ 0x80c6, 0x88 },
	{ 0x80c8, 0x88 },
	{ 0x80ca, 0x88 },
	{ 0x80ce, 0x46 },
	{ 0x80cf, 0x49 },
	{ 0x80d0, 0x7a },
	{ 0x80d1, 0x7b },
	{ 0x80d2, 0x46 },
	{ 0x80d3, 0x49 },
	{ 0x80d4, 0x7a },
	{ 0x80d5, 0x7b },
	{ 0x80d6, 0x32 },
	{ 0x80d7, 0x7c },
	{ 0x80d8, 0x88 },
	{ 0x80d9, 0x20 },
	{ 0x80da, 0x3a },
	{ 0x80db, 0x3f },
	{ 0x80dc, 0x6e },
	{ 0x80dd, 0x73 },
	{ 0x80e0, 0x4f },
	{ 0x80e1, 0x50 },
	{ 0x80e2, 0x84 },
	{ 0x80e3, 0x86 },
	{ 0x80e6, 0x4f },
	{ 0x80e7, 0x50 },
	{ 0x80e8, 0x84 },
	{ 0x80e9, 0x86 },
	{ 0x80ec, 0x51 },
	{ 0x80ed, 0x52 },
	{ 0x80ee, 0x88 },
	{ 0x80ef, 0x89 },
	{ 0x80f2, 0x37 },
	{ 0x80f3, 0x38 },
	{ 0x8100, 0x7e },
	{ 0x8101, 0x88 },
	{ 0x8104, 0x88 },
	{ 0x8105, 0x4d },
	{ 0x8108, 0x4a },
	{ 0x8109, 0x88 },
	{ 0x810d, 0x88 },
	{ 0x810f, 0x88 },
	{ 0x8122, 0x89 },
	{ 0x8123, 0x8a },
	{ 0x808b, 0x7a },
	{ 0x80c0, 0x01 },
	{ 0x80c1, 0x20 },
	{ 0x8089, 0x46 },
	{ 0x801d, 0x1d },
	{ 0x8032, 0x44 },
	{ 0x6000, 0x40 },
	{ 0x6001, 0x40 },
	{ 0x6002, 0x00 },
	{ 0x6003, 0x00 },
	{ 0x6004, 0x00 },
	{ 0x6005, 0x00 },
	{ 0x6006, 0x00 },
	{ 0x6007, 0x00 },
	{ 0x6008, 0x00 },
	{ 0x6009, 0x00 },
	{ 0x301b, 0xb4 },
	{ 0x3712, 0xc2 },
	{ 0x3a04, 0x01 },
	{ 0x386c, 0x46 },
	{ 0x3900, 0x10 },
	{ 0x3900, 0x13 },
	{ 0x3255, 0x00 },
	{ 0x3631, 0x80 },
	{ 0x363d, 0x11 },
	{ 0x363e, 0x21 },
	{ 0x395e, 0xc0 },
	{ 0x5000, 0x3c },
	{ 0x5080, 0x3f },
	{ 0x5081, 0xff },
	{ 0x5185, 0x00 },
	{ 0x5186, 0x01 },
	{ 0x5188, 0x01 },
	{ 0x518d, 0x08 },
	{ 0x5252, 0x82 },
	{ 0x5257, 0x8f },
	{ 0x5259, 0x03 },
	{ 0x502b, 0x33 },
	{ 0x5034, 0x34 },
	{ 0x4008, 0x02 },
	{ 0x4009, 0x0d },
	{ 0x4050, 0x01 },
	{ 0x4051, 0x06 },
	{ 0x4d06, 0xc8 },
	{ 0x396e, 0x80 },
	{ 0x3727, 0x46 },
	{ 0x3728, 0x46 },
	{ 0x3729, 0x46 },
	{ 0x3631, 0xf0 },
	{ 0x3642, 0xc0 },
	{ 0x37e5, 0x77 },
	{ 0x372e, 0x15 },
	{ 0x3730, 0x15 },
	{ 0x3732, 0x15 },
	{ 0x372d, 0x15 },
	{ 0x372f, 0x15 },
	{ 0x3731, 0x15 },
	{ 0x371e, 0x08 },
	{ 0x3401, 0x00 },
	{ 0x3622, 0x4e },
	{ 0x3cfe, 0x01 },
	{ 0x4000, 0xf9 },
	{ 0x3642, 0xa8 },
	{ 0x4034, 0x00 },
	{ 0x4035, 0x49 },
	{ 0x4036, 0x00 },
	{ 0x4037, 0x49 },
	{ 0x4038, 0x00 },
	{ 0x4039, 0x49 },
	{ 0x403a, 0x00 },
	{ 0x403b, 0x49 },
	{ 0x4052, 0x00 },
	{ 0x4053, 0x49 },
	{ 0x4054, 0x00 },
	{ 0x4055, 0x49 },
	{ 0x4056, 0x00 },
	{ 0x4057, 0x49 },
	{ 0x4058, 0x00 },
	{ 0x4059, 0x49 },
	{ 0x0326, 0x17 },
	{ 0x0324, 0x00 },
	{ 0x0325, 0xc0 },
	{ 0x367b, 0x50 },
	{ 0x3608, 0x68 },
	{ 0x3604, 0x30 },
	{ 0x3603, 0x67 },
	{ 0x3b9d, 0x60 },
	{ 0x3652, 0x10 },
	{ 0x3654, 0x10 },
	{ 0x3651, 0x9f },
	{ 0x3af0, 0x04 },
	{ 0x3655, 0x68 },
	{ 0x3739, 0x06 },
	{ 0x4e19, 0x0e },
	{ 0x4e1a, 0x0e },
	{ 0x4e1b, 0xf6 },
	{ 0x4e1c, 0xe3 },
	{ 0x4e18, 0x22 },
	{ 0x0441, 0x04 },
	{ 0x3707, 0x22 },
	{ 0x3726, 0x22 },
	{ 0x3727, 0x22 },
	{ 0x3728, 0x22 },
	{ 0x3729, 0x22 },
	{ 0x372a, 0x22 },
	{ 0x3972, 0x22 },
	{ 0x3971, 0x00 },
	{ 0x3970, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x396e, 0x00 },
	{ 0x396d, 0x00 },
	{ 0x400a, 0x07 },
	{ 0x381d, 0x01 },
	{ 0x3748, 0x18 },
	{ 0x3979, 0x00 },
	{ 0x397a, 0x00 },
	{ 0x397b, 0x00 },
	{ 0x397c, 0x00 },
	{ 0x397d, 0x00 },
	{ 0x397e, 0x00 },
	{ 0x397f, 0x00 },
	{ 0x3980, 0x00 },
	{ 0x3981, 0x00 },
	{ 0x3982, 0x00 },
	{ 0x3983, 0x00 },
	{ 0x3984, 0x00 },
	{ 0x3985, 0x00 },
	{ 0x3986, 0x00 },
	{ 0x3987, 0x00 },
	{ 0x3988, 0x00 },
	{ 0x37e5, 0x47 },
	{ 0x3708, 0x0a },
	{ 0x3709, 0x0a },
	{ 0x370a, 0x05 },
	{ 0x370b, 0x05 },
	{ 0x372b, 0x1f },
	{ 0x372c, 0x0a },
	{ 0x372d, 0x0a },
	{ 0x372e, 0x0a },
	{ 0x372f, 0x0a },
	{ 0x3730, 0x0a },
	{ 0x3731, 0x0a },
	{ 0x3732, 0x0a },
	{ 0x3733, 0x05 },
	{ 0x3734, 0x05 },
	{ 0x3735, 0x05 },
	{ 0x3736, 0x05 },
	{ 0x3737, 0x05 },
	{ 0x3738, 0x05 },
	{ 0x3739, 0x05 },
	{ 0x373a, 0x1f },
	{ 0x373b, 0x1f },
	{ 0x3638, 0xac },
	{ 0x3642, 0xa8 },
	{ 0x3643, 0xa8 },
	{ 0x371e, 0x08 },
	{ 0x371f, 0x18 },
	{ 0x3720, 0x18 },
	{ 0x3721, 0x18 },
	{ 0x3722, 0x18 },
	{ 0x3638, 0xbb },
	{ 0x3636, 0x98 },
	{ 0x3963, 0x0f },
	{ 0x396a, 0x66 },
	{ 0x396b, 0x44 },
	{ 0x396d, 0x00 },
	{ 0x396e, 0x00 },
	{ 0x396f, 0x22 },
	{ 0x3970, 0x00 },
	{ 0x3971, 0x00 },
	{ 0x3972, 0x11 },
	{ 0x400a, 0x03 },
	{ 0x400b, 0x00 },
	{ 0x3735, 0x03 },
	{ 0x3736, 0x03 },
	{ 0x3737, 0x00 },
	{ 0x3738, 0x00 },
	{ 0x8100, 0x4b },
	{ 0x8101, 0x51 },
	{ 0x4501, 0x04 },
	{ 0x37e5, 0x67 },
	{ 0x3603, 0x77 },
	{ 0x3501, 0x05 },
	{ 0x3502, 0x60 },
	{ 0x5140, 0x04 },
	{ 0x5141, 0x00 },
	{ 0x5142, 0x04 },
	{ 0x5143, 0x00 },
	{ 0x5144, 0x04 },
	{ 0x5145, 0x00 },
	{ 0x5146, 0x04 },
	{ 0x5147, 0x00 },
	{ 0x3010, 0x11 },
	{ 0x380c, 0x04 },
	{ 0x380d, 0xa0 },
	{ 0x380e, 0x05 },
	{ 0x380f, 0x2e },
	{ 0x3800, 0x00 },
	{ 0x3801, 0x00 },
	{ 0x3802, 0x00 },
	{ 0x3803, 0x00 },
	{ 0x3804, 0x05 },
	{ 0x3805, 0xaf },
	{ 0x3806, 0x04 },
	{ 0x3807, 0x43 },
	{ 0x3808, 0x02 },
	{ 0x3809, 0x80 },
	{ 0x380a, 0x01 },
	{ 0x380b, 0xe0 },
	{ 0x3810, 0x00 },
	{ 0x3811, 0x2c },
	{ 0x3812, 0x00 },
	{ 0x3813, 0x21 },
	{ 0x5000, 0x3f },
	{ 0x4c00, 0x00 },
	{ 0x4c01, 0x00 },
	{ 0x4c02, 0x00 },
	{ 0x4c03, 0x00 },
	{ 0x4c04, 0x02 },
	{ 0x4c05, 0x80 },
	{ 0x4c06, 0x01 },
	{ 0x4c07, 0xe0 },
	{ 0x46a6, 0x05 },
	{ 0x46a7, 0x0f },
	{ 0x46a8, 0x05 },
	{ 0x46a9, 0x0f },
	{ 0x46aa, 0x00 },
	{ 0x46ab, 0x04 },
	{ 0x46ac, 0x00 },
	{ 0x46ad, 0x04 },
	{OG01H1B_TABLE_END, 0x0000}
};

/* Supported sensor mode configurations */
static const struct og01h1b_mode supported_modes[MODE_MIPI_LANE_NUM][MODE_NUM] = {
	{
		[MODE_1440_1080] = {
			.width = 1440,
			.height =1080,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_1440x1080_120fps),
				.regs = mode_1440x1080_120fps,
			},
		},
		[MODE_1280_960] = {
			.width = 1280,
			.height =960,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_1280x960_120fps),
				.regs = mode_1280x960_120fps,
			},
		},
		[MODE_640_480] = {
			.width = 640,
			.height =480,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_640x480_180fps),
				.regs = mode_640x480_180fps,
			},
		}
	},
	{
		[MODE_1440_1080] = {
			.width = 1440,
			.height =1080,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_1440x1080_60fps),
				.regs = mode_1440x1080_60fps,
			},
		},
		[MODE_1280_960] = {
			.width = 1280,
			.height =960,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_1280x960_60fps),
				.regs = mode_1280x960_60fps,
			},
		},
		[MODE_640_480] = {
			.width = 640,
			.height =480,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_640x480_60fps),
				.regs = mode_640x480_60fps,
			},
		}
	},
	{
		[MODE_1440_1080] = {
			.width = 1440,
			.height =1080,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_1440x1080_30fps),
				.regs = mode_1440x1080_30fps,
			},
		},
		[MODE_1280_960] = {
			.width = 1280,
			.height =960,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_1280x960_30fps),
				.regs = mode_1280x960_30fps,
			},
		},
		[MODE_640_480] = {
			.width = 640,
			.height =480,
			.reg_list = {
				.num_of_regs = ARRAY_SIZE(mode_640x480_30fps),
				.regs = mode_640x480_30fps,
			},
		}
	}
};

/**
 * to_og01h1b() - og01h1b V4L2 sub-device to og01h1b device.
 * @subdev: pointer to og01h1b V4L2 sub-device
 *
 * Return: pointer to og01h1b device
 */
static inline struct og01h1b *to_og01h1b(struct v4l2_subdev *subdev)
{
	return container_of(subdev, struct og01h1b, sd);
}

/**
 * og01h1b_read_reg() - Read registers.
 * @og01h1b: pointer to og01h1b device
 * @reg: register address
 * @len: length of bytes to read. Max supported bytes is 4
 * @val: pointer to register value to be filled.
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_read_reg(struct og01h1b *og01h1b, u16 reg, u32 *val)
{
	struct i2c_client *client = v4l2_get_subdevdata(&og01h1b->sd);
	int err = 0;
	u32 reg_val = 0;

	err = regmap_read(og01h1b->regmap, reg, &reg_val);
	if (err)
		dev_err(&client->dev, "%s: i2c read failed, %x\n",
			__func__, reg);
	
	*val = reg_val;

	return 0;
}

/**
 * og01h1b_write_reg() - Write register
 * @og01h1b: pointer to og01h1b device
 * @reg: register address
 * @len: length of bytes. Max supported bytes is 4
 * @val: register value
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_write_reg(struct og01h1b *og01h1b, u16 reg, u32 val)
{
	struct i2c_client *client = v4l2_get_subdevdata(&og01h1b->sd);
	int err = 0;
	err = regmap_write(og01h1b->regmap, reg, val);
	if (err)
		dev_err(&client->dev, "%s: i2c write failed, %x = %x\n",
			__func__, reg, val);

	return 0;
}

/**
 * og01h1b_write_regs() - Write a list of registers
 * @og01h1b: pointer to og01h1b device
 * @regs: list of registers to be written
 * @len: length of registers array
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_write_regs(struct og01h1b *og01h1b,
				 const struct og01h1b_reg *regs, u32 len)
{
	unsigned int i;
	int ret;

	for (i = 0; i < len; i++) {
				/* Handle special address values */
		if (regs[i].address == OG01H1B_TABLE_END)
			break;

		if (regs[i].address == OG01H1B_TABLE_WAIT_MS) {
			usleep_range(regs[i].val * 1000, regs[i].val * 1000 + 500);
			continue;
		}
		ret = og01h1b_write_reg(og01h1b, regs[i].address, regs[i].val);
		if (ret)
			return ret;
	}

	return 0;
}

/**
 * og01h1b_update_controls() - Update control ranges based on streaming mode
 * @og01h1b: pointer to og01h1b device
 * @mode: pointer to og01h1b_mode sensor mode
 * @fmt: pointer to the requested mode
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_update_controls(struct og01h1b *og01h1b,
				  const struct og01h1b_mode *mode,
				  const struct v4l2_subdev_format *fmt)
{
	s64 pixel_rate;
	int ret;

	ret = __v4l2_ctrl_s_ctrl(og01h1b->link_freq_ctrl, mode->link_freq_idx);
	if (ret)
		return ret;
	
	if (fmt->format.code == MEDIA_BUS_FMT_Y10_1X10)
	{
		if(og01h1b->mode == MODE_MIPI_LANE_4)
		{
			pixel_rate = OG01H1B_PIXEL_RATE_10BIT_LANE_4;
		}else if(og01h1b->mode == MODE_MIPI_LANE_2)
		{
			pixel_rate = OG01H1B_PIXEL_RATE_10BIT_LANE_2;
		}else
		{
			pixel_rate = OG01H1B_PIXEL_RATE_10BIT_LANE_1;
		}
	}else
	{
		if(og01h1b->mode == MODE_MIPI_LANE_4)
		{
			pixel_rate = OG01H1B_PIXEL_RATE_8BIT_LANE_4;
		}else if(og01h1b->mode == MODE_MIPI_LANE_2)
		{
			pixel_rate = OG01H1B_PIXEL_RATE_8BIT_LANE_2;
		}else
		{
			pixel_rate = OG01H1B_PIXEL_RATE_8BIT_LANE_1;
		}
	}
	
	ret = __v4l2_ctrl_modify_range(og01h1b->pixel_rate, pixel_rate,
					   pixel_rate, 1, pixel_rate);

	return ret;
}

/**
 * og01h1b_set_exp_auto() - Set exposure auto
 * @og01h1b: pointer to og01h1b device
 * @exposure_auto: updated exposure_auto value
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_set_exp_auto(struct og01h1b *og01h1b, u32 exposure_auto)
{
	u32 current_val;
	int ret = og01h1b_read_reg(og01h1b, OG01H1B_REG_AUTO_AEAGC, &current_val);
	if (ret)
		return ret;

	if (exposure_auto == V4L2_EXPOSURE_AUTO)
		current_val &= ~OG01H1B_AEAGC_BIT;
	else
		current_val |= OG01H1B_AEAGC_BIT;

	return og01h1b_write_reg(og01h1b, OG01H1B_REG_AUTO_AEAGC, current_val);
}

/**
 * og01h1b_set_exp() - Set updated exposure
 * @og01h1b: pointer to og01h1b device
 * @exposure: updated exposure value
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_set_exp(struct og01h1b *og01h1b, u32 exposure)
{
	int ret;
	
	exposure = exposure << 4;
	
	ret = og01h1b_write_reg(og01h1b, OG01H1B_REG_EXPOSURE_L, exposure & 0x000000FF);
	if (ret)
		goto error_release_group_hold;

	ret = og01h1b_write_reg(og01h1b, OG01H1B_REG_EXPOSURE_M, (exposure >> 8) & 0x000000FF);
	if (ret)
		goto error_release_group_hold;
	
	ret = og01h1b_write_reg(og01h1b, OG01H1B_REG_EXPOSURE_H, (exposure >> 16) & 0x000000FF);
	if (ret)
		goto error_release_group_hold;

error_release_group_hold:

	return ret;
}

/**
 * og01h1b_set_gain() - Set  gain
 * @og01h1b: pointer to og01h1b device
 * @gain: updated analog gain value
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_set_gain(struct og01h1b *og01h1b, u32 gain)
{
	int ret;

	ret = og01h1b_write_reg(og01h1b, OG01H1B_REG_AGAIN, gain);

	return ret;
}

static int og01h1b_set_ctrl_hflip(struct og01h1b *og01h1b, int value)
{
	u32 current_val;
	int ret = og01h1b_read_reg(og01h1b, OG01H1B_REG_TIMING_FORMAT_2, &current_val);
	if (ret)
		return ret;

	if (value)
		current_val &= ~OG01H1B_FLIP_BIT;
	else
		current_val |= OG01H1B_FLIP_BIT;

	return og01h1b_write_reg(og01h1b, OG01H1B_REG_TIMING_FORMAT_2, current_val);
}

static int og01h1b_set_ctrl_vflip(struct og01h1b *og01h1b, int value)
{
	u32 current_val;
	int ret = og01h1b_read_reg(og01h1b, OG01H1B_REG_TIMING_FORMAT_1, &current_val);
	if (ret)
		return ret;

	if (value)
		current_val |= OG01H1B_FLIP_BIT;
	else
		current_val &= ~OG01H1B_FLIP_BIT;

	return og01h1b_write_reg(og01h1b, OG01H1B_REG_TIMING_FORMAT_1, current_val);
}

/**
 * og01h1b_set_ctrl() - Set subdevice control
 * @ctrl: pointer to v4l2_ctrl structure
 *
 * Supported controls:
 * - V4L2_CID_VBLANK
 * - cluster controls:
 *   - V4L2_CID_ANALOGUE_GAIN
 *   - V4L2_CID_EXPOSURE
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_set_ctrl(struct v4l2_ctrl *ctrl)
{
	struct og01h1b *og01h1b =
		container_of(ctrl->handler, struct og01h1b, ctrl_handler);
	int ret;
	
	printk("og01h1b_set_ctrl:__v4l2_ctrl_modify_range");
	
	/* Set controls only if sensor is in power on state */
	if (!pm_runtime_get_if_in_use(og01h1b->dev))
		return 0;

	switch (ctrl->id) {
	case V4L2_CID_EXPOSURE_AUTO:
		ret = og01h1b_set_exp_auto(og01h1b, ctrl->val);
		break;
	case V4L2_CID_EXPOSURE:
		ret = og01h1b_set_exp(og01h1b, ctrl->val);
		break;
	case V4L2_CID_GAIN:
		ret = og01h1b_set_gain(og01h1b, ctrl->val);
		break;
	case V4L2_CID_HFLIP:
		ret = og01h1b_set_ctrl_hflip(og01h1b, ctrl->val);
		break;
	case V4L2_CID_VFLIP:
		ret = og01h1b_set_ctrl_vflip(og01h1b, ctrl->val);
		break;
	default:
		dev_err(og01h1b->dev, "Invalid control %d", ctrl->id);
		ret = -EINVAL;
	}

	pm_runtime_put(og01h1b->dev);

	return ret;
}

/* V4l2 subdevice control ops*/
static const struct v4l2_ctrl_ops og01h1b_ctrl_ops = {
	.s_ctrl = og01h1b_set_ctrl,
};

/**
 * og01h1b_enum_mbus_code() - Enumerate V4L2 sub-device mbus codes
 * @sd: pointer to og01h1b V4L2 sub-device structure
 * @sd_state: V4L2 sub-device configuration
 * @code: V4L2 sub-device code enumeration need to be filled
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_enum_mbus_code(struct v4l2_subdev *sd,
				 struct v4l2_subdev_pad_config *sd_state,
				 struct v4l2_subdev_mbus_code_enum *code)
{
	switch (code->index) {
	case 0:
		code->code = MEDIA_BUS_FMT_Y10_1X10;
		break;
	case 1:
		code->code = MEDIA_BUS_FMT_Y8_1X8;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

/**
 * og01h1b_enum_frame_size() - Enumerate V4L2 sub-device frame sizes
 * @sd: pointer to og01h1b V4L2 sub-device structure
 * @sd_state: V4L2 sub-device configuration
 * @fsize: V4L2 sub-device size enumeration need to be filled
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_enum_frame_size(struct v4l2_subdev *sd,
				  struct v4l2_subdev_pad_config *sd_state,
				  struct v4l2_subdev_frame_size_enum *fsize)
{
	struct og01h1b *og01h1b = to_og01h1b(sd);

	if (fsize->index >= ARRAY_SIZE(supported_modes[og01h1b->mode]))
		return -EINVAL;

	if (fsize->code != MEDIA_BUS_FMT_Y10_1X10 &&
		fsize->code != MEDIA_BUS_FMT_Y8_1X8)
		return -EINVAL;

	fsize->min_width = supported_modes[og01h1b->mode][fsize->index].width;
	fsize->max_width = fsize->min_width;
	fsize->min_height = supported_modes[og01h1b->mode][fsize->index].height;
	fsize->max_height = fsize->min_height;

	return 0;
}

/**
 * og01h1b_fill_pad_format() - Fill subdevice pad format
 *                            from selected sensor mode
 * @og01h1b: pointer to og01h1b device
 * @mode: pointer to og01h1b_mode sensor mode
 * @code: mbus code to be stored
 * @fmt: V4L2 sub-device format need to be filled
 */
static void og01h1b_fill_pad_format(struct og01h1b *og01h1b,
				   const struct og01h1b_mode *mode,
				   u32 code,
				   struct v4l2_subdev_format *fmt)
{
	fmt->format.width = mode->width;
	fmt->format.height = mode->height;
	fmt->format.code = code;
	fmt->format.field = V4L2_FIELD_NONE;
	fmt->format.colorspace = V4L2_COLORSPACE_RAW;
	fmt->format.ycbcr_enc = V4L2_YCBCR_ENC_DEFAULT;
	fmt->format.quantization = V4L2_QUANTIZATION_DEFAULT;
	fmt->format.xfer_func = V4L2_XFER_FUNC_NONE;
}

/**
 * og01h1b_get_pad_format() - Get subdevice pad format
 * @sd: pointer to og01h1b V4L2 sub-device structure
 * @sd_state: V4L2 sub-device configuration
 * @fmt: V4L2 sub-device format need to be set
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_get_pad_format(struct v4l2_subdev *sd,
				 struct v4l2_subdev_pad_config *sd_state,
				 struct v4l2_subdev_format *fmt)
{
	struct og01h1b *og01h1b = to_og01h1b(sd);

	mutex_lock(&og01h1b->mutex);

	if (fmt->which == V4L2_SUBDEV_FORMAT_TRY) {
		struct v4l2_mbus_framefmt *framefmt;

		framefmt = v4l2_subdev_get_try_format(sd, sd_state, fmt->pad);
		fmt->format = *framefmt;
	} else {
		og01h1b_fill_pad_format(og01h1b, og01h1b->cur_mode, og01h1b->code,
					   fmt);
	}

	mutex_unlock(&og01h1b->mutex);

	return 0;
}

/**
 * og01h1b_set_pad_format() - Set subdevice pad format
 * @sd: pointer to og01h1b V4L2 sub-device structure
 * @sd_state: V4L2 sub-device configuration
 * @fmt: V4L2 sub-device format need to be set
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_set_pad_format(struct v4l2_subdev *sd,
				 struct v4l2_subdev_pad_config *sd_state,
				 struct v4l2_subdev_format *fmt)
{
	struct og01h1b *og01h1b = to_og01h1b(sd);
	const struct og01h1b_mode *mode;
	u32 code;
	int ret = 0;

	mutex_lock(&og01h1b->mutex);

	mode = v4l2_find_nearest_size(supported_modes[og01h1b->mode],
					  ARRAY_SIZE(supported_modes[og01h1b->mode]),
					  width, height,
					  fmt->format.width,
					  fmt->format.height);
	if (fmt->format.code == MEDIA_BUS_FMT_Y8_1X8)
		code = MEDIA_BUS_FMT_Y8_1X8;
	else
		code = MEDIA_BUS_FMT_Y10_1X10;

	og01h1b_fill_pad_format(og01h1b, og01h1b->cur_mode, code, fmt);

	if (fmt->which == V4L2_SUBDEV_FORMAT_TRY) {
		struct v4l2_mbus_framefmt *framefmt;

		framefmt = v4l2_subdev_get_try_format(sd, sd_state, fmt->pad);
		*framefmt = fmt->format;
	} else {
		ret = og01h1b_update_controls(og01h1b, og01h1b->cur_mode, fmt);
		if (!ret) {
			og01h1b->cur_mode = mode;
			og01h1b->code = code;
		}
	}

	mutex_unlock(&og01h1b->mutex);

	return ret;
}

/**
 * og01h1b_init_pad_cfg() - Initialize sub-device pad configuration
 * @sd: pointer to og01h1b V4L2 sub-device structure
 * @sd_state: V4L2 sub-device configuration
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_init_pad_cfg(struct v4l2_subdev *sd,
			       struct v4l2_subdev_pad_config *sd_state)
{
	struct og01h1b *og01h1b = to_og01h1b(sd);
	struct v4l2_subdev_format fmt = { 0 };

	fmt.which = sd_state ? V4L2_SUBDEV_FORMAT_TRY : V4L2_SUBDEV_FORMAT_ACTIVE;
	og01h1b_fill_pad_format(og01h1b, &supported_modes[og01h1b->mode][0],
			       og01h1b->code, &fmt);

	return og01h1b_set_pad_format(sd, sd_state, &fmt);
}

static const struct v4l2_rect *
__og01h1b_get_pad_crop(struct og01h1b *og01h1b,
		      struct v4l2_subdev_pad_config *sd_state,
		      unsigned int pad, enum v4l2_subdev_format_whence which)
{
	switch (which) {
	case V4L2_SUBDEV_FORMAT_TRY:
		return v4l2_subdev_get_try_crop(&og01h1b->sd, sd_state, pad);
	case V4L2_SUBDEV_FORMAT_ACTIVE:
		return &og01h1b->cur_mode->crop;
	}

	return NULL;
}

static int og01h1b_get_selection(struct v4l2_subdev *sd,
				struct v4l2_subdev_pad_config *sd_state,
				struct v4l2_subdev_selection *sel)
{
	return 0;
}

/**
 * og01h1b_start_streaming() - Start sensor stream
 * @og01h1b: pointer to og01h1b device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_start_streaming(struct og01h1b *og01h1b)
{
	const struct og01h1b_reg bitdepth_regs[2] = {
		{OG01H1B_REG_PLL_CTRL_0D, OG01H1B_PLL_CTRL_0D_RAW10},
		{OG01H1B_REG_PLL_CTRL_0D, OG01H1B_PLL_CTRL_0D_RAW8},
	};
	
	const struct og01h1b_reg_list *reg_list;
	int bitdepth_index;
	int ret;
	
	/* Write sensor mode registers */
	reg_list = &og01h1b->cur_mode->reg_list;
	ret = og01h1b_write_regs(og01h1b, reg_list->regs, reg_list->num_of_regs);
	if (ret) {
		dev_err(og01h1b->dev, "fail to write initial registers");
		return ret;
	}
	
	/* Write common registers */
	printk("og01h1b_start_streaming:common_regs");
	ret = og01h1b_write_regs(og01h1b, common_regs_list.regs,
				common_regs_list.num_of_regs);
	if (ret) {
		dev_err(og01h1b->dev, "fail to write common registers");
		return ret;
	}
	
	printk("og01h1b_start_streaming:bitdepth_regs");
	bitdepth_index = og01h1b->code == MEDIA_BUS_FMT_Y10_1X10 ? 0 : 1;
	ret = og01h1b_write_regs(og01h1b, &bitdepth_regs[bitdepth_index], 1);
	if (ret) {
		dev_err(og01h1b->dev, "fail to write bitdepth regs");
		return ret;
	}
	
	/* Setup handler will write actual exposure and gain */
	ret =  __v4l2_ctrl_handler_setup(og01h1b->sd.ctrl_handler);
	if (ret) {
		dev_err(og01h1b->dev, "fail to setup handler");
		return ret;
	}

	/* Start streaming */
	ret = og01h1b_write_reg(og01h1b, OG01H1B_REG_MODE_SELECT, OG01H1B_MODE_STREAMING);
	if (ret) {
		dev_err(og01h1b->dev, "fail to start streaming");
		return ret;
	}

	return 0;
}

/**
 * og01h1b_stop_streaming() - Stop sensor stream
 * @og01h1b: pointer to og01h1b device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_stop_streaming(struct og01h1b *og01h1b)
{
	return og01h1b_write_reg(og01h1b, OG01H1B_REG_MODE_SELECT, OG01H1B_MODE_STANDBY);
}

/**
 * og01h1b_set_stream() - Enable sensor streaming
 * @sd: pointer to og01h1b subdevice
 * @enable: set to enable sensor streaming
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_set_stream(struct v4l2_subdev *sd, int enable)
{
	struct og01h1b *og01h1b = to_og01h1b(sd);
	int ret;

	mutex_lock(&og01h1b->mutex);

	if (og01h1b->streaming == enable) {
		mutex_unlock(&og01h1b->mutex);
		return 0;
	}

	if (enable) {
		ret = pm_runtime_resume_and_get(og01h1b->dev);
		if (ret)
			goto error_unlock;

		ret = og01h1b_start_streaming(og01h1b);
		if (ret)
			goto error_power_off;
	} else {
		og01h1b_stop_streaming(og01h1b);
		pm_runtime_put(og01h1b->dev);
	}

	og01h1b->streaming = enable;

	mutex_unlock(&og01h1b->mutex);

	return 0;

error_power_off:
	pm_runtime_put(og01h1b->dev);
error_unlock:
	mutex_unlock(&og01h1b->mutex);

	return ret;
}

/**
 * og01h1b_detect() - Detect og01h1b sensor
 * @og01h1b: pointer to og01h1b device
 *
 * Return: 0 if successful, -EIO if sensor id does not match
 */
static int og01h1b_detect(struct og01h1b *og01h1b)
{
	int ret;
	u32 val, msb;

	ret = og01h1b_read_reg(og01h1b, OG01H1B_REG_ID + 1, &val);
	if (ret)
		return ret;
	ret = og01h1b_read_reg(og01h1b, OG01H1B_REG_ID, &msb);
	if (ret)
		return ret;

	val |= (msb << 8);
	if (val != OG01H1B_ID) {
		dev_err(og01h1b->dev, "chip id mismatch: %x!=%x",
			OG01H1B_ID, val);
		return -ENXIO;
	}

	return 0;
}

static int og01h1b_configure_regulators(struct og01h1b *og01h1b)
{
	unsigned int i;

	for (i = 0; i < OG01H1B_NUM_SUPPLIES; i++)
		og01h1b->supplies[i].supply = og01h1b_supply_names[i];

	return devm_regulator_bulk_get(og01h1b->dev,
					   OG01H1B_NUM_SUPPLIES,
					   og01h1b->supplies);
}

/**
 * og01h1b_parse_hw_config() - Parse HW configuration and check if supported
 * @og01h1b: pointer to og01h1b device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_parse_hw_config(struct og01h1b *og01h1b)
{
	struct fwnode_handle *fwnode = dev_fwnode(og01h1b->dev);
	struct v4l2_fwnode_endpoint bus_cfg = {
		.bus_type = V4L2_MBUS_CSI2_DPHY
	};
	struct fwnode_handle *ep;
	unsigned long rate;
	unsigned int i;
	int ret;

	if (!fwnode)
		return -ENXIO;

	/* Request optional reset pin */
	og01h1b->pwdn = devm_gpiod_get_optional(og01h1b->dev, "reset",
						     GPIOD_OUT_LOW);
	if (IS_ERR(og01h1b->pwdn)) {
		dev_err(og01h1b->dev, "failed to get reset gpio %ld",
			PTR_ERR(og01h1b->pwdn));
		return PTR_ERR(og01h1b->pwdn);
	}

	/* Get sensor input clock */
	og01h1b->inclk = devm_clk_get(og01h1b->dev, NULL);
	if (IS_ERR(og01h1b->inclk)) {
		dev_err(og01h1b->dev, "could not get inclk");
		return PTR_ERR(og01h1b->inclk);
	}

	ret = og01h1b_configure_regulators(og01h1b);
	if (ret)
		return dev_err_probe(og01h1b->dev, ret,
				     "Failed to get power regulators\n");

	rate = clk_get_rate(og01h1b->inclk);
	if (rate != OG01H1B_INCLK_RATE) {
		dev_err(og01h1b->dev, "inclk frequency mismatch");
		return -EINVAL;
	}

	ep = fwnode_graph_get_next_endpoint(fwnode, NULL);
	if (!ep)
		return -ENXIO;

	ret = v4l2_fwnode_endpoint_alloc_parse(ep, &bus_cfg);
	fwnode_handle_put(ep);
	if (ret)
		return ret;

	og01h1b->noncontinuous_clock =
		bus_cfg.bus.mipi_csi2.flags & V4L2_MBUS_CSI2_NONCONTINUOUS_CLOCK;

	if (bus_cfg.bus.mipi_csi2.num_data_lanes == 1) {
		og01h1b->mode = MODE_MIPI_LANE_1;
	}
	else if (bus_cfg.bus.mipi_csi2.num_data_lanes == 2) {
		og01h1b->mode = MODE_MIPI_LANE_2;
	}
	else if (bus_cfg.bus.mipi_csi2.num_data_lanes == 4) {
		og01h1b->mode = MODE_MIPI_LANE_4;
	}
	else {
		dev_err(og01h1b->dev,
			"number of CSI2 data lanes %d is not supported",
			bus_cfg.bus.mipi_csi2.num_data_lanes);
		ret = -EINVAL;
		goto done_endpoint_free;
	}

	if (!bus_cfg.nr_of_link_frequencies) {
		dev_err(og01h1b->dev, "no link frequencies defined");
		ret = -EINVAL;
		goto done_endpoint_free;
	}

	for (i = 0; i < bus_cfg.nr_of_link_frequencies; i++)
		if (bus_cfg.link_frequencies[i] == OG01H1B_LINK_FREQ)
			goto done_endpoint_free;

	ret = -EINVAL;

done_endpoint_free:
	v4l2_fwnode_endpoint_free(&bus_cfg);

	return ret;
}

/* V4l2 subdevice ops */
static const struct v4l2_subdev_core_ops og01h1b_core_ops = {
	.subscribe_event = v4l2_ctrl_subdev_subscribe_event,
	.unsubscribe_event = v4l2_event_subdev_unsubscribe,
};

static const struct v4l2_subdev_video_ops og01h1b_video_ops = {
	.s_stream = og01h1b_set_stream,
};

static const struct v4l2_subdev_pad_ops og01h1b_pad_ops = {
	.init_cfg = og01h1b_init_pad_cfg,
	.enum_mbus_code = og01h1b_enum_mbus_code,
	.enum_frame_size = og01h1b_enum_frame_size,
	.get_fmt = og01h1b_get_pad_format,
	.set_fmt = og01h1b_set_pad_format,
	.get_selection = og01h1b_get_selection,
};

static const struct v4l2_subdev_ops og01h1b_subdev_ops = {
	.core = &og01h1b_core_ops,
	.video = &og01h1b_video_ops,
	.pad = &og01h1b_pad_ops,
};

/**
 * og01h1b_power_on() - Sensor power on sequence
 * @dev: pointer to i2c device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_power_on(struct device *dev)
{
	struct v4l2_subdev *sd = dev_get_drvdata(dev);
	struct og01h1b *og01h1b = to_og01h1b(sd);
	int ret;

	ret = regulator_bulk_enable(OG01H1B_NUM_SUPPLIES, og01h1b->supplies);
	if (ret < 0) {
		dev_err(dev, "Failed to enable regulators\n");
		return ret;
	}

	usleep_range(400, 600);

	gpiod_set_value_cansleep(og01h1b->pwdn, 1);

	ret = clk_prepare_enable(og01h1b->inclk);
	if (ret) {
		dev_err(og01h1b->dev, "fail to enable inclk");
		goto error_reset;
	}

	msleep(100);

	usleep_range(400, 600);

	ret = og01h1b_write_reg(og01h1b, OG01H1B_REG_MIPI_CTRL00, 
				   og01h1b->noncontinuous_clock ?
					OG01H1B_GATED_CLOCK : 0);
	if (ret) {
		dev_err(og01h1b->dev, "fail to write MIPI_CTRL00");
		goto error_clk;
	}

	return 0;

error_clk:
	clk_disable_unprepare(og01h1b->inclk);
error_reset:
	gpiod_set_value_cansleep(og01h1b->pwdn, 0);

	regulator_bulk_disable(OG01H1B_NUM_SUPPLIES, og01h1b->supplies);

	return ret;
}

/**
 * og01h1b_power_off() - Sensor power off sequence
 * @dev: pointer to i2c device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_power_off(struct device *dev)
{
	struct v4l2_subdev *sd = dev_get_drvdata(dev);
	struct og01h1b *og01h1b = to_og01h1b(sd);

	gpiod_set_value_cansleep(og01h1b->pwdn, 0);

	clk_disable_unprepare(og01h1b->inclk);

	regulator_bulk_disable(OG01H1B_NUM_SUPPLIES, og01h1b->supplies);

	return 0;
}

/**
 * og01h1b_init_controls() - Initialize sensor subdevice controls
 * @og01h1b: pointer to og01h1b device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_init_controls(struct og01h1b *og01h1b)
{
	struct v4l2_ctrl_handler *ctrl_hdlr = &og01h1b->ctrl_handler;
	const struct og01h1b_mode *mode = og01h1b->cur_mode;
	struct v4l2_fwnode_device_properties props;
	int ret;
	s64 pixel_rate;
	
	if (og01h1b->code == MEDIA_BUS_FMT_Y10_1X10)
	{
		if(og01h1b->mode == MODE_MIPI_LANE_4)
		{
			pixel_rate = OG01H1B_PIXEL_RATE_10BIT_LANE_4;
		}else if(og01h1b->mode == MODE_MIPI_LANE_2)
		{
			pixel_rate = OG01H1B_PIXEL_RATE_10BIT_LANE_2;
		}else
		{
			pixel_rate = OG01H1B_PIXEL_RATE_10BIT_LANE_1;
		}
	}else
	{
	 	if(og01h1b->mode == MODE_MIPI_LANE_4)
	 	{
	 		pixel_rate = OG01H1B_PIXEL_RATE_8BIT_LANE_4;
	 	}else if(og01h1b->mode == MODE_MIPI_LANE_2)
	 	{
	 		pixel_rate = OG01H1B_PIXEL_RATE_8BIT_LANE_2;
	 	}else
	 	{
	 		pixel_rate = OG01H1B_PIXEL_RATE_8BIT_LANE_1;
	 	}
	}

	ret = v4l2_ctrl_handler_init(ctrl_hdlr, 7);
	if (ret)
		return ret;

	/* Serialize controls with sensor device */
	ctrl_hdlr->lock = &og01h1b->mutex;

	/* Initialize exposure and gain */
	v4l2_ctrl_new_std_menu(ctrl_hdlr, &og01h1b_ctrl_ops, V4L2_CID_EXPOSURE_AUTO,
							1, 0, V4L2_EXPOSURE_AUTO);
	
	og01h1b->exp_ctrl = v4l2_ctrl_new_std(ctrl_hdlr,
						 &og01h1b_ctrl_ops,
						 V4L2_CID_EXPOSURE,
						 OG01H1B_EXPOSURE_MIN,
						 mode->height - OG01H1B_EXPOSURE_OFFSET,
						 OG01H1B_EXPOSURE_STEP,
						 OG01H1B_EXPOSURE_DEFAULT);

	og01h1b->again_ctrl = v4l2_ctrl_new_std(ctrl_hdlr,
						 &og01h1b_ctrl_ops,
						 V4L2_CID_GAIN,
						 OG01H1B_AGAIN_MIN,
						 OG01H1B_AGAIN_MAX,
						 OG01H1B_AGAIN_STEP,
						 OG01H1B_AGAIN_DEFAULT);

	v4l2_ctrl_new_std(ctrl_hdlr, &og01h1b_ctrl_ops, V4L2_CID_VFLIP,
			  0, 1, 1, 0);

	v4l2_ctrl_new_std(ctrl_hdlr, &og01h1b_ctrl_ops, V4L2_CID_HFLIP,
			  0, 1, 1, 0);

	/* Read only controls */
	og01h1b->pixel_rate = v4l2_ctrl_new_std(ctrl_hdlr, &og01h1b_ctrl_ops,
						   V4L2_CID_PIXEL_RATE,
						   pixel_rate,
						   pixel_rate, 1,
						   pixel_rate);

	og01h1b->link_freq_ctrl = v4l2_ctrl_new_int_menu(ctrl_hdlr,
							&og01h1b_ctrl_ops,
							V4L2_CID_LINK_FREQ,
							ARRAY_SIZE(link_freq) -
							1,
							mode->link_freq_idx,
							link_freq);
	if (og01h1b->link_freq_ctrl)
		og01h1b->link_freq_ctrl->flags |= V4L2_CTRL_FLAG_READ_ONLY;

	ret = v4l2_fwnode_device_parse(og01h1b->dev, &props);
	if (!ret) {
		/* Failure sets ctrl_hdlr->error, which we check afterwards anyway */
		v4l2_ctrl_new_fwnode_properties(ctrl_hdlr, &og01h1b_ctrl_ops,
						&props);
	}

	if (ctrl_hdlr->error || ret) {
		dev_err(og01h1b->dev, "control init failed: %d",
			ctrl_hdlr->error);
		v4l2_ctrl_handler_free(ctrl_hdlr);
		return ctrl_hdlr->error;
	}

	og01h1b->sd.ctrl_handler = ctrl_hdlr;

	return 0;
}

/**
 * og01h1b_probe() - I2C client device binding
 * @client: pointer to i2c client device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_probe(struct i2c_client *client, const struct i2c_device_id *device_id)
{
	struct og01h1b *og01h1b;
	int ret;

	og01h1b = devm_kzalloc(&client->dev, sizeof(*og01h1b), GFP_KERNEL);
	if (!og01h1b)
		return -ENOMEM;
	printk("og01h1b_probe:devm_kzalloc");
	og01h1b->dev = &client->dev;

	
	/* Initialize subdev */
	v4l2_i2c_subdev_init(&og01h1b->sd, client, &og01h1b_subdev_ops);
	v4l2_i2c_subdev_set_name(&og01h1b->sd, client,
				 device_get_match_data(og01h1b->dev), NULL);

	printk("og01h1b_probe:Initialize subdev");
	ret = og01h1b_parse_hw_config(og01h1b);
	if (ret) {
		dev_err(og01h1b->dev, "HW configuration is not supported");
		return ret;
	}

	og01h1b->regmap = devm_regmap_init_i2c(client, &sensor_regmap_config);
	if (IS_ERR(og01h1b->regmap)) {
		dev_err(&client->dev, "regmap init failed: %ld\n", PTR_ERR(og01h1b->regmap));
		return -ENODEV;
	}

	mutex_init(&og01h1b->mutex);

	ret = og01h1b_power_on(og01h1b->dev);
	if (ret) {
		dev_err(og01h1b->dev, "failed to power-on the sensor");
		goto error_mutex_destroy;
	}

	/* Check module identity */
	ret = og01h1b_detect(og01h1b);
	if (ret) {
		dev_err(og01h1b->dev, "failed to find sensor: %d", ret);
		goto error_power_off;
	}
	printk("og01h1b_probe:og01h1b_detect");

	/* Set default mode to first mode */
	og01h1b->cur_mode = &supported_modes[og01h1b->mode][0];
	og01h1b->code = MEDIA_BUS_FMT_Y10_1X10;

	ret = og01h1b_init_controls(og01h1b);
	if (ret) {
		dev_err(og01h1b->dev, "failed to init controls: %d", ret);
		goto error_power_off;
	}
	printk("og01h1b_probe:og01h1b_init_controls");

	/* Initialize subdev */
	og01h1b->sd.flags |= V4L2_SUBDEV_FL_HAS_DEVNODE |
			    V4L2_SUBDEV_FL_HAS_EVENTS;
	og01h1b->sd.entity.function = MEDIA_ENT_F_CAM_SENSOR;

	/* Initialize source pad */
	og01h1b->pad.flags = MEDIA_PAD_FL_SOURCE;
	ret = media_entity_pads_init(&og01h1b->sd.entity, 1, &og01h1b->pad);
	if (ret) {
		dev_err(og01h1b->dev, "failed to init entity pads: %d", ret);
		goto error_handler_free;
	}

	ret = v4l2_async_register_subdev_sensor_common(&og01h1b->sd);
	if (ret < 0) {
		dev_err(og01h1b->dev,
			"failed to register async subdev: %d", ret);
		goto error_media_entity;
	}

	pm_runtime_set_active(og01h1b->dev);
	pm_runtime_enable(og01h1b->dev);
	pm_runtime_idle(og01h1b->dev);
	
	printk("og01h1b_probe:pm_runtime_set_active");
	
	return 0;

error_media_entity:
	media_entity_cleanup(&og01h1b->sd.entity);
error_handler_free:
	v4l2_ctrl_handler_free(og01h1b->sd.ctrl_handler);
error_power_off:
	og01h1b_power_off(og01h1b->dev);
error_mutex_destroy:
	mutex_destroy(&og01h1b->mutex);

	return ret;
}

/**
 * og01h1b_remove() - I2C client device unbinding
 * @client: pointer to I2C client device
 *
 * Return: 0 if successful, error code otherwise.
 */
static int og01h1b_remove(struct i2c_client *client)
{
	struct v4l2_subdev *sd = i2c_get_clientdata(client);
	struct og01h1b *og01h1b = to_og01h1b(sd);

	v4l2_async_unregister_subdev(sd);
	media_entity_cleanup(&sd->entity);
	v4l2_ctrl_handler_free(sd->ctrl_handler);

	pm_runtime_disable(&client->dev);
	if (!pm_runtime_status_suspended(&client->dev))
		og01h1b_power_off(&client->dev);
	pm_runtime_set_suspended(&client->dev);

	mutex_destroy(&og01h1b->mutex);
	printk("og01h1b_remove:mutex_destroy");

	return 0;
}

static const struct dev_pm_ops og01h1b_pm_ops = {
	SET_RUNTIME_PM_OPS(og01h1b_power_off, og01h1b_power_on, NULL)
};

static const struct of_device_id og01h1b_of_match[] = {
	{ .compatible = "omnivison,og01h1b", .data = "og01h1b" },
	{ }
};

MODULE_DEVICE_TABLE(of, og01h1b_of_match);

static struct i2c_driver og01h1b_driver = {
	.probe = og01h1b_probe,
	.remove = og01h1b_remove,
	.driver = {
		.name = "og01h1b",
		.pm = &og01h1b_pm_ops,
		.of_match_table = og01h1b_of_match,
	},
};

module_i2c_driver(og01h1b_driver);

MODULE_DESCRIPTION("V4L2 driver for OG01H1B");
MODULE_AUTHOR("Shikino High-Tech");
MODULE_LICENSE("GPL v2");

