// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>

#include "device-info.h"

#define IMX8MP_SOC_ID	"i.MX8MP"
#define IMX6ULL_SOC_ID	"i.MX6ULL"
#define IMX8ULP_SOC_ID	"i.MX8ULP"

enum soc get_soc(void)
{
	FILE *fp;
	char soc_id[16];

	fp = fopen("/sys/devices/soc0/soc_id", "r");
	if (fp == NULL) {
		fprintf(stderr, "fopen error: Failed to open /sys/devices/soc0/soc_id: %m\n");
		return SOC_UNKNOWN;
	}


	if (fgets(soc_id, 16, fp) == NULL) {
		fclose(fp);
		fprintf(stderr, "fgets error: %m\n");
		return SOC_UNKNOWN;
	}

	fclose(fp);

	if (strstr(soc_id, IMX8MP_SOC_ID) != NULL)
		return IMX8MP;
	if (strstr(soc_id, IMX6ULL_SOC_ID) != NULL)
		return IMX6ULL;
	if (strstr(soc_id, IMX8ULP_SOC_ID) != NULL)
		return IMX8ULP;

	fprintf(stderr, "unsupported SoC: %s\n", soc_id);
	return SOC_UNKNOWN;
}

int imx_ocotp_read(struct armadillo *device)
{
	struct stat finfo;

	_cleanup_(closep) int fd = open(device->nvmem_path, O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "Open %s failed: %m\n", device->nvmem_path);
		return -1;
	}

	fstat(fd, &finfo);
	device->ocotp_data = malloc(finfo.st_size);
	if (device->ocotp_data == NULL) {
		fprintf(stderr, "failed to allocate memory (%jd)\n", finfo.st_size);
		return -1;
	}

	if (read_full(fd, device->ocotp_data, finfo.st_size) < 0) {
		fprintf(stderr, "Could not read %s: %m\n", device->nvmem_path);
		return -1;
	}

	return 0;
}

int close_precheck_read(struct armadillo *device)
{
	_cleanup_(closep) int fd = open(device->close_precheck_path, O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "Open %s failed: %m\n", device->close_precheck_path);
		return -1;
	}

	if (read_to_end(fd, device->close_precheck_data, CLOSE_PRECHECK_SIZE) < 0) {
		fprintf(stderr, "Could not read %s: %m\n", device->close_precheck_data);
		return -1;
	}
	return 0;
}
