// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#include "device-info.h"
#include "security_imx8ulp.h"

enum secure_status secure_status_jtag_imx8ulp(unsigned char *ocotp_data)
{
	uint32_t sjc_disable_mem;

	memcpy(&sjc_disable_mem,
		ocotp_data + IMX8ULP_SJC_DISABLE_OFFSET, 4);

	if ((sjc_disable_mem
	    & IMX8ULP_SJC_DISABLE_MASK)
	    != IMX8ULP_SJC_DISABLE_TRUE) {
		return INSECURE;
	}

	return SECURE;
}

int is_secure_fuse_jtag_imx8ulp(unsigned char *ocotp_data)
{
	if (secure_status_jtag_imx8ulp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

char *secure_status_to_string_jtag_imx8ulp(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "open";
	default: return "disabled";
	}
}

int print_fuse_status_jtag_imx8ulp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("jtag %s\n",
		secure_status_to_string_jtag_imx8ulp(
			secure_status_jtag_imx8ulp(ocotp_data)
		)
	);
	return 0;
}

int write_fuse_jtag_imx8ulp(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX8ULP_SJC_DISABLE_TRUE;
	if (pwrite(fd, &buf, sizeof(buf), IMX8ULP_SJC_DISABLE_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM_IMX8ULP, IMX8ULP_SJC_DISABLE_OFFSET);
		return 1;
	}

	return 0;
}

enum secure_status secure_status_sd_boot_imx8ulp(unsigned char *ocotp_data)
{
	uint32_t btFuseSel_mem;
	uint32_t usdhc_mem;

	memcpy(&btFuseSel_mem,
		ocotp_data + IMX8ULP_BT_FUSE_SEL_OFFSET, 4);

	memcpy(&usdhc_mem,
		ocotp_data + IMX8ULP_USDHC_OFFSET, 4);

	if ((btFuseSel_mem
	    & IMX8ULP_BT_FUSE_SEL_MASK)
	    != IMX8ULP_BT_FUSE_SEL_PROGRAMMED) {
		return INSECURE;
	}

	if ((usdhc_mem
	    & IMX8ULP_USDHC_MASK)
	    != IMX8ULP_USDHC_EMMC) {
		return UNKNOWN;
	}

	if ((usdhc_mem
	    & IMX8ULP_USDHC_DEVICE_TYPE_MASK)
	    != IMX8ULP_USDHC_DEVICE_TYPE_EMMC) {
		return UNKNOWN;
	}

	return SECURE;
}

int is_secure_fuse_sd_boot_imx8ulp(unsigned char *ocotp_data)
{
	if (secure_status_sd_boot_imx8ulp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

char *secure_status_to_string_sd_boot_imx8ulp(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "allowed";
	case SECURE: return "disabled";
	default: return "unknown";
	}
}

int print_fuse_status_sd_boot_imx8ulp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("sd_boot %s\n",
		secure_status_to_string_sd_boot_imx8ulp(
			secure_status_sd_boot_imx8ulp(ocotp_data)
		)
	);
	return 0;
}

int write_fuse_sd_boot_imx8ulp(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX8ULP_BT_FUSE_SEL_PROGRAMMED;
	if (pwrite(fd, &buf, sizeof(buf), IMX8ULP_BT_FUSE_SEL_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8ULP_BT_FUSE_SEL_OFFSET);
		return 1;
	}

	return 0;
}

static int read_secureboot_path(char *srk_close_str)
{
	_cleanup_(closep) int fd = open(IMX8ULP_SECUREBOOT_PATH, O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "Open %s failed: %m\n", IMX8ULP_SECUREBOOT_PATH);
		return 1;
	}

	if (read_to_end(fd, srk_close_str, CLOSE_PRECHECK_SIZE) < 0) {
		fprintf(stderr, "Could not read %s: %m\n", IMX8ULP_SECUREBOOT_PATH);
		return 1;
	}
	return 0;
}

enum secure_status secure_status_secureboot_imx8ulp(unsigned char *ocotp_data)
{
	int i, rc;
	uint32_t hash_mem[8];
	char srk_close_str[CLOSE_PRECHECK_SIZE];

	// value of SRK region.
	for (i = 0; i < 8; i++)
		memcpy(&hash_mem[i],
			ocotp_data + IMX8ULP_SRK_HASH_OFFSET + 4*i, 4);

	rc = read_secureboot_path(srk_close_str);
	if (rc != 0) {
		fprintf(stderr, "Could not read %s\n", IMX8ULP_SECUREBOOT_PATH);
		return 1;
	}

	for (i = 0; i < 8; i++) {
		if (hash_mem[i] != 0)
			break;
		if (i == 7)
			return INSECURE;
	}

	if (strcmp(srk_close_str, "close\n") == 0)
		return SECURE;

	return UNKNOWN;
}

static char *secure_status_to_string(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "disabled";
	case SECURE: return "configured";
	default: return "configured (not enforced)";
	}
}

int print_fuse_status_secureboot_imx8ulp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("secureboot %s\n",
		secure_status_to_string(secure_status_secureboot_imx8ulp(ocotp_data))
	);
	return 0;
}

int is_secure_fuse_secureboot_imx8ulp(unsigned char *ocotp_data)
{
	if (secure_status_secureboot_imx8ulp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

int write_fuse_secureboot_srk_imx8ulp(int fd)
{
	int i;

	// write SRK to fuse of SRK region.
	for (i = 0; i < 8; i++) {
		if (pwrite(fd, &srk_hashes[i], sizeof(srk_hashes[i]),
		    IMX8ULP_SRK_HASH_OFFSET + 4*i) < 0) {
			fprintf(stderr,
				"Could not write 0x%08X to %s (offset is 0x%02X, SRK%d) : %m\n",
				srk_hashes[i], IMX_OCOTP_NVMEM, IMX8ULP_SRK_HASH_OFFSET + 4*i, i
			);
			return 1;
		}
	}

	return 0;
}

int cmp_fuse_secureboot_srk_imx8ulp(unsigned char *ocotp_data)
{
	int i;
	uint32_t buf = 0x0;

	for (i = 0; i < 8; i++) {
		memcpy(&buf,
			ocotp_data + IMX8ULP_SRK_HASH_OFFSET + 4*i, 4);

		if (srk_hashes[i] != buf)
			return 1;
	}

	return 0;
}

int close_precheck_imx8ulp(char *close_precheck_data)
{
	if (strcmp(close_precheck_data, "open\n") != 0
	    && strcmp(close_precheck_data, "close\n") != 0) {
		fprintf(stderr,
			"Secureboot HAB failure logged.\n"
		);
		return EXIT_FAILURE;
	}

	// If /boot/Image is a plain non-signed kernel then in open mode then
	// uboot will load it without validation, leaving close_precheck of success
	// but not being able to boot after close.
	// User must ensure there is no /boot/Image to check /dev/mmcblkXbootY
	// was used.
	if (access("/boot/Image", F_OK) == 0) {
		fprintf(stderr,
			"/boot/Image exists. If /boot/Image is a non-signed kernel, Armadillo\n"
			"can not boot after the close operation is enforced.\n"
			"Remove file and reboot to confirm status is OK.\n"
		);
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}

int write_fuse_secureboot_close_imx8ulp(void)
{
	char *buf = "close\n";

	_cleanup_(closep) int fd = open(IMX8ULP_SECUREBOOT_PATH, O_WRONLY);
	if (fd < 0) {
		fprintf(stderr, "Open %s failed: %m\n", IMX8ULP_SECUREBOOT_PATH);
		return 1;
	}

	int byte_num = write(fd, buf, strlen(buf));

	if (byte_num == -1) {
		fprintf(stderr, "could not write 'close' to %s: %m\n", IMX8ULP_SECUREBOOT_PATH);
		return 1;
	}

	return 0;
}
