// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#include "device-info.h"
#include "security_imx8mp.h"

enum secure_status secure_status_boot_cfg_lock_imx8mp(unsigned char *ocotp_data)
{
	uint32_t status_mem;

	memcpy(&status_mem,
		ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET, 4);

	if ((status_mem
	    & IMX8MP_BOOT_CFG_LOCK_MASK)
	    == IMX8MP_BOOT_CFG_LOCK_OPWP) {
		return SECURE;
	}

	return INSECURE;
}

int is_secure_fuse_boot_cfg_lock_imx8mp(unsigned char *ocotp_data)
{
	if (secure_status_boot_cfg_lock_imx8mp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

enum secure_status secure_status_cst_srk_lock_imx8mp(unsigned char *ocotp_data)
{
	uint32_t status_mem;

	memcpy(&status_mem,
		ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET, 4);

	if ((status_mem
	    & IMX8MP_CST_SRK_LOCK_MASK)
	    == IMX8MP_CST_SRK_LOCK_ENABLE) {
		return SECURE;
	}

	return INSECURE;
}

char *secure_status_to_string_lock_imx8mp(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "open";
	default: return "write protected";
	}
}

int print_fuse_status_lock_imx8mp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("boot_cfg_lock %s\n",
		secure_status_to_string_lock_imx8mp(
			secure_status_boot_cfg_lock_imx8mp(ocotp_data)
		)
	);

	PRINTF_CHECK("cst_srk_lock %s\n",
		secure_status_to_string_lock_imx8mp(
			secure_status_cst_srk_lock_imx8mp(ocotp_data)
		)
	);

	return 0;
}

int write_fuse_lock_imx8mp(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX8MP_BOOT_CFG_LOCK_OPWP;
	if (pwrite(fd, &buf, sizeof(buf), IMX8MP_BOOT_CFG_LOCK_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8MP_BOOT_CFG_LOCK_OFFSET
		);
		return 1;
	}

	return 0;
}

char *fuse_status_jtag_imx8mp(unsigned char *ocotp_data)
{
	uint32_t status_mem;

	memcpy(&status_mem,
		ocotp_data + IMX8MP_JTAG_OFFSET, 4);

	if ((status_mem & IMX8MP_SJC_DISABLE_MASK)
	    == IMX8MP_SJC_DISABLE_TRUE)
		return "disabled";

	if ((status_mem & IMX8MP_JTAG_SMODE_MASK)
	    == IMX8MP_JTAG_SMODE_JTAG_ENABLE)
		return "open";

	switch (status_mem &
		(IMX8MP_JTAG_SMODE_MASK | IMX8MP_KTE_MASK)) {
	case IMX8MP_JTAG_SMODE_SECURE:
		return "secure mode. WARNING: bus tracing is allowed";
	case IMX8MP_JTAG_SMODE_SECURE
	    | IMX8MP_KTE_KILL:
		return "secure mode";
	case IMX8MP_JTAG_SMODE_NO_DEBUG:
		return "bus tracing only allowed";
	case IMX8MP_JTAG_SMODE_NO_DEBUG
	    | IMX8MP_KTE_KILL:
		return "disabled";
	}

	return "unknown";
}

int print_fuse_status_jtag_imx8mp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("jtag %s\n",
		fuse_status_jtag_imx8mp(ocotp_data)
	);

	return 0;
}

enum secure_status secure_status_jtag_imx8mp(unsigned char *ocotp_data)
{
	char *status_str;

	status_str = fuse_status_jtag_imx8mp(ocotp_data);

	if ((strcmp(status_str, "secure mode") == 0)
	    || (strcmp(status_str, "bus tracing only allowed") == 0)
	    || (strcmp(status_str, "disabled") == 0))
		return SECURE;

	if (strcmp(status_str, "unknown") == 0)
		return UNKNOWN;

	return INSECURE;
}

int is_secure_fuse_jtag_imx8mp(unsigned char *ocotp_data)
{
	if (secure_status_jtag_imx8mp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

int write_fuse_jtag_imx8mp(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX8MP_SJC_DISABLE_TRUE;
	if (pwrite(fd, &buf, sizeof(buf), IMX8MP_JTAG_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8MP_JTAG_OFFSET);
		return 1;
	}

	return 0;
}

enum secure_status secure_status_sd_boot_imx8mp(unsigned char *ocotp_data)
{
	uint32_t bootModeFuses_btFuseSel_mem;
	uint32_t forceBtFromFuse_mem;

	memcpy(&bootModeFuses_btFuseSel_mem,
		ocotp_data + IMX8MP_BOOT_MODE_FUSES_OFFSET, 4);

	memcpy(&forceBtFromFuse_mem,
		ocotp_data + IMX8MP_FORCE_BT_FROM_FUSE_OFFSET, 4);

	if ((bootModeFuses_btFuseSel_mem
	    & IMX8MP_BT_FUSE_SEL_MASK)
	    != IMX8MP_BT_FUSE_SEL_PROGRAMMED) {
		return INSECURE;
	}

	if ((forceBtFromFuse_mem
	    & IMX8MP_FORCE_BT_FROM_FUSE_MASK)
	    != IMX8MP_FORCE_BT_FROM_FUSE_ENABLE) {
		return INSECURE;
	}

	if ((bootModeFuses_btFuseSel_mem
	    & IMX8MP_BOOT_MODE_FUSES_MASK)
	    != IMX8MP_BOOT_MODE_FUSES_EMMC) {
		return UNKNOWN;
	}

	return SECURE;
}

int is_secure_fuse_sd_boot_imx8mp(unsigned char *ocotp_data)
{
	if (secure_status_sd_boot_imx8mp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

char *secure_status_to_string_sd_boot_imx8mp(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "allowed";
	case SECURE: return "disabled";
	default: return "unknown";
	}
}

int print_fuse_status_sd_boot_imx8mp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("sd_boot %s\n",
		secure_status_to_string_sd_boot_imx8mp(
			secure_status_sd_boot_imx8mp(ocotp_data)
		)
	);
	return 0;
}

int write_fuse_sd_boot_imx8mp(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX8MP_BT_FUSE_SEL_PROGRAMMED | IMX8MP_BOOT_MODE_FUSES_EMMC;
	if (pwrite(fd, &buf, sizeof(buf), IMX8MP_BOOT_MODE_FUSES_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8MP_BOOT_MODE_FUSES_OFFSET);
		return 1;
	}

	buf = IMX8MP_FORCE_BT_FROM_FUSE_ENABLE;
	if (pwrite(fd, &buf, sizeof(buf), IMX8MP_FORCE_BT_FROM_FUSE_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8MP_FORCE_BT_FROM_FUSE_OFFSET);
		return 1;
	}

	return 0;
}

enum secure_status secure_status_secureboot_imx8mp(unsigned char *ocotp_data)
{
	int i;
	uint32_t hash_mem[8];
	uint32_t sec_config_mem;

	// value of SRK region.
	for (i = 0; i < 8; i++)
		memcpy(&hash_mem[i],
			ocotp_data + IMX8MP_SRK_HASH_OFFSET + 4*i, 4);

	// close process. If the signature confirmation fails, the startup will fail.
	memcpy(&sec_config_mem,
			ocotp_data + IMX8MP_SEC_CONFIG_OFFSET, 4);


	for (i = 0; i < 8; i++) {
		if (hash_mem[i] != 0)
			break;
		if (i == 7)
			return INSECURE;
	}

	if ((sec_config_mem
	    & IMX8MP_SEC_CONFIG_MASK)
	    != IMX8MP_SEC_CONFIG_ENABLE) {
		return UNKNOWN;
	}

	return SECURE;
}

int is_secure_fuse_secureboot_imx8mp(unsigned char *ocotp_data)
{
	if (secure_status_secureboot_imx8mp(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

char *secure_status_to_string_secureboot_imx8mp(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "disabled";
	case SECURE: return "configured";
	default: return "configured (not enforced)";
	}
}

int print_fuse_status_secureboot_imx8mp(unsigned char *ocotp_data)
{
	PRINTF_CHECK("secureboot %s\n",
		secure_status_to_string_secureboot_imx8mp(
			secure_status_secureboot_imx8mp(ocotp_data)
		)
	);
	return 0;
}

int write_fuse_secureboot_srk_imx8mp(int fd)
{
	int i;
	uint32_t buf = 0x0;

	// value of SRK region.
	for (i = 0; i < 8; i++) {
		if (pwrite(fd, &srk_hashes[i], sizeof(srk_hashes[i]),
		    IMX8MP_SRK_HASH_OFFSET + 4*i) < 0) {
			fprintf(stderr,
				"Could not write 0x%08X to %s (offset is 0x%02X, SRK%d) : %m\n",
				srk_hashes[i], IMX_OCOTP_NVMEM, IMX8MP_SRK_HASH_OFFSET + 4*i, i
			);
			return 1;
		}
	}

	// cst_srk_lock.
	buf = IMX8MP_CST_SRK_LOCK_ENABLE;
	if (pwrite(fd, &buf, 4, IMX8MP_BOOT_CFG_LOCK_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X, SEC_CONFIG) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8MP_BOOT_CFG_LOCK_OFFSET
		);
		return 1;
	}

	return 0;
}

int write_fuse_secureboot_close_imx8mp(int fd)
{
	uint32_t buf = 0x0;

	// close.
	buf = IMX8MP_SEC_CONFIG_ENABLE;
	if (pwrite(fd, &buf, 4, IMX8MP_SEC_CONFIG_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X, SEC_CONFIG) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX8MP_SEC_CONFIG_OFFSET
		);
		return 1;
	}

	return 0;
}

int cmp_fuse_secureboot_srk_imx8mp(unsigned char *ocotp_data)
{
	int i;
	uint32_t buf = 0x0;

	for (i = 0; i < 8; i++) {
		memcpy(&buf,
			ocotp_data + IMX8MP_SRK_HASH_OFFSET + 4*i, 4);

		if (srk_hashes[i] != buf)
			return 1;
	}

	return 0;
}

int close_precheck_imx8mp(char *close_precheck_data)
{
	if (strcmp(close_precheck_data, "success\n") != 0) {
		fprintf(stderr,
			"Secureboot HAB failure logged.\n");
		return EXIT_FAILURE;
	}

	// If /boot/Image is a plain non-signed kernel then in open mode then
	// uboot will load it without validation, leaving hab_status of success
	// but not being able to boot after close.
	// User must ensure there is no /boot/Image to check /dev/mmcblkXbootY
	// was used.
	if (access("/boot/Image", F_OK) == 0) {
		fprintf(stderr,
			"/boot/Image exists. If /boot/Image is a non-signed kernel, Armadillo\n"
			"can not boot after the close operation is enforced.\n"
			"Remove file and reboot to confirm status is OK.\n"
		);
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
