// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#include "device-info.h"
#include "security_imx6ull.h"

enum secure_status secure_status_boot_cfg_lock_imx6ull(unsigned char *ocotp_data)
{
	uint32_t status_mem;

	memcpy(&status_mem,
		ocotp_data + IMX6ULL_BOOT_CFG_LOCK_OFFSET, 4);

	if ((status_mem
	    & IMX6ULL_BOOT_CFG_LOCK_MASK)
	    == IMX6ULL_BOOT_CFG_LOCK_OPWP) {
		return SECURE;
	}

	return INSECURE;
}

int is_secure_fuse_boot_cfg_lock_imx6ull(unsigned char *ocotp_data)
{
	if (secure_status_boot_cfg_lock_imx6ull(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

char *secure_status_to_string_lock_imx6ull(enum secure_status status)
{
	switch (status) {
	case INSECURE: return "open";
	default: return "write protected";
	}
}

int print_fuse_status_lock_imx6ull(unsigned char *ocotp_data)
{
	PRINTF_CHECK("boot_cfg_lock %s\n",
		secure_status_to_string_lock_imx6ull(
			secure_status_boot_cfg_lock_imx6ull(ocotp_data)
		)
	);

	return 0;
}

int write_fuse_lock_imx6ull(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX6ULL_BOOT_CFG_LOCK_OPWP;
	if (pwrite(fd, &buf, sizeof(buf), IMX6ULL_BOOT_CFG_LOCK_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX6ULL_BOOT_CFG_LOCK_OFFSET);
		return 1;
	}

	return 0;
}

char *fuse_status_jtag_imx6ull(unsigned char *ocotp_data)
{
	uint32_t status_mem;

	memcpy(&status_mem,
		ocotp_data + IMX6ULL_JTAG_OFFSET, 4);

	if ((status_mem & IMX6ULL_SJC_DISABLE_MASK)
	    == IMX6ULL_SJC_DISABLE_TRUE) {
		return "disabled";
	}

	switch (status_mem & IMX6ULL_JTAG_SMODE_MASK) {
	case IMX6ULL_JTAG_SMODE_JTAG_ENABLE:
		return "open";
	case IMX6ULL_JTAG_SMODE_UNKNOWN:
		return "unknown";
	}

	switch (status_mem &
		(IMX6ULL_JTAG_SMODE_MASK | IMX6ULL_KTE_MASK)) {
	case IMX6ULL_JTAG_SMODE_SECURE:
		return "secure mode. WARNING: bus tracing is allowed";
	case IMX6ULL_JTAG_SMODE_SECURE
		| IMX6ULL_KTE_KILL:
		return "secure mode";
	case IMX6ULL_JTAG_SMODE_NO_DEBUG:
		return "bus tracing only allowed";
	case IMX6ULL_JTAG_SMODE_NO_DEBUG
		| IMX6ULL_KTE_KILL:
		return "disabled";
	}

	return "unknown";
}

int print_fuse_status_jtag_imx6ull(unsigned char *ocotp_data)
{
	PRINTF_CHECK("jtag %s\n",
		fuse_status_jtag_imx6ull(ocotp_data)
	);

	return 0;
}

enum secure_status secure_status_jtag_imx6ull(unsigned char *ocotp_data)
{
	char *status_str;

	status_str = fuse_status_jtag_imx6ull(ocotp_data);

	if ((strcmp(status_str, "secure mode") == 0)
	    || (strcmp(status_str, "bus tracing only allowed") == 0)
	    || (strcmp(status_str, "disabled") == 0))
		return SECURE;

	if (strcmp(status_str, "unknown") == 0)
		return UNKNOWN;

	return INSECURE;
}

int is_secure_fuse_jtag_imx6ull(unsigned char *ocotp_data)
{
	if (secure_status_jtag_imx6ull(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

int write_fuse_jtag_imx6ull(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX6ULL_SJC_DISABLE_TRUE;
	if (pwrite(fd, &buf, sizeof(buf), IMX6ULL_JTAG_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX6ULL_JTAG_OFFSET);
		return 1;
	}

	return 0;
}

char *fuse_status_sd_boot_imx6ull(unsigned char *ocotp_data)
{
	uint32_t bootdev_usdhc_bus_mem;
	uint32_t btFuseSel_mem;

	memcpy(&bootdev_usdhc_bus_mem,
		ocotp_data + IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET, 4);

	memcpy(&btFuseSel_mem, ocotp_data + IMX6ULL_BT_FUSE_SEL_OFFSET, 4);

	if ((btFuseSel_mem
	    & IMX6ULL_BT_FUSE_SEL_MASK)
	    != IMX6ULL_BT_FUSE_SEL_PROGRAMMED) {
		return "allowed";
	}

	if ((bootdev_usdhc_bus_mem
	    & (IMX6ULL_BOOT_DEVICE_MASK
	    | IMX6ULL_USDHC_MASK))
	    != (IMX6ULL_BOOT_DEVICE_EMMC
	    | IMX6ULL_USDHC_EMMC)) {
		return "unknown";
	}

	if ((bootdev_usdhc_bus_mem
	    & IMX6ULL_BUS_WIDTH_MASK)
	    != IMX6ULL_BUS_WIDTH_8BIT) {
		return "disabled. WARNING: eMMC bus width is not 8-bit";
	}

	return "disabled";
}

int print_fuse_status_sd_boot_imx6ull(unsigned char *ocotp_data)
{
	PRINTF_CHECK("sd_boot %s\n",
		fuse_status_sd_boot_imx6ull(ocotp_data)
	);
	return 0;
}

enum secure_status secure_status_sd_boot_imx6ull(unsigned char *ocotp_data)
{
	char *status_str;

	status_str = fuse_status_sd_boot_imx6ull(ocotp_data);

	if (strcmp(status_str, "disabled") == 0)
		return SECURE;

	if (strcmp(status_str, "unknown") == 0)
		return UNKNOWN;

	return INSECURE;
}

int is_secure_fuse_sd_boot_imx6ull(unsigned char *ocotp_data)
{
	if (secure_status_sd_boot_imx6ull(ocotp_data) == SECURE)
		return EXIT_SUCCESS;
	return EXIT_FAILURE;
}

int write_fuse_sd_boot_imx6ull(int fd)
{
	uint32_t buf = 0x0;

	buf = IMX6ULL_BT_FUSE_SEL_PROGRAMMED;
	if (pwrite(fd, &buf, sizeof(buf), IMX6ULL_BT_FUSE_SEL_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX6ULL_BT_FUSE_SEL_OFFSET);
		return 1;
	}

	buf = IMX6ULL_BOOT_DEVICE_EMMC | IMX6ULL_BUS_WIDTH_8BIT;
	if (pwrite(fd, &buf, sizeof(buf), IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET) < 0) {
		fprintf(stderr,
			"Could not write 0x%08X to %s (offset is 0x%02X) : %m\n",
			buf, IMX_OCOTP_NVMEM, IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET);
		return 1;
	}

	return 0;
}
