// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#pragma once

#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>

#define IMX_OCOTP_NVMEM "/sys/bus/nvmem/devices/imx-ocotp0/nvmem"
#define IMX_OCOTP_NVMEM_IMX8ULP "/sys/bus/nvmem/devices/fsb_s400_fuse0/nvmem"

#define IMX8MP_HAB_STATUS "/sys/devices/platform/soc@0/hab_status"
#define IMX8ULP_SECUREBOOT_PATH "/sys/devices/platform/soc@0/27010000.efuse/secureboot"
#define CLOSE_PRECHECK_SIZE 20


// utils.c
int read_full(int fd, void *buf, int count);
int read_to_end(int fd, void *buf, int count);
void xfail(const char *message) __attribute__((noreturn));
char *xstrdup(const char *str);

// soc.c
enum soc {
	SOC_UNKNOWN = 0,
	IMX6ULL,
	IMX8MP,
	IMX8ULP,
};

enum soc get_soc(void);

// armadillo.c
int print_product_name(void);
int print_serial_num(void);
int print_mac_addr(void);
int print_se_param(void);
int print_ext_board_info(void);
int print_fuse_status(void);
int is_secure_fuse(void);
int write_fuse(void);
int show_srk(void);
int close_precheck(void);

enum board_type {
	A640_ES_PRODUCT_ID = 0x0097,
	A640_PRODUCT_ID = 0x009C,
	A610_ES_PRODUCT_ID = 0x00B4,
	A610_PRODUCT_ID = 0x00B7,
	G4_EVA_PRODUCT_ID = 0x00C6,
	G4_ES1_PRODUCT_ID = 0x00C7,
	G4_ES2_PRODUCT_ID = 0x00C8,
	G4_PRODUCT_ID = 0x00C9,
	G4_ES3_PRODUCT_ID = 0x00CB,
	HIGH_G1_ES1_PRODUCT_ID = 0x00E4,
	A6E_PRODUCT_ID = 0x00CE,
	X2_ES1_PRODUCT_ID = 0x00D1,
	X2_ES2_PRODUCT_ID = 0x00D2,
	X2_PRODUCT_ID = 0x00D3,
	A900_ES_PRODUCT_ID = 0x00E2,
	A900_PRODUCT_ID = 0x00E3,
	A9E_PRODUCT_ID = 0x0101,
	BOARD_TYPE_UNKNOWN,
};

struct armadillo {
	enum soc soc;
	enum board_type board_type;
	int mac_addr_num;
	const char *product_name;
	const char *se_param;
	uint32_t ext_board_count;
	const char *ext_board_name;
	const char *nvmem_path;
	unsigned char *ocotp_data;
	const char *close_precheck_path;
	char close_precheck_data[CLOSE_PRECHECK_SIZE];
};

extern struct armadillo armadillo;

int imx_ocotp_read(struct armadillo *device);
int close_precheck_read(struct armadillo *device);

// abos.c
int print_abos_version(void);

// device-info.c
extern bool is_print_env;
extern bool is_lock;
extern bool is_jtag;
extern bool is_sd_boot;
extern bool is_secureboot;
extern bool is_lock_write_fuse;
extern bool is_jtag_write_fuse;
extern bool is_sd_boot_write_fuse;
extern bool is_secureboot_srk_write_fuse;
extern bool is_secureboot_close_write_fuse;
extern bool is_lock_is_secure;
extern bool is_jtag_is_secure;
extern bool is_sd_boot_is_secure;
extern bool is_secureboot_is_secure;
extern uint32_t srk_hashes[8];

enum secure_status {
	INSECURE,
	SECURE,
	UNKNOWN
};

#define _cleanup_(f) __attribute__((cleanup(f)))

static inline void closep(int *fd)
{
	if (*fd < 3)
		return;
	close(*fd);
	*fd = -1;
}

#define PRINTF_CHECK(...)	do { if (printf(__VA_ARGS__) < 0) return 1; } while (0)
