// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>

#include "device-info.h"

char *version = "1.4.2";
bool is_print_env = false;
bool is_yes_i_know;
bool is_lock;
bool is_jtag;
bool is_sd_boot;
bool is_secureboot;
bool is_lock_write_fuse;
bool is_jtag_write_fuse;
bool is_sd_boot_write_fuse;
bool is_secureboot_srk_write_fuse;
bool is_secureboot_close_write_fuse;
bool is_lock_is_secure;
bool is_jtag_is_secure;
bool is_sd_boot_is_secure;
bool is_secureboot_is_secure;
bool is_show_srk;

uint32_t srk_hashes[8];

/*
 * Long options without short options
 * are defined by enums starting with CHAR_MAX + 1
 */
enum {
	SE_PARAM_OPTION = CHAR_MAX + 1,
	SRK_HASH_OPTION,
	YES_I_KNOW_OPTION,
	SHOW_SRK_OPTION,
	CLOSE_PRECHECK
};

static struct option long_options[] = {
	{"abos-version",	no_argument,	NULL,	'a'},
	{"additional-board",	no_argument,	NULL,	'b'},
	{"env",	no_argument,	NULL,	'e'},
	{"fuse-status",	required_argument,	NULL,	'f'},
	{"help",	no_argument,	NULL,	'h'},
	{"is-secure",	required_argument,	NULL,	'i'},
	{"mac-address",	no_argument,	NULL,	'm'},
	{"product",	no_argument,	NULL,	'p'},
	{"serial",	no_argument,	NULL,	's'},
	{"version",	no_argument,	NULL,	'v'},
	{"write-fuse",	required_argument,	NULL,	'w'},
	{"close-precheck",	no_argument,	NULL,	CLOSE_PRECHECK},
	{"se-param",	no_argument,	NULL,	SE_PARAM_OPTION},
	{"show-srk",	no_argument,	NULL,	SHOW_SRK_OPTION},
	{"srk-hash",	required_argument,	NULL,	SRK_HASH_OPTION},
	{"yes-i-know-what-i-am-doing",	no_argument,	NULL,	YES_I_KNOW_OPTION},
	{0,			0,				0,	0}
};

int parse_one_security_fuse_optargs(char *argument, char *option)
{
	if (strcmp(option, "fuse-status") == 0) {
		if (strcmp(argument, "lock") == 0) {
			is_lock = true;
			return 0;
		}

		if (strcmp(argument, "jtag") == 0) {
			is_jtag = true;
			return 0;
		}

		if (strcmp(argument, "sd_boot") == 0) {
			is_sd_boot = true;
			return 0;
		}

		if (strcmp(argument, "secureboot") == 0) {
			is_secureboot = true;
			return 0;
		}
	} else if (strcmp(option, "write-fuse") == 0) {
		if (strcmp(argument, "lock") == 0) {
			is_lock_write_fuse = true;
			return 0;
		}

		if (strcmp(argument, "jtag") == 0) {
			is_jtag_write_fuse = true;
			return 0;
		}

		if (strcmp(argument, "sd_boot") == 0) {
			is_sd_boot_write_fuse = true;
			return 0;
		}

		if (strcmp(argument, "secureboot_srk") == 0) {
			is_secureboot_srk_write_fuse = true;
			return 0;
		}

		if (strcmp(argument, "secureboot_close") == 0) {
			is_secureboot_close_write_fuse = true;
			return 0;
		}
	} else if (strcmp(option, "is-secure") == 0) {
		if (strcmp(argument, "lock") == 0) {
			is_lock_is_secure = true;
			return 0;
		}

		if (strcmp(argument, "jtag") == 0) {
			is_jtag_is_secure = true;
			return 0;
		}

		if (strcmp(argument, "sd_boot") == 0) {
			is_sd_boot_is_secure = true;
			return 0;
		}

		if (strcmp(argument, "secureboot") == 0) {
			is_secureboot_is_secure = true;
			return 0;
		}
	}

	return 1;
}

int parse_security_fuse_selection(char *arguments, char *option)
{
	char *argument;

	argument = strtok(arguments, ",");
	if (argument == NULL)
		return 1;

	if (parse_one_security_fuse_optargs(argument, option) == 1)
		return 1;

	// strtok() takes null as its first argument for the second and subsequent times.
	while ((argument = strtok(NULL, ",")) != NULL)
		if (parse_one_security_fuse_optargs(argument, option) == 1)
			return 1;

	return 0;
}

static int parse_srk_hash(char *str_srk_hashes, uint32_t *srk_hashes, int *srk_hash_count)
{
	char *str_srk_hash;
	uint32_t srk_hash;

	str_srk_hash = strtok(str_srk_hashes, ",");
	if (str_srk_hash == NULL)
		return 1;
	if (strncmp(str_srk_hash, "0x", 2) != 0)
		return 2;
	srk_hash = strtoul(str_srk_hash, NULL, 16);
	memcpy(&srk_hashes[*srk_hash_count], &srk_hash, 4);
	*srk_hash_count += 1;

	// // strtok() takes null as its first argument for the second and subsequent times.
	while ((str_srk_hash = strtok(NULL, ",")) != NULL) {
		if (str_srk_hash == NULL)
			return 1;
		if (strncmp(str_srk_hash, "0x", 2) != 0)
			return 2;
		srk_hash = strtoul(str_srk_hash, NULL, 16);
		memcpy(&srk_hashes[*srk_hash_count], &srk_hash, 4);
		*srk_hash_count += 1;
	}

	if (*srk_hash_count != 8)
		return 3;

	return 0;
}

static void __attribute__ ((__noreturn__)) print_version(char *argv0) {
	printf("%s version %s\n", argv0, version);
	exit(EXIT_SUCCESS);
}

static void __attribute__ ((__noreturn__)) help(char *argv0) {
	printf("Usage: %s [options]\n", argv0);
	printf("Options:\n");
	printf("  -a, --abos-version: show ABOS version\n");
	printf("  -b, --additional-board: show additional board info\n");
	printf("  -e, --env: show as environmental variables\n");
	printf("  -f, --fuse-status: Print the fuse`s status related to the security.\n");
	printf("                     This option takes the following arguments:\n");
	printf("                     lock, jtag, sd_boot, secureboot\n");
	printf("  -h, --help: show this help\n");
	printf("  -i, --is-secure: Return 0 if fuse`s settings related to the security\n");
	printf("                     is secure. it can not be used with other options.\n");
	printf("                     This option takes the following one argument:\n");
	printf("                     lock, jtag, sd_boot, secureboot\n");
	printf("  -m, --mac-address: print the MAC address of this board\n");
	printf("  -p, --product: print the product name of this board\n");
	printf("  -s, --serial: print the serial number of this board\n");
	printf("  -v, --version: show device-info version\n");
	printf("  -w, --write-fuse: Write fuses related to the security.\n");
	printf("                    Once written to fuses, it cannot be reverted again.\n");
	printf("                    When writing, specify the '--yes-i-know-what-i-am-doing'\n");
	printf("                    option.\n");
	printf("                    This option takes the following arguments:\n");
	printf("                    lock, jtag, sd_boot, secureboot_srk, secureboot_close\n");
	printf("  --close-precheck: return 0 if it is acceptable to perform the close handling.\n");
	printf("  --se-param: print param of secure element in a format that\n");
	printf("              can be exported directly to EX_SSS_BOOT_SSS_PORT\n");
	printf("  --show-srk: show SRK Hash.\n");
	printf("  --srk-hash: Specify SRK_HASH to be written to eFuse.\n");
	printf("              This option is used with the 'secureboot_srk' argument of\n");
	printf("              the '--write-fuse' option.\n");
	printf("  --yes-i-know-what-i-am-doing: Specify when making irreversible changes.\n");
	exit(EXIT_SUCCESS);
}

int main(int argc, char *argv[])
{
	int c;
	int option_index = 0;
	int ret = 0;
	int srk_hash_count = 0;

	bool is_print_abos_version = false;
	bool is_print_fuse_status = false;
	bool is_secure = false;
	bool is_print_mac_addr = false;
	bool is_print_product_name = false;
	bool is_print_serial_num = false;
	bool is_print_se_param = false;
	bool is_print_ext_board_info = false;
	bool is_srk_hash = false;
	bool is_write_fuse = false;
	bool is_show_srk = false;
	bool is_close_precheck = false;

	while ((c = getopt_long(argc, argv, "abef:hi:mpsvw:", long_options, &option_index)) > 0) {
		switch (c) {
		case 'a':
			is_print_abos_version = true;
			break;
		case 'b':
			is_print_ext_board_info = true;
			break;
		case 'e':
			is_print_env = true;
			break;
		case 'f':
			is_print_fuse_status = true;
			if (parse_security_fuse_selection(optarg, "fuse-status") == 1) {
				fprintf(stderr, "--fuse-status: unrecognized selection.\n");
				fprintf(stderr, "See help option.\n");
				exit(EXIT_FAILURE);
			}
			break;
		case 'h':
			help(argv[0]);
		case 'i':
			is_secure = true;
			if (parse_one_security_fuse_optargs(optarg, "is-secure") == 1) {
				fprintf(stderr, "--is-secure: unrecognized selection.\n");
				fprintf(stderr, "See help option.\n");
				exit(EXIT_FAILURE);
			}
			break;
		case 'm':
			is_print_mac_addr = true;
			break;
		case 'p':
			is_print_product_name = true;
			break;
		case 's':
			is_print_serial_num = true;
			break;
		case 'v':
			print_version(argv[0]);
		case 'w':
			is_write_fuse = true;
			if (parse_security_fuse_selection(optarg, "write-fuse") == 1) {
				fprintf(stderr, "--write-fuse: unrecognized selection.\n");
				fprintf(stderr, "See help option.\n");
				exit(EXIT_FAILURE);
			}
			break;
		case CLOSE_PRECHECK:
			is_close_precheck = true;
			break;
		case SE_PARAM_OPTION:
			is_print_se_param = true;
			break;
		case SRK_HASH_OPTION:
			is_srk_hash = true;
			switch (parse_srk_hash(optarg, srk_hashes, &srk_hash_count)) {
			case 1:
				fprintf(stderr, "--srk-hash: unrecognized the srk hash value.\n");
				fprintf(stderr, "See help option.\n");
				exit(EXIT_FAILURE);
			case 2:
				fprintf(stderr,
					"--srk-hash: the srk hash (%d/8) value must start with '0x'.\n",
					srk_hash_count + 1
				);
				exit(EXIT_FAILURE);
			case 3:
				fprintf(stderr,
					"--srk-hash: the number of srk hash values does not match. "
				);
				fprintf(stderr,
					"number of srk hash found (%d/8).\n",
					srk_hash_count
				);
				exit(EXIT_FAILURE);
			default:
				break;
			}
			break;
		case YES_I_KNOW_OPTION:
			is_yes_i_know = true;
			break;
		case SHOW_SRK_OPTION:
			is_show_srk = true;
			break;
		default:
			fprintf(stderr, "Execute '%s --help'.\n", argv[0]);
			return EXIT_FAILURE;
		}
	}

	if (argc == 1)
		help(argv[0]);

	if (is_secure && argc > 4) {
		fprintf(stderr, "Can not use --is-secure (-i) with other options.\n");
		return EXIT_FAILURE;
	}

	if (is_show_srk && argc > 3) {
		fprintf(stderr, "Can not use --show-srk with other options.\n");
		return EXIT_FAILURE;
	}

	if (is_close_precheck && argc > 3) {
		fprintf(stderr, "Can not use --hab-status with other options.\n");
		return EXIT_FAILURE;
	}

	if (optind < argc) {
		printf("Unrecognized option: ");
		while (optind < argc)
			printf("%s ", argv[optind++]);
		printf("\n");
	}

	if (is_print_env && is_print_fuse_status) {
		fprintf(stderr, "Do not use --fuse-status (-f) with --env (-e).\n");
		return EXIT_FAILURE;
	}

	if (is_print_env && is_write_fuse) {
		fprintf(stderr, "Do not use --write-fuse (-w) with --env (-e).\n");
		return EXIT_FAILURE;
	}

	if (is_write_fuse && !is_yes_i_know) {
		fprintf(stderr,
			"'--yes-i-know-what-i-am-doing' option is needed to write values to fuses.\n"
		);
		fprintf(stderr, "selected arguments:");
		if (is_lock_write_fuse)
			fprintf(stderr, " lock");
		if (is_jtag_write_fuse)
			fprintf(stderr, " jtag");
		if (is_sd_boot_write_fuse)
			fprintf(stderr, " sd_boot");
		if (is_secureboot_srk_write_fuse)
			fprintf(stderr, " secureboot_srk");
		if (is_secureboot_close_write_fuse)
			fprintf(stderr, " secureboot_close");
		fprintf(stderr, "\n");
		return EXIT_FAILURE;
	}

	if (is_write_fuse && is_secureboot_srk_write_fuse && !is_srk_hash) {
		fprintf(stderr,
			"'--srk-hash' option is needed to write SRK hash to fuses for secureboot.\n"
		);
		return EXIT_FAILURE;
	}

	if (is_print_abos_version)
		ret |= print_abos_version();
	if (is_write_fuse)
		ret |= write_fuse();
	if (is_print_fuse_status)
		ret |= print_fuse_status();
	if (is_print_mac_addr)
		ret |= print_mac_addr();
	if (is_print_product_name)
		ret |= print_product_name();
	if (is_print_serial_num)
		ret |= print_serial_num();
	if (is_print_se_param)
		ret |= print_se_param();
	if (is_print_ext_board_info)
		ret |= print_ext_board_info();
	if (is_secure)
		ret |= is_secure_fuse();
	if (is_show_srk)
		ret |= show_srk();
	if (is_close_precheck)
		ret |= close_precheck();

	free(armadillo.ocotp_data);
	return ret;
}
