// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */

#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <endian.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>

#include "device-info.h"
#include "security_imx8mp.h"
#include "security_imx6ull.h"
#include "security_imx8ulp.h"

#define IMX_OCOTP_MVMEM_IMX8MP_PRODUCT   0xE0
#define IMX_OCOTP_MVMEM_IMX6ULL_PRODUCT  0xA0
#define IMX_OCOTP_MVMEM_IMX8ULP_PRODUCT  0x658
#define IMX_OCOTP_NVMEM_A900_PRODUCT     0x656
#define IMX_OCOTP_MVMEM_PRODUCT_SIZE     2

#define IMX_OCOTP_MVMEM_IMX8MP_LOT_SN    0xE4
#define IMX_OCOTP_MVMEM_IMX6ULL_LOT_SN   0xA4
#define IMX_OCOTP_MVMEM_IMX8ULP_LOT_SN   0x65C
#define IMX_OCOTP_MVMEM_LOT_SN_SIZE      4

#define IMX_OCOTP_MVMEM_IMX8MP_MAC_ADDR  0x90
#define IMX_OCOTP_MVMEM_IMX6ULL_MAC_ADDR 0x88
#define IMX_OCOTP_MVMEM_IMX8ULP_MAC_ADDR 0xAC
#define IMX_OCOTP_MVMEM_MAC_ADDR_SIZE    6

#define A610_MIN_SERIAL_NUM_HAVE_SE      0x00B700430001
#define A640_MIN_SERIAL_NUM_HAVE_SE      0x009C01490001

#define SET_DEFAULT_PRODUCT_NAME(name) \
	do { \
		if (armadillo.product_name == NULL) \
			armadillo.product_name = (name); \
	} while (0)

enum ext_board_type {
	A6E_D8A4_PRODUCT_ID = 0x000e,
	EXT_BOARD_TYPE_UNKNOWN,
};

#define EXT_BOARD_COUNT_PATH "/proc/device-tree/chosen/atmark,extboard-count"
#define EXT_BOARD_PRODUCT_PATH "/proc/device-tree/chosen/atmark,extboard-info"

struct armadillo armadillo;

static enum board_type get_board_type(void)
{
	uint16_t board_type;

	switch (armadillo.soc) {
	case IMX8MP:
		memcpy(&board_type, armadillo.ocotp_data + IMX_OCOTP_MVMEM_IMX8MP_PRODUCT,
			IMX_OCOTP_MVMEM_PRODUCT_SIZE);
		break;
	case IMX6ULL:
		memcpy(&board_type, armadillo.ocotp_data + IMX_OCOTP_MVMEM_IMX6ULL_PRODUCT,
			IMX_OCOTP_MVMEM_PRODUCT_SIZE);
		break;
	case IMX8ULP:
		memcpy(&board_type, armadillo.ocotp_data + IMX_OCOTP_MVMEM_IMX8ULP_PRODUCT,
			IMX_OCOTP_MVMEM_PRODUCT_SIZE);
		break;
	default:
		return BOARD_TYPE_UNKNOWN;
	}

	return board_type;
}

static uint64_t get_serial_num(void)
{
	unsigned char *buf;
	uint32_t lot_sn;

	switch (armadillo.soc) {
	case IMX6ULL:
		buf = armadillo.ocotp_data + IMX_OCOTP_MVMEM_IMX6ULL_LOT_SN;
		break;
	case IMX8MP:
		buf = armadillo.ocotp_data + IMX_OCOTP_MVMEM_IMX8MP_LOT_SN;
		break;
	case IMX8ULP:
		buf = armadillo.ocotp_data + IMX_OCOTP_MVMEM_IMX8ULP_LOT_SN;
		break;
	default:
		// should never happen after init_armadillo() success...
		xfail("Invalid soc after init_armadillo() success");
	}

	memcpy(&lot_sn, buf, 4);
	return ((uint64_t)armadillo.board_type << 32) + lot_sn;
}

static uint32_t read_ext_board_data(const char *target_path)
{
	FILE *fp;
	int ret;
	uint32_t data;

	fp = fopen(target_path, "r");
	if (fp == NULL)
		return 0;

	ret = fread(&data, sizeof(data), 1, fp);
	fclose(fp);

	if (ret <= 0)
		return 0;

	return be32toh(data);
}

static void get_ext_board_info(void)
{
	enum ext_board_type ext_board_product_id;

	ext_board_product_id = read_ext_board_data(EXT_BOARD_PRODUCT_PATH);
	switch (ext_board_product_id) {
	case A6E_D8A4_PRODUCT_ID:
		armadillo.ext_board_name = "Armadillo-IoT A6E +Di8+Ai4 Additional Board";
		break;
	default:
		return;
	}

	armadillo.ext_board_count = read_ext_board_data(EXT_BOARD_COUNT_PATH);
}

static void get_a900_product_name(void)
{
	uint16_t board_type;

	memcpy(&board_type, armadillo.ocotp_data + IMX_OCOTP_NVMEM_A900_PRODUCT,
		IMX_OCOTP_MVMEM_PRODUCT_SIZE);

	switch (board_type) {
	case A9E_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-IoT A9E");
		break;
	default:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-900");
	}
}

static int parse_conf(void)
{
	FILE *conffile;
	int rc = 0;
	const char *config_file_path = "/usr/lib/device-info/device-info.conf";
	const char *env_config = getenv("DEVICE_INFO_CONF");

	if (env_config != NULL)
		config_file_path = env_config;

	conffile = fopen(config_file_path, "r");
	if (conffile == NULL)
		return 0;

	char *line = NULL;
	size_t line_size = 0;
	ssize_t n;

	while (errno = 0, (n = getline(&line, &line_size, conffile)) >= 0) {
		char *key, *val;
		size_t key_len;

		/* Parse key=value format */
		key = strtok(line, "=");
		if (key == NULL) /* blank line */
			continue;
		if (*key == '#') /* comment */
			continue;

		/* Remove trailing spaces from key */
		key_len = strlen(key);
		while (key_len > 0 && isspace(key[key_len - 1])) {
			key[key_len - 1] = 0;
			key_len--;
		}

		if (key_len == 0) /* should never happen */
			continue;

		/* rest of line */
		val = strtok(NULL, "");

		if (!val)
			continue;

		while (isspace(*val))
			val++;

		/* Remove trailing whitespace including newlines */
		size_t val_len = strlen(val);

		while (val_len > 0 && isspace(val[val_len - 1])) {
			val[val_len - 1] = '\0';
			val_len--;
		}

		if (*val == '\0')
			continue;

		/* Check if this is PRODUCT_NAME (case-insensitive) */
		if (!strcasecmp(key, "PRODUCT_NAME")) {
			free((void *)armadillo.product_name);
			armadillo.product_name = xstrdup(val);
			continue;
		}
	}

	if (n < 0 && errno != 0) {
		rc = -errno;
		fprintf(stderr, "Error reading file: %s\n", strerror(errno));
	}

	free(line);
	fclose(conffile);
	return rc;
}

static int init_armadillo(void)
{
	armadillo.soc = get_soc();
	if (armadillo.soc == SOC_UNKNOWN) {
		return false;
	}

	if (armadillo.soc == IMX8ULP)
		armadillo.nvmem_path = IMX_OCOTP_NVMEM_IMX8ULP;
	else
		armadillo.nvmem_path = IMX_OCOTP_NVMEM;

	if (imx_ocotp_read(&armadillo) < 0) {
		return false;
	}

	if (armadillo.soc == IMX8ULP)
		armadillo.close_precheck_path = IMX8ULP_SECUREBOOT_PATH;
	else if (armadillo.soc == IMX8MP)
		armadillo.close_precheck_path = IMX8MP_HAB_STATUS;
	else
		armadillo.close_precheck_path = NULL;

	armadillo.board_type = get_board_type();
	armadillo.mac_addr_num = 1;
	parse_conf();
	switch (armadillo.board_type) {
	case A640_ES_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-640");
		break;
	case A640_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-640");
		if (get_serial_num() >= A640_MIN_SERIAL_NUM_HAVE_SE)
			armadillo.se_param = "/dev/i2c-0:0x48";
		break;
	case A610_ES_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-610");
		break;
	case A610_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-610");
		if (get_serial_num() >= A610_MIN_SERIAL_NUM_HAVE_SE)
			armadillo.se_param = "/dev/i2c-0:0x48";
		break;
	case G4_EVA_PRODUCT_ID:
	case G4_ES1_PRODUCT_ID:
	case G4_ES2_PRODUCT_ID:
	case G4_PRODUCT_ID:
	case G4_ES3_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-IoT G4");
		armadillo.mac_addr_num = 2;
		armadillo.se_param = "/dev/i2c-2:0x48";
		break;
	case HIGH_G1_ES1_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("high-g1-es1");
		armadillo.se_param = "/dev/i2c-2:0x48";
		break;
	case A6E_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-IoT A6E");
		armadillo.se_param = "/dev/i2c-1:0x48";
		get_ext_board_info();
		break;
	case X2_ES1_PRODUCT_ID:
	case X2_ES2_PRODUCT_ID:
	case X2_PRODUCT_ID:
		SET_DEFAULT_PRODUCT_NAME("Armadillo-X2");
		armadillo.se_param = "/dev/i2c-2:0x48";
		break;
	case A900_ES_PRODUCT_ID:
	case A900_PRODUCT_ID:
		get_a900_product_name();
		armadillo.se_param = "/dev/i2c-6:0x48";
		break;
	default:
		fprintf(stderr, "unsupported products: %X\n", armadillo.board_type);
		return false;
	}

	return true;
}

int print_product_name(void)
{
	if (!init_armadillo())
		return EXIT_FAILURE;

	if (is_print_env)
		PRINTF_CHECK("export AT_PRODUCT_NAME='%s'\n", armadillo.product_name);
	else
		PRINTF_CHECK("%s\n", armadillo.product_name);

	return EXIT_SUCCESS;
}

int print_serial_num(void)
{
	uint64_t serial_num;

	if (!init_armadillo())
		return EXIT_FAILURE;

	serial_num = get_serial_num();

	if (is_print_env)
		PRINTF_CHECK("export AT_SERIAL_NUMBER=%012"PRIX64"\n", serial_num);
	else
		PRINTF_CHECK("%012"PRIX64"\n", serial_num);

	return EXIT_SUCCESS;
}

int print_mac_addr(void)
{
	int i, offset = 0;

	if (!init_armadillo())
		return EXIT_FAILURE;

	switch (armadillo.soc) {
	case IMX6ULL:
		offset = IMX_OCOTP_MVMEM_IMX6ULL_MAC_ADDR;
		break;
	case IMX8MP:
		offset = IMX_OCOTP_MVMEM_IMX8MP_MAC_ADDR;
		break;
	case IMX8ULP:
		offset = IMX_OCOTP_MVMEM_IMX8ULP_MAC_ADDR;
		goto imx8ulp_mac;
		break;
	default:
		// should never happen after init_armadillo() success...
		xfail("Invalid soc after init_armadillo() success");
	}

	if (armadillo.board_type == HIGH_G1_ES1_PRODUCT_ID)
		offset += 6;

	for (i = 0; i < armadillo.mac_addr_num; i++) {
		unsigned char *buf = armadillo.ocotp_data + offset + i * 6;

		if (is_print_env)
			PRINTF_CHECK("export AT_LAN_MAC%d=%02X:%02X:%02X:%02X:%02X:%02X\n",
				    i+1, buf[5], buf[4], buf[3], buf[2], buf[1], buf[0]);
		else
			PRINTF_CHECK("eth%d: %02X:%02X:%02X:%02X:%02X:%02X\n",
				    i, buf[5], buf[4], buf[3], buf[2], buf[1], buf[0]);
	}
	return EXIT_SUCCESS;

imx8ulp_mac:
	for (i = 0; i < armadillo.mac_addr_num; i++) {
		unsigned char *buf = armadillo.ocotp_data + offset + i * 6;

		if (is_print_env)
			PRINTF_CHECK("export AT_LAN_MAC%d=%02X:%02X:%02X:%02X:%02X:%02X\n",
				    i+1, buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
		else
			PRINTF_CHECK("eth%d: %02X:%02X:%02X:%02X:%02X:%02X\n",
				    i, buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
	}
	return EXIT_SUCCESS;
}

int print_se_param(void)
{
	if (!init_armadillo())
		return EXIT_FAILURE;

	/* Old Armadillo 640 or 610 do not have se050,
	 * but we do not want things like podman_start's add_armadillo_env to fail.
	 * As rule of thumb: if we print as environment variable, the caller likely
	 * requested other env vars and can check it is not empty.
	 * If this was not env, then likely only se050 was requested, e.g.
	 * `EX_SSS_BOOT_SSS_PORT="$(device-info --se-param)"`, and we should fail */
	if (armadillo.se_param == NULL && !is_print_env)
		return EXIT_FAILURE;

	if (is_print_env)
		PRINTF_CHECK("export AT_SE_PARAM=%s\n", armadillo.se_param ?: "");
	else
		PRINTF_CHECK("%s\n", armadillo.se_param);

	return EXIT_SUCCESS;
}

int print_ext_board_info(void)
{
	uint32_t i;

	if (!init_armadillo())
		return EXIT_FAILURE;

	if (is_print_env) {
		for (i = 0; i < armadillo.ext_board_count; i++) {
			PRINTF_CHECK("export AT_ADD_BOARD_NAME_%u='%s'\n",
				    i + 1, armadillo.ext_board_name);
		}
		PRINTF_CHECK("export AT_ADD_BOARD_COUNT='%u'\n", armadillo.ext_board_count);
	} else {
		for (i = 0; i < armadillo.ext_board_count; i++) {
			PRINTF_CHECK("additional board%u: %s\n", i + 1, armadillo.ext_board_name);
		}
		PRINTF_CHECK("additional board count: %u\n", armadillo.ext_board_count);
	}

	return EXIT_SUCCESS;
}

int print_fuse_status(void)
{
	int rc = 1;

	if (!init_armadillo())
		return EXIT_FAILURE;

	switch (armadillo.soc) {
	case IMX6ULL:
		if (is_lock) {
			rc = print_fuse_status_lock_imx6ull(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_jtag) {
			rc = print_fuse_status_jtag_imx6ull(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_sd_boot) {
			rc = print_fuse_status_sd_boot_imx6ull(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		break;
	case IMX8MP:
		if (is_lock) {
			rc = print_fuse_status_lock_imx8mp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_jtag) {
			rc = print_fuse_status_jtag_imx8mp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_sd_boot) {
			rc = print_fuse_status_sd_boot_imx8mp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_secureboot) {
			rc = print_fuse_status_secureboot_imx8mp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		break;
	case IMX8ULP:
		if (is_lock) {
			fprintf(stderr, "displaying lock status is not supported for this SoC.\n");
			return EXIT_FAILURE;
		}
		if (is_jtag) {
			rc = print_fuse_status_jtag_imx8ulp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_sd_boot) {
			rc = print_fuse_status_sd_boot_imx8ulp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		if (is_secureboot) {
			rc = print_fuse_status_secureboot_imx8ulp(armadillo.ocotp_data);
			if (rc != 0)
				return EXIT_FAILURE;
		}
		break;
	default:
		fprintf(stderr, "fuse status printing is not supported for this SoC.\n");
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}

int write_fuse(void)
{
	int rc = 1;

	if (!init_armadillo())
		return EXIT_FAILURE;

	_cleanup_(closep) int fd = open(armadillo.nvmem_path, O_RDWR);

	if (fd < 0) {
		fprintf(stderr, "failed to open %s: %m\n", armadillo.nvmem_path);
		return EXIT_FAILURE;
	}

	switch (armadillo.soc) {
	case IMX6ULL:
		if (is_jtag_write_fuse) {
			switch (secure_status_jtag_imx6ull(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_jtag_imx6ull(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("JTAG disabled.\n");
				break;
			case SECURE:
				PRINTF_CHECK("JTAG already disabled.\n");
				break;
			default:
				fprintf(stderr,
					"Something else written on fuses related to JTAG.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_sd_boot_write_fuse) {
			switch (secure_status_sd_boot_imx6ull(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_sd_boot_imx6ull(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("SD boot disabled.\n");
				break;
			case SECURE:
				PRINTF_CHECK("SD boot already disabled.\n");
				break;
			default:
				fprintf(stderr,
					"Something else written on fuses related to SD boot.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_lock_write_fuse) {
			if ((secure_status_jtag_imx6ull(armadillo.ocotp_data) == SECURE)
			    && (secure_status_sd_boot_imx6ull(armadillo.ocotp_data) == SECURE)) {
				switch (secure_status_boot_cfg_lock_imx6ull(armadillo.ocotp_data)) {
				case INSECURE:
					rc = write_fuse_lock_imx6ull(fd);
					if (rc != 0)
						return EXIT_FAILURE;
					PRINTF_CHECK(
						"Write-protected fuse areas related to JTAG and SD boot.\n"
					);
					break;
				case SECURE:
					PRINTF_CHECK(
						"Already Write-protected fuse areas related to JTAG and SD boot.\n");
					break;
				default:
					fprintf(stderr,
						"Something is written in the fuse that locks the BOOT-related fuses.\n"
					);
					return EXIT_FAILURE;
				}
			} else {
				fprintf(stderr,
					"Refusing to write to BOOT_CFG_LOCK: JTAG or SD boot have not been restricted.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_secureboot_srk_write_fuse) {
			fprintf(stderr, "Secure boot is not supported for this SoC.\n");
			return EXIT_FAILURE;
		}
		if (is_secureboot_close_write_fuse) {
			fprintf(stderr, "Secure boot is not supported for this SoC.\n");
			return EXIT_FAILURE;
		}
		break;
	case IMX8MP:
		if (is_jtag_write_fuse) {
			switch (secure_status_jtag_imx8mp(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_jtag_imx8mp(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("JTAG disabled.\n");
				break;
			case SECURE:
				PRINTF_CHECK("JTAG already disabled.\n");
				break;
			default:
				fprintf(stderr,
					"Something else written on fuses related to JTAG.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_sd_boot_write_fuse) {
			switch (secure_status_sd_boot_imx8mp(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_sd_boot_imx8mp(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("SD boot disabled.\n");
				break;
			case SECURE:
				PRINTF_CHECK("SD boot already disabled.\n");
				break;
			default:
				fprintf(stderr,
					"Something else written on fuses related to SD boot.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_lock_write_fuse) {
			if ((secure_status_jtag_imx8mp(armadillo.ocotp_data) == SECURE)
			    && (secure_status_sd_boot_imx8mp(armadillo.ocotp_data) == SECURE)) {
				switch (secure_status_boot_cfg_lock_imx8mp(armadillo.ocotp_data)) {
				case INSECURE:
					rc = write_fuse_lock_imx8mp(fd);
					if (rc != 0)
						return EXIT_FAILURE;
					PRINTF_CHECK(
						"Write-protected fuse areas related to JTAG and SD boot.\n"
					);
					break;
				case SECURE:
					PRINTF_CHECK(
						"Already Write-protected fuse areas related to JTAG and SD boot.\n"
					);
					break;
				default:
					fprintf(stderr,
						"Something is written in the fuse that locks the BOOT-related fuses.\n"
					);
					return EXIT_FAILURE;
				}
			} else {
				fprintf(stderr,
				    "Refusing to write to BOOT_CFG_LOCK: JTAG or SD boot have not been restricted.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_secureboot_srk_write_fuse) {
			switch (secure_status_secureboot_imx8mp(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_secureboot_srk_imx8mp(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK(
					"Secure boot configured (not enforced).\n"
				);
				break;
			default:
				if (cmp_fuse_secureboot_srk_imx8mp(armadillo.ocotp_data) != 0) {
					fprintf(stderr,
						"Refusing to write to SRK Hash: different SRK Hash is already written.\n"
					);
					return EXIT_FAILURE;
				}
				break;
			}
		}
		if (is_secureboot_close_write_fuse) {
			switch (secure_status_secureboot_imx8mp(armadillo.ocotp_data)) {
			case INSECURE:
				fprintf(stderr,
					"Refusing to close: SRK Hash is not written.\n"
				);
				return EXIT_FAILURE;
			case SECURE:
				fprintf(stderr,
					"Refusing to close: secure boot is already enabled.\n"
				);
				return EXIT_FAILURE;
			default:
				rc = write_fuse_secureboot_close_imx8mp(fd);
				if (rc != 0)
					return 1;
				PRINTF_CHECK("Secure boot configured.\n");
				break;
			}
		}
		break;
	case IMX8ULP:
		if (is_jtag_write_fuse) {
			switch (secure_status_jtag_imx8ulp(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_jtag_imx8ulp(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("JTAG disabled.\n");
				break;
			default:
				PRINTF_CHECK("JTAG already disabled.\n");
				break;
			}
		}
		if (is_sd_boot_write_fuse) {
			switch (secure_status_sd_boot_imx8ulp(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_sd_boot_imx8ulp(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("SD boot disabled.\n");
				break;
			case SECURE:
				PRINTF_CHECK("SD boot already disabled.\n");
				break;
			default:
				fprintf(stderr,
					"Something else written on fuses related to SD boot.\n"
				);
				return EXIT_FAILURE;
			}
		}
		if (is_lock_write_fuse) {
			fprintf(stderr,
				"writing for boot config lock is not supported for this SoC.\n"
			);
			return EXIT_FAILURE;
		}
		if (is_secureboot_srk_write_fuse) {
			switch (secure_status_secureboot_imx8ulp(armadillo.ocotp_data)) {
			case INSECURE:
				rc = write_fuse_secureboot_srk_imx8ulp(fd);
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK(
					"Secure boot configured (not enforced).\n"
				);
				break;
			default:
				if (cmp_fuse_secureboot_srk_imx8ulp(armadillo.ocotp_data) != 0) {
					fprintf(stderr,
						"Refusing to write to SRK Hash: different SRK Hash is already written.\n"
					);
					return EXIT_FAILURE;
				}
				break;
			}
		}
		if (is_secureboot_close_write_fuse) {
			switch (secure_status_secureboot_imx8ulp(armadillo.ocotp_data)) {
			case INSECURE:
				fprintf(stderr,
					"Refusing to close: SRK Hash is not written.\n"
				);
				return EXIT_FAILURE;
			case SECURE:
				fprintf(stderr,
					"Refusing to close: secure boot is already enabled.\n"
				);
				return EXIT_FAILURE;
			default:
				rc = write_fuse_secureboot_close_imx8ulp();
				if (rc != 0)
					return EXIT_FAILURE;
				PRINTF_CHECK("Secure boot configured.\n");
				break;
			}
		}
		break;
	default:
		fprintf(stderr, "fuse writing is not supported for this SoC.\n");
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}

int is_secure_fuse(void)
{
	if (!init_armadillo())
		return EXIT_FAILURE;

	switch (armadillo.soc) {
	case IMX6ULL:
		if (is_jtag_is_secure)
			return is_secure_fuse_jtag_imx6ull(armadillo.ocotp_data);
		if (is_sd_boot_is_secure)
			return is_secure_fuse_sd_boot_imx6ull(armadillo.ocotp_data);
		if (is_lock_is_secure)
			return is_secure_fuse_boot_cfg_lock_imx6ull(armadillo.ocotp_data);
		if (is_secureboot_is_secure)
			return EXIT_FAILURE;
		break;
	case IMX8MP:
		if (is_jtag_is_secure)
			return is_secure_fuse_jtag_imx8mp(armadillo.ocotp_data);
		if (is_sd_boot_is_secure)
			return is_secure_fuse_sd_boot_imx8mp(armadillo.ocotp_data);
		if (is_lock_is_secure)
			return is_secure_fuse_boot_cfg_lock_imx8mp(armadillo.ocotp_data);
		if (is_secureboot_is_secure)
			return is_secure_fuse_secureboot_imx8mp(armadillo.ocotp_data);
		break;
	case IMX8ULP:
		if (is_jtag_is_secure)
			return is_secure_fuse_jtag_imx8ulp(armadillo.ocotp_data);
		if (is_sd_boot_is_secure)
			return is_secure_fuse_sd_boot_imx8ulp(armadillo.ocotp_data);
		if (is_lock_is_secure)
			return EXIT_FAILURE;
		if (is_secureboot_is_secure)
			return is_secure_fuse_secureboot_imx8ulp(armadillo.ocotp_data);
		break;
	default:
		fprintf(stderr, "fuse status checking is not supported for this SoC.\n");
		return EXIT_FAILURE;
	}
	return EXIT_FAILURE;
}

int show_srk(void)
{
	int offset, i;
	uint32_t buf = 0x0;

	if (!init_armadillo())
		return EXIT_FAILURE;

	switch (armadillo.soc) {
	case IMX8MP:
		offset = IMX8MP_SRK_HASH_OFFSET;
		break;
	case IMX8ULP:
		offset = IMX8ULP_SRK_HASH_OFFSET;
		break;
	default:
		fprintf(stderr, "Showing SRK Hash is not supported for this SoC.\n");
		return EXIT_FAILURE;
	}

	for (i = 0; i < 8; i++) {
		memcpy(&buf,
			armadillo.ocotp_data + offset + 4*i, 4);
		PRINTF_CHECK("0x%08X", buf);

		if (i < 7)
			PRINTF_CHECK(",");
	}

	PRINTF_CHECK("\n");

	return EXIT_SUCCESS;
}

int close_precheck(void)
{
	if (!init_armadillo())
		return EXIT_FAILURE;

	if (armadillo.close_precheck_path == NULL) {
		fprintf(stderr, "--close-precheck is not supported for this SoC.\n");
		return EXIT_FAILURE;
	}

	if (close_precheck_read(&armadillo) < 0)
		return EXIT_FAILURE;

	switch (armadillo.soc) {
	case IMX8MP:
		return close_precheck_imx8mp(armadillo.close_precheck_data);
	case IMX8ULP:
		return close_precheck_imx8ulp(armadillo.close_precheck_data);
	default:
		fprintf(stderr, "--close-precheck is not supported for this SoC.\n");
		return EXIT_FAILURE;
	}
}
