// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */
#include <assert.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>

#include "../security_imx8mp.h"

// same data size as "/sys/bus/nvmem/devices/imx-ocotp0/nvmem"
#define DATA_SIZE 1536

// for avoiding undefined reference to `srk_hashes'
uint32_t srk_hashes[8];

enum secure_status {
	INSECURE,
	SECURE,
	UNKNOWN
};

int test_boot_cfg_lock_settings_imx8mp(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t buf;

	buf = IMX8MP_BOOT_CFG_LOCK_OPWP;
	memcpy(ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET,
	    &buf, 4
	);
	status_char = secure_status_to_string_lock_imx8mp(
			secure_status_boot_cfg_lock_imx8mp(ocotp_data)
	);
	printf("%s\n", status_char);
	printf("%08X\n", *(ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET));
	assert((strcmp(status_char, "write protected") == 0));

	buf = 0x8;
	memcpy(ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET, &buf, 4);

	status_char = secure_status_to_string_lock_imx8mp(
			secure_status_boot_cfg_lock_imx8mp(ocotp_data)
	);
	printf("%s\n", status_char);
	printf("%08X\n", *(ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET));
	assert((strcmp(status_char, "open") == 0));

	return 0;
}

int test_cst_srk_lock_settings_imx8mp(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t buf;

	buf = IMX8MP_CST_SRK_LOCK_ENABLE;
	memcpy(ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET, &buf, 4);
	status_char = secure_status_to_string_lock_imx8mp(
			secure_status_cst_srk_lock_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "write protected") == 0));

	buf = 0x0;
	memcpy(ocotp_data + IMX8MP_BOOT_CFG_LOCK_OFFSET, &buf, 4);
	status_char = secure_status_to_string_lock_imx8mp(
			secure_status_cst_srk_lock_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "open") == 0));

	return 0;
}

int test_jtag_settings_imx8mp(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t buf;

	buf = IMX8MP_SJC_DISABLE_TRUE
	    | IMX8MP_JTAG_SMODE_SECURE
	    | IMX8MP_KTE_KILL;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "disabled") == 0));

	buf = IMX8MP_JTAG_SMODE_JTAG_ENABLE | IMX8MP_KTE_KILL;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "open") == 0));

	// JTAG_SMODE is "reserved" mode.
	buf = 0x800000 | IMX8MP_KTE_KILL;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "unknown") == 0));

	buf = IMX8MP_JTAG_SMODE_SECURE;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "secure mode. WARNING: bus tracing is allowed") == 0));

	buf = IMX8MP_JTAG_SMODE_SECURE | IMX8MP_KTE_KILL;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "secure mode") == 0));

	buf = IMX8MP_JTAG_SMODE_NO_DEBUG;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "bus tracing only allowed") == 0));

	buf = IMX8MP_JTAG_SMODE_NO_DEBUG | IMX8MP_KTE_KILL;
	memcpy(ocotp_data + IMX8MP_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx8mp(ocotp_data);
	assert((strcmp(status_char, "disabled") == 0));

	return 0;
}

int test_sd_boot_settings_imx8mp(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t bootModeFuses_btFuseSel_mem;
	uint32_t forceBtFromFuse_mem;

	bootModeFuses_btFuseSel_mem = IMX8MP_BOOT_MODE_FUSES_EMMC;
	forceBtFromFuse_mem = IMX8MP_FORCE_BT_FROM_FUSE_OFFSET;
	memcpy(ocotp_data + IMX8MP_BOOT_MODE_FUSES_OFFSET,
		&bootModeFuses_btFuseSel_mem, 4);
	memcpy(ocotp_data + IMX8MP_FORCE_BT_FROM_FUSE_OFFSET,
		&forceBtFromFuse_mem, 4);
	status_char = secure_status_to_string_sd_boot_imx8mp(
			secure_status_sd_boot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "allowed") == 0));

	bootModeFuses_btFuseSel_mem = IMX8MP_BT_FUSE_SEL_PROGRAMMED
		| IMX8MP_BOOT_MODE_FUSES_EMMC;
	forceBtFromFuse_mem = 0x0;
	memcpy(ocotp_data + IMX8MP_BOOT_MODE_FUSES_OFFSET,
		&bootModeFuses_btFuseSel_mem, 4);
	memcpy(ocotp_data + IMX8MP_FORCE_BT_FROM_FUSE_OFFSET,
		&forceBtFromFuse_mem, 4);
	status_char = secure_status_to_string_sd_boot_imx8mp(
			secure_status_sd_boot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "allowed") == 0));

	bootModeFuses_btFuseSel_mem = IMX8MP_BT_FUSE_SEL_PROGRAMMED;
	forceBtFromFuse_mem = IMX8MP_FORCE_BT_FROM_FUSE_ENABLE;
	memcpy(ocotp_data + IMX8MP_BOOT_MODE_FUSES_OFFSET,
		&bootModeFuses_btFuseSel_mem, 4);
	memcpy(ocotp_data + IMX8MP_FORCE_BT_FROM_FUSE_OFFSET,
		&forceBtFromFuse_mem, 4);
	status_char = secure_status_to_string_sd_boot_imx8mp(
			secure_status_sd_boot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "unknown") == 0));

	bootModeFuses_btFuseSel_mem = IMX8MP_BT_FUSE_SEL_PROGRAMMED
		| IMX8MP_BOOT_MODE_FUSES_EMMC;
	forceBtFromFuse_mem = IMX8MP_FORCE_BT_FROM_FUSE_ENABLE;
	memcpy(ocotp_data + IMX8MP_BOOT_MODE_FUSES_OFFSET,
		&bootModeFuses_btFuseSel_mem, 4);
	memcpy(ocotp_data + IMX8MP_FORCE_BT_FROM_FUSE_OFFSET,
		&forceBtFromFuse_mem, 4);
	status_char = secure_status_to_string_sd_boot_imx8mp(
			secure_status_sd_boot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "disabled") == 0));

	return 0;
}

int test_secureboot_settings_imx8mp(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t hash_mem;
	uint32_t sec_config_mem;
	int i;

	for (i = 0; i < 8; i++) {
		hash_mem = 0x0;
		memcpy(ocotp_data + IMX8MP_SRK_HASH_OFFSET + 4*i, &hash_mem, 4);
	}
	sec_config_mem = IMX8MP_SEC_CONFIG_ENABLE;
	memcpy(ocotp_data + IMX8MP_SEC_CONFIG_OFFSET,
		&sec_config_mem, 4);
	status_char = secure_status_to_string_secureboot_imx8mp(
			secure_status_secureboot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "disabled") == 0));

	for (i = 0; i < 8; i++) {
		hash_mem = 0x1;
		memcpy(ocotp_data + IMX8MP_SRK_HASH_OFFSET + 4*i, &hash_mem, 4);
	}
	sec_config_mem = 0x0;
	memcpy(ocotp_data + IMX8MP_SEC_CONFIG_OFFSET,
		&sec_config_mem, 4);
	status_char = secure_status_to_string_secureboot_imx8mp(
			secure_status_secureboot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "configured (not enforced)") == 0));

	for (i = 0; i < 8; i++) {
		hash_mem = 0x1;
		memcpy(ocotp_data + IMX8MP_SRK_HASH_OFFSET + 4*i, &hash_mem, 4);
	}
	sec_config_mem = IMX8MP_SEC_CONFIG_ENABLE;
	memcpy(ocotp_data + IMX8MP_SEC_CONFIG_OFFSET,
		&sec_config_mem, 4);
	status_char = secure_status_to_string_secureboot_imx8mp(
			secure_status_secureboot_imx8mp(ocotp_data)
	);
	assert((strcmp(status_char, "configured") == 0));

	return 0;
}

int main(void)
{
	unsigned char ocotp_data[DATA_SIZE];

	memset(ocotp_data, 0x0, DATA_SIZE);
	printf("-------- test security_imx8mp ---------\n");
	test_boot_cfg_lock_settings_imx8mp(ocotp_data);
	test_cst_srk_lock_settings_imx8mp(ocotp_data);
	printf("lock: ok\n");
	test_jtag_settings_imx8mp(ocotp_data);
	printf("jtag: ok\n");
	test_sd_boot_settings_imx8mp(ocotp_data);
	printf("sd_boot: ok\n");
	test_secureboot_settings_imx8mp(ocotp_data);
	printf("secureboot: ok\n");

	return 0;
}
