// SPDX-License-Identifier: MIT
/*
 * Copyright (c) 2023 Atmark Techno,Inc.
 */
#include <assert.h>
#include "../security_imx6ull.h"

// same data size as "/sys/bus/nvmem/devices/imx-ocotp0/nvmem"
#define DATA_SIZE 1536

enum secure_status {
	INSECURE,
	SECURE,
	UNKNOWN
};

int test_boot_cfg_lock_settings_imx6ull(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t buf;

	buf = IMX6ULL_BOOT_CFG_LOCK_OPWP;
	memcpy(ocotp_data + IMX6ULL_BOOT_CFG_LOCK_OFFSET, &buf, 4);
	status_char = secure_status_to_string_lock_imx6ull(
			secure_status_boot_cfg_lock_imx6ull(ocotp_data)
	);
	assert((strcmp(status_char, "write protected") == 0));

	buf = 0x8;
	memcpy(ocotp_data + IMX6ULL_BOOT_CFG_LOCK_OFFSET, &buf, 4);
	status_char = secure_status_to_string_lock_imx6ull(
			secure_status_boot_cfg_lock_imx6ull(ocotp_data)
	);
	assert((strcmp(status_char, "open") == 0));

	return 0;
}

int test_jtag_settings_imx6ull(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t buf;

	buf = IMX6ULL_SJC_DISABLE_TRUE
	    | IMX6ULL_JTAG_SMODE_SECURE
	    | IMX6ULL_KTE_KILL;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "disabled") == 0));

	buf = IMX6ULL_JTAG_SMODE_JTAG_ENABLE | IMX6ULL_KTE_KILL;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "open") == 0));

	// JTAG_SMODE is "reserved" mode.
	buf = 0x800000 | IMX6ULL_KTE_KILL;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "unknown") == 0));

	buf = IMX6ULL_JTAG_SMODE_SECURE;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "secure mode. WARNING: bus tracing is allowed") == 0));

	buf = IMX6ULL_JTAG_SMODE_SECURE | IMX6ULL_KTE_KILL;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "secure mode") == 0));

	buf = IMX6ULL_JTAG_SMODE_NO_DEBUG;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "bus tracing only allowed") == 0));

	buf = IMX6ULL_JTAG_SMODE_NO_DEBUG | IMX6ULL_KTE_KILL;
	memcpy(ocotp_data + IMX6ULL_JTAG_OFFSET, &buf, 4);
	status_char = fuse_status_jtag_imx6ull(ocotp_data);
	assert((strcmp(status_char, "disabled") == 0));

	return 0;
}

int test_sd_boot_settings_imx6ull(unsigned char *ocotp_data)
{
	char *status_char;
	uint32_t bootdev_usdhc_bus_mem;
	uint32_t btFuseSel_mem;

	bootdev_usdhc_bus_mem = IMX6ULL_BOOT_DEVICE_EMMC
		| IMX6ULL_USDHC_EMMC
		| IMX6ULL_BUS_WIDTH_8BIT;
	btFuseSel_mem = 0x0;
	memcpy(ocotp_data + IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET,
		&bootdev_usdhc_bus_mem, 4);
	memcpy(ocotp_data + IMX6ULL_BT_FUSE_SEL_OFFSET,
		&btFuseSel_mem, 4);
	status_char = fuse_status_sd_boot_imx6ull(ocotp_data);
	assert((strcmp(status_char, "allowed") == 0));

	bootdev_usdhc_bus_mem = IMX6ULL_BOOT_DEVICE_EMMC
		| 0x800
		| IMX6ULL_BUS_WIDTH_8BIT;
	btFuseSel_mem = IMX6ULL_BT_FUSE_SEL_PROGRAMMED;
	memcpy(ocotp_data + IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET,
		&bootdev_usdhc_bus_mem, 4);
	memcpy(ocotp_data + IMX6ULL_BT_FUSE_SEL_OFFSET,
		&btFuseSel_mem, 4);
	status_char = fuse_status_sd_boot_imx6ull(ocotp_data);
	assert((strcmp(status_char, "unknown") == 0));

	bootdev_usdhc_bus_mem = 0xC0
		| IMX6ULL_USDHC_EMMC
		| IMX6ULL_BUS_WIDTH_8BIT;
	btFuseSel_mem = IMX6ULL_BT_FUSE_SEL_PROGRAMMED;
	memcpy(ocotp_data + IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET,
		&bootdev_usdhc_bus_mem, 4);
	memcpy(ocotp_data + IMX6ULL_BT_FUSE_SEL_OFFSET,
		&btFuseSel_mem, 4);
	status_char = fuse_status_sd_boot_imx6ull(ocotp_data);
	assert((strcmp(status_char, "unknown") == 0));

	bootdev_usdhc_bus_mem = IMX6ULL_BOOT_DEVICE_EMMC
		| IMX6ULL_USDHC_EMMC
		| 0xC000;
	btFuseSel_mem = IMX6ULL_BT_FUSE_SEL_PROGRAMMED;
	memcpy(ocotp_data + IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET,
		&bootdev_usdhc_bus_mem, 4);
	memcpy(ocotp_data + IMX6ULL_BT_FUSE_SEL_OFFSET,
		&btFuseSel_mem, 4);
	status_char = fuse_status_sd_boot_imx6ull(ocotp_data);
	assert((strcmp(status_char, "disabled. WARNING: eMMC bus width is not 8-bit") == 0));

	bootdev_usdhc_bus_mem = IMX6ULL_BOOT_DEVICE_EMMC
		| IMX6ULL_USDHC_EMMC
		| IMX6ULL_BUS_WIDTH_8BIT;
	btFuseSel_mem = IMX6ULL_BT_FUSE_SEL_PROGRAMMED;
	memcpy(ocotp_data + IMX6ULL_BOOT_DEVICE_USDHC_BUS_OFFSET,
		&bootdev_usdhc_bus_mem, 4);
	memcpy(ocotp_data + IMX6ULL_BT_FUSE_SEL_OFFSET,
		&btFuseSel_mem, 4);
	status_char = fuse_status_sd_boot_imx6ull(ocotp_data);
	assert((strcmp(status_char, "disabled") == 0));

	return 0;
}

int main(void)
{
	unsigned char ocotp_data[DATA_SIZE];

	memset(ocotp_data, 0x0, DATA_SIZE);
	printf("-------- test security_imx6ull ---------\n");
	test_boot_cfg_lock_settings_imx6ull(ocotp_data);
	printf("lock: ok\n");
	test_jtag_settings_imx6ull(ocotp_data);
	printf("jtag: ok\n");
	test_sd_boot_settings_imx6ull(ocotp_data);
	printf("sd_boot: ok\n");

	return 0;
}
