// SPDX-License-Identifier: MIT

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>

#define SERIAL_BAUDRATE B115200
#define EMS31_SERIAL_BAUDRATE B460800
#define WAIT_1ST_SEC 5
#define WAIT_DEFAULT_SEC 1
/* this size is based on with AT^SIND=? on EMS31-J (336 bytes) */
#define AT_BUF_SIZE 350
#define URC_MAX_LENGTH 256
#define URC_TIMEOUT_SEC_MIN 3
#define URC_TIMEOUT_SEC_MAX 180

static char _recv_buffer[AT_BUF_SIZE];
static int _recv_length = 0;
static struct termios tio_orig;
static bool has_tio_orig = false;

static void restore(int fd)
{
	if (has_tio_orig)
		tcsetattr(fd, TCSANOW, &tio_orig);

	has_tio_orig = false;
}

static bool setup(int fd, bool enable_flowctl, speed_t baudrate)
{
	struct termios tio;
	int ret;

	ret = tcgetattr(fd, &tio_orig);
	if (ret) {
		perror("tcgetattr");
		return false;
	}
	has_tio_orig = true;

	memset (&tio, 0, sizeof(struct termios));

	tio.c_cflag &= ~(CBAUD | CSIZE | CSTOPB | PARENB | PARODD);
	if (enable_flowctl)
		tio.c_cflag |= CRTSCTS;

	tio.c_iflag &= ~(IGNCR | ICRNL | IUCLC | INPCK | IXON | IXOFF | IXANY );
	tio.c_oflag &= ~(OPOST | OLCUC | OCRNL | ONLCR | ONLRET);
	tio.c_lflag &= ~(ICANON | ECHO | ECHOE | ECHONL);
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;
	tio.c_cc[VEOF] = 1;
	tio.c_iflag |= IGNPAR;
	tio.c_cflag |= (CS8 | CLOCAL | CREAD);

	ret = cfsetispeed(&tio, baudrate);
	if (ret < 0) {
		perror("cfsetispeed");
		return false;
	}

	ret = cfsetospeed(&tio, baudrate);
	if (ret < 0) {
		perror("cfsetospeed");
		return false;
	}

	ret = tcflush(fd, TCIFLUSH);
	if (ret < 0)
		perror("tcflush");

	ret = tcsetattr(fd, TCSANOW, &tio);
	if (ret)
		perror("tcsetattr");

	return true;
}

static void close_serial(int fd);
static int open_serial(const char *port_name, bool enable_flowctl, speed_t baudrate)
{
	int fd;

	fd = open(port_name, O_RDWR);
	if (fd == -1) {
		perror("open serial");
		return -1;
	}

	if (!setup(fd, enable_flowctl, baudrate)) {
		close_serial(fd);
		return -1;
	}

	return fd;
}

static void close_serial(int fd)
{
	restore(fd);
	close(fd);
}

static int copy_valid_command(char *buf, char *end, int cut)
{
	int valid_length;
	int length_with_crlf;

	valid_length = end - _recv_buffer;
	length_with_crlf = valid_length + cut;

	memcpy(buf, _recv_buffer, valid_length);

	if (length_with_crlf == _recv_length) {
		memset(_recv_buffer, 0, AT_BUF_SIZE);
		_recv_length = 0;
	} else {
		memmove(_recv_buffer, &(_recv_buffer[length_with_crlf]),
			AT_BUF_SIZE - length_with_crlf);
		_recv_length -= length_with_crlf;
		memset(&(_recv_buffer[AT_BUF_SIZE - length_with_crlf]),
		       0, length_with_crlf);
	}
	return valid_length;
}

static int copy_at_command(char *buf)
{
	char *p = NULL;
	int cut = 0;
	int result;

	while (1) {
		p = strstr(_recv_buffer, "\r\n");
		if (p)
			cut = 2;
		else {
			p = strchr(_recv_buffer, '\r');
			if (p)
				cut = 1;
			else {
				p = strchr(_recv_buffer, '\n');
				if (p)
					cut = 1;
				else
					return -1;
			}
		}

		result = copy_valid_command(buf, p, cut);
		if (result == 0)	/* retry when only \r \n */
			continue;

		return result;
	}
}

static bool has_received_command()
{
	return (_recv_length > 0);
}

int wait_data(int fd, int sec)
{
	struct timeval tv;
	fd_set readfds;
	int ret;

	FD_ZERO(&readfds);
	FD_SET(fd, &readfds);

	if (sec == 0) {
		tv.tv_sec = 0;
		tv.tv_usec = 1;
	} else {
		tv.tv_sec = sec;
		tv.tv_usec = 0;
	}

	ret = select(fd + 1, &readfds, NULL, NULL, &tv);

	if (ret == -1) {
		perror("select");
		return -1;
	}

	if (!ret)
		return -1;

	if (FD_ISSET(fd, &readfds))
		return 0;

	return -1;
}

static int read_line(int fd, char *buf, int timeout_sec)
{
	int current_length = 0;
	int sec = timeout_sec;
	int result;

	if ((_recv_length > 0) &&
	    (strchr(_recv_buffer, '\r') || strchr(_recv_buffer, '\n'))) {
		result = copy_at_command(buf);
		if (result > 0)
			return result;
	}

	while (wait_data(fd, sec) == 0) {
		current_length = read(fd, &(_recv_buffer[_recv_length]),
				      AT_BUF_SIZE - 1 - _recv_length);
		if (current_length < 0) {
			perror("read");
			return -1;
		}

		if (current_length == 0)
			return -1;

		_recv_length += current_length;

		if (strchr(_recv_buffer, '\r') || strchr(_recv_buffer, '\n')) {
			result = copy_at_command(buf);
			if (result > 0)
				return result;
		}

		sec = WAIT_DEFAULT_SEC;
	}
	return -1;
}

static int atcmd_send(const char *port_name, const char *command, bool echo,
		      bool enable_flowctl, speed_t baudrate)
{
	int read_size;
	int result = -1;
	char buf[AT_BUF_SIZE];
	int sec = WAIT_1ST_SEC;
	int fd;

	fd = open_serial(port_name, enable_flowctl, baudrate);
	if (fd < 0)
		return -1;

	/* dummy read */
	memset(buf, 0, AT_BUF_SIZE);
	do {
		read_size = read_line(fd, buf, 0);
	} while (read_size > 0);

	/* send command */
	snprintf(buf, AT_BUF_SIZE, "%s\r\n", command);
	write(fd, buf, strlen(command)+2);

	/* read response */
	while (1) {
		if (!has_received_command() && (wait_data(fd, sec) != 0)) {
			result = -1;
			break;
		}

		sec = WAIT_DEFAULT_SEC;
		memset(buf, 0, AT_BUF_SIZE);
		read_size = read_line(fd, buf, sec);
		if (read_size <= 0) {
			fprintf(stderr, "error: timed out.\n");
			result = -1;
			break;
		}

		sec = WAIT_DEFAULT_SEC;
		if (strcmp(buf, command) == 0) /* command echo */
			continue;
		else if (strstr(buf, "OK")) {
			result = 0;
			fprintf(stderr, "OK\n");
			break;
		} else if (strstr(buf, "ERROR")) {
			result = -1;
			fprintf(stderr, "ERROR\n");
			break;
		} else if (echo)
			fprintf(stderr, "%s\n", buf);
	}

	close_serial(fd);
	return result;
}

static int atcmd_send_and_wait_urc(const char *port_name, const char *command,
				   bool enable_flowctl, speed_t baudrate, const char *urc,
				   int urc_timeout_sec)
{
	int read_size;
	int result = -1;
	char buf[AT_BUF_SIZE];
	int sec = urc_timeout_sec;
	int fd;
	struct timespec tp;
	time_t ending_time;

	if (!urc)
		return -1;

	fd = open_serial(port_name, enable_flowctl, baudrate);
	if (fd < 0)
		return -1;

	/* dummy read */
	memset(buf, 0, AT_BUF_SIZE);
	do {
		read_size = read_line(fd, buf, 0);
	} while (read_size > 0);

	/* send command */
	snprintf(buf, AT_BUF_SIZE, "%s\r\n", command);
	write(fd, buf, strlen(command)+2);

	/* set wait URC ending time */
	clock_gettime(CLOCK_MONOTONIC_RAW, &tp);
	ending_time = tp.tv_sec + urc_timeout_sec;

	/* read response */
	while (1) {
		if (!has_received_command() && (wait_data(fd, sec) != 0)) {
			result = -1;
			break;
		}

		clock_gettime(CLOCK_MONOTONIC_RAW, &tp);
		sec = (int)(ending_time - tp.tv_sec);
		if (sec <= 0) {
			fprintf(stderr, "error: timed out.\n");
			result = -1;
			break;
		}
		memset(buf, 0, AT_BUF_SIZE);
		read_size = read_line(fd, buf, sec);
		if (read_size <= 0) {
			fprintf(stderr, "error: timed out.\n");
			result = -1;
			break;
		}

		if (strcmp(buf, command) == 0) /* command echo */
			continue;
		else if (strstr(buf, urc)) {
			result = 0;
			break;
		}
	}

	close_serial(fd);
	return result;
}

static bool is_string(const char *str)
{
	return str && str[0] &&
	       (strnlen(str, URC_MAX_LENGTH+1) < URC_MAX_LENGTH+1);
}

static bool is_correct_urc_timeout_sec (const char *str,
					unsigned int *timeout_sec)
{
	unsigned long num;

	if (!timeout_sec)
		return false;

	errno = 0;
	num = strtoul (str, NULL, 10);
	if (errno) {
		return false;
	}

	if (URC_TIMEOUT_SEC_MIN > num || num > URC_TIMEOUT_SEC_MAX)
		return false;

	*timeout_sec = (unsigned int)num;

	return true;
}

static void usage()
{
	fprintf(stderr, "Usage: send-at port command\n");
	fprintf(stderr, "       send-at port command echo\n");
	fprintf(stderr, "       send-at port command ems31\n");
	fprintf(stderr, "       send-at port command echo ems31\n");
	fprintf(stderr, "       send-at port command urc urc_str timeout\n");
	fprintf(stderr, "       send-at port command urc urc_str timeout ems31\n");
	fprintf(stderr, "     port: ex. /dev/ttymxc3\n");
	fprintf(stderr, "  command: ex. \"AT\"\n");
	fprintf(stderr, "     echo: print command response.\n");
	fprintf(stderr, "    ems31: change baudrate for ems31 (460800)\n");
	fprintf(stderr, "  urc_str: desire URC string ex \"^SHUTDOWN\"\n");
	fprintf(stderr, "  timeout: wait URC timeout seconds (3-180)\n");
}

int main(int argc, char *argv[])
{
	int result = -1;
	unsigned int urc_timeout_sec = 0;

	switch (argc) {
	case 3:
		result = atcmd_send(argv[1], argv[2], false, true, SERIAL_BAUDRATE);
		break;

	case 4:
		if (strcmp("echo", argv[3]) == 0)
			result = atcmd_send(argv[1], argv[2], true, true,
					    SERIAL_BAUDRATE);
		else if (strcmp("ems31", argv[3]) == 0)
			result = atcmd_send(argv[1], argv[2], false, true,
					    EMS31_SERIAL_BAUDRATE);
		else if (strcmp("sim7672", argv[3]) == 0)
			/* sim7672 option is used internally in sim7672-boot command */
			/* to configure hardware flow control for the LTE module, */
			/* so it is not documented in the help. */
			result = atcmd_send(argv[1], argv[2], false, false,
					    SERIAL_BAUDRATE);
		else
			usage();
		break;

	case 5:
		if ((strcmp("echo", argv[3]) == 0) &&
		    (strcmp("ems31", argv[4]) == 0))
			result = atcmd_send(argv[1], argv[2], true, true,
					    EMS31_SERIAL_BAUDRATE);
		else if ((strcmp("echo", argv[3]) == 0) &&
			 (strcmp("sim7672", argv[4]) == 0))
			result = atcmd_send(argv[1], argv[2], true, false,
					    SERIAL_BAUDRATE);
		else
			usage();
		break;

	case 6:
		if ((strcmp("urc", argv[3]) == 0) &&
		    is_string(argv[4]) &&
		    is_correct_urc_timeout_sec(argv[5], &urc_timeout_sec)) 
			result = atcmd_send_and_wait_urc(argv[1], argv[2],
							 true,
							 SERIAL_BAUDRATE,
							 argv[4],
							 urc_timeout_sec);
		else
			usage();
		break;

	case 7:
		if ((strcmp("urc", argv[3]) == 0) &&
		    is_string(argv[4]) &&
		    is_correct_urc_timeout_sec(argv[5], &urc_timeout_sec) &&
		    (strcmp("ems31", argv[6]) == 0))
			result = atcmd_send_and_wait_urc(argv[1], argv[2],
							 true,
							 EMS31_SERIAL_BAUDRATE,
							 argv[4],
							 urc_timeout_sec);
		else if ((strcmp("urc", argv[3]) == 0) &&
			 is_string(argv[4]) &&
			 is_correct_urc_timeout_sec(argv[5], &urc_timeout_sec) &&
			 (strcmp("sim7672", argv[6]) == 0))
			result = atcmd_send_and_wait_urc(argv[1], argv[2],
							 false,
							 SERIAL_BAUDRATE,
							 argv[4],
							 urc_timeout_sec);
		else
			usage();
		break;

	default:
		usage();
		break;
	}

	exit(result?1:0);
}
