//
// power_state_handler.c
//
#include "power_state_handler.h"

#include <errno.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/eventfd.h>
#include <sys/socket.h>
#include <sys/syscall.h>
#include <sys/un.h>
#include <poll.h>
#include <pthread.h>
#include <signal.h>
#include <unistd.h>

#include "swupdate_locker.h"


#define REQ_START_INHIBIT_SLEEP	"request_start_inhibit_sleep"
#define REQ_FINISH_INHIBIT_SLEEP	"request_finish_inhibit_sleep"
#define REQ_START_SLEEP	"request_start_sleep"
#define NFY_FINISH_SLEEP	"notify_finish_sleep"
#define REQ_DISABLE_SLEEP	"request_disable_sleep"
#define REQ_ENABLE_SLEEP	"request_enable_sleep"

#define NOTIFIER_NAME	"atmark-power-nfy"
#define NFY_ENTER_SLEEP	"enter_sleep"
#define NFY_LEAVE_SLEEP	"leave_sleep"

#define ADDR_UN_LEN(addr_un)	(sizeof((addr_un).sun_family) + strlen((addr_un).sun_path + 1) + 2)

typedef struct {
	struct sockaddr_un	address;
} Client;
#define CLIENT_NAME(c)	((c)->address.sun_path + 1)
#define CLIENT_ADDR_LEN(c)	ADDR_UN_LEN((c)->address)

typedef struct {
	Client* body;
	int	capacity;
	int	count;
} ClientList;

typedef struct {
	int	service_sock;	// クライアントからのリクエスト受信・応答用
	int	notify_sock;	// スリープ遷移通知送信・応答受信用
	pthread_t	thread;	// スレッド
	bool	quit_requested;	// 終了要求されたか
	int	wait_event;	// 終了要求時の通知用
	EPowerHandlerState	curr_state;	// 現在の電源状態
	bool	is_sleep_requester_exists;	// スリープ遷移を要求されているか
	Client	sleep_requester;
	ClientList* inhibit_lockers;	// スリープの一時停止を要求中のクライアント集合
	ClientList* sleep_resume_listners;	// スリープ開始／終了の通知先クライアント集合
	ClientList* resp_wait_list;
} PowerStateHandler;


static void
init_pollfd(struct pollfd* ioPollFd, int targetFd)
{
	ioPollFd->fd = targetFd;
	ioPollFd->events = POLLIN;
	ioPollFd->revents = 0;
}

static uint64_t
GetCurrentTimeInMillis()
{
	struct timespec	ts;

	(void)clock_gettime(CLOCK_MONOTONIC_RAW, &ts);

	return (ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
}


//
// Client
//
pid_t
Client_GetPID(const Client* client)
{
	const char* name = CLIENT_NAME(client);
	const char* curs = strrchr(name, '_');

	if (NULL == curs) {
		fprintf(stderr, "invalid argument for %s\n", __FUNCTION__);
		return (pid_t)-1;
	}
	return atoi(curs + 1);
}


//
// ClientList
//
ClientList*
ClientList_Create(int initialCapacity)
{
	ClientList* obj = (ClientList*)calloc(1, sizeof(ClientList));

	if (initialCapacity < 0) {
		fprintf(stderr, "WARNING; invalid initialCapacity: %d, treat as 0.\n", initialCapacity);
		initialCapacity = 0;
	}
	if (NULL == obj) {
		fprintf(stderr, "failed to calloc().\n");
		return NULL;
	}
	if (initialCapacity > 0) {
		obj->body = (Client*)calloc(initialCapacity, sizeof(Client));
		if (NULL == obj->body) {
			fprintf(stderr, "failed to calloc(), size= %zu.\n",
				sizeof(Client) * initialCapacity);
			free(obj);
			return NULL;
		}
		obj->capacity = initialCapacity;
	}

	return obj;
}

void
ClientList_Destroy(ClientList* obj)
{
	if (NULL != obj) {
		free(obj->body);
		free(obj);
	}
}

void
ClientList_Clear(ClientList* obj)
{
	obj->count = 0;
	bzero(obj->body, sizeof(Client) * obj->capacity);
}

int
ClientList_FindElem(ClientList* obj, const char* name)
{
	for (int i = 0; i < obj->count; ++i) {
		if (strcmp(CLIENT_NAME(&obj->body[i]), name) == 0) {
			return i;
		}
	}

	return -1;  // not found
}

int
ClientList_Count(ClientList* obj)
{
	return obj->count;
}

const Client*
ClientList_ElemAt(ClientList* obj, int index)
{
	if (index < 0 || obj->count <= index) {
		fprintf(stderr, "invalid argument for %s().\n", __FUNCTION__);
		return NULL;
	}
	return (obj->body + index);
}

int
ClientList_AddElem(ClientList* obj, const char* name)
{
	if (obj->count == obj->capacity) {
		int newCapacity = (obj->capacity != 0 ? obj->capacity * 2 : 10);
		Client* newBody = (Client*)realloc(
			obj->body, sizeof(Client) * newCapacity);

		if (NULL == newBody) {
			fprintf(stderr, "failed to realloc(), size= %zu.\n",
				sizeof(Client) * newCapacity);
			return -1;
		}
		obj->body = newBody;
		obj->capacity = newCapacity;
	}
	obj->body[obj->count].address.sun_family = AF_LOCAL;
	strcpy(CLIENT_NAME(&obj->body[obj->count++]), name);

	return 0;
}

void
ClientList_RmvElem(ClientList* obj, int index)
{
	if (index < 0 || obj->count <= index) {
		printf("invalid argument for %s().\n", __FUNCTION__);
		return;
	}
	if (index < obj->count - 1) {
		memmove(obj->body + index, obj->body + index + 1,
			(obj->count - index - 1) * sizeof(Client));
	}
	obj->count -= 1;
}

static void
BroadcastToClients(int sockfd, ClientList* targets, const char* msg)
{
	for (int i = 0, n = ClientList_Count(targets); i < n; ++i) {
		const Client* c = ClientList_ElemAt(targets, i);

		if (sendto(sockfd, msg, strlen(msg) + 1, 0,
				(struct sockaddr*)&c->address, ADDR_UN_LEN(c->address)) < 0) {
			if (ENOTCONN == errno || ECONNREFUSED == errno) {
				ClientList_RmvElem(targets, i);
				--i;
				--n;
			} else {
				fprintf(stderr, "failed to sendto(), errno= %d, the client address : '%s'.\n",
					errno, CLIENT_NAME(c));
			}
		}
	}
}


//
// PowerStateHandler
//
static void* ThreadProc(void* arg);
static void	PowerStateHandler_CheckSleepInhibitLockers(PowerStateHandler* obj);
static void	PowerStateHandler_HandleRequest(PowerStateHandler* obj);
static void	PowerStateHandler_SendReply(PowerStateHandler* obj,
	const char* replyMsg, const struct sockaddr_un* clientAddr, socklen_t addrLen);
static void PowerStateHandler_HandleEnterSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen);
static void	PowerStateHandler_HandleLeaveSleepNotification(PowerStateHandler* obj);
static void PowerStateHandler_HandleInhibitSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen);
static void PowerStateHandler_HandleAllowSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen);
static void PowerStateHandler_HandleDisableSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen);
static void PowerStateHandler_HandleEnableSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen);
static void	PowerStateHandler_TransientToSleepState(PowerStateHandler* obj);
static void	PowerStateHandler_BeforeSleepAction(PowerStateHandler* obj);
static void	PowerStateHandler_AfterResumeAction(PowerStateHandler* obj);

static void*
ThreadProc(void* arg)
{
	PowerStateHandler* obj = (PowerStateHandler*)arg;
	struct pollfd	pollfdSet[2];

	init_pollfd(&pollfdSet[0], obj->service_sock);
	init_pollfd(&pollfdSet[1], obj->wait_event);

	while (!obj->quit_requested) {
		int	timeout = (ClientList_Count(obj->inhibit_lockers) != 0 ? 1000 : -1);
		int	num_evt = poll(pollfdSet, 2, timeout);

		if (num_evt == 0) {
			PowerStateHandler_CheckSleepInhibitLockers(obj);
		}
		else {
			if (obj->quit_requested) {
				break;
			}
			if (pollfdSet[0].revents != 0) {
				if ((pollfdSet[0].revents & POLLIN) != 0) {
					PowerStateHandler_HandleRequest(obj);
				}
				pollfdSet[0].revents = 0;
			}
		}
	}
//	printf("leave ThreadProc().\n");

	return (void*)0;
}

static void
PowerStateHandler_CheckSleepInhibitLockers(PowerStateHandler* obj)
{
	for (int i = 0, n = ClientList_Count(obj->inhibit_lockers); i < n; ++i) {
		const Client* client = ClientList_ElemAt(obj->inhibit_lockers, i);
		pid_t	pid = Client_GetPID(client);

		if (kill(pid, 0) < 0 && errno == ESRCH) {
			ClientList_RmvElem(obj->inhibit_lockers, i);
			--i;
			--n;
			if (0 == n) {
				if (POWER_SLEEP_INHIBITED == obj->curr_state) {
					if (obj->is_sleep_requester_exists) {
						PowerStateHandler_TransientToSleepState(obj);
					} else {
						obj->curr_state = POWER_NORMAL;
					}
				}
			}
		}
	}
}

static void
PowerStateHandler_HandleRequest(PowerStateHandler* obj)
{
	char	msgBuf[64];
	int	recvLen;
	struct sockaddr_un	clientAddr;
	socklen_t	addrLen = sizeof(clientAddr);

	recvLen = recvfrom(
		obj->service_sock, msgBuf, sizeof(msgBuf), 0, (struct sockaddr*)&clientAddr, &addrLen);
	if (recvLen < 0) {
		fprintf(stderr, "failed to recvfrom().\n");
		return;
	}

	if (0 == strcmp(msgBuf, REQ_START_INHIBIT_SLEEP)) {
		PowerStateHandler_HandleInhibitSleepRequest(obj, &clientAddr, addrLen);
	}
	else if (0 == strcmp(msgBuf, REQ_FINISH_INHIBIT_SLEEP)) {
		PowerStateHandler_HandleAllowSleepRequest(obj, &clientAddr, addrLen);
	}
	else if (0 == strcmp(msgBuf, REQ_START_SLEEP)) {
		PowerStateHandler_HandleEnterSleepRequest(obj, &clientAddr, addrLen);
	}
	else if (0 == strcmp(msgBuf, NFY_FINISH_SLEEP)) {
		PowerStateHandler_HandleLeaveSleepNotification(obj);
	}
	else if (0 == strcmp(msgBuf, REQ_POWER_STATE_NOTIFICATION)) {
		(void)ClientList_AddElem(obj->sleep_resume_listners, clientAddr.sun_path + 1);
		PowerStateHandler_SendReply(obj, "OK", &clientAddr, addrLen);
	}
	else if (0 == strcmp(msgBuf, REQ_DISABLE_SLEEP)) {
		PowerStateHandler_HandleDisableSleepRequest(obj, &clientAddr, addrLen);
	}
	else if (0 == strcmp(msgBuf, REQ_ENABLE_SLEEP)) {
		PowerStateHandler_HandleEnableSleepRequest(obj, &clientAddr, addrLen);
	}
	else {
		PowerStateHandler_SendReply(obj, "NG; unknown request", &clientAddr, addrLen);
	}
}

static void
PowerStateHandler_SendReply(PowerStateHandler* obj,
	const char* replyMsg, const struct sockaddr_un* clientAddr, socklen_t addrLen)
{
	if (sendto(obj->service_sock,
			replyMsg, strlen(replyMsg) + 1, 0, (struct sockaddr*)clientAddr, addrLen) < 0) {
		fprintf(stderr, "failed to sendto(), errno= %d, client= '%s', replyMsg: '%s'.\n",
			errno, clientAddr->sun_path + 1, replyMsg);
	}
}

static void
PowerStateHandler_HandleEnterSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen)
{
	char	respMsg[64];

	switch (obj->curr_state) {
	case POWER_PRE_SLEEP:
		sprintf(respMsg, "NG; prepare sleeping");
		break;
	case POWER_SLEEPING:
		sprintf(respMsg, "NG; already sleeping");
		break;
	case POWER_SLEEP_DISABLED:
		sprintf(respMsg, "NG; sleeping is disabled");
		break;
	case POWER_SLEEP_INHIBITED:
		if (obj->is_sleep_requester_exists) {
			sprintf(respMsg, "NG; already sleep requested");
		} else {
			obj->is_sleep_requester_exists = true;
			obj->sleep_requester.address = *clientAddr;
			return;  // send reply when after inhibiting finished
		}
		break;
	case POWER_NORMAL:
		obj->is_sleep_requester_exists = true;
		obj->sleep_requester.address = *clientAddr;
		PowerStateHandler_TransientToSleepState(obj);
		return;  // already sent the reply in PowerStateHandler_TransientToSleepState()
	default:
		sprintf(respMsg, "NG; unknown state (internal error)");
	}

	PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
}

static void
PowerStateHandler_HandleLeaveSleepNotification(PowerStateHandler* obj)
{
	if (POWER_SLEEPING == obj->curr_state) {
		obj->curr_state = POWER_NORMAL;
		PowerStateHandler_AfterResumeAction(obj);
	} else {
		fprintf(stderr, "invalid state change: (%d)->(%d)\n", obj->curr_state, POWER_NORMAL);
	}
}

static void
PowerStateHandler_HandleInhibitSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen)
{
	char	respMsg[64];

	switch (obj->curr_state) {
	case POWER_PRE_SLEEP:
		sprintf(respMsg, "NG; prepare sleeping");
		break;
	case POWER_SLEEPING:
		sprintf(respMsg, "NG; already sleeping");
		break;
	default:
		sprintf(respMsg, "OK");
		(void)ClientList_AddElem(obj->inhibit_lockers, clientAddr->sun_path + 1);
		switch (obj->curr_state) {
		case POWER_NORMAL:
			obj->curr_state = POWER_SLEEP_INHIBITED;
			break;
		case POWER_SLEEP_DISABLED:
		case POWER_SLEEP_INHIBITED:
			break;  // do nothing
		default:
			sprintf(respMsg, "NG; unknown state (internal error)");
		}
	}

	PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
}

static void
PowerStateHandler_HandleAllowSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen)
{
	char	respMsg[64];
	int	index = ClientList_FindElem(obj->inhibit_lockers, clientAddr->sun_path + 1);

	if (index < 0) {
		sprintf(respMsg, "NG; unknown client");
		PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
		return;
	}
	ClientList_RmvElem(obj->inhibit_lockers, index);
	sprintf(respMsg, "OK");

	switch (obj->curr_state) {
	case POWER_NORMAL:
	case POWER_PRE_SLEEP:
	case POWER_SLEEPING:
		sprintf(respMsg, "NG; invalid state change");
		break;
	case POWER_SLEEP_DISABLED:
		break;  // do nothing
	case POWER_SLEEP_INHIBITED:
		if (0 == ClientList_Count(obj->inhibit_lockers)) {
			if (obj->is_sleep_requester_exists) {
				PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
				PowerStateHandler_TransientToSleepState(obj);
				return;  // already sent the reply at above
			} else {
				obj->curr_state = POWER_NORMAL;
			}
		}
		break;
	default:
		sprintf(respMsg, "NG; unknown state (internal error)");
	}

	PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
}

static void
PowerStateHandler_HandleDisableSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen)
{
	char	respMsg[64];

	switch (obj->curr_state) {
	case POWER_PRE_SLEEP:
	case POWER_SLEEPING:
		sprintf(respMsg, "NG; already sleeping");
		break;
	case POWER_SLEEP_DISABLED:
		fprintf(stderr, "Warning; already disabled sleeping.\n");
		// fall through
	default:
		sprintf(respMsg, "OK");
		obj->curr_state = POWER_SLEEP_DISABLED;
		break;
	}

	PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
}

static void
PowerStateHandler_HandleEnableSleepRequest(
	PowerStateHandler* obj, const struct sockaddr_un* clientAddr, socklen_t addrLen)
{
	char	respMsg[64];

	sprintf(respMsg, "OK");
	if (obj->curr_state != POWER_SLEEP_DISABLED) {
		fprintf(stderr, "Warning; already enabled sleeping");
	} else {
		obj->curr_state =
			(0 != ClientList_Count(obj->inhibit_lockers) ? POWER_SLEEP_INHIBITED : POWER_NORMAL);
		if (obj->curr_state == POWER_NORMAL && obj->is_sleep_requester_exists) {
			PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
			PowerStateHandler_TransientToSleepState(obj);
			return;  // already sent the reply at above
		}
	}

	PowerStateHandler_SendReply(obj, respMsg, clientAddr, addrLen);
}

static void
PowerStateHandler_TransientToSleepState(PowerStateHandler* obj)
{
	obj->curr_state = POWER_PRE_SLEEP;
	PowerStateHandler_BeforeSleepAction(obj);
	obj->is_sleep_requester_exists = false;
	obj->curr_state = POWER_SLEEPING;
}

static void
PowerStateHandler_BeforeSleepAction(PowerStateHandler* obj)
{
#define TIMEOUT_SECS	30
	struct pollfd	pollfdSet[2];
	char	msgBuf[64];
	struct sockaddr_un	rmtAddr;
	int	timeout = TIMEOUT_SECS;
	uint64_t	startTime;

	switch (acquire_swupdate_lock()) {
	case SWUPDATE_LOCKED:
		break;  // OK
	case SWUPDATE_REBOOTING:
		printf("rebooting after swupdate in progress, so wait that...\n");
		release_swupdate_lock();
		sleep(60);
		break;
	case SWUPDATE_ERROR:
		printf("system error: failed to acquire_swupdate_lock(). Continuing anyway!\n");
		break;
	}

	init_pollfd(&pollfdSet[0], obj->notify_sock);
	init_pollfd(&pollfdSet[1], obj->service_sock);

	BroadcastToClients(obj->notify_sock, obj->sleep_resume_listners, NFY_ENTER_SLEEP);
	startTime = GetCurrentTimeInMillis();

	ClientList_Clear(obj->resp_wait_list);
	for (int i = 0, n = ClientList_Count(obj->sleep_resume_listners); i < n; ++i) {
		const Client* c = ClientList_ElemAt(obj->sleep_resume_listners, i);
		(void)ClientList_AddElem(obj->resp_wait_list, CLIENT_NAME(c));
	}

	while (ClientList_Count(obj->resp_wait_list) != 0) {
		int	num_evt = poll(pollfdSet, 2, timeout);

		if (num_evt > 0) {
			if (pollfdSet[0].revents != 0) {
				if ((pollfdSet[0].revents & POLLIN) != 0) {
					socklen_t	addrLen = sizeof(rmtAddr);
					int	n = recvfrom(obj->notify_sock, msgBuf, sizeof(msgBuf), 0,
						(struct sockaddr*)&rmtAddr, &addrLen);

					if (n > 0) {
						const char* clientName = rmtAddr.sun_path + 1;

						if (strcmp(msgBuf, "done") == 0) {
							int	index = ClientList_FindElem(obj->resp_wait_list, clientName);

							if (index >= 0) {
								ClientList_RmvElem(obj->resp_wait_list, index);
							} else {
								fprintf(stderr, "unknown or dropped client: '%s'\n", clientName);
							}
						} else {
							fprintf(stderr, "unexpected response: '%s', client= '%s'\n",
								msgBuf, clientName);
						}
					} else {
						fprintf(stderr, "failed to recvfrom().\n");
					}
				}
				pollfdSet[0].revents = 0;
			}
			if (pollfdSet[1].revents != 0) {
				if ((pollfdSet[1].revents & POLLIN) != 0) {
					socklen_t	addrLen = sizeof(rmtAddr);
					int	n = recvfrom(obj->service_sock, msgBuf, sizeof(msgBuf), 0,
						(struct sockaddr*)&rmtAddr, &addrLen);

					if (n > 0) {
						sprintf(msgBuf, "NG; preparing sleep");
						if (sendto(obj->service_sock, msgBuf, strlen(msgBuf) + 1, 0,
								(struct sockaddr*)&rmtAddr, addrLen) < 0) {
							fprintf(stderr, "failed to sendto(), errno= %d, client= '%s'\n",
								errno, rmtAddr.sun_path + 1);
						}
					} else {
						fprintf(stderr, "failed to recvfrom().\n");
					}
				}
				pollfdSet[1].revents = 0;
				//
				// 注意：PowerStateHandler_BeforeSleepAction() を実行している間に 
				//       REQ_START_INHIBIT_SLEEP メッセージが届いた場合、それに対してエラーを
				//       返さなければならない。
			}
		}

		timeout = TIMEOUT_SECS - ((GetCurrentTimeInMillis() - startTime) / 1000);
		if (timeout <= 0) {
			fprintf(stderr, "timed out for waiting the sleep confirmation.\n");
			break;
		}
	}

	sprintf(msgBuf, "OK");
	if (sendto(obj->service_sock, msgBuf, strlen(msgBuf) + 1, 0,
			(struct sockaddr*)&obj->sleep_requester.address, CLIENT_ADDR_LEN(&obj->sleep_requester)) < 0) {
		fprintf(stderr, "failed to sendto(), errno= %d, sleep_requester= '%s'\n",
			errno, CLIENT_NAME(&obj->sleep_requester));
	}
}

static void
PowerStateHandler_AfterResumeAction(PowerStateHandler* obj)
{
	release_swupdate_lock();
	BroadcastToClients(obj->notify_sock, obj->sleep_resume_listners, NFY_LEAVE_SLEEP);
}

static int
MakeNamedSocket(const char* name)
{
	int	sockfd = socket(AF_LOCAL, SOCK_DGRAM|SOCK_CLOEXEC, 0);
	struct sockaddr_un	sockAddr;

	if (sockfd < 0) {
		return -1;
	}
	bzero(&sockAddr, sizeof(sockAddr));
	sockAddr.sun_family = AF_LOCAL;
	strcpy((char*)sockAddr.sun_path + 1, name);
	if (bind(sockfd, (struct sockaddr*)&sockAddr, ADDR_UN_LEN(sockAddr)) < 0) {
		(void)close(sockfd);
		return -1;
	}

	return sockfd;
}

power_state_handler_t
PowerStateHandler_Create()
{
	PowerStateHandler* newObj = (PowerStateHandler*)calloc(1, sizeof(PowerStateHandler));

	if (NULL == newObj) {
		fprintf(stderr, "failed to calloc().\n");
		return NULL;
	}
	newObj->wait_event = eventfd(0, 0);
	if (newObj->wait_event < 0) {
		fprintf(stderr, "failed to eventfd().\n");
		free(newObj);
		return NULL;
	}
	newObj->service_sock = MakeNamedSocket(SERVICE_NAME);
	if (newObj->service_sock < 0) {
		fprintf(stderr, "failed to make the named socket: '%s'\n", SERVICE_NAME);
		(void)close(newObj->wait_event);
		free(newObj);
		return NULL;
	}
	newObj->notify_sock = MakeNamedSocket(NOTIFIER_NAME);
	if (newObj->notify_sock < 0) {
		fprintf(stderr, "failed to make the named socket: '%s'\n", NOTIFIER_NAME);
		(void)close(newObj->service_sock);
		(void)close(newObj->wait_event);
		free(newObj);
		return NULL;
	}

	newObj->inhibit_lockers = ClientList_Create(4);
	newObj->sleep_resume_listners = ClientList_Create(4);
	newObj->resp_wait_list = ClientList_Create(4);
	if (NULL == newObj->inhibit_lockers
			|| NULL == newObj->sleep_resume_listners
			|| NULL == newObj->resp_wait_list) {
		fprintf(stderr, "failed to ClientList_Create().\n");
		PowerStateHandler_Destroy(newObj);
		return NULL;
	}
	newObj->curr_state = POWER_NORMAL;

	return newObj;
}

void
PowerStateHandler_Destroy(power_state_handler_t handler)
{
	PowerStateHandler* obj = (PowerStateHandler*)handler;

	if (PowerStateHandler_IsRunnig(handler)) {
		PowerStateHandler_Stop(handler);
	}
	ClientList_Destroy(obj->inhibit_lockers);
	ClientList_Destroy(obj->sleep_resume_listners);
	ClientList_Destroy(obj->resp_wait_list);
	(void)close(obj->notify_sock);
	(void)close(obj->service_sock);
	(void)close(obj->wait_event);
	free(obj);
}

bool
PowerStateHandler_Start(power_state_handler_t handler)
{
	PowerStateHandler* obj = (PowerStateHandler*)handler;

	if (0 != obj->thread) {
		return false;  // already started
	}
	if (0 != pthread_create(&obj->thread, NULL, ThreadProc, obj)) {
		fprintf(stderr, "failed to pthread_create().\n");
		return false;
	}

	return true;
}

void
PowerStateHandler_Stop(power_state_handler_t handler)
{
	if (PowerStateHandler_IsRunnig(handler)) {
		PowerStateHandler* obj = (PowerStateHandler*)handler;
		void* retval = NULL;
		int	sts;

		obj->quit_requested = true;
		(void)eventfd_write(obj->wait_event, 1);
		sts = pthread_join(obj->thread, &retval);
		if (0 != sts) {
			fprintf(stderr, "failed to pthread_join().\n");
		}
		obj->thread = 0;
		// obj->sleep_requester などに対する後始末（未了）
		// xxx
	}
}

bool
PowerStateHandler_IsRunnig(power_state_handler_t handler)
{
	PowerStateHandler* obj = (PowerStateHandler*)handler;
	return (obj->thread != 0);
}

EPowerHandlerState
PowerStateHandler_CurrentState(power_state_handler_t handler)
{
	PowerStateHandler* obj = (PowerStateHandler*)handler;
	return obj->curr_state;
}


//
// for the client process
//
static void
MakeSockAddr(struct sockaddr_un* addrBuf, const char* name)
{
	bzero(addrBuf, sizeof(*addrBuf));
	addrBuf->sun_family = AF_LOCAL;
	strcpy((char*)addrBuf->sun_path + 1, name);
}

static bool
ExecRequestOrNotification(const char* msg, bool isRequest)
{
	bool	isOK = false;
	char	clientID[32];
	struct sockaddr_un	selfAddr;
	struct sockaddr_un	serverAddr;
	int	sockfd;

	sockfd = socket(AF_LOCAL, SOCK_DGRAM, 0);
	sprintf(clientID, "atmark-power-req_%d", getppid());
	MakeSockAddr(&selfAddr, clientID);
	if (bind(sockfd, (struct sockaddr*)&selfAddr, ADDR_UN_LEN(selfAddr)) < 0) {
		fprintf(stderr, "failed to bind().\n");
		goto finish;
	}
	MakeSockAddr(&serverAddr, SERVICE_NAME);
	if (connect(sockfd, (struct sockaddr*)&serverAddr, ADDR_UN_LEN(serverAddr)) < 0) {
		fprintf(stderr, "failed to connect().\n");
		goto finish;
	}
	if (send(sockfd, msg, strlen(msg) + 1, 0) < 0) {
		fprintf(stderr, "failed to send*().\n");
		goto finish;
	}

	if (isRequest) {
		char	msgBuf[64];

		if (recvfrom(sockfd, msgBuf, sizeof(msgBuf), 0, NULL, NULL) < 0) {
			fprintf(stderr, "failed to recvfrom().\n");
		} else if (0 != strcmp(msgBuf, "OK")) {
			printf("request rejected, reply: '%s'\n", msgBuf);
		} else {
			isOK = true;
		}
	} else {
		isOK = true;
	}

finish:
	(void)close(sockfd);
	return isOK;
}

bool
Request_PowerStateHandler_StartInhibitSleep()
{
	return ExecRequestOrNotification(REQ_START_INHIBIT_SLEEP, true);
}

bool
Request_PowerStateHandler_FinishInhibitSleep()
{
	return ExecRequestOrNotification(REQ_FINISH_INHIBIT_SLEEP, true);
}

bool
Request_PowerStateHandler_StartSleep()
{
	return ExecRequestOrNotification(REQ_START_SLEEP, true);
}

bool
Notify_PowerStateHandler_FinishSleep()
{
	return ExecRequestOrNotification(NFY_FINISH_SLEEP, false);
}

bool
Request_PowerStateHandler_DisableSleep()
{
	return ExecRequestOrNotification(REQ_DISABLE_SLEEP, true);
}

bool
Request_PowerStateHandler_EnableSleep()
{
	return ExecRequestOrNotification(REQ_ENABLE_SLEEP, true);
}


#ifndef ENABLE_UNIT_TEST
//# define ENABLE_UNIT_TEST	1
# define ENABLE_UNIT_TEST	0
#endif

#if ENABLE_UNIT_TEST
static bool
RequestPowerStateNotification(int sockfd)
{
	bool	isOK = false;
	struct sockaddr_un	serverAddr;
	const char* msg = REQ_POWER_STATE_NOTIFICATION;

	MakeSockAddr(&serverAddr, SERVICE_NAME);
	if (sendto(sockfd, msg, strlen(msg) + 1, 0,
			(struct sockaddr*)&serverAddr, ADDR_UN_LEN(serverAddr)) < 0) {
		fprintf(stderr, "failed to sendto(), errno= %d, serverr= '%s'.\n",
			errno, serverAddr.sun_path + 1);
		return -1;
	} else {
		char	msgBuf[64];
		struct sockaddr_un	rmtAddr;
		socklen_t	len = sizeof(rmtAddr);

		if (recvfrom(sockfd, msgBuf, sizeof(msgBuf), 0, (struct sockaddr*)&rmtAddr, &len) < 0) {
			fprintf(stderr, "failed to recvfrom().\n");
		} else if (0 != strcmp(msgBuf, "OK")) {
			printf("request rejected, reply: '%s'\n", msgBuf);
		} else {
			isOK = true;
		}
	}

	return isOK;
}

static void
DoMonitorPowerState()
{
	char	client_id[64];
	int	sockfd;

	sprintf(client_id, "atmark-power-listener_%d", getpid());
	sockfd = MakeNamedSocket(client_id);
	if (sockfd < 0) {
		fprintf(stderr, "failed to MakeNamedSocket().\n");
		return;
	}

	if (!RequestPowerStateNotification(sockfd)) {
		fprintf(stderr, "failed to RequestPowerStateNotification().\n");
		(void)close(sockfd);
		return;
	} else {
		char	msgBuf[64];
		struct sockaddr_un	rmtAddr;
		socklen_t	len;

		printf("wait for notification...");
		fflush(stdout);
		bzero(&rmtAddr, sizeof(rmtAddr));
		len = sizeof(rmtAddr);
		if (recvfrom(sockfd, msgBuf, sizeof(msgBuf), 0, (struct sockaddr*)&rmtAddr, &len) < 0) {
			fprintf(stderr, "failed to recvfrom().\n");
			(void)close(sockfd);
			return;
		} else {
			printf("recevied msg: '%s', peer addr: '%s'\n", msgBuf, rmtAddr.sun_path + 1);
			printf("wait a while...");
			fflush(stdout);
			sleep(3);
			printf("now, send the reply.\n");
			if (sendto(sockfd, "done", strlen("done") + 1, 0, (struct sockaddr*)&rmtAddr, len) < 0) {
				fprintf(stderr, "failed to sendto().\n");
/*
				printf("remote addr: '%s', len= %d\n", rmtAddr.sun_path + 1, len);
				for (int i = 0; i < len; ++i) {
					printf("%02x", ((const unsigned char*)&rmtAddr)[i]);
				}
				fflush(stdout);
*/
			}
			(void)close(sockfd);
		}
	}
}

int
main(int argc, char** argv)
{
	if (argc == 2) {
		bool	isOK = false;

		if (0 == strcmp(argv[1], "inhibit_sleep")) {
			isOK = Request_PowerStateHandler_StartInhibitSleep();
		} else if (0 == strcmp(argv[1], "allow_sleep")) {
			isOK = Request_PowerStateHandler_FinishInhibitSleep();
		} else if (0 == strcmp(argv[1], "enter_sleep")) {
			isOK = Request_PowerStateHandler_StartSleep();
		} else if (0 == strcmp(argv[1], "leave_sleep")) {
			isOK = Notify_PowerStateHandler_FinishSleep();
		} else if (0 == strcmp(argv[1], "disable_sleep")) {
			isOK = Request_PowerStateHandler_DisableSleep();
		} else if (0 == strcmp(argv[1], "enable_sleep")) {
			isOK = Request_PowerStateHandler_EnableSleep();
		} else if (0 == strcmp(argv[1], "monitor")) {
			DoMonitorPowerState();
			return 0;
		} else {
			fprintf(stderr, "unknown sub command: '%s'\n", argv[1]);
			return -1;
		}
		return (isOK ? 0 : -1);
	} else {
		power_state_handler_t	handler = PowerStateHandler_Create();

		if (!PowerStateHandler_Start(handler)) {
			fprintf(stderr, "failedd to PowerStateHandler_Start().\n");
		}
		else {
			printf("the state machine successfully started.\n");
			sleep(10 * 60);  // wait 10minutes
			PowerStateHandler_Stop(handler);
		}
		PowerStateHandler_Destroy(handler);

		return 0;
	}
}
#endif  // ENABLE_UNIT_TEST

//
// End of File
//
