//
// swupdate_locker.c
//

#define _POSIX_C_SOURCE 200809L

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <unistd.h>

#include "swupdate_locker.h"

#define LOCK_FILE	"/var/lock/swupdate.lock"
#define REBOOT_FILE	"/run/swupdate_rebooting"

static int lock_fd = -1;

enum swupdate_status
acquire_swupdate_lock()
{
	struct stat statbuf_fd, statbuf_path;

	// this should never happen in practice
	if (lock_fd >= 0)
		goto sanity_checks;

again:
	lock_fd = open(LOCK_FILE, O_WRONLY|O_CREAT, 0644);
	if (lock_fd < 0) {
		printf("Could not open mkswu lock file %s: %m\n", LOCK_FILE);
		return SWUPDATE_ERROR;
	}
	if (flock(lock_fd, LOCK_EX|LOCK_NB) < 0) {
		if (errno != EAGAIN || errno != EWOULDBLOCK) {
			printf("Could not take mkswu lock: %m\n");
			goto out_close;
		}
		printf("Waiting for mkswu lock...\n");
		while (1) {
			if (flock(lock_fd, LOCK_EX) >= 0)
				break;
			if (errno == EAGAIN)
				continue;
			printf("Could not take mkswu lock: %m\n");
			goto out_close;
		}
	}
sanity_checks:
	if (fstat(lock_fd, &statbuf_fd) < 0) {
		// should never happen...
		printf("Could not stat mkswu lock (fd): %m\n");
		goto out_close;
	}
	if (lstat(LOCK_FILE, &statbuf_path) < 0 || statbuf_fd.st_ino != statbuf_path.st_ino) {
		printf("lock file changed, grabbing again\n");
		close(lock_fd);
		goto again;
	}
	if (access(REBOOT_FILE, F_OK) == 0) {
		return SWUPDATE_REBOOTING;
	}
	// write PID for debugging. This is only informational so we do
	// not care about errors.
	lseek(lock_fd, 0, SEEK_SET);
	ftruncate(lock_fd, 0);
	dprintf(lock_fd, "%d\n", getpid());
	return SWUPDATE_LOCKED;

out_close:
	close(lock_fd);
	lock_fd = -1;
	return SWUPDATE_ERROR;
}

void
release_swupdate_lock(void)
{
	if (lock_fd < 0)
		return;
	unlink(LOCK_FILE);
	close(lock_fd);
	lock_fd = -1;
}

extern void
swupdate_mark_reboot(void)
{
	int fd;

	fd = open(REBOOT_FILE, O_WRONLY|O_CREAT, 0644);
	if (fd < 0) {
		printf("Could not create reboot marker %s: %m\n", REBOOT_FILE);
		return;
	}
	close(fd);
}

//
// End of File
//

