//
// pwu_config.cpp
//

#include "pwu_config.h"

#include <string.h>
#include <vector>

#include <minIni.h>

#if WIN32
# define strdup	_strdup
#endif  // WIN32


#define TARGET	"TARGET"
#define MODE	"MODE"
#define WAKEUP	"WAKEUP"

#define RTC_PREFIX	"'RTC:"


typedef struct {
	PWU_WAKEUP  cause;
	void*       param;
	bool        param_must_free;
} PWU_WakeupCause;

typedef struct {
	std::vector<char*>*             targets;
	PWU_MODE                        mode;
	std::vector<PWU_WakeupCause>*   wakeups;
	bool                            is_valid;
} PWU_Config;


static void
PWU_WakeupCause_Init(PWU_WakeupCause& obj, PWU_WAKEUP val, void* param, bool must_free)
{
	if (param == NULL) {
		must_free = false;
	}
	obj.cause = val;
	obj.param = param;
	obj.param_must_free = must_free;
}

static void
PWU_WakeupCause_Cleanup(PWU_WakeupCause& obj)
{
	if (obj.param_must_free) {
		free(obj.param);
	}
}

static void
CleanupWakeups(PWU_Config* confObj)
{
	for (std::vector<PWU_WakeupCause>::iterator iter = confObj->wakeups->begin();
	iter < confObj->wakeups->end(); iter++) {
		PWU_WakeupCause_Cleanup(*iter);
	}
	delete confObj->wakeups;
	confObj->wakeups = NULL;
}

static char*
ValidateValue(char* token)
{
	char*	curs = token;

	while (isspace(*curs)) {
		++curs;
	}

	return curs;
}

static bool
ExtractValues(const char* valuesStr, std::vector<const char*>& outVec)
{
	char* token;

	outVec.clear();
	token = strtok(const_cast<char*>(valuesStr), ",");
	if ((token = ValidateValue(token)) == NULL) {
		return false;
	}
	outVec.push_back(token);
	while ((token = strtok(NULL, ",")) != NULL) {
		if ((token = ValidateValue(token)) == NULL) {
			return false;
		}
		outVec.push_back(token);
	}

	return true;
}

static PWU_MODE
ValidateMode(const char* mode)
{
	if (strcmp(mode, "'NONE'") == 0) {
		return PWU_MODE_NONE;
	} else if (strcmp(mode, "'SHUTDOWN'") == 0) {
		return PWU_MODE_SHUTDOWN;
	} else if (strcmp(mode, "'SLEEP'") == 0) {
		return PWU_MODE_SLEEP;
	} else {
		fprintf(stderr, "unkown mode: %s\n", mode);
		return PWU_MODE_UNKNOWN;
	}
}

static bool
ValidateWakeupCause(const char* strVal, PWU_WakeupCause* outCause)
{
	if (strncmp(strVal, RTC_PREFIX, strlen(RTC_PREFIX)) == 0) {
		long	secs = atol(strVal + strlen(RTC_PREFIX));

		if (secs < 60) {
			fprintf(stderr, "Warning; rtc wakeup time must greater than 60[sec].\n");
			secs = 60;
		}
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_RTC, (void*)secs, false);
	} else if (strcmp(strVal, "'AIN'") == 0) {
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_AIN, NULL, false);
	} else if (strcmp(strVal, "'SW1'") == 0) {
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_SW1, NULL, false);
	} else if (strcmp(strVal, "'GPIO'") == 0) {
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_GPIO, NULL, false);
	} else if (strcmp(strVal, "'USB'") == 0) {
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_USB, NULL, false);
	} else if (strcmp(strVal, "'UART'") == 0) {
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_UART, NULL, false);
	} else if (strcmp(strVal, "'SMS'") == 0) {
		PWU_WakeupCause_Init(*outCause, PWU_WAKEUP_SMS, NULL, false);
	} else {
		fprintf(stderr, "unknown wakeup: %s\n", strVal);
		return false;
	}

	return true;
}

static int
IniParserCallback(const mTCHAR* /*Section*/,
	const mTCHAR* Key, const mTCHAR* Value, void* UserData)
{
	PWU_Config* confObj = (PWU_Config*)UserData;

	if (strcmp(Key, TARGET) == 0) {
		if (confObj->targets != NULL) {
			goto err_duplicate;
		}
		std::vector<const char*>	vals;

		if (ExtractValues(Value, vals)) {
			confObj->targets = new std::vector<char*>();
			confObj->targets->reserve(vals.size());
			for (std::vector<const char*>::iterator iter = vals.begin();
			iter < vals.end(); iter++) {
				confObj->targets->push_back(strdup(*iter));
			}
		}
	} else if (strcmp(Key, MODE) == 0) {
		if (confObj->mode != PWU_MODE_UNKNOWN) {
			goto err_duplicate;
		}
		confObj->mode = ValidateMode(Value);
	} else if (strcmp(Key, WAKEUP) == 0) {
		if (confObj->wakeups != NULL) {
			goto err_duplicate;
		}
		std::vector<const char*>	vals;

		if (ExtractValues(Value, vals)) {
			confObj->wakeups = new std::vector<PWU_WakeupCause>();
			confObj->wakeups->reserve(vals.size());
			for (std::vector<const char*>::iterator iter = vals.begin();
			iter < vals.end(); iter++) {
				PWU_WakeupCause	pseudo;

				if (! ValidateWakeupCause(*iter, &pseudo)) {
					CleanupWakeups(confObj);
					confObj->is_valid = false;
					return 1;
				}
				confObj->wakeups->push_back(pseudo);
			}
		}
	}

	return 1;
err_duplicate:
	fprintf(stderr, "%s is already specified, ignore secondaries.\n", Key);
	return 1;
}

int
pwu_config_load(const char* conf_file, pwu_config_t* outConf)
{
	PWU_Config* newObj = (PWU_Config*)malloc(sizeof(PWU_Config));

	if (newObj == NULL) {
		return -1;  // out of memory
	}
	newObj->targets = NULL;
	newObj->mode    = PWU_MODE_UNKNOWN;
	newObj->wakeups = NULL;
	newObj->is_valid = true;
	ini_browse(IniParserCallback, newObj, conf_file);
	*outConf = newObj;

	return (newObj->is_valid ? 0 : -2);
}

void
pwu_config_dispose(pwu_config_t conf)
{
	PWU_Config* obj = (PWU_Config*)conf;

	if (obj->targets != NULL) {
		for (std::vector<char*>::iterator iter = obj->targets->begin();
		iter < obj->targets->end(); iter++) {
			free(*iter);
		}
		delete obj->targets;
		obj->targets = NULL;
	}
	if (obj->wakeups != NULL) {
		CleanupWakeups(obj);
	}
	free(obj);
}

const char*
pwu_config_target_container(pwu_config_t conf)
{
	return pwu_config_target_at(conf, 0);
}

PWU_MODE
pwu_config_mode(pwu_config_t conf)
{
	PWU_Config* obj = (PWU_Config*)conf;
	return obj->mode;
}

PWU_WAKEUP
pwu_config_wakeup(pwu_config_t conf, const void** outParam)
{
	return pwu_config_wakeup_at(conf, 0, outParam);
}

size_t
pwu_config_count_targets(pwu_config_t conf)
{
	PWU_Config* obj = (PWU_Config*)conf;
	return (obj->targets != NULL ? obj->targets->size() : 0);
}

const char*
pwu_config_target_at(pwu_config_t conf, size_t index)
{
	size_t	count = pwu_config_count_targets(conf);

	if (count <= 0 || count <= index) {
		return NULL;
	} else {
		PWU_Config* obj = (PWU_Config*)conf;
		return (*obj->targets)[index];
	}
}

size_t
pwu_config_count_wakeups(pwu_config_t conf)
{
	PWU_Config* obj = (PWU_Config*)conf;
	return (obj->wakeups != NULL ? obj->wakeups->size() : 0);
}

PWU_WAKEUP
pwu_config_wakeup_at(pwu_config_t conf, size_t index, const void** outParam)
{
	size_t	count = pwu_config_count_wakeups(conf);

	if (count <= 0 || count <= index) {
		return PWU_WAKEUP_UNKNOWN;
	} else {
		PWU_Config* obj = (PWU_Config*)conf;
		const PWU_WakeupCause&	elem = (*obj->wakeups)[index];

		if (outParam != NULL) {
			*outParam = elem.param;
		}
		return elem.cause;
	}
}

int
pwu_config_get_wakeup_param(pwu_config_t conf, PWU_WAKEUP wakeup, const void** outParam)
{
	PWU_Config* obj = (PWU_Config*)conf;

	for (std::vector<PWU_WakeupCause>::iterator iter = obj->wakeups->begin();
	iter < obj->wakeups->end(); iter++) {
		if (iter->cause == wakeup) {
			*outParam = iter->param;
			return 0;
		}
	}

	return -1;
}

//
// End of File
//
