//
// power-utils.c
//

#include <fcntl.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>
#include <unistd.h>

#include "power_funcs.h"
#include "power_state_handler.h"
#include "pwu_config.h"
#include "un_socket.h"


extern int	pwu_notify_container(const char* msg);

#define CONTAINER_STOP	"container_stop"
#define QUIT_SERVICE	"quit"

#define CONF_FILE_PATH	"/etc/atmark/power-utils.conf"
#define SLEEP_SECS_PATH	"/etc/atmark/power-utils/sleep_secs"

typedef enum PowerMode {
	PM_NONE,
	PM_SHUTDOWN,
	PM_SLEEP,
} PowerMode;


static pwu_config_t	sConfig = 0;
static power_state_handler_t	sPowerStateHandler = NULL;

static int
start_containers(pwu_config_t conf)
{
	const char* target;

	// 設定ファイルで指定されたコンテナを podman_start で起動
	target = pwu_config_target_container(conf);
	if (target != NULL) {
		char	cmd[256];

		snprintf(cmd, sizeof(cmd) - 1, "/usr/bin/podman_start %s", target);
		cmd[sizeof(cmd) - 1] = '\0';
		(void)system(cmd);
	// NOTE: 少なくとも最初の版では、pod をサポートしない
	//       <- pod 内のコンテナが全て終了した時に pod の poststop hook が実行されるか
	//          否かが不明なため
	}

	return 0;
}

static void
do_sleep_and_resume(bool isNormalMode)
{
	// 起床要因を設定
	(void)power_disable_wakeup_all();
	if (isNormalMode) {
		if (power_enable_wakeup_causes(sConfig) != 0) {
			printf("failed to power_enable_wakeup_causes().\n");
			goto finish;
		}
	} else {
		unsigned long	secs = 0;
		char	secsBuf[20+1];
		int	fd;

		fd = open(SLEEP_SECS_PATH, O_RDONLY);
		if (fd >= 0) {
			size_t	readLen = read(fd, secsBuf, sizeof(secsBuf) - 1);

			(void)close(fd);
			if (readLen > 0) {
				char*	endptr = NULL;

				secsBuf[readLen] = '\0';
				if (secsBuf[readLen - 1] == '\n') {
					secsBuf[readLen - 1] = '\0';
				}
				secs = strtoul(secsBuf, &endptr, 10);
				if (endptr != NULL && *endptr != '\0') {
					secs = 0;
				}
			}
		}
		if (secs < 5 || 3600 < secs) {
			printf("invalid sleep secs: %lu\n", secs);
			goto finish;
		} else if (power_enable_wakeup_by_snvs_rtc(secs) != 0) {
			printf("failed to power_enable_wakeup_by_snvs_rtc().\n");
			goto finish;
		}
	}

	// OS をサスペンド
	if (power_suspend() != 0) {
		printf("failed to power_suspend().\n");
		goto finish;
	}
	printf("back from suspend.\n");

finish:
	// サスペンドから復帰したらコンテナを起動（レジューム動作）
	(void)unlink(SLEEP_SECS_PATH);
	start_containers(sConfig);
}

static unsigned long
get_poweroff_secs(pwu_config_t conf)
{
	const void* param;

	if (pwu_config_get_wakeup_param(conf, PWU_WAKEUP_RTC, &param) != 0
	|| (unsigned long)param < 180) {
		return 180;
	} else{
		return (unsigned long)param;
	}
}

static bool
check_sleep_secs_requested()
{
	struct stat	statbuf;

	return (0 == stat(SLEEP_SECS_PATH, &statbuf) && S_IFREG == (statbuf.st_mode & S_IFMT));
}

static void
do_power_utils_service(int sockfd)
{
	unsigned char	msgBuf[128];

	for (;;) {
		size_t	recvSize = sock_recv(sockfd, msgBuf, sizeof(msgBuf));

		if (recvSize == 0 || recvSize == (size_t)-1) {
			break;
		}
		msgBuf[recvSize] = '\0';
		if (strcmp((char*)msgBuf, CONTAINER_STOP) == 0) {
			pwu_config_t	conf;
			PWU_MODE	mode;
			bool	isNormalMode = true;

			printf("the container is stopped.\n");
			if (pwu_config_load(CONF_FILE_PATH, &conf) == 0) {  // 設定ファイルを再ロード
				pwu_config_dispose(sConfig);
				sConfig = conf;
			} else {
				pwu_config_dispose(conf);
			}
			if (check_sleep_secs_requested()) {
				mode = PWU_MODE_SLEEP;
				isNormalMode = false;
			} else {
				mode = pwu_config_mode(sConfig);
				switch (mode) {
				case PWU_MODE_NONE:
				case PWU_MODE_UNKNOWN:
					continue;  // do nothing
				default:
					break;
				}
			}

			// 設定ファイルの設定項目 'targets 'で指定したコンテナ全てが停止したかどうかを判定
				// NOTE: この版では、target を単一とする
			switch (mode) {
			case PWU_MODE_SHUTDOWN:
				(void)power_shutdown(get_poweroff_secs(sConfig));
				break;
			case PWU_MODE_SLEEP:
				do_sleep_and_resume(isNormalMode);
				break;
			default:
				break;
			}
		} else if (strcmp((char*)msgBuf, QUIT_SERVICE) == 0) {
			printf("quit the service...\n");
			break;
		}
	}
}

int
main(int argc, char** argv)
{
	int	retVal = 0;
	char*	name = strrchr(argv[0], '/');
	char*	curs;

	if (name == NULL) {
		name = argv[0];
	} else {
		++name;
	}
	curs = strchr(name, '.');
	if (curs != NULL) {
		*curs = '\0';
	}
	if (strcmp(name, "pwu_notify") == 0) {
		const char* msg;

		if (argc != 2) {
			printf("Usage: pwu_notify <message>\n");
			return -1;
		} else {
			msg = argv[1];
		}
		if ((strcmp(msg, CONTAINER_STOP) != 0)
		&& (strcmp(msg, QUIT_SERVICE) != 0)) {
			printf("unknown message: '%s'\n", msg);
			return -1;
		}
		pwu_notify_container(argv[1]);
	} else if (argc >= 2) {
		if (0 == strcmp(argv[1], "inhibit_sleep")) {
			return (Request_PowerStateHandler_StartInhibitSleep() ? 0 : -1);
		} else if (0 == strcmp(argv[1], "allow_sleep")) {
			return (Request_PowerStateHandler_FinishInhibitSleep() ? 0 : -1);
		} else if (0 == strcmp(argv[1], "disable_sleep")) {
			return (Request_PowerStateHandler_DisableSleep() ? 0 : -1);
		} else if (0 == strcmp(argv[1], "enable_sleep")) {
			return (Request_PowerStateHandler_EnableSleep() ? 0 : -1);
		} else if (0 == strcmp(argv[1], "enter_sleep")) {  // for aiot script only
			return (Request_PowerStateHandler_StartSleep() ? 0 : -1);
		} else if (0 == strcmp(argv[1], "leave_sleep")) {  // for aiot script only
			return (Notify_PowerStateHandler_FinishSleep() ? 0 : -1);
		} else {
			printf("Usage: power-utils [ACTION]\n"
				"ACTION:\n"
				"  inhibit_sleep\n"
				"  allow_sleep\n"
				"  disable_sleep\n"
				"  enable_sleep\n");
			return -1;
		}
	} else {
		int	sockfd;

		if (pwu_config_load(CONF_FILE_PATH, &sConfig) != 0) {
			fprintf(stderr, "failed to pwu_config_load().\n");
			return -1;
		}
		sockfd = create_server_socket();
		if (sockfd < 0) {
			fprintf(stderr, "failed to create_server_socket().\n");
			return -1;
		}
		sPowerStateHandler = PowerStateHandler_Create();
		if (NULL == sPowerStateHandler) {
			fprintf(stderr, "failed to PowerStateHandler_Create().\n");
			retVal = -1;
		} else if (! PowerStateHandler_Start(sPowerStateHandler)) {
			fprintf(stderr, "failed to PowerStateHandler_Start().\n");
			PowerStateHandler_Destroy(sPowerStateHandler);
			retVal = -1;
		} else {
			do_power_utils_service(sockfd);
			PowerStateHandler_Stop(sPowerStateHandler);
			PowerStateHandler_Destroy(sPowerStateHandler);
		}
		dispose_socket(sockfd);
		pwu_config_dispose(sConfig);
	}

	return retVal;
}

//
// End of File
//
