import asyncio
import time
import traceback
from enum import Enum
from .log import Log
from .io.dio_manager import STATE as DIO_Mgr_STATE


class DI_config:
    def __init__(self, port, interval, edge_type):
        self._port = port
        self._interval = interval
        edge = EDGE.NONE
        if edge_type == "falling":
            edge = EDGE.FALL
        elif edge_type == "rising":
            edge = EDGE.RISE
        elif edge_type == "both":
            edge = EDGE.BOTH
        self._edge_type = edge
        self._state = 0
        self._start_time = time.time()
        self._edge_flag = EDGE.NONE
        self._notifi_edge = EDGE.NONE


class DO_config:
    def __init__(self,
                 port_state,
                 start,
                 port,
                 output_state,
                 output_time,
                 output_delay_time,
                 ):

        self._enable = self.check_do_enable(port_state, output_state)
        self._start = start
        self._state = STATE.NONE
        self._start_time = time.time()
        self._port = port
        self._output_state_isHigh = self.check_do_output_state(output_state)
        self._output_time = output_time
        self._output_delay_time = output_delay_time

    def check_do_output_state(self, output_state):
        return True if output_state != "low" else False

    def check_do_enable(self, port_state, output_state):
        return True if port_state == DIO_Mgr_STATE.ENABLE and output_state != "disable" else False


class EDGE(Enum):
    NONE = 0
    FALL = 1
    RISE = 2
    BOTH = 3


class STATE(Enum):
    NONE = 0
    START = 1
    OUTPUT = 2
    END = 3


class SensingMgr:

    def _set_xdo_output(self, port, is_high):
        if port == 0:
            self._dio_mgr.set_VOUT_Output(is_high)
        else:
            self._dio_mgr.set_DO_Output(port, is_high)

    def _xdout_proc(self, conf_list):
        for conf in conf_list:
            if conf._enable is False or conf._start is False:
                continue
            if conf._state is STATE.END:
                continue

            curr_time = time.time()
            elapsed_time = curr_time - conf._start_time

            if conf._state is STATE.NONE:
                conf._start_time = curr_time
                if conf._output_delay_time > 0:
                    self._set_xdo_output(
                        conf._port, not conf._output_state_isHigh)
                else:
                    self._set_xdo_output(
                        conf._port, conf._output_state_isHigh)
                conf._state = STATE.START

            elif conf._state is STATE.START:
                if elapsed_time > float(conf._output_delay_time):
                    self._set_xdo_output(
                        conf._port, conf._output_state_isHigh)
                    conf._start_time = curr_time
                    if conf._output_time == 0:
                        conf._state = STATE.END
                    else:
                        conf._state = STATE.OUTPUT

            elif conf._state is STATE.OUTPUT:
                if elapsed_time > float(conf._output_time):
                    self._set_xdo_output(
                        conf._port, not conf._output_state_isHigh)
                    conf._state = STATE.END

    async def __async_loop(self, timeout=None):
        self.__curr_task = asyncio.current_task()

        while not self.__quit_requested:

            for conf in self._di_conf:
                curr_state = self._dio_mgr.get_DI(conf._port)

                # check edge
                if conf._state is not curr_state:
                    if conf._state is True:
                        conf._edge_flag = EDGE.FALL
                    else:
                        conf._edge_flag = EDGE.RISE

                if conf._notifi_edge is EDGE.NONE:
                    conf._notifi_edge = conf._edge_flag

                # set edge
                if (time.time() - conf._start_time) > conf._interval:
                    conf._start_time = time.time()
                    if conf._notifi_edge is EDGE.FALL and \
                            conf._edge_type is EDGE.FALL:
                        self._dio_mgr.set_DI_edge(conf._port, 1)
                    elif conf._notifi_edge is EDGE.RISE and \
                            conf._edge_type is EDGE.RISE:
                        self._dio_mgr.set_DI_edge(conf._port, 1)
                    elif conf._notifi_edge is not EDGE.NONE and \
                            conf._edge_type is EDGE.BOTH:
                        self._dio_mgr.set_DI_edge(conf._port, 1)
                    else:
                        self._dio_mgr.set_DI_edge(conf._port, 0)
                    conf._notifi_edge = EDGE.NONE

            # DO
            self._xdout_proc(self._do_conf)

            # VOUT
            self._xdout_proc(self._vout_conf)

            try:
                await asyncio.sleep(0.1)
            except asyncio.CancelledError:
                break

    def start_loop(self, timeout=None):
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        try:
            loop.run_until_complete(self.__async_loop(timeout))
            loop.close()
        except Exception as e:
            log = Log.instance()
            log.error("sensing_manager error")
            log.error(e.args)
            log.error(traceback.format_exc())
            log.preserved_error_log()

    def set_di_conf(self, port, interval, edge_type):
        for conf in self._di_conf:
            if port is conf._port:
                new_conf = conf
                new_conf._interval = interval
                new_conf._edge_type = edge_type
                new_conf._state = self._dio_mgr.get_DI(port)
                self._di_conf.remove(conf)
                self._di_conf.append(new_conf)
                break

    def set_do_conf(
            self,
            port,
            output_state,
            output_time,
            output_delay_time,
            ):

        for conf in self._do_conf:
            if port is conf._port:
                port_state = self._dio_mgr.get_DO_status(port)

                if port_state == DIO_Mgr_STATE.NONE and (output_state == 'high' or output_state == "low"):
                    val = conf.check_do_output_state(output_state)
                    self._dio_mgr.reset_DO(port=port, val=not val)
                    port_state = self._dio_mgr.get_DO_status(port)

                if port_state == DIO_Mgr_STATE.ENABLE:
                    new_conf = conf
                    new_conf._start = False
                    new_conf._output_state_isHigh = conf.check_do_output_state(output_state)
                    new_conf._enable = conf.check_do_enable(port_state, output_state)
                    new_conf._output_time = output_time
                    new_conf._output_delay_time = output_delay_time

                    self._do_conf.remove(conf)
                    self._do_conf.append(new_conf)
                break

    def start_do(self, port):
        for conf in self._do_conf:
            if port is conf._port:
                if conf._enable:
                    conf._start = True
                    conf._state = STATE.NONE
                    conf._start_time = time.time()
                    return True
                else:
                    return False
        return False

    def stop_do(self, port):
        for conf in self._do_conf:
            if port is conf._port:
                if conf._enable:
                    conf._start = False
                    return True
                else:
                    return False
        return False

    def _get_vout_conf(self):
        if len(self._vout_conf) == 0:
            return None

        return self._vout_conf[0]

    def set_vout_conf(self, output_state, output_time, output_delay_time):
        conf = self._get_vout_conf()
        if conf is None:
            return

        port_state = self._dio_mgr.get_VOUT_status()
        if port_state == DIO_Mgr_STATE.NONE and (output_state == 'high' or output_state == "low"):
            val = conf.check_do_output_state(output_state)
            self._dio_mgr.reset_VOUT(val=not val)
            port_state = self._dio_mgr.get_VOUT_status()

        if port_state == DIO_Mgr_STATE.ENABLE:
            new_conf = conf
            new_conf._start = False
            new_conf._output_state_isHigh = conf.check_do_output_state(output_state)
            new_conf._enable = conf.check_do_enable(port_state, output_state)
            new_conf._output_time = output_time
            new_conf._output_delay_time = output_delay_time

            self._vout_conf.remove(conf)
            self._vout_conf.append(new_conf)

    def start_vout(self):
        conf = self._get_vout_conf()
        if conf is None:
            return False

        if conf._enable:
            conf._start = True
            conf._state = STATE.NONE
            conf._start_time = time.time()
            return True

        return False

    def stop_vout(self):
        conf = self._get_vout_conf()
        if conf is None:
            return False

        if conf is not None and conf._enable:
            conf._start = False
            return True

        return False

    def request_stop(self):
        if self.__curr_task is not None:
            self.__quit_requested = True
            self.__curr_task.cancel()

    def create_di_conf(self, port, interval, edge_type):
        di_conf = DI_config(port, interval, edge_type)
        di_conf._state = self._dio_mgr.get_DI(port)
        self._di_conf.append(di_conf)

    def create_do_conf(
            self,
            port,
            port_state,
            start,
            output_state,
            output_time,
            output_delay_time,
            ):
        do_conf = DO_config(port_state, start, port, output_state, output_time,
                            output_delay_time
                            )
        self._do_conf.append(do_conf)

    def create_vout_conf(
            self,
            port_state,
            start,
            output_state,
            output_time,
            output_delay_time,
            ):
        vout_conf = DO_config(port_state,
                              start,
                              0,
                              output_state,
                              output_time,
                              output_delay_time
                              )
        self._vout_conf.append(vout_conf)

    def __init__(self, dio_mgr):
        self._dio_mgr = dio_mgr
        self._di_conf = []
        self._do_conf = []
        self._vout_conf = []
        self.__quit_requested = False
        self.__curr_task = None

#
# End of File
#
