from .model_dev_base import ModelDevBase
from .reporters.cpu_temp_reporter import CpuTempReporter
from .reporters.di_reporter import DIReporter
from .reporters.rs485_reporter import RS485Reporter
from .io.rs485_manager import RS485_Mgr
from .reporters.sw_event_reporter import SwEventReporter
from .sensing_manager import SensingMgr
from .log import Log
from .io.led_device import LedDevice
from .io.ain_operator import AinOperator
from .reporters.ain_reporter import AinReporter
from .io.vin_operator import VinOperator
from .reporters.vin_reporter import VinReporter
from .product import Product


class ModelDevGW(ModelDevBase):
    def __init__(self, modelConfig, cloud_config, send_cloud):
        super().__init__(modelConfig, send_cloud)
        self._dio_mgr = None
        self._rs485_mgr = None
        self._sensing_mgr = None
        self._ain_operator = None
        self._vin_operator = None
        self._reporters = {}
        self._model_config = modelConfig
        self._cloud_config = cloud_config
        self._sw_event = "/dev/input/by-path/platform-gpio-keys-event"
        self._product = Product()
        self._dio_mgr = self._product.dio_mgr()
        self._rs485_path = self._product.rs485_path()
        self._vin_iios = self._product.vin_iios()

    def on_recv_data(self, data):
        log = Log.instance()
        if data is None:
            log.info("data from cloud is empty.")
            return False

        if self._cloud_config:
            log.info("update config from cloud.")
            self.update_reporter(self._model_config, data)
        else:
            log.info("config don't update from cloud.")

        return self.exec_command(self._model_config, data)

    def update_reporter(self, config, data):
        log = Log.instance()
        update_list = config.update_config(data)

        if not update_list:
            return

        for section, conf in update_list.items():
            if section.startswith(config.DI):
                reporter = self._reporters[section]
                if reporter is None:
                    log.error("update_reporter: unknwon section:{section}")
                    continue

                reporter.set_config(conf.get(config.TYPE_IS_POLLING))
                reporter.set_interval(conf.get(config.INTERVAL))
                if conf.get(config.START):
                    log.info("{section} start.")
                    reporter.start()
                else:
                    log.info("{section} stop.")
                    reporter.stop()

            elif section == config.RS485_DATA1 or \
                    section == config.RS485_DATA2 or \
                    section == config.RS485_DATA3 or \
                    section == config.RS485_DATA4:

                reporter = self._reporters[section]
                if section == config.RS485_DATA1:
                    data_num = self._rs485_mgr.DATA1
                elif section == config.RS485_DATA2:
                    data_num = self._rs485_mgr.DATA2
                elif section == config.RS485_DATA3:
                    data_num = self._rs485_mgr.DATA3
                elif section == config.RS485_DATA4:
                    data_num = self._rs485_mgr.DATA4

                reporter.set_config(
                    data_num=data_num,
                    method=conf.get(config.METHOD),
                    data_size=conf.get(config.DATA_SIZE),
                    baudrate=conf.get(config.BAUDRATE),
                    device_id=conf.get(config.DEVICE_ID),
                    stop_bits=conf.get(config.STOP),
                    parity=conf.get(config.PARITY),
                    reg_addr=conf.get(config.REGISTER_ADDR),
                    func_code=conf.get(config.FUNC_CODE),
                    count=conf.get(config.REGISTER_COUNT),
                    endian=conf.get(config.ENDIAN),
                    data_offset=conf.get(config.DATA_OFFSET),
                    data_multiply=conf.get(config.DATA_MULTIPLY),
                    data_divider=conf.get(config.DATA_DIVIDER)
                    )
                reporter.set_interval(conf.get(config.INTERVAL))
                reporter.start()
            elif section == config.DO1 or section == config.DO2:
                port = 0
                if section == config.DO1:
                    port = self._dio_mgr.do_port(1)
                elif section == config.DO2:
                    port = self._dio_mgr.do_port(2)

                self._sensing_mgr.set_do_conf(
                    port=port,
                    output_state=conf.get(config.OUTPUT_STATE),
                    output_time=conf.get(config.OUTPUT_TIME),
                    output_delay_time=conf.get(config.OUTPUT_DELAY_TIME),
                    )
            elif section == config.VOUT:
                self._sensing_mgr.set_vout_conf(
                    output_state=conf.get(config.OUTPUT_STATE),
                    output_time=conf.get(config.OUTPUT_TIME),
                    output_delay_time=conf.get(config.OUTPUT_DELAY_TIME),
                    )
            elif section.startswith(config.AIN):
                reporter = self._reporters[section]
                if reporter is None:
                    log.error("update_reporter: unknwon section:{section}")
                    continue

                reporter.set_interval(conf.get(config.INTERVAL))
                if conf.get(config.START):
                    log.info("{section} start.")
                    reporter.start()
                else:
                    log.info("{section} stop.")
                    reporter.stop()
            elif section == config.VIN:
                reporter = self._reporters[section]
                if reporter is None:
                    log.error("update_reporter: unknwon section:{section}")
                    continue

                reporter.set_interval(conf.get(config.INTERVAL))
                if conf.get(config.START):
                    log.info("vin start.")
                    reporter.start()
                else:
                    log.info("vin stop.")
                    reporter.stop()

    def exec_command(self, config, data):
        log = Log.instance()
        ret = True
        exec_result = True
        for section, conf in data.items():
            index = section.find('_command')
            section = section[:index]

            if conf == "":
                continue

            if section == config.DO1 or section == config.DO2:
                port = 0
                if section == config.DO1:
                    port = self._dio_mgr.do_port(1)
                elif section == config.DO2:
                    port = self._dio_mgr.do_port(2)

                key = "action"
                if key in conf.keys():
                    log.info(f"do is {key}:{conf[key]}")

                    if conf[key] == "start":
                        exec_result = self._sensing_mgr.start_do(port)
                    elif conf[key] == 'stop':
                        exec_result = self._sensing_mgr.stop_do(port)
                    else:
                        ret = False
                        log.error(
                            "DO_command valid value is start or stop.")

                    if not exec_result:
                        ret = False
                        Log.error(log, "DO_Command invalid port.")
                else:
                    ret = False
                    log.error("DO_command don't have [action]")

            elif section == "RS485":
                try:
                    self._rs485_mgr.write_command(
                        method=conf.get(config.METHOD),
                        baudrate=conf.get(config.BAUDRATE),
                        data_size=conf.get(config.DATA_SIZE),
                        stop_bits=conf.get(config.STOP),
                        parity=conf.get(config.PARITY),
                        device_id=conf.get(config.DEVICE_ID),
                        reg_addr=conf.get(config.REGISTER_ADDR),
                        func_code=conf.get(config.FUNC_CODE),
                        write_byte=conf.get("write_byte")
                        )
                except Exception as e:
                    ret = False
                    log.error("RS485_command is invalid parameter")
                    log.error(e.args)
            elif section == "led":
                key = "state"

                if key in conf.keys():
                    log.info(f"led is {key}:{conf[key]}")

                    if conf[key] == "on":
                        with LedDevice.get("app") as led:
                            led.on_or_off(True)
                    elif conf[key] == 'off':
                        with LedDevice.get("app") as led:
                            led.on_or_off(False)
                    else:
                        ret = False
                        log.error(
                            "led_command valid value is on or off.")
                else:
                    ret = False
                    log.error("led_command don't have [state]")
            elif section == config.VOUT:
                key = "action"
                if key in conf.keys():
                    log.info(f"vout is {key}:{conf[key]}")

                    if conf[key] == "start":
                        exec_result = self._sensing_mgr.start_vout()
                    elif conf[key] == 'stop':
                        exec_result = self._sensing_mgr.stop_vout()
                    else:
                        ret = False
                        log.error(
                            "VOUT_command valid value is start or stop.")

                    if not exec_result:
                        ret = False
                        Log.error(log, "VOUT_Command error.")
                else:
                    ret = False
                    log.error("VOUT_command don't have [action]")

        return ret

    def send_interval(self, config):
        conf = config.get_default_conf()
        return conf.get(config.SEND_INTERVAL)

    def _create_active_reporters(self, config, report_queue, report_repository):
        reporters = []
        swreporter = SwEventReporter(
            report_queue, report_repository, self._sw_event)

        reporters.append(swreporter)
        return reporters

    def _default_do_val(self, conf, config):
        default_val = None
        if conf.get(config.OUTPUT_STATE) == "high" or \
                conf.get(config.OUTPUT_STATE) == "low":
            default_val = 0

            if conf.get(config.OUTPUT_DELAY_TIME) == 0 \
                    and conf.get(config.OUTPUT_STATE) == "high":
                default_val = 1
            elif conf.get(config.OUTPUT_DELAY_TIME) > 0 \
                    and conf.get(config.OUTPUT_STATE) == "low":
                default_val = 1
        elif conf.get(config.OUTPUT_STATE) == "":
            default_val = -1

        return default_val

    def _create_di_reporters(self, config):
        di_reporters = []
        # DI/DO
        do_list = dict()
        port = self._dio_mgr.do_port(1)
        while port <= self._dio_mgr.do_port(2):
            conf = config.get_do_conf(port)
            do_list[port] = None
            if conf is not None:
                default_val = self._default_do_val(conf, config)
                if default_val is not None:
                    self._dio_mgr.setup_do(port, default_val)
            port += 1

        # DI reporter
        for i in range(self._dio_mgr.DI_PORT_MIN, self._dio_mgr.DI_PORT_MAX+1):
            conf = config.get_di_conf(i)
            if conf is not None:
                self._dio_mgr.setup_di(i)
                diReporter = DIReporter(
                    dio_mgr=self._dio_mgr,
                    start=conf.get(config.START),
                    type_isPolling=conf.get(config.TYPE_IS_POLLING),
                    port=i
                    )
                diReporter.set_interval(conf.get(config.INTERVAL))
                di_reporters.append(diReporter)
                self._reporters[config.DI+str(i)] = diReporter
        return di_reporters

    def _create_rs485_reporters(self, config):
        rs485_reporters = []
        rs485_mgr = RS485_Mgr(self._rs485_path)
        self._rs485_mgr = rs485_mgr
        data_num = self._rs485_mgr.DATA1
        while data_num <= self._rs485_mgr.DATA4:
            conf = config.get_rs485_conf(data_num)
            if conf is not None:
                rs485Reporter = RS485Reporter(
                    rs485_mgr=rs485_mgr,
                    start=conf.get(config.START),
                    data_num=data_num,
                    method=conf.get(config.METHOD),
                    baudrate=conf.get(config.BAUDRATE),
                    device_id=conf.get(config.DEVICE_ID),
                    data_size=conf.get(config.DATA_SIZE),
                    stop_bits=conf.get(config.STOP),
                    parity=conf.get(config.PARITY),
                    reg_addr=conf.get(config.REGISTER_ADDR),
                    func_code=conf.get(config.FUNC_CODE),
                    count=conf.get(config.REGISTER_COUNT),
                    endian=conf.get(config.ENDIAN),
                    data_offset=conf.get(config.DATA_OFFSET),
                    data_multiply=conf.get(config.DATA_MULTIPLY),
                    data_divider=conf.get(config.DATA_DIVIDER)
                    )
                rs485Reporter.set_interval(conf.get(config.INTERVAL))
                rs485_reporters.append(rs485Reporter)
                if data_num == self._rs485_mgr.DATA1:
                    self._reporters[config.RS485_DATA1] = rs485Reporter
                elif data_num == self._rs485_mgr.DATA2:
                    self._reporters[config.RS485_DATA2] = rs485Reporter
                elif data_num == self._rs485_mgr.DATA3:
                    self._reporters[config.RS485_DATA3] = rs485Reporter
                elif data_num == self._rs485_mgr.DATA4:
                    self._reporters[config.RS485_DATA4] = rs485Reporter
            data_num += 1
        return rs485_reporters

    def _create_ain_reporters(self, config):
        log = Log.instance()
        ain_reporters = []
        for i in range(AinOperator.AIN_PORT_MIN, AinOperator.AIN_PORT_MAX+1):
            conf = config.get_ain_conf(i)
            if conf is not None:
                if self._ain_operator is None:
                    self._ain_operator = AinOperator()
                if not self._ain_operator.has_map():
                    log.warning("Cannot find iio devices support AIN.")
                    return ain_reporters

                type_is_current = False
                if conf.get(config.TYPE) == "current":
                    type_is_current = True

                ain_reporter = AinReporter(
                    start=conf.get(config.START),
                    ain_operator=self._ain_operator,
                    port=i,
                    type_is_current=type_is_current
                    )
                ain_reporter.set_interval(conf.get(config.INTERVAL))
                ain_reporters.append(ain_reporter)
                self._reporters[config.DI+str(i)] = ain_reporter
        return ain_reporters

    def _create_vin_reporters(self, config):
        log = Log.instance()
        vin_reporters = []
        conf = config.get_vin_conf()
        if conf is not None:
            if self._vin_operator is None:
                valid_vin_iios = VinOperator.find_valid_vin_iio(self._vin_iios)
                if valid_vin_iios == []:
                    log.warning("Cannot find iio devices support VIN.")
                    return vin_reporters
                else:
                    self._vin_operator = VinOperator(valid_vin_iios[0])
                    log.info(f"Use iio device of {self._vin_operator.vin_iio.description} as a VIN.")

            vin_reporter = VinReporter(
                start=conf.get(config.START),
                vin_operator=self._vin_operator,
                )
            vin_reporter.set_interval(conf.get(config.INTERVAL))
            vin_reporters.append(vin_reporter)
            self._reporters[config.VIN] = vin_reporter
        return vin_reporters

    def _create_reporters(self, config):
        # hide super()'s CpuTempReporter
        reporters = []
        conf = config.get_cpu_temp_conf()

        tempReporter = CpuTempReporter(
            start=conf.get(config.START)
            )
        tempReporter.set_interval(conf.get(config.POLLING_INTERVAL))
        reporters.append(tempReporter)

        # DI reporter
        reporters += self._create_di_reporters(config)

        # VOUT
        conf = config.get_vout_conf()
        if conf is not None:
            default_val = self._default_do_val(conf, config)
            if default_val is not None:
                self._dio_mgr.setup_vout(default_val)

        # RS485 reporter
        reporters += self._create_rs485_reporters(config)
        # AIN reporter
        reporters += self._create_ain_reporters(config)
        # VIN reporter
        reporters += self._create_vin_reporters(config)
        return reporters

    def _create_sensing_manager(self, config):
        sensing_mgr = SensingMgr(self._dio_mgr)
        # DI edge
        for i in range(self._dio_mgr.DI_PORT_MIN, self._dio_mgr.DI_PORT_MAX+1):
            conf = config.get_di_conf(i)
            if conf is not None:
                if not conf.get(config.TYPE_IS_POLLING):
                    sensing_mgr.create_di_conf(
                        port=i,
                        interval=conf.get(config.INTERVAL),
                        edge_type=conf.get(config.EDGE_TYPE))

        port = self._dio_mgr.do_port(1)
        while port <= self._dio_mgr.do_port(2):
            conf = config.get_do_conf(port)
            if conf is not None:
                sensing_mgr.create_do_conf(
                    port=port,
                    port_state=self._dio_mgr.get_DO_status(port),
                    start=conf.get(config.START),
                    output_state=conf.get(config.OUTPUT_STATE),
                    output_time=conf.get(config.OUTPUT_TIME),
                    output_delay_time=conf.get(config.OUTPUT_DELAY_TIME)
                    )
            port += 1

        conf = config.get_vout_conf()
        if conf is not None:
            sensing_mgr.create_vout_conf(
                port_state=self._dio_mgr.get_VOUT_status(),
                start=conf.get(config.START),
                output_state=conf.get(config.OUTPUT_STATE),
                output_time=conf.get(config.OUTPUT_TIME),
                output_delay_time=conf.get(config.OUTPUT_DELAY_TIME)
                )

        self._sensing_mgr = sensing_mgr
        return sensing_mgr

#
# End of File
#
