import configparser
from functools import partial
from .log import Log
from enum import Enum, IntEnum
from dataclasses import dataclass

@dataclass
class CheckAttr:
    data_name: str
    required: bool
    data_is_value: bool

@dataclass
class CheckAttrList():
    data_name_list = []
    required_list = []
    data_is_value_list = []

    def clear(self):
        self.data_name_list.clear()
        self.required_list.clear()
        self.data_is_value_list.clear()

    def append(self, CheckAttr):
        self.data_name_list.append(CheckAttr.data_name)
        self.required_list.append(CheckAttr.required)
        self.data_is_value_list.append(CheckAttr.data_is_value)


class RET_CODE(Enum):
    OK = 1
    ERROR = 2
    DISABLE = 3


class A6E_DI_PORT_NUM(IntEnum):
    MIN = 1
    MAX = 34


class A6E_AIN_PORT_NUM(IntEnum):
    MIN = 1
    MAX = 16


class ModelConfigBase:
    # DEFAULT section
    CLOUD_CONFIG = "cloud_config"
    SEND_CLOUD = "send_cloud"
    CACHE = "cache"
    SEND_INTERVAL = "send_interval"
    DATA_SEND_ONESHOT = "data_send_oneshot"
    WAIT_CONTAINER_STOP = "wait_container_stop"

    # LOG section
    LOG = "LOG"
    FILE = "file"
    STREAM = "stream"

    # DI section
    DI = "DI"
    TYPE = "type"
    TYPE_IS_POLLING = "type_isPolling"
    INTERVAL = "interval"
    EDGE_TYPE = "edge_type"
    START = "start"

    # DO section
    DO1 = "DO1"
    DO2 = "DO2"
    OUTPUT_STATE = "output_state"
    OUTPUT_TIME = "output_time"
    OUTPUT_DELAY_TIME = "output_delay_time"

    # RS485 section
    RS485_DATA1 = "RS485_Data1"
    RS485_DATA2 = "RS485_Data2"
    RS485_DATA3 = "RS485_Data3"
    RS485_DATA4 = "RS485_Data4"
    METHOD = "method"
    BAUDRATE = "baudrate"
    DATA_SIZE = "data_size"
    PARITY = "parity"
    STOP = "stop"
    DEVICE_ID = "device_id"
    FUNC_CODE = "func_code"
    REGISTER_ADDR = "register_addr"
    REGISTER_COUNT = "register_count"
    ENDIAN = "endian"
    # ENDIAN_IS_BIG = "endian_is_big"
    # INTERVAL = "interval"
    DATA_OFFSET = "data_offset"
    DATA_MULTIPLY = "data_multiply"
    DATA_DIVIDER = "data_devider"

    # CPU_temp section
    CPU_TEMP = "CPU_temp"
    POLLING_INTERVAL = "polling_interval"

    MAX_INTERVAL = 3600

    # AIN section
    AIN = "AIN"

    # VIN section
    VIN = "VIN"

    # VOUT section
    VOUT = "VOUT"

    def __init__(self):
        self._configs = {}
        self._output_log_flag = True

    def init_output_log_flag(self):
        self._output_log_flag = True

    def enable_output_log_flag(self, enable=True):
        if enable:
            self._output_log_flag = True
        else:
            self._output_log_flag = False

    def load(self, config_file_path):
        log = Log.instance()
        config = configparser.ConfigParser()
        config.read(config_file_path)

        if not self.parse_default_config(config):
            log.error("config file not found [DEFAULT] section.")
            return False

        self.parse_log_config(config)
        try:
            self.parse_di_config(config)
        except Exception as e:
            log.error("[DI] section is invalid.")
            log.error(e.args)
        try:
            self.parse_do_config(config)
        except Exception as e:
            log.error("[DO] section is invalid.")
            log.error(e.args)
        try:
            self.parse_rs485_config(config)
        except Exception as e:
            log.error("[RS485] section is invalid.")
            log.error(e.args)
        self.parse_cpu_temp_config(config)
        try:
            self.parse_ain_config(config)
        except Exception as e:
            log.error("[AIN] section is invalid.")
            log.error(e.args)
        try:
            self.parse_vin_config(config)
        except Exception as e:
            log.error("[VIN] section is invalid.")
            log.error(e.args)
        try:
            self.parse_vout_config(config)
        except Exception as e:
            log.error("[VOUT] section is invalid.")
            log.error(e.args)
        return True

    def check_max_interval(self, val):
        log = Log.instance()
        try:
            if val == "":
                return 1

            if isinstance(val, str):
                val = int(val)
            if val > self.MAX_INTERVAL:
                log.info(
                    f"interval reduced from {val} to {self.MAX_INTERVAL}[s].")
                val = self.MAX_INTERVAL
            elif val < 1:
                val = 1
            return val
        except Exception as e:
            log.error("max interval check error.")
            log.error(e.args)
            return 1

    def parse_default_config(self, config):
        default_dict = {}
        section = "DEFAULT"
        log = Log.instance()
        try:
            cloud_config = config[section].getboolean(
                ModelConfigBase.CLOUD_CONFIG)
            default_dict[ModelConfigBase.CLOUD_CONFIG] = cloud_config

            send_cloud = config[section].getboolean(
                ModelConfigBase.SEND_CLOUD)
            default_dict[ModelConfigBase.SEND_CLOUD] = send_cloud

            cache = config[section].getboolean(
                ModelConfigBase.CACHE)
            default_dict[ModelConfigBase.CACHE] = cache

            send_interval = self.check_max_interval(
                config[section].getint(ModelConfigBase.SEND_INTERVAL, 10))
            if send_interval < 1:
                send_interval = 1
            elif send_interval > 10:
                send_interval = 10
            default_dict[ModelConfigBase.SEND_INTERVAL] = send_interval

            oneshot = config[section].getboolean(
                ModelConfigBase.DATA_SEND_ONESHOT, False)
            default_dict[ModelConfigBase.DATA_SEND_ONESHOT] = oneshot

            stop = config[section].getint(
                ModelConfigBase.WAIT_CONTAINER_STOP, 0)
            if stop < 0:
                stop = 0
            elif stop > 60:
                stop = 60
            default_dict[ModelConfigBase.WAIT_CONTAINER_STOP] = stop

            self._configs[section] = default_dict
            return True
        except Exception as e:
            log.error(
                "[DEFAULT] section need [cloud_config], [send_cloud], \
                [cache], [send_interval], [data_send_oneshot],\
                [wait_container_stop]")
            log.error(e.args)
        return False

    def parse_log_config(self, config):
        log_dict = {}
        section = ModelConfigBase.LOG
        file = config[section].getboolean(ModelConfigBase.FILE, True)
        log_dict[ModelConfigBase.FILE] = file
        stream = config[section].getboolean(ModelConfigBase.STREAM, False)
        log_dict[ModelConfigBase.STREAM] = stream
        self._configs[section] = log_dict

    # CPU temp
    def parse_cpu_temp_config(self, config):
        cpu_temp_dict = {}
        section = ModelConfigBase.CPU_TEMP
        cpu_temp_dict[ModelConfigBase.START] = False
        cpu_temp_dict[ModelConfigBase.TYPE] = 'none'
        cpu_temp_dict[ModelConfigBase.POLLING_INTERVAL] = 1
        param_valid = True

        disable = self.check_cpu_temp_param(
                cpu_temp_dict, config, section,
                ModelConfigBase.TYPE, True, False)

        if disable == None:
            self._configs[section] = cpu_temp_dict
            return
        elif not disable:
            param_valid = False

        if not self.check_cpu_temp_param(
                cpu_temp_dict, config, section,
                ModelConfigBase.POLLING_INTERVAL, True, True):
            param_valid = False

        if param_valid:
            cpu_temp_dict[ModelConfigBase.START] = True
        self._configs[section] = cpu_temp_dict

    def check_cpu_temp_param(self, cpu_temp_dict, config, section_name, data_name, required, data_is_value):
        log = Log.instance()
        valid = True

        try:
            if data_is_value:
                data = config[section_name][data_name]
            else:
                if data_name == ModelConfigBase.TYPE and \
                        not config.has_option(section_name, data_name):
                    data = 'polling'
                else:
                    data = config[section_name][data_name].lower()

            if not required and data == "" and data_is_value:
                return 0
            elif required and data == "":
                log.warning(f"{section_name} {data_name} parameter is empty.")
                return False

            if data_name == ModelConfigBase.TYPE:
                if data == 'none':
                    log.info(f"Getting {section_name} is disable.")
                    return None
                elif data != 'none' and data != 'polling':
                    valid = False
            elif data_name == ModelConfigBase.POLLING_INTERVAL:
                data = self.check_max_interval(data)

            if not valid:
                log.warning(f"{section_name} {data_name} parameter is invalid : {data}")
                return False
            else:
                cpu_temp_dict[data_name] = data
            return True
        except:
            log.warning(f"{section_name} {data_name} parameter is invalid : {data}")
            return False

    # DI
    def parse_di_config_dict(self, section, config):
        di_dict = {}
        type_isPolling = True

        di_dict[ModelConfigBase.START] = True

        if config[section][ModelConfigBase.TYPE] == "edge":
            type_isPolling = False
        elif config[section][ModelConfigBase.TYPE] == "" or \
                config[section][ModelConfigBase.TYPE] == "none":
            di_dict[ModelConfigBase.START] = False

        type = config[section].get(ModelConfigBase.TYPE, "polling")
        di_dict[ModelConfigBase.TYPE] = type
        di_dict[ModelConfigBase.TYPE_IS_POLLING] = type_isPolling
        interval = self.check_max_interval(
            config[section].get(ModelConfigBase.INTERVAL, 1))
        di_dict[ModelConfigBase.INTERVAL] = interval
        # option
        edge_type = config[section].get(ModelConfigBase.EDGE_TYPE, "")
        di_dict[ModelConfigBase.EDGE_TYPE] = edge_type
        return di_dict

    def parse_di_config(self, config):
        for i in range(A6E_DI_PORT_NUM.MIN, A6E_DI_PORT_NUM.MAX+1):
            section = ModelConfigBase.DI + str(i)
            if config.has_section(section):
                self._configs[section] = self.parse_di_config_dict(section, config)

    # DO
    def parse_do_config_dict(self, section, config):
        do_dict = self.create_default_do_conf()

        start = True

        if not self.check_do_param(
                do_dict, config, section,
                ModelConfigBase.OUTPUT_STATE, True, False):
            start = False

        if not self.check_do_param(
                do_dict, config, section,
                ModelConfigBase.OUTPUT_TIME, True, True):
            start = False

        if not self.check_do_param(
                do_dict, config, section,
                ModelConfigBase.OUTPUT_DELAY_TIME, True, True):
            start = False

        do_dict[ModelConfigBase.START] = start

        return do_dict

    def check_do_param(self, do_dict, config, section_name, data_name, required, data_is_value):
        log = Log.instance()
        valid = True
        start = True

        try:
            if data_is_value:
                data = config[section_name][data_name]
            else:
                data = config[section_name][data_name].lower()

            if not required and data == "" and data_is_value:
                return 0
            elif required and data == "":
                log.warning(f"{section_name} {data_name} parameter is empty.")
                return False

            if data_is_value and isinstance(data, str):
                if '.' in data:
                    data = float(data)
                else:
                    data = int(data)

            do_dict[data_name] = data

            if data_name == ModelConfigBase.OUTPUT_STATE:
                if not (data == "high" or data == "low"):
                    start = False

            if not valid:
                log.warning(f"{section_name} {data_name} parameter is invalid : {data}")
                return False
            elif not start:
                return False

            return True
        except:
            log.warning(f"{section_name} {data_name} parameter is invalid : {data}")
            return False

    def parse_do_config(self, config):
        section = ModelConfigBase.DO1
        if config.has_section(section):
            self._configs[section] = self.parse_do_config_dict(section, config)
        section = ModelConfigBase.DO2
        if config.has_section(section):
            self._configs[section] = self.parse_do_config_dict(section, config)

    # vout
    def parse_vout_config(self, config):
        section = ModelConfigBase.VOUT
        if config.has_section(section):
            self._configs[section] = self.parse_do_config_dict(section, config)

    # RS485
    def parse_rs485_config_dict(self, section, config):
        rs485_dict = self.create_default_rs485_conf()

        rs485CheckAttr = CheckAttrList()
        rs485CheckAttr.clear()
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.METHOD, required=True, data_is_value=False))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.BAUDRATE, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.PARITY, required=True, data_is_value=False))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.ENDIAN, required=True, data_is_value=False))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.DATA_SIZE, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.STOP, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.DEVICE_ID, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.FUNC_CODE, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.REGISTER_ADDR, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.REGISTER_COUNT, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.INTERVAL, required=True, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.DATA_OFFSET, required=False, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.DATA_MULTIPLY, required=False, data_is_value=True))
        rs485CheckAttr.append(CheckAttr(ModelConfigBase.DATA_DIVIDER, required=False, data_is_value=True))

        self.init_output_log_flag()
        results = (list(map(partial(self.check_param, rs485_dict, config, section),
                       rs485CheckAttr.data_name_list, rs485CheckAttr.required_list, rs485CheckAttr.data_is_value_list)))

        if not (RET_CODE.DISABLE in results or RET_CODE.ERROR in results):
            rs485_dict[ModelConfigBase.START] = True

        return rs485_dict

    # AIN
    def parse_ain_config_dict(self, section, config):
        ain_dict = {}
        ain_dict[ModelConfigBase.START] = True
        type = config[section].get(ModelConfigBase.TYPE, "voltage")
        ain_dict[ModelConfigBase.TYPE] = type
        interval = self.check_max_interval(
            config[section].get(ModelConfigBase.INTERVAL, 1))
        ain_dict[ModelConfigBase.INTERVAL] = interval
        return ain_dict

    def parse_ain_config(self, config):
        for i in range(A6E_AIN_PORT_NUM.MIN, A6E_AIN_PORT_NUM.MAX+1):
            section = ModelConfigBase.AIN + str(i)
            if config.has_section(section):
                self._configs[section] = self.parse_ain_config_dict(section, config)

    # VIN
    def parse_vin_config_dict(self, section, config):
        vin_dict = {}
        vin_dict[ModelConfigBase.START] = True
        interval = self.check_max_interval(
            config[section].get(ModelConfigBase.INTERVAL, 1))
        vin_dict[ModelConfigBase.INTERVAL] = interval
        return vin_dict

    def parse_vin_config(self, config):
        section = ModelConfigBase.VIN
        if config.has_section(section):
            self._configs[section] = self.parse_vin_config_dict(section, config)

    def check_data_type(self, config, section_name, data_name, required=True, data_is_value=True):
        log = Log.instance()

        try:
            if data_is_value:
                data = config[section_name][data_name]
            else:
                if section_name == ModelConfigBase.CPU_TEMP and \
                        data_name == ModelConfigBase.TYPE and \
                        not config.has_option(section_name, data_name):
                    data = 'polling'
                else:
                    data = config[section_name][data_name].lower()

            if not required and data == "" and data_is_value:
                data = 0
            elif required and data == "":
                if self._output_log_flag:
                    log.error(f"{section_name} {data_name} parameter is empty.")
                return None

            if data_is_value and isinstance(data, str):
                if '.' in data:
                    data = float(data)
                else:
                    data = int(data)
            return data
        except:
            log.error(f"{section_name} {data_name} data type is exception : {data}")
            return None

    def check_param(self, dict, config, section_name, data_name, required=True, data_is_value=True):
        log = Log.instance()
        valid = True

        try:
            data = self.check_data_type(
                config, section_name, data_name, required, data_is_value)

            if data is None:
                return RET_CODE.ERROR

            if data_name == ModelConfigBase.TYPE:
                if data == 'none':
                    log.info(f"Getting {section_name} is disable.")
                    self.enable_output_log_flag(False)
                    dict[data_name] = data
                    return RET_CODE.DISABLE
                elif section_name == ModelConfigBase.CPU_TEMP and \
                        data != 'none' and data != 'polling':
                    valid = False
            elif data_name == ModelConfigBase.EDGE_TYPE:
                if data == "" and dict[ModelConfigBase.TYPE] == 'edge':
                    valid = False
            elif data_name == ModelConfigBase.METHOD:
                if data == 'none':
                    log.info(f"Getting {section_name} is disable.")
                    dict[data_name] = data
                    self.enable_output_log_flag(False)
                    return RET_CODE.DISABLE
                elif data != 'rtu':
                    log.warning(f"{section_name} {data_name} parameter is not Modbus rtu.")
                    valid = False
            elif data_name == ModelConfigBase.PARITY:
                if not (data == 'none' or data == 'even' or data == 'odd'):
                    valid = False
            elif data_name == ModelConfigBase.ENDIAN:
                if not (data == 'big' or data == 'little'):
                    valid = False
            elif data_name == ModelConfigBase.DATA_SIZE:
                if not (data == 8 or data == 7):
                    valid = False
            elif data_name == ModelConfigBase.STOP:
                if not (data == 1 or data == 2):
                    valid = False
            elif data_name == ModelConfigBase.INTERVAL or\
                    data_name == ModelConfigBase.POLLING_INTERVAL:
                data = self.check_max_interval(data)

            if not valid:
                if self._output_log_flag:
                    log.error(f"{section_name} {data_name} parameter is invalid : {data}")
                    return RET_CODE.ERROR
            else:
                dict[data_name] = data
            return RET_CODE.OK
        except:
            log.error(f"{section_name} {data_name} parameter is exception : {data}")
            return RET_CODE.ERROR

    def parse_rs485_config(self, config):
        section = ModelConfigBase.RS485_DATA1
        if config.has_section(section):
            self._configs[section] = self.parse_rs485_config_dict(
                section, config)
        section = ModelConfigBase.RS485_DATA2
        if config.has_section(section):
            self._configs[section] = self.parse_rs485_config_dict(
                section, config)
        section = ModelConfigBase.RS485_DATA3
        if config.has_section(section):
            self._configs[section] = self.parse_rs485_config_dict(
                section, config)
        section = ModelConfigBase.RS485_DATA4
        if config.has_section(section):
            self._configs[section] = self.parse_rs485_config_dict(
                section, config)

    # global
    def get_default_conf(self):
        return self._configs["DEFAULT"]

    def get_cpu_temp_conf(self):
        return self._configs[ModelConfigBase.CPU_TEMP]

    def get_log_conf(self):
        return self._configs[ModelConfigBase.LOG]

    # DI
    def get_di_conf(self, port):
        log = Log.instance()
        port_name = ""
        if port < A6E_DI_PORT_NUM.MIN or port > A6E_DI_PORT_NUM.MAX:
            log.error(f'out of range, port number:{port}')
            return None

        port_name = ModelConfigBase.DI + str(port)
        if port_name in self._configs.keys():
            return self._configs[port_name]
        elif port == 1 or port == 2:
            conf = self.create_default_di_conf()
            self._configs[port_name] = conf
            return conf
        else:
            return None

    def create_default_di_conf(self):
        di_dict = {}
        di_dict[ModelConfigBase.START] = False
        di_dict[ModelConfigBase.TYPE] = "polling"
        di_dict[ModelConfigBase.TYPE_IS_POLLING] = True
        di_dict[ModelConfigBase.INTERVAL] = 1
        di_dict[ModelConfigBase.EDGE_TYPE] = ""
        return di_dict

    # DO
    def get_do_conf(self, port):
        port_name = ""
        if port == 1:
            port_name = ModelConfigBase.DO1
        elif port == 2:
            port_name = ModelConfigBase.DO2

        if port_name in self._configs.keys():
            return self._configs[port_name]
        else:
            conf = self.create_default_do_conf()
            self._configs[port_name] = conf
            return conf

    def create_default_do_conf(self):
        do_dict = {}
        do_dict[ModelConfigBase.START] = False
        do_dict[ModelConfigBase.OUTPUT_STATE] = ""
        do_dict[ModelConfigBase.OUTPUT_TIME] = 0
        do_dict[ModelConfigBase.OUTPUT_DELAY_TIME] = 0
        return do_dict

    def get_vout_conf(self):
        if "VOUT" in self._configs.keys():
            return self._configs["VOUT"]
        else:
            return None

    def get_rs485_conf(self, data_num):
        data_name = ""
        if data_num == 1:
            data_name = ModelConfigBase.RS485_DATA1
        elif data_num == 2:
            data_name = ModelConfigBase.RS485_DATA2
        elif data_num == 3:
            data_name = ModelConfigBase.RS485_DATA3
        elif data_num == 4:
            data_name = ModelConfigBase.RS485_DATA4

        if data_name in self._configs.keys():
            return self._configs[data_name]
        else:
            conf = self.create_default_rs485_conf()
            self._configs[data_name] = conf
            return conf

    def create_default_rs485_conf(self):
        rs485_dict = {}
        rs485_dict[ModelConfigBase.START] = False
        rs485_dict[ModelConfigBase.METHOD] = "rtu"
        rs485_dict[ModelConfigBase.BAUDRATE] = 0
        rs485_dict[ModelConfigBase.PARITY] = "none"
        rs485_dict[ModelConfigBase.STOP] = 1
        rs485_dict[ModelConfigBase.DATA_SIZE] = 8
        rs485_dict[ModelConfigBase.DEVICE_ID] = 0
        rs485_dict[ModelConfigBase.FUNC_CODE] = 0
        rs485_dict[ModelConfigBase.REGISTER_ADDR] = 0
        rs485_dict[ModelConfigBase.REGISTER_COUNT] = 1
        rs485_dict[ModelConfigBase.ENDIAN] = "big"
        rs485_dict[ModelConfigBase.INTERVAL] = 1
        # option
        rs485_dict[ModelConfigBase.DATA_OFFSET] = 0
        rs485_dict[ModelConfigBase.DATA_MULTIPLY] = 0
        rs485_dict[ModelConfigBase.DATA_DIVIDER] = 0
        return rs485_dict

    # AIN
    def get_ain_conf(self, port):
        log = Log.instance()
        port_name = ""
        if port < A6E_AIN_PORT_NUM.MIN or port > A6E_AIN_PORT_NUM.MAX:
            log.error(f'out of range, port number:{port}')
            return None

        port_name = ModelConfigBase.AIN + str(port)
        if port_name in self._configs.keys():
            return self._configs[port_name]
        else:
            conf = self.create_default_ain_conf()
            self._configs[port_name] = conf
            return conf

    def create_default_ain_conf(self):
        ain_dict = {}
        ain_dict[ModelConfigBase.START] = False
        ain_dict[ModelConfigBase.INTERVAL] = 1
        return ain_dict

    def send_interval(self):
        conf = self.get_default_conf()
        return conf.get(ModelConfigBase.SEND_INTERVAL)

    # VIN
    def get_vin_conf(self):
        port_name = ModelConfigBase.VIN
        if port_name in self._configs.keys():
            return self._configs[port_name]
        else:
            conf = self.create_default_vin_conf()
            self._configs[port_name] = conf
            return conf

    def create_default_vin_conf(self):
        vin_dict = {}
        vin_dict[ModelConfigBase.START] = False
        vin_dict[ModelConfigBase.INTERVAL] = 1
        return vin_dict

    # update
    def update_di_conf(self, key, dict):
        log = Log.instance()
        old_conf = None
        section = ""
        for i in range(A6E_DI_PORT_NUM.MIN, A6E_DI_PORT_NUM.MAX+1):
            temp_key = ModelConfigBase.DI + str(i)
            if key == temp_key:
                old_conf = self.get_di_conf(i)
                section = temp_key
                break

        if old_conf is None:
            log.error("DI config key is invalid.")
            return None

        for key, value in dict.items():
            if key == ModelConfigBase.TYPE:
                old_conf[ModelConfigBase.START] = True
                type_isPolling = True
                if value == "edge":
                    type_isPolling = False
                elif value == "none" or value == "":
                    old_conf[ModelConfigBase.START] = False
                old_conf[ModelConfigBase.TYPE_IS_POLLING] = type_isPolling
            old_conf[key] = value

        self._configs[section] = old_conf
        return section

    def update_do_conf(self, key, dict):
        log = Log.instance()
        old_conf = None
        section = ""
        if key == ModelConfigBase.DO1:
            old_conf = self.get_do_conf(1)
            section = ModelConfigBase.DO1
        elif key == ModelConfigBase.DO2:
            old_conf = self.get_do_conf(2)
            section = ModelConfigBase.DO2
        else:
            log.error("DO config key is invalid.")
            return None

        for key, value in dict.items():
            old_conf[key] = value

        self._configs[section] = old_conf
        return section

    def update_rs485_conf(self, key, dict):
        log = Log.instance()
        old_conf = None

        section = ""
        if key == ModelConfigBase.RS485_DATA1:
            old_conf = self.get_rs485_conf(1)
            section = ModelConfigBase.RS485_DATA1
        elif key == ModelConfigBase.RS485_DATA2:
            old_conf = self.get_rs485_conf(2)
            section = ModelConfigBase.RS485_DATA2
        elif key == ModelConfigBase.RS485_DATA3:
            old_conf = self.get_rs485_conf(3)
            section = ModelConfigBase.RS485_DATA3
        elif key == ModelConfigBase.RS485_DATA4:
            old_conf = self.get_rs485_conf(4)
            section = ModelConfigBase.RS485_DATA4
        else:
            log.error("RS485 config key is invalid.")
            return None

        for key, value in dict.items():
            old_conf[key] = value

        self._configs[section] = old_conf
        return section

    def update_ain_conf(self, key, dict):
        log = Log.instance()
        old_conf = None
        section = ""
        for i in range(A6E_AIN_PORT_NUM.MIN, A6E_AIN_PORT_NUM.MAX+1):
            temp_key = ModelConfigBase.AIN + str(i)
            if key == temp_key:
                old_conf = self.get_ain_conf(i)
                section = temp_key
                break

        if old_conf is None:
            log.error("AIN config key is invalid.")
            return None

        for key, value in dict.items():
            old_conf[key] = value

        self._configs[section] = old_conf
        return section

    def update_vin_conf(self, key, dict):
        log = Log.instance()
        old_conf = None
        section = ""
        temp_key = ModelConfigBase.VIN
        if key == temp_key:
            old_conf = self.get_vin_conf()
            section = temp_key

        if old_conf is None:
            log.error("AIN config key is invalid.")
            return None

        for key, value in dict.items():
            old_conf[key] = value

        self._configs[section] = old_conf
        return section

    def update_vout_conf(self, key, dict):
        log = Log.instance()
        old_conf = None
        section = ""
        temp_key = ModelConfigBase.VOUT
        if key == temp_key:
            old_conf = self.get_vout_conf()
            section = temp_key

        if old_conf is None:
            log.error("VOUT config key is invalid.")
            return None

        for key, value in dict.items():
            old_conf[key] = value

        self._configs[section] = old_conf
        return section

    def update_config(self, data):
        update_list = {}

        for key, value in data.items():
            section = None
            if "_config" not in key:
                continue

            index = key.find('_config')
            key = key[:index]

            if "DI" in key:
                section = self.update_di_conf(key, value)
            elif "DO" in key:
                section = self.update_do_conf(key, value)
            elif "RS485" in key:
                section = self.update_rs485_conf(key, value)
            elif "AIN" in key:
                section = self.update_ain_conf(key, value)
            elif "VIN" in key:
                section = self.update_vin_conf(key, value)
            elif "VOUT" in key:
                section = self.update_vout_conf(key, value)

            if section is not None:
                update_list[section] = self._configs[section]

        return update_list

#
# End of File
#
