import os.path
from dataclasses import dataclass
from pathlib import Path

from ..log import Log


@dataclass
class VinIIO:
    dev_id_name: str
    dev_id_value: str
    vin_raw_name: str
    vin_scale_name: str
    r1: int
    r2: int
    description: str
    dev_dir: Path | None = None

class VinOperator():
    @staticmethod
    def find_valid_vin_iio(vin_iios: list[VinIIO], base_path=Path('/sys/bus/iio/devices/')):
        valid_vin_iio: list[VinIIO] = []
        for vin_iio in vin_iios:
            pattern = os.path.join('*', vin_iio.dev_id_name)
            paths_match_id = [f for f in base_path.glob(pattern)
                              if f.is_file() and
                              f.read_text().strip() == vin_iio.dev_id_value]
            if paths_match_id == []:
                continue
            vin_iio.dev_dir = paths_match_id[0].parent
            valid_vin_iio.append(vin_iio)

        return valid_vin_iio

    def __init__(self, vin_iio: VinIIO):
        self.vin_iio = vin_iio
        self.path_raw = self.vin_iio.dev_dir.joinpath(self.vin_iio.vin_raw_name)
        self.path_scale = self.vin_iio.dev_dir.joinpath(self.vin_iio.vin_scale_name)
        self.r1 = self.vin_iio.r1
        self.r2 = self.vin_iio.r2

    def value(self):
        log = Log.instance()
        raw = 0
        scale = 1.0

        try:
            raw = int(self.path_raw.read_text())
        except FileNotFoundError:
            log.error('VinOperator::raw: file not found')
            return None

        try:
            scale = float(self.path_scale.read_text())
        except FileNotFoundError:
            log.error('VinOperator::scale: file not found')
            return None

        return int(raw * scale * (self.r1 + self.r2) / self.r2)

#
# End of File
#
