from abc import ABC
import gpiod
from dataclasses import dataclass
from ..log import Log
from enum import Enum


class STATE(Enum):
    NONE = 0
    ENABLE = 1
    DISABLE = 2

@dataclass
class DInAttr:
    chip: int
    line: int
    gpiod_line: type(gpiod.libgpiodcxx.line)
    edge: bool


@dataclass
class DOutAttr:
    chip: int
    line: int
    gpiod_line: type(gpiod.libgpiodcxx.line)
    state: Enum
    val: int


class DIO_Mgr(ABC):
    def __init__(self):
        self._dins = dict()
        self._douts = dict()
        self._vout = dict()

    @staticmethod
    def _create_do_attr(chip, line, desire=-1):
        return DOutAttr(chip, line, None, STATE.ENABLE if desire >= 0 else STATE.NONE, desire)

    def _di_port_to_gpio(self, port):
        chip = -1
        line = -1
        if port == 1 or port == 2:
            chip = 5
            line = port - 1
        elif port > 2 and port <= self.DI_PORT_MAX:
            chip = int((port - 3) / 8 + 6)
            line = (port - 3) % 8
        return chip, line

    def _do_port_to_gpio(self, port):
        chip = -1
        line = -1
        if port == 1 or port == 2:
            chip = 5
            line = port + 1

        return chip, line

    def setup_di(self, port):
        log = Log.instance()
        try:
            (chip, line) = self._di_port_to_gpio(port)
            self._dins[str(port)] = DInAttr(chip, line, None, False)
            config = gpiod.line_request()
            config.consumer = "DIO_Mgr"
            config.request_type = gpiod.line_request.DIRECTION_INPUT
            self._setup_port(self._dins, config, str(port), False)
        except Exception as e:
            Log.error(log, "DIO_Mgr: setup_di({port}) is Error!")
            Log.error(log, e.args)

    def setup_do(self, port, desire):
        log = Log.instance()
        try:
            (chip, line) = self._do_port_to_gpio(port)
            self._douts[str(port)] = self._create_do_attr(chip, line, desire)
            config = gpiod.line_request()
            config.consumer = "DIO_Mgr"
            config.request_type = gpiod.line_request.DIRECTION_OUTPUT
            if self._douts[str(port)].state == STATE.ENABLE:
                self._douts[str(port)].state = self._setup_port(self._douts, config, str(port), True)
        except Exception as e:
            Log.error(log, "DIO_Mgr: setup_do() is Error!")
            Log.error(log, e.args)

    def get_DI(self, port):
        log = Log.instance()
        try:
            return self._dins[str(port)].gpiod_line.get_value()
        except Exception:
            log.warning('get_DI: has not DI{0} informations.'.format(port))
            return None

    def set_DO1(self, isHigh):
        log = Log.instance()
        try:
            self._douts['1'].gpiod_line.set_value(1 if isHigh else 0)
        except Exception:
            log.warning('set_DO1: has not DO1 informations.')

    def set_DO2(self, isHigh):
        log = Log.instance()
        try:
            self._douts['2'].gpiod_line.set_value(1 if isHigh else 0)
        except Exception:
            log.warning('set_DO2: has not DO2 informations.')

    def set_DO_Output(self, port, isHigh):
        log = Log.instance()
        try:
            self._douts[str(port)].gpiod_line.set_value(1 if isHigh else 0)
        except Exception:
            log.warning('set_DO_Output: has not DO{0} informations.'.format(port))

    def get_DI_edge(self, port):
        log = Log.instance()
        try:
            return self._dins[str(port)].edge
        except Exception:
            log.warning('get_DI_edge: has not DI{0} informations.'.format(port))
            return None

    def set_DI_edge(self, port, val):
        log = Log.instance()
        try:
            self._dins[str(port)].edge = val
        except Exception:
            log.warning('set_DI_edge: has not DI{0} informations.'.format(port))

    def get_DO_status(self, port):
        log = Log.instance()
        try:
            return self._douts[str(port)].state
        except Exception:
            log.warning('get_DO_status: has not DO{0} informations.'.format(port))
            return STATE.DISABLE

    def reset_DO(self, port, val):
        log = Log.instance()
        config = gpiod.line_request()
        config.consumer = "DIO_Mgr"
        config.request_type = gpiod.line_request.DIRECTION_OUTPUT

        try:
            if port == self.DO1_PORT and self._douts['1'].state == STATE.NONE:
                self._douts['1'].val = val
                self._douts['1'].state = self._setup_port(self._douts, config, '1', True)

            if port == self.DO2_PORT and self._douts['2'].state == STATE.NONE:
                self._douts['2'].val = val
                self._douts['2'].state = self._setup_port(self._douts, config, '2', True)
        except Exception:
            log.warning('reset_DO: has not DO{0} informations.'.format(port))

    @staticmethod
    def _setup_port(port_dic, config, port, isOutput):
        log = Log.instance()
        ret = STATE.ENABLE
        try:
            dio_attrs = port_dic[port]
            dio_attrs.gpiod_line = gpiod.chip(
                dio_attrs.chip).get_line(dio_attrs.line)
            if isOutput:
                dio_attrs.gpiod_line.request(config, dio_attrs.val)
            else:
                dio_attrs.gpiod_line.request(config)
        except Exception as e:
            Log.error(log, "DIO_Mgr: _setup_port() is Error!")
            Log.error(log, f"chip:{dio_attrs.chip}, line:{dio_attrs.line} - {e}")
            ret = STATE.NONE
        return ret

#
# EOF
#
