import os
#import os.path
#import os.readlink
from dataclasses import dataclass
from ..log import Log
import subprocess


@dataclass
class AinMap:
    path_raw: str
    path_scale: str
    register: int


class AinOperator():
    AIN_PORT_MIN = 1
    AIN_PORT_UNIT = 4
    AIN_PORT_MAX = 16
    AIN_BOARD_MIN = 1
    AIN_BOARD_MAX = 4
    DEV_PATH_REF = '/sys/bus/iio/devices/iio:device'
    DEV_NAME = 'ads1015'

    I2C_CHIP = 3
    I2C_ADDRESS = ['0x57', '0x56', '0x55', '0x53']
    # A6E add-on module information
    # [ Vendor_id, Product_id ]
    BOARD_DATA = [
        [0x0001, 0x000e]
    ]
    REVISON_MIN = 0x0001
    REVISON_MAX = 0xFFFF

    def __init__(self):
        self._map = {}
        self._create_map()

    def _create_map(self):
        for i in range(0, 64):  # 64 is somehow
            board = -1
            if os.path.isdir(self.DEV_PATH_REF+str(i)):
                try:
                    with open(self.DEV_PATH_REF+str(i)+'/name') as f:
                        name = f.read()
                        if self.DEV_NAME in name:
                            reg = os.readlink(self.DEV_PATH_REF+str(i)+'/of_node')
                            if reg.endswith('49'):
                                board = 1
                            elif reg.endswith('48'):
                                board = 2
                            elif reg.endswith('4B') or reg.endswith('4b'):
                                board = 3
                            elif reg.endswith('4A') or reg.endswith('4a'):
                                board = 4
                except FileNotFoundError:
                    break
            else:
                break

            if board != -1:
                register = self._get_register(board-1)
                self._map_port(board, i, register)

    def _map_port(self, board, found, register):
        for j in range(self.AIN_PORT_MIN, self.AIN_PORT_UNIT+1):
            map = AinMap(path_raw=self.DEV_PATH_REF+str(found)+'/in_voltage'+str(j-1)+'_raw',
                         path_scale=self.DEV_PATH_REF+str(found)+'/in_voltage'+str(j-1)+'_scale',
                         register=register)
            self._map[(board-1)*self.AIN_PORT_UNIT+j-1] = map

    def _port_to_map(self, port):
        return self._map[port-1]

    def _get_register(self, board):
        register = 1
        i2c_address = self.I2C_ADDRESS[board]
        vendorid = self._get_board_data(i2c_address, 0, 1)
        productid = self._get_board_data(i2c_address, 2, 3)
        revision = self._get_board_data(i2c_address, 4, 5)

        if revision < self.REVISON_MIN or revision > self.REVISON_MAX:
            return register

        i = 0
        for data in self.BOARD_DATA:
            if vendorid == data[0] and productid == data[1]:
                # AT4414A
                if i == 0:
                    register = 249
                    break
            i += 1

        return register

    def _get_board_data(self, i2c_address, read_addr_top, read_addr_bottom):
        return (self._exec_i2cget(i2c_address, read_addr_top) << 2) + \
            self._exec_i2cget(i2c_address, read_addr_bottom)

    def _exec_i2cget(self, i2c_address , read_addr):
        data = '0x00'
        status, value = subprocess.getstatusoutput(
                f'i2cget -y {self.I2C_CHIP} {i2c_address} {read_addr}')

        if status == 0:
            data = value

        return int(data, 16)

    def calc_value(self, port):
        log = Log.instance()
        raw = 0.0
        scale = 1.0

        if (port - 1) not in self._map.keys():
            log.warning('AinOperator::value: port:{0} is not found'.format(port))
            return None

        try:
            with open(self._port_to_map(port).path_raw, "r") as f:
                raw = int(f.read())
        except FileNotFoundError:
            log.error('AinOperator::value: file not found port:{0}'.format(port))
            return None

        try:
            with open(self._port_to_map(port).path_scale, "r") as f:
                scale = float(f.read())
        except FileNotFoundError:
            log.error('AinOperator::value: file not found port:{0}'.format(port))
            return None

        if scale == 0.0:
            return 0

        return raw * scale

    def has_map(self):
        return not (not self._map)

    def voltage(self, port):
        value = self.calc_value(port)
        if value is not None:
            voltage_value = int(value)
        return voltage_value

    def current(self, port):
        value = self.calc_value(port)
        if value is not None:
            current_value = int(value / self._port_to_map(port).register)
        return current_value

    def set_scale(self, port, scale):
        log = Log.instance()
        if scale not in (0.1875, 0.125, 0.0625, 0.03125, 0.015625, 0.007813):
            return
        try:
            with open(self._port_to_map(port).path_scale, "w") as f:
                 f.write(float(scale))
        except FileNotFoundError:
            log.error('AinOperator::set_scale: file not found port:{port}')
            return None

#
# End of File
#
